/*
 * Copyright (C) 2002, John Leuner.
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Code for implementing static initializers as per 2.17.5 of the VM spec.
 */

#include "config.h"


#include "vm/classfil.h"
#include "vm/loading_errors.h"
#include "vm/cptuplelist.h"
#include "vm/interp.h"

#include "vm/jni.h"
#include "lib/indigenous/java.lang/Class.h"
#include "lib/indigenous/java.lang/VMClass.h"
#include "initialize.h"
#include "vm/classfile_methods.h"
#include "vm/interp_methods.h"

extern tClassLoaderTuple* pstVMClassType;


jobject RunInitialize(JNIEnv* env, tClassLoaderTuple* pstClass)
{
  tMethod* pstMethod;
  tOBREF pstExOb;
  int32 args[2];

  assert(pstVMClassType);
  if (pstVMClassType == pstClass) {
    traceInit("RunInitialize: skipping VMClass.initialize() for class %s", 
	      pstClass->uidName);
    pstClass->pstClass->bInitialised = 1;
    return NULL;
  }
   
  // Initialise the superclass by calling VMClass.initialize()
  pstMethod = CLASSFILE_FindMethod(env, pstVMClassType, "initialize", "(I)V");
  assert(pstMethod);

  traceInit("RunInitialize: calling VMClass.initialize() for class %s", 
	    pstClass->uidName);

  args[0] = (int32) (pstClass->vmClassObject);
  args[1] = sys_current_thread();

  pstExOb = INTERP_RunVirtualMethodFromPtr(env, pstMethod, args);
  if (pstExOb != NULL) {
    traceInit("VMClass.initialize() for class %s returned exception %s", 
	      pstClass->uidName, (DEREF(pstExOb))->pstType->uidName);
  }
  else {
    traceInit("VMClass.initialize() for class %s returned normally", 
	      pstClass->uidName);
  }
  return pstExOb;
}


jobject INITIALIZE_InitializeClass(JNIEnv* env, jobject classObject)
{
  tClassLoaderTuple* pstClass = CLASS_GetClassStruct(env, classObject);
    
  if (pstClass->pstClass->bInitialised) {
    return;
  }
  return RunInitialize(env, pstClass);
}


void Java_java_lang_VMClass_step7(JNIEnv* env, jobject vmClass)
{
  tClassLoaderTuple* pstClass;
  tClassLoaderTuple* pstSuperClass;
  jobject pstExOb;
  tMethod* pstMethod;
  
  pstClass = CLASS_GetVMClassStruct(env, vmClass);

  if ((pstClass->pstClass->u16AccessFlags & ACC_INTERFACE) != 0) {
    // Don't recursively initialise an interface's supers
    return;
  }
  pstSuperClass = pstClass->pstClass->pstSuperClass;
  if (pstSuperClass == NULL) {
    // XXX ... I think this means that the superclass is java.lang.Object
    return;
  }
  if (pstSuperClass->pstClass->bInitialised) {
    return;
  }

  traceInit("Recursing for superclass of class %s", pstClass->uidName);
  pstExOb = RunInitialize(env, pstSuperClass);

  if (pstExOb) {
    (*env)->Throw(env, pstExOb);
  }
}


jboolean Java_java_lang_VMClass_isInitialized(JNIEnv* env, jobject vmClass)
{
  if (CLASS_GetVMClassStruct(env, vmClass)->pstClass->bInitialised) {
    return JNI_TRUE;
  }
  else {
    return JNI_FALSE;
  }
}


void Java_java_lang_VMClass_setInitialized(JNIEnv* env, jobject vmClass)
{
  CLASS_GetVMClassStruct(env, vmClass)->pstClass->bInitialised = 1;
}


void Java_java_lang_VMClass_step8(JNIEnv* env, jobject vmClass)
{
  tClassLoaderTuple* pstClass;
  jobject pstExOb;
  tMethod* pstMethod;

  pstClass = CLASS_GetVMClassStruct(env, vmClass);
  /* initialise natives for the class before calling <clinit> because
     the latter might use a native method */
  NATIVES_InitNatives(pstClass);

  /* call the class initialising method */
  pstMethod = CLASSFILE_FindMethod(env, pstClass, "<clinit>", "()V");
  if (pstMethod) {
    traceInit("Running static initialization for class %s", 
	      pstClass -> uidName);
    pstExOb = INTERP_RunStaticMethodFromPtr(env, pstMethod, NULL);
    if (pstExOb) {
      traceInit("Initialization of class %s threw a %s exception", 
		pstClass -> uidName, DEREF(pstExOb)->pstType->uidName);
      (*env)->Throw(env, pstExOb);
    }
    else {
      traceInit("Initialization of class %s succeeded", pstClass -> uidName);
    }
  }
  else {
    traceInit("Class %s has no static initialization", pstClass -> uidName);
  }
  return;
}
