#include "config.h"

#ifdef ALLOCATOR

#include "vm/garbage/public_methods.h"

#include "vm/thread.h"
#include "vm/garbage_defs.h"
#include "vm/garbage/bitfield.h"
#include "vm/garbage/mark/mark.h"

#include "vm/classfile_methods.h"
#ifndef TEASEME
#include <stdio.h>
#endif

#define HEAPBOUND(x) ((((int32*) x) >= heap->pi32YoungHeap) && (((int32*) x) < (heap->pi32OldHeap + heap->i32OldHeapSize)))


int ALLOCATOR_MARK_MarkThreadStacks(tAllocatorHeap* heap)
{
  // First we iterate over each thread
  tThreadNode* iter = THREAD_lockAndGetThreadList();
  
  while (iter != NULL) {
    // For each thread we iterate through all the initial stack frames
    int numFrames = iter->pstInitialFrames->count;
    int i;
    
    if (iter->state != THREAD_STATE_TERMINATED) {
      assert(iter->stackBottom >= iter->stackTop);
      for (i = 0; i < numFrames; i++) {
	tStackFrame* stackFrame = 
	  LLIST_ElementAt(iter->pstInitialFrames, i);
	
	// And for that stack frame we mark it and its children
	while (stackFrame != NULL) {
	  // And for each frame we must examine the current object
	  if (stackFrame->pstCurrObject) {
	    ALLOCATOR_MARK_RecursivelyMark(heap, 
					   stackFrame->pstCurrObject, 
					   0, GARBAGE_REACHABLE, 
					   &(stackFrame->pstCurrObject));
	  }
	  
	  // And the java operand stack
	  if (stackFrame->bIsNativeFrame == 0 && stackFrame->pstCurrMethod) {
	    int j;
	    for (j = 0; 
		 j < stackFrame->pstCurrMethod->pstCode->u16MaxStack + 1;
		 j++) {
	      if (ALLOCATOR_ISHANDLE(heap, 
				     (tOBREF) stackFrame->pi32OpBot[j])) {
		ALLOCATOR_MARK_RecursivelyMark(heap, 
					       (tOBREF) stackFrame->pi32OpBot[j], 
					       1, GARBAGE_REACHABLE, 
					       (tOBREF *) &(stackFrame->pi32OpBot[j]));
	      }
	    }
	  }
	  
	  // And the local variables
	  if (stackFrame->pstCurrMethod && stackFrame->pi32Vars) {
	    int k;
	    int i32NumLocals =
	      (stackFrame->pstCurrMethod->u16ArgSize + 
	       stackFrame->pstCurrMethod->pstCode->u16MaxLocals);
	    for (k = 0; k < i32NumLocals; k++) {
	      if (ALLOCATOR_ISHANDLE(heap, (tOBREF) stackFrame->pi32Vars[k])) {
		ALLOCATOR_MARK_RecursivelyMark(heap, 
					       (tOBREF) stackFrame->pi32Vars[k], 
					       0, GARBAGE_REACHABLE, 
					       (tOBREF *) &(stackFrame->pi32Vars[k]));
	      }
	    }
	    
	  }
	  
	  stackFrame = stackFrame->pstChildFrame;
	}    
      }
    }
    iter = iter->next;
  }
  THREAD_unlockThreadList();
  return 0;
}


int ALLOCATOR_MARK_MarkCStacks(tAllocatorHeap* heap)
{
  // First we iterate over each thread
  tThreadNode* iter = THREAD_lockAndGetThreadList();
  // tThreadNode* iter = pstThreadList;
  while (iter != NULL) {
    int32* pi32TempPtr;
    
#ifdef GARBAGE_VERBOSE
    traceGC("Stack bottom is %x\n", (int) iter->stackBottom);
#endif
    
    for (pi32TempPtr = iter->stackBottom; 
	 pi32TempPtr > (int32*) (iter->stackTop); 
	 pi32TempPtr--) {
      if (ALLOCATOR_ISHANDLE(heap, (tOBREF) *pi32TempPtr)) {
	ALLOCATOR_MARK_RecursivelyMark(heap, (tOBREF) *pi32TempPtr, 
				       1, GARBAGE_REACHABLE, 
				       (tOBREF *) pi32TempPtr);
      }
      else {
#ifdef GARBAGE_VERBOSE
	if (HEAPBOUND(*pi32TempPtr)) {
	  /* if the field points into the heap but its contents don't
	     then it might be a direct pointer into the heap */
	  traceGC("Found direct pointer on C stack (thread %i) %p %p (%s)\n", 
		  (int) iter->pthread, pi32TempPtr, (void*) *pi32TempPtr, 
		  "none"); 
	}
#endif							
      }
    }
    
    iter = iter->next;
  }  
  THREAD_unlockThreadList();
  return 0;
}


/*
 * @doc FUNC
 * @func
 * This function is used to mark items that are reachable. If the item that
 * is specified is an object then this function calls itself on its fields if
 * they are objects or arrays. If the specified item is an array of arrays or
 * an array of objects then it calls itself on each of the arrays or objects
 * in that array. If iBFCheck is 1 then a bitfield check is performed to see
 * if the item points to the beginning of an object if it points into the
 * heap (used when checking the C stack and Java stack).  oSlot usually
 * points to the location from which the GC got the object reference. 
 *
 * Returns 0 if no errors occurred
 */
int ALLOCATOR_MARK_RecursivelyMark(tAllocatorHeap* heap, tOBREF o, 
				   int iBFCheck, int32 i32Colour, 
				   tOBREF* oSlot)
{
  tClassLoaderTuple* pstTempClass;
  int32 i;
  int32 i32StartOffset;

  /* check if current value is a handle in the heap handle block or
     the RPOT */
  if (ALLOCATOR_ISHANDLE(heap, o)) {
    i32StartOffset = ((int32*) *o) - heap->pi32OldHeap;
    
    /* check the bitfield to see if it's really the start of an object */
    if (!(IS_OLD_BIT_SET(i32StartOffset))) {
      /* return if there is no bit set in the bitfield for this address */
      /* XXX - should we continue?  Isn't this a bad object? */
      eprintf("No bit set for ref %p->%p\n", o, *o);
      return 0;
    }
#ifdef DEBUG_OBJECT_FOR_GC
    INTERP_CheckObject(PDEREF(o));
#endif

#ifdef PERSIST
    /* If persistence is enabled, the mark phase is responsible for
       updating references for newly promoted persistent objects so
       that they use the new handle in the RPOT, not the old handle
       that the object had before promotion.  If we are marking the
       RPOT, oSlot will be NULL.  NOTE: we do this BEFORE we check if 
       the object has been marked!
    */
    if (oSlot) {
      *oSlot = PDEREF(o)->hHandle;
    }
#endif

    /* check to see if item has already been marked... return if so */
    if (PDEREF(o)->i32Flags & GARBAGE_REACHABILITY_MASK) {
      return 0;
    }
    else {
      /* mark it with the current colour */
      PDEREF(o)->i32Flags |= i32Colour;
    }
    
    /* if it's an object then mark all its members if appropriate */
    if ((PDEREF(o)->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_OBJECT) {
      /* don't do static fields here - they get done separately */
      
      /* go through the instance fields for the class and superclasses */
      pstTempClass = PDEREF(o)->pstType;
      
      while (pstTempClass) {
	for (i = 0; i < pstTempClass->pstClass->u16InstCount; i++) {
	  /* if signature says this is an object or array, mark it */
	  if ((pstTempClass->pstClass->
	       pstInstOffsets[i].uidFieldSig[0] == '[') ||
	      (pstTempClass->pstClass->
	       pstInstOffsets[i].uidFieldSig[0] == 'L')) {
	    tOBREF* childSlot = 
	      (tOBREF *) &(PDEREF(o)->pi32Vars[pstTempClass->pstClass->
					      pstInstOffsets[i].u16Offset]);
	    tOBREF childRef = *childSlot;
	    if (childRef 
#ifdef PERSIST
		&& !ISPID(childRef)
#endif
		) /* if */ {
	      if (ALLOCATOR_ISHANDLE(heap, childRef)) {
#ifdef DEBUG_OBJECT_FOR_GC
		INTERP_CheckObject(DEREF(childRef));
#endif
		ALLOCATOR_MARK_RecursivelyMark(heap, childRef, 0,
					       i32Colour, childSlot);
	      }
	      else {
		// This handle isn't in the heap or the RPOT
		panic("Stray child reference %p->%p\n", childRef, *childRef);
		return 1;
	      }
	    }
	  }
	}
	pstTempClass = pstTempClass->pstClass->pstSuperClass;
      }
    }
    /* if it's an array of arrays or of objects then mark them too */
    else if ((PDEREF(o)->i32Flags & GARBAGE_TYPEMASK) == GARBAGE_ARRAY) {
      tARREF a = (tARREF) o;
      if ((PDEREF(a)->enType == T_ARRAY) ||
	  (PDEREF(a)->enType == T_OBJECT)) {
	for (i = 0; i < PDEREF(a)->i32Number; i++) {
	  tOBREF* refSlot = (tOBREF*) &(((tOBREF*) PDEREF(a)->pvElements)[i]);
	  tOBREF ref = *refSlot;
	  if (ref && HEAPBOUND((int32*) *ref)) {
	    ALLOCATOR_MARK_RecursivelyMark(heap, ref, 0, i32Colour, refSlot);
	  }
	}
      }
    }
  }
  else {
    // This handle isn't in the heap or the RPOT
    panic("Stray reference %p->%p\n", o, *o);
    return 1;
  }
  
  return 0;
}

#endif
