/* This is the start of a new garbage collector (replacing vm/garbage.c)
 *
 * That was a simple conservative mark and sweep collector (with compaction).
 *
 * I want to implement a generational collector with two heaps, called 
 * young heap and old heap.  Allocation from young heap is done for all 
 * objects < 50 words in size. All bigger objects automatically go to Old Heap.
 *
 * The Young Heap uses the notion of a Thread Local Heap (TLH) to avoid 
 * acquiring a lock on allocation of every object.
 *
 * tlh.h describes the structure of a TLH. The Young Heap has a list of 
 * free TLHs, a thread can acquire one of these for itself by doing an 
 * atomic test & set on the lock word of the TLH.
 *
 * Only the owning thread may manipulate a TLH. Each TLH has 375 words 
 * for objects.
 *
 * If a thread can't acquire a new TLH, it has to do a conventional 
 * allocation from a free list of spaces.
 *
 * The young heap is collected more often that the old, and survivors 
 * of X collections are promoted to the old heap. The Old heap is compacted 
 * after collection.
 */

#include "config.h"

#ifdef ALLOCATOR

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#endif


#include "vm/wrappers.h"
#include "vm/cplist.h"

#include "vm/thread.h"
#include "vm/jni.h"
#include "vm/hash.h"
#include "vm/cplist.h"
#include "vm/cptuplelist.h"
#include "vm/classfile_methods.h"

#include "vm/interp_methods.h"

#include "vm/garbage/public_methods.h"
#include "vm/garbage/handles.h"

#include "vm/garbage/youngheap.h"
#include "vm/garbage/oldheap.h"

#include "vm/garbage_defs.h"
#include "vm/garbage/mark/mark.h"


#include "lib/indigenous/java.lang/VMClassLoader.h"

#define ALLOCATOR_SIZE (10 * 1024 * 1024) //megabytes
#define ALLOCATOR_HANDLES_SIZE 200000
#define OLD_TO_YOUNG_RATIO 100   //the old heap is ratio times as large as the young heap

#define FINALIZABLE_SIZE 32

#ifdef DEBUG
#define CORRUPTION_CHECK() doCorruptionCheck()
#else
#define CORRUPTION_CHECK() 
#endif

extern tOBREF OOMExceptionObject;

static int allocator_size = 0;


/* 
 * Must be called before ALLOCATOR_Init()
 */
void ALLOCATOR_SetHeapSize(int nbytes)
{
  allocator_size = nbytes;
}

/* 
 * Allocates the bitfield, young heap and old heap in one large block. 
 * The handle block is in a separate area
 */

tAllocatorHeap* ALLOCATOR_Init()
{
  tAllocatorHeap* heap = (tAllocatorHeap*) sys_malloc(sizeof(tAllocatorHeap));
  if (heap == NULL) {
    return NULL;
  }
  
  heap->iDuringGC = 0;
  if (allocator_size == 0) {
    allocator_size = ALLOCATOR_SIZE;
  }

  assert(allocator_size > 0);

  heap->pi32Block = (int32*) sys_malloc(allocator_size );
  if (heap->pi32Block == NULL) {
    sys_free(heap);
    return NULL;
  }
  
  // Now allocate the handle block 
  heap->pi32HandleBlock = 
    (int32*) sys_malloc(ALLOCATOR_HANDLES_SIZE * sizeof(int32*));
  if (heap->pi32HandleBlock == NULL) {
    sys_free(heap->pi32Block);
    sys_free(heap);
    return NULL;
  }

  heap->i32HandleBlockSize = (ALLOCATOR_HANDLES_SIZE);
  
  // Now initialize the handle block 
  if (HANDLES_Init(heap->pi32HandleBlock, heap->i32HandleBlockSize) != 0) {
    sys_free(heap->pi32Block);
    sys_free(heap->pi32HandleBlock);
    sys_free(heap);
    return NULL;
  }
  
  heap->hNextFreeHandle = HANDLES_FirstHandle(heap->pi32HandleBlock);
  
  heap->i32BlockSize = allocator_size / sizeof(int32);
  
  // Determine the size of the bitfield
  // We need 1 bit for every word left in the block
  // So the bitfield will represent i32BlockSize words - 1/32nd 
  // ie i32BlockSize words * 31/32
  heap->i32BFSize = ((heap->i32BlockSize * 31) / 32) / 32;
   
  // Determine the size of the young heap and old heap

  heap->i32OldHeapSize = //temp val
    (heap->i32BlockSize - heap->i32BFSize) / (OLD_TO_YOUNG_RATIO + 1); 
  heap->i32YoungHeapSize = heap->i32OldHeapSize;
  heap->i32OldHeapSize = 
    heap->i32BlockSize - heap->i32YoungHeapSize - heap->i32BFSize;
  
  // might be one too little?
  heap->i32OldHeapBFSize = heap->i32OldHeapSize / 32; 
  heap->i32YoungHeapBFSize = heap->i32BFSize - heap->i32OldHeapBFSize;
  
  // now set up the pointers
  heap->pi32YoungHeap = heap->pi32Block + heap->i32BFSize;
  heap->pi32OldHeap = heap->pi32YoungHeap + heap->i32YoungHeapSize;
  
  assert((heap->pi32OldHeap + heap->i32OldHeapSize) == 
	 (heap->pi32Block + heap->i32BlockSize));
  
  // create the heap locks
  heap->youngMutex = sys_mutex_create(0);
  heap->oldMutex = sys_mutex_create(0);

  // create the finalizable object table
  heap->hFinalizable = (tOBREF *) sys_malloc(ALLOCATOR_SIZE * sizeof(tOBREF));
  assert(heap->hFinalizable);
  heap->i32FinalizableCount = 0;
  heap->i32FinalizableLimit = ALLOCATOR_SIZE;
  
  // Now initialize the young heap
  if (YOUNGHEAP_Init(heap) != 0) {
    //XXX we must free all that's been allocated 
    return NULL;
  }
  
  // Now initialize the old heap
  if (OLDHEAP_Init(heap) != 0) {
    //XXX we must free all that's been allocated 
    return NULL;
  }
  
  traceGC("Young heap size is %i and old heap size %i\n", 
	  (int) heap->i32YoungHeapSize, (int) heap->i32OldHeapSize);
  traceGC("Young heap BF size is %i and old heap BF size %i\n", 
	  (int) heap->i32YoungHeapBFSize, (int) heap->i32OldHeapBFSize);
  traceGC("Reserved %i regular handles\n", heap->i32HandleBlockSize);

  //   OLDHEAP_TestBitfield( heap );
  return heap;
}

#ifdef GARBAGE_VERBOSE
void ALLOCATOR_showStats(tAllocatorHeap* heap, char *desc)
{
  traceGC("*** Allocator statistics - %s ***", desc);
  traceGC("Handle block is %li words from %p to %p", 
	  heap->i32HandleBlockSize, heap->pi32HandleBlock,
	  heap->pi32HandleBlock + heap->i32HandleBlockSize);
  traceGC("Allocator heap size %i bytes from %p to %p", 
	  allocator_size, heap->pi32Block, 
	  ((byte*) heap->pi32Block) + allocator_size);
  traceGC("Young heap BF is %li words  BF from %p to %p",
	  heap->i32YoungHeapBFSize, heap->pi32Block,
	  heap->pi32Block + heap->i32YoungHeapBFSize);
  traceGC("Old heap BF is %li words BF from %p to %p", 
	  heap->i32OldHeapBFSize, heap->pi32Block + heap->i32YoungHeapBFSize,
	  heap->pi32Block + heap->i32YoungHeapBFSize + heap->i32OldHeapBFSize);
  traceGC("Young heap is %li words from %p to %p", 
	  heap->i32YoungHeapSize, heap->pi32YoungHeap,
	  heap->pi32YoungHeap + heap->i32YoungHeapSize);
  traceGC("Old heap is %li words from %p to %p", 
	  heap->i32OldHeapSize, heap->pi32OldHeap,
	  heap->pi32OldHeap + heap->i32OldHeapSize);
}
#endif

//local methods
static int ALLOCATOR_MarkReachableFromMachine(tAllocatorHeap* heap);
static int ALLOCATOR_MarkFinalizerReachable(tAllocatorHeap* heap);
static int ALLOCATOR_MarkReachableFromRPOT(tAllocatorHeap* heap);
static int ALLOCATOR_SweepAndCompact(tAllocatorHeap* heap);
static int ALLOCATOR_MoveFinalizationCandidates(tAllocatorHeap* heap);
static tOBREF ALLOCATOR_ConstructGCThreadObject(JNIEnv* env);
static int ALLOCATOR_TidyFinalizedObjects(tAllocatorHeap* heap);

int ALLOCATOR_ISHANDLE(tAllocatorHeap* heap, tOBREF h);


#ifdef DEBUG
void doCorruptionCheck()
{
  // Insert your test code here ...
}
#endif

/*
 * Does a garbage collection of the heaps 
 */
int ALLOCATOR_gc(tAllocatorHeap* heap)
{
  traceGC0("ALLOCATOR_gc: started");
  if (heap->iDuringGC) {
    panic0("Can't do recursive GC!");
    return -1; 
  } 
  
  heap->iDuringGC = 1;

#ifdef GARBAGE_VERBOSE
  ALLOCATOR_showStats(heap, "before running GC");
#endif
  CORRUPTION_CHECK();
  
  /*
    first mark all objects reachable from the thread stacks, JNI
    local and global refs, and other places
   */
  traceGC0("Calling MarkReachableFromMachine");
  if (ALLOCATOR_MarkReachableFromMachine(heap) != 0) {
    return -1;
  }

  CORRUPTION_CHECK();

#ifdef PERSIST
  /* mark all objects reachable from the RPOT (only) */
  traceGC0("Calling MarkReachableFromRPOT");
  if (ALLOCATOR_MarkReachableFromRPOT(heap) != 0) {
    return -2;
  }

  CORRUPTION_CHECK();
#endif

  /* mark any objects that are F_REACHABLE (only) */
  traceGC0("Calling MarkFinalizerReachable");
  if (ALLOCATOR_MarkFinalizerReachable(heap) != 0) {
    return -3;
  }
  CORRUPTION_CHECK();

  /* move candidates for finalization to the end of the finalizable
     object list */
  traceGC0("Calling MoveFinalizationCandidates");
  if (ALLOCATOR_MoveFinalizationCandidates(heap) != 0) {
    return -4;
  }

  /* sweep, compact and clear the object marks */
  traceGC0("Calling SweepAndCompact");
  if (ALLOCATOR_SweepAndCompact(heap) != 0) {
    return -5;
  }
  
  CORRUPTION_CHECK();

  heap->iDuringGC = 0;

#ifdef GARBAGE_VERBOSE  
  ALLOCATOR_showStats(heap, "after running GC");
#endif
  traceGC0("ALLOCATOR_gc: finished");
  return 0;
}


static int ALLOCATOR_MarkReachableFromMachine(tAllocatorHeap* heap)
{
  JNI_JNIDataIterator iter;
  int threadIndex;
  int i = 0;
  tClassLoaderTuple* tuple;
  tJNIData*  pstJNIData;

  if (ALLOCATOR_MARK_MarkThreadStacks(heap) != 0) {
    return -1;
  }
  
  /* go through the JNI local references for the current thread... in our
     single-processor environment it's the only one that can be executing
     native code at the moment... will be NULL if native code is not being
     executed 

     oh-oh, what happens with MT?
     
     Update for MT: Well this is the GC thread running, so it has NO
     native state.  The other threads must never be in a native call,
     although there may be a native call higher up the chain.  This
     means we must scan the pstJNIData for each thread 
  */
  pstJNIData = JNI_firstJNIData(&iter);
  while (pstJNIData) {
    for (i = 0; i < pstJNIData->u16NumLocalRefsUsed; i++) {
      if (pstJNIData->ppstLocalRefs[i]) {
	ALLOCATOR_MARK_RecursivelyMark(heap, 
				       (tOBREF) pstJNIData->ppstLocalRefs[i],
				       0, GARBAGE_REACHABLE,
				       (tOBREF *) &(pstJNIData->ppstLocalRefs[i]));
      }
    }
    pstJNIData = JNI_nextJNIData(&iter);
  }

  /* go through the JNI global references */
  for (i = 0; i < JNI_u16NumGlobalRefsUsed; i++) {
    if (JNI_ppstGlobalRefs[i]) {
      ALLOCATOR_MARK_RecursivelyMark(heap, (tOBREF) JNI_ppstGlobalRefs[i], 
				     0, GARBAGE_REACHABLE, 
				     (tOBREF *) &(JNI_ppstGlobalRefs[i]));
    }
  }

  /* do the preallocated OutOfMemoryException object */
  if (OOMExceptionObject) {
    ALLOCATOR_MARK_RecursivelyMark(heap, OOMExceptionObject, 
				   0, GARBAGE_REACHABLE, 
				   &OOMExceptionObject);
  }

  /* go through C stack looking for more objects */
  ALLOCATOR_MARK_MarkCStacks(heap);
  
  /* go through all regular classes looking for relevant static fields 
     also look to see if a java.lang.Class object has been created for 
     the class */
  for (CPTUPLELIST_GoTop(); CPTUPLELIST_IsMore(); CPTUPLELIST_Skip()) {
    tuple = CPTUPLELIST_GetCurrentClass();
    for (i = 0; i < tuple->pstClass->u16StatCount; i++) {
      uint16 j;

      /* if signature says this isn't an object or array, don't mark it */
      if ((tuple->pstClass->pstStatOffsets[i].uidFieldSig[0] != '[') &&
	  (tuple->pstClass->pstStatOffsets[i].uidFieldSig[0] != 'L')) {
	continue;
      }
      j = tuple->pstClass->pstStatOffsets[i].u16Offset;
      if (tuple->pi32StatVars[j] == 0) {
	continue;
      }
      // XXX -- I don't know why it happens, but it seems that kissme
      // uses non-pointer values Strings in statics: scc
      if (ALLOCATOR_ISHANDLE(heap, (tOBREF) tuple->pi32StatVars[j])) {
	ALLOCATOR_MARK_RecursivelyMark(heap, 
				       (tOBREF) tuple->pi32StatVars[j],
				       1, GARBAGE_REACHABLE, 
				       (tOBREF*) &(tuple->pi32StatVars[j]));
      }
    }
    if (tuple->classObject) {
      ALLOCATOR_MARK_RecursivelyMark(heap, tuple->classObject,
				     0, GARBAGE_REACHABLE, 
				     &(tuple->classObject));
    }
  }

  /* finally, mark the primitive and wrapper class objects */
  for (i = 0; i < 9; i++) {
    tOBREF* loc;
    loc = java_lang_VMClassLoader_getPrimitiveClassLoc(i);
    if (*loc) {
      ALLOCATOR_MARK_RecursivelyMark(heap, *loc, 0, GARBAGE_REACHABLE, loc);
    }
    loc = java_lang_VMClassLoader_getPrimitiveWrapperClassLoc(i);
    if (*loc) {
      ALLOCATOR_MARK_RecursivelyMark(heap, *loc, 0, GARBAGE_REACHABLE, loc);
    }
  }
  return 0;
}


static int ALLOCATOR_MarkFinalizerReachable(tAllocatorHeap* heap)
{
  int i;

  for (i = 0; i < heap->i32FinalizableCount; i++) {
    if (heap->hFinalizable[i]) {
      ALLOCATOR_MARK_RecursivelyMark(heap, heap->hFinalizable[i],
				     0, GARBAGE_F_REACHABLE, NULL);
    }
  }
  return 0;
}


static int ALLOCATOR_MoveFinalizationCandidates(tAllocatorHeap* heap)
{
  int i, j;
  tOBREF tmp;

  sys_mutex_lock(heap->gc_finalization_mutex);
  // Move the objects that now require finalization to the end of the
  // finalizable list so that we can clear the GC marks.
  for (i = 0, j = heap->i32FinalizableCount - 1; i < j; /* */) {
    if (heap->hFinalizable[i] == NULL) {
      // Fill a hole by shuffling objects
      heap->hFinalizable[i] = heap->hFinalizable[j];
      heap->hFinalizable[j] = 
	heap->hFinalizable[--(heap->i32FinalizableCount)];
      j--;
    }
    else if ((DEREF(heap->hFinalizable[i])->i32Flags & 
	      GARBAGE_REACHABILITY_MASK) == GARBAGE_F_REACHABLE) {
      // Swap a candidate with an object we haven't checked yet
      tmp = heap->hFinalizable[i];
      heap->hFinalizable[i] = heap->hFinalizable[j];
      heap->hFinalizable[j] = tmp;
      j--;
    }
    else {
      // Leave a non-candidate alone
      i++;
    }
  }
  // Set the pointers for the finalization thread
  heap->i32FinalizationFirst = heap->i32FinalizationNext = j + 1;
  heap->i32FinalizationLimit = heap->i32FinalizableCount;
  
  if (heap->i32FinalizationFirst < heap->i32FinalizationLimit) {
    // Wake the finalization thread
    sys_condition_signal(heap->gc_finalization_cond);
  }
  sys_mutex_unlock(heap->gc_finalization_mutex);
  return 0;
}


/**
 * Get an object that is ready to be finalized.  Returns NULL if
 * no more objects need finalizing right now.
 */
tOBREF ALLOCATOR_GetFinalizationCandidate(tAllocatorHeap* heap)
{
  tOBREF obj;

  while (heap->i32FinalizationNext < heap->i32FinalizationLimit) {
    obj = heap->hFinalizable[heap->i32FinalizationNext++];
    if (obj != NULL) {
      return obj;
    }
  }
  return NULL;
}


#ifdef PERSIST
static int ALLOCATOR_MarkReachableFromRPOT(tAllocatorHeap* heap)
{
  int i;
  tRPOTElement* temp;

  if (RPOT_table == NULL) {
    return 0;
  }

  for (i = 0; i < RPOT_TABLESIZE; i++) {
#ifdef REVRPOT
    for (temp = RPOT_table[i]; temp; temp = temp->next) {
      if (((temp->la->i32Flags & PERSIST_DirtyMask) != 0)) {
	ALLOCATOR_MARK_RecursivelyMark(heap, &(temp->la),
				       0, GARBAGE_REACHABLEFROMRPOT, NULL);
      }
    }
#else
    if (RPOT_table[i].pid) {
      for (temp = &(RPOT_table[i]); temp; temp = temp->next) {
	if (((temp->la->i32Flags & PERSIST_DirtyMask) != 0)) {
	  ALLOCATOR_MARK_RecursivelyMark(heap, &(temp->la),
					 0, GARBAGE_REACHABLEFROMRPOT, NULL);
	}
      }
    }
#endif
  }
  return 0;
}
#endif


static int ALLOCATOR_SweepAndCompact(tAllocatorHeap* heap)
{
  if (OLDHEAP_SweepAndCompact(heap) != 0) {
    return -1;
  }
  return 0;
}


int ALLOCATOR_ISHANDLE(tAllocatorHeap* heap, tOBREF h)
{
  if (OLDHEAP_ISHANDLE(heap, h)) {
    return 1;
  }

  if (YOUNGHEAP_ISHANDLE(heap, h)) {
    return 2;
  }

#ifdef PERSIST
  if (RPOT_InRPOT(h)) {
    return 3;
  }
#endif

  return 0;
}

/**
 * Add 'obj' to the table of objects that will need finalizing
 */
void ALLOCATOR_RegisterFinalizable(tOBREF obj, tAllocatorHeap* heap)
{
  assert(obj);
  sys_mutex_lock(heap->gc_finalization_mutex);
  GARBAGE_LOCK(heap);
  if (heap->i32FinalizableCount == heap->i32FinalizableLimit) {
    heap->i32FinalizableLimit *= 2;
    heap->hFinalizable = (tOBREF *) 
      sys_realloc(heap->hFinalizable, 
		  heap->i32FinalizableLimit * sizeof(tOBREF));
    assert(heap->hFinalizable);
  }
  heap->hFinalizable[heap->i32FinalizableCount++] = obj;
  GARBAGE_UNLOCK(heap);
  sys_mutex_unlock(heap->gc_finalization_mutex);
}

/**
 * Remove 'obj' from the table of objects that will need finalizing
 */
void ALLOCATOR_DeregisterFinalizable(tOBREF obj, tAllocatorHeap* heap)
{
  int i;

  assert(obj);
  if (heap->i32FinalizationNext > 0 && 
      heap->hFinalizable[heap->i32FinalizationNext - 1] == obj) {
    // The object will most likely be one slot before the next candidate
    heap->hFinalizable[heap->i32FinalizationNext - 1] = NULL;
  }
  else {
    // ... but if the GC ran before the finalization thread had finished,
    // the finalization candidates may have been shuffled abot.
    for (i = heap->i32FinalizationFirst; i < heap->i32FinalizationLimit; i++) {
      if (heap->hFinalizable[i] == obj) {
	heap->hFinalizable[i] = NULL;
	break;
      }
    }
  }
}


int ALLOCATOR_getNumTotalWords(tAllocatorHeap* heap)
{
  return heap->i32OldHeapSize;
}


int ALLOCATOR_getNumFreeWords(tAllocatorHeap* heap)
{
  int ret = 0;

  int32* freePointer = heap->pi32OldFirstFree;
  while (freePointer) {
    ret += *freePointer;
    freePointer = (int32*) *(freePointer + 1);
  }
  return ret;
}

#endif
