/*
 * Copyright (C) 2002, Stephen Crawley
 *
 * This file is part of the kissme project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "vm/diag.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>


struct diagChannelStruct {
  char *dcs_name;
  char *dcs_description;
  int   dcs_enabled;
  FILE *dcs_file;
};


struct diagChannelStruct channels[NOS_DIAG_CHANNELS];


/* Exit with an assertion failure message */
void DIAG_AssertFail(char *file, unsigned int line, char *condition)
{
  eprintf("Kissme assertion '%s' failed: %s line %u\n",
	  condition, file, line);
  fflush(stderr);
  DIAG_BailOut();
}


/* Exit by printing a message and calling BailOut. */
void DIAG_Panic0(char* message)
{
  eprintf("Kissme Panic: %s\n", message);
  fflush(stderr);
  DIAG_BailOut();
}


/* Exit by printing a formatted message and calling BailOut. */
void DIAG_Panic(char* format, ...)
{
  va_list ap;

  eprint("Kissme Panic: ");
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  eprint("\n");
  fflush(stderr);
  DIAG_BailOut();
}


/* Terminate immediately, generating a core dump (or whatever) */
void DIAG_BailOut()
{
  eprint("Attempting to dump core\n");
  fflush(stderr);
#ifdef KISSME_LINUX_USER
  abort();
#else
#error how do we abort?
#endif
}


/* 
 * Exit, after printing an error message and then calling 
 * shutdown_machine(NULL, 1, rc)
 */
int DIAG_FatalError0(int rc, char* message)
{
  eprintf("Kissme Fatal Internal Error: %s\n", message);
  fflush(stderr);
  shutdown_machine(NULL, 1, rc);
#ifdef KISSME_LINUX_USER
  exit(rc);
#else
#error how do we exit?
#endif
}


/*
 * Exit, after printing a formatted error message and then calling 
 * shutdown_machine(NULL, 1, rc)
 */ 
int DIAG_FatalError(int rc, char* format, ...)
{
  va_list ap;

  eprint("Kissme Fatal Internal Error: ");
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  eprintf("\n");
  fflush(stderr);
  va_end(ap);
  return shutdown_machine(NULL, 1, rc);
}

/* Print an error message and continue */
void DIAG_Error0(char* message)
{
  eprintf("Kissme Internal Error: %s\n", message);
  fflush(stderr);
}


/* Print a formatted error message and continue */
void DIAG_Error(char* format, ...)
{
  va_list ap;

  eprint("Kissme Internal Error: ");
  va_start(ap, format);
  vfprintf(stderr, format, ap);
  va_end(ap);
  eprintf("\n");
  fflush(stderr);
}


/* Send a message to a logging channel */
void DIAG_Log0(int channel, char* message)
{
  assert(channel >= 0 && channel <= LAST_DIAG_CHANNEL);
  if (channels[channel].dcs_enabled) {
    if (channels[channel].dcs_name) {
      fprintf(channels[channel].dcs_file, 
	      "%s: %s\n", channels[channel].dcs_name, message);
    }
    else {
      fprintf(channels[channel].dcs_file, 
	      "log channel %d: %s\n", channel, message);
    }
  }
}


/* Send a formatted message to a logging channel */
void DIAG_Log(int channel, char* format, ...)
{
  va_list ap;
  
  assert(channel >= 0 && channel <= LAST_DIAG_CHANNEL);
  if (channels[channel].dcs_enabled) {
    FILE *file = channels[channel].dcs_file;
    if (channels[channel].dcs_name) {
      fputs(channels[channel].dcs_name, file);
      fputs(": ", file);
    }
    else {
      fprintf(file, "log channel %d: ", channel);
    }
    va_start(ap, format);
    vfprintf(file, format, ap);
    va_end(ap);
    fputs("\n", file);
  }
}


/* 
 * Define a logging channel with a known channel number.  The 'name'
 * is prefixed to log messages, and the 'description' is displayed
 * by DIAG_ListChannels.
 */ 
void DIAG_DefineStaticChannel(int channel, char *name, char *description)
{
  assert(channel >= 0 && channel < NOS_STATIC_DIAG_CHANNELS);
  assert(channels[channel].dcs_name == 0);
  channels[channel].dcs_name = name;
  channels[channel].dcs_description = description;
}


/*
 * Same as DIAG_DefineStaticChannel, except that the channel number is
 * allocated on the fly.
 */
int DIAG_DefineDynamicChannel(char *name, char *description)
{
  int i;
  for (i = NOS_STATIC_DIAG_CHANNELS; i <= LAST_DIAG_CHANNEL; i++) {
    if (channels[i].dcs_name == 0) {
      channels[i].dcs_name = name;
      channels[i].dcs_description = description;
      return i;
    }
  }
  nonfatalError0("all dynamic channel slots in used");
  return -1;
}


/*
 * Enable or disable a channel.  The result is the previous state.
 */
int DIAG_SetChannelState(int channel, int enabled)
{
  int wasEnabled;

  assert(channel >= 0 && channel <= LAST_DIAG_CHANNEL);
  wasEnabled = channels[channel].dcs_enabled;
  channels[channel].dcs_enabled = enabled;
  return wasEnabled;
}


/*
 * Get a channel's current state.
 */
int DIAG_ChannelState(int channel)
{
  assert(channel >= 0 && channel <= LAST_DIAG_CHANNEL);
  return channels[channel].dcs_enabled;
}


/*
 * Set the FILE associated with a channel.
 */
void DIAG_SetChannelFile(int channel, FILE *file)
{
  assert(channel >= 0 && channel <= LAST_DIAG_CHANNEL);
  fflush(channels[channel].dcs_file);
  channels[channel].dcs_file = file;
}


/*
 * Flush all trace print channels
 */
void DIAG_FlushChannels()
{
  int i;
  for (i = 0; i <= LAST_DIAG_CHANNEL; i++) {
    fflush(channels[i].dcs_file);
  }
}


/*
 * List the channel names, descriptions and their states.
 */
void DIAG_ListChannels() 
{
  int i;
  for (i = 0; i <= LAST_DIAG_CHANNEL; i++) {
    if (channels[i].dcs_name) {
      eprintf("  %s (%d): %s - %s\n", channels[i].dcs_name, i, 
	      channels[i].dcs_description, 
	      channels[i].dcs_enabled ? "enabled" : "disabled");
    }
    else if (channels[i].dcs_enabled) {
      eprintf("  <unnamed channel> (%d) - enabled\n", i);
    }
  }
}


/*
 * Map a channel name to a channel number.  Returns the channel number
 * or -1 if the channel is not defined.
 */
int DIAG_LookupChannel(char *name) 
{
  int i;
  for (i = 0; i <= LAST_DIAG_CHANNEL; i++) {
    if (channels[i].dcs_name &&
	strcasecmp(name, channels[i].dcs_name) == 0) {
      return i;
    }
  }
  return -1;
}


/*
 * Initialise logging channels.
 */
void DIAG_InitChannels()
{
  int i;
  for (i = 0; i <= LAST_DIAG_CHANNEL; i++) {
    channels[i].dcs_name = NULL;
    channels[i].dcs_description = NULL; 
    channels[i].dcs_enabled = 0;
    channels[i].dcs_file = stderr;
  }
  DIAG_DefineStaticChannel(DEFAULT_DIAG_CHANNEL, "trace",
			   "default tracing");
  DIAG_DefineStaticChannel(LOADER_DIAG_CHANNEL, "loading",
			   "tracing of class loading and resolution");
  DIAG_DefineStaticChannel(CALL_DIAG_CHANNEL, "calls",
			   "tracing of Java method calls");
  DIAG_DefineStaticChannel(JNI_DIAG_CHANNEL, "jni",
			   "tracing of Native method calls");
  DIAG_DefineStaticChannel(OPCODE_DIAG_CHANNEL, "opcodes",
			   "tracing of Java opcodes executed");
  DIAG_DefineStaticChannel(OPSTACK_DIAG_CHANNEL, "opstacks",
			   "tracing of the Java operand stack");
  DIAG_DefineStaticChannel(FRAME_DIAG_CHANNEL, "frames",
			   "tracing of Java method frames");
  DIAG_DefineStaticChannel(EXCEPTION_DIAG_CHANNEL, "exceptions",
			   "tracing of Java exception throwing and catching");
  DIAG_DefineStaticChannel(ALLOC_DIAG_CHANNEL, "alloc",
			   "tracing of Java object allocation");
  DIAG_DefineStaticChannel(GC_DIAG_CHANNEL, "gc",
			   "tracing of garbage collection");
  DIAG_DefineStaticChannel(GCTHREAD_DIAG_CHANNEL, "gcsync",
			   "tracing of GC thread synchronization");
  DIAG_DefineStaticChannel(THREAD_DIAG_CHANNEL, "threads",
			   "tracing of thread management");
  DIAG_DefineStaticChannel(LOCK_DIAG_CHANNEL, "locks",
			   "tracing of locking");
  DIAG_DefineStaticChannel(STARTUP_DIAG_CHANNEL, "startup",
			   "tracing of VM startup and shutdown");
  DIAG_DefineStaticChannel(PERSIST_DIAG_CHANNEL, "persist",
			   "tracing of persistence activity");
  DIAG_DefineStaticChannel(INIT_DIAG_CHANNEL, "init",
			   "tracing of Java class initialization");
}


