/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project which in turn is 
 * part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

#include "vm/jni.h"
#include "vm/loading_errors.h"
#include "vm/classfile_internal.h"
#include "vm/classfile_methods.h"
#include "vm/classfil.h"
#include "vm/cptuplelist.h"
#include "vm/jutils.h"
#include "vm/classloader_tuple.h"

#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <math.h>
#endif



extern tMutex* classfileMutex;

static char* pszObjectString = "java/lang/Object";
static char* pszSerializableString = "java/io/Serializable";
static char* pszCloneableString = "java/lang/Cloneable";

static int AscizCmp(tClass*, int, char*);
static int BuildFieldTables(tClassLoaderTuple*, tFieldInfo*, uint16);
static void BuildVirtualTable(tClass* c);
static void CalcMethodArgLen(tClass* c); 
static void AddInterfaceMethods(tClass *c, tClass *i);
static int ClassHasFinalizer(tClassLoaderTuple*);


#define STRING_TEMP_SIZE 256  // size of temp string storage during loading 


/*
 * @doc FUNC
 * @func
 * This function is used to calculate the argument size of every method
 * in a class and store the information in a field in the tMethod structure.
 */
static void CalcMethodArgLen(tClass* c /* @parm Class whose methods 
					  we want to check */)
{
  int i;
  int iStatic;
  
  for (i = 0; i < c->u16MethodsCount; i++) {
    if (c->pstMethods[i].u16AccessFlags & ACC_STATIC) {
      iStatic = 1;
    }
    else {
      iStatic = 0;
    }
    c->pstMethods[i].u16ArgSize = 
      CLASSFILE_ParseArgumentLength(c->pstMethods[i].uidSignature, iStatic);
    c->pstMethods[i].u16RetSize =
      CLASSFILE_ParseReturnLength(c->pstMethods[i].uidSignature);

    // eprintf("Arg size for %s.%s is %i\n", c->uidName, 
    //         c->pstMethods[i].uidName, c->pstMethods[i].u16ArgSize);
  }
}


/*
 * @doc FUNC
 * @func
 * This function compares a C ASCII string to a string at the specified
 * index in the specified class's constant pool. It is used internally
 * by the class loading function to check the names of attributes during
 * loading.
 *
 * @rdesc Returns one of the following:
 *
 * @flag 0 | Successful comparison
 *
 * @flag other | Failed comparison
 */
static int AscizCmp(tClass* pstClass, 
		    /* @parm Use this class's constant pool */
		    int iIndex,   
		    /* @parm The index in the constant pool */
		    char* pszS      
		    /* @parm The C string to compare against */)
{
  assert(iIndex <= pstClass->u16ConstPoolCount);
  assert(CONSTTAG(pstClass, iIndex) == CONSTANT_Utf8);
  return strcmp(pszS, CONSTGET_UidAsciz(pstClass, iIndex));
}


/*
 * @doc FUNC
 * @func
 * This function reads a single byte from memory. It's used to be uniform
 * with the getu2 and getu4 functions.
 *
 * @rdesc The byte that was read.
 */
byte memgetu1(byte* pbArray, int32* pi32ArrayPos)
{
  byte value;
  value = pbArray[*pi32ArrayPos];
  (*pi32ArrayPos)++;
  return value;
}


/*
 * @doc FUNC
 * @func
 * This function reads two bytes from memory in big-endian order and stores
 * them in a native unsigned 16-bit integer.
 *
 * @rdesc The 16-bit word that was read.
 */
uint16 memgetu2(byte* pbArray, int32* pi32ArrayPos)
{
  uint16 value;

  value = pbArray[*pi32ArrayPos];
  (*pi32ArrayPos)++;
  value <<= 8;
  value += pbArray[*pi32ArrayPos];
  (*pi32ArrayPos)++;
  return value;
}


/*
 * @doc FUNC
 * @func
 * This function reads four bytes from memory in big-endian order and stores
 * them in a native unsigned 32-bit integer.
 *
 * @rdesc The 32-bit word that was read.
 */
uint32 memgetu4(byte* pbArray, int32* pi32ArrayPos)
{
  uint32 value = 0;
  int i;

  for (i = 0; i < 4; i++) {
    value <<= 8;
    value += pbArray[*pi32ArrayPos];
    (*pi32ArrayPos)++;
  }
  return value;
}


/*
 * @doc FUNC
 * @func
 * This function reads eight bytes from memory in big-endian order and stores
 * them in a native unsigned 64 bit long long.
 */
unsigned long long memgetu8(byte* pbArray, int32* pi32ArrayPos)
{
  unsigned long long value = 0;
  int i;
  
  for (i = 0; i < 8; i++) {
    value <<= 8;
    value += pbArray[*pi32ArrayPos];
    (*pi32ArrayPos)++;
  }
  return value;
}


/* 
 * Another helper method, this one reads info about methods 
 */
int readMethodsInfo(JNIEnv* env, tClassLoaderTuple* tuple, 
		    uint32* pi32FilePos, uint16* pu16ConstRealSize, 
		    int32* pi32ArrayPos, byte* pbArray) 
{
  tClass* c = tuple->pstClass;
  int i = 0, j = 0, k = 0, l = 0;
  uint16 u16AttributesCount0;
  uint16 u16AttributesCount1;
  uint16 tempu2;
  
  /* read in pstMethods count */
  c->u16MethodsCount = locgetu2();

#ifdef CLASSFILE_TRACE
  eprintf("Methods: %d\n", c->u16MethodsCount);
#endif
  
  /* read in pstMethods */
  if (c->u16MethodsCount > 0) {
    c->pstMethods = 
      (tMethod*) sys_malloc(sizeof(tMethod) * c->u16MethodsCount);
    
    if (c->pstMethods == NULL) {
      return -1;
    }
    memset(c->pstMethods, 0, sizeof(tMethod) * c->u16MethodsCount);
    for (i = 0; i < c->u16MethodsCount; i++) {
#ifdef CLASSPERSIST
      c->pstMethods[i].pid = NULL;
#endif
      c->pstMethods[i].pstClass = tuple; /*c;*/
      c->pstMethods[i].u16AccessFlags = locgetu2();
      tempu2 = locgetu2();
      c->pstMethods[i].uidName = CONSTGET_UidAsciz(c, tempu2);
      tempu2 = locgetu2();
      c->pstMethods[i].uidSignature = CONSTGET_UidAsciz(c, tempu2);
      assert(c->pstMethods[i].uidName);
      assert(c->pstMethods[i].uidSignature);
      assert(UID_GetUid(c->pstMethods[i].uidName) == 
	     c->pstMethods[i].uidName);
      assert(UID_GetUid(c->pstMethods[i].uidSignature) == 
	     c->pstMethods[i].uidSignature);
      c->pstMethods[i].bHaveResolvedExceptions = 0;
      c->pstMethods[i].u16NumExceptions = 0;
      c->pstMethods[i].pstExceptions = NULL;

      /* Set the VTIndex to -1 for now.  If this a method of a class, it
	 will be set to the real virtual table index later. */
      c->pstMethods[i].u16VTIndex = -1;
      
      /* this is set here so that it is NULL if the method is native and does
	 has not have any bytecode */
      c->pstMethods[i].pstCode = NULL;
      
      // Each method can have Code, Synthetic, Exceptions and
      // Deprecated attributes, we are only required to process Code
      // and Exceptions
      
      /* Now do the attributes */
      
      u16AttributesCount0 = locgetu2();
      
#ifdef CLASSFILE_TRACE
      eprintf("Method no. %i\n", i);
      eprintf("	access: %x\n", c->pstMethods[i].u16AccessFlags);
      eprintf("	name: %s\n", c->pstMethods[i].uidName);
      eprintf("	sig: %s\n", c->pstMethods[i].uidSignature);
      eprintf("	attr#: %i\n", u16AttributesCount0);
#endif
      
      if (u16AttributesCount0 > 0) {
	for (j = 0; j < u16AttributesCount0; j++) {
	  tempu2 = locgetu2();		/* attr_name_index */
	  if (tempu2 == 0) {
	    eprintf("Illegal constant pool index while loading %s\n", 
		    tuple->uidName);
	    return -2;
	  }
	  if (AscizCmp(c, tempu2, "Code") == 0) {
	    uint32 tempu4;
	    
	    c->pstMethods[i].pstCode = (tCode*) sys_malloc(sizeof(tCode));
	    assert(c->pstMethods[i].pstCode);
	    c->pstMethods[i].pstCode->pstLineNumberTable = NULL;
	    c->pstMethods[i].pstCode->pstExceptionTable = NULL;
	    
	    tempu4 = locgetu4(); //This is the attribute length
	    
	    tempu2 = locgetu2();
	    c->pstMethods[i].pstCode->u16MaxStack = tempu2; //Change from 1
	    tempu2 = locgetu2();
	    c->pstMethods[i].pstCode->u16MaxLocals = tempu2; //Change from 1
	    
#ifdef CLASSFILE_TRACE
	    eprintf("	max stack: %i\n", 
		    c->pstMethods[i].pstCode->u16MaxStack);
	    eprintf("	max locals:%i\n", 
		    c->pstMethods[i].pstCode->u16MaxLocals);
#endif
	    
	    c->pstMethods[i].pstCode->u32CodeLength = locgetu4();
#ifdef STACKMAP
	    c->pstMethods[i].pstCode->pstStackMap = NULL;
#endif
	    
#ifdef CLASSFILE_TRACE
	    eprintf("	code len: %i\n", 
		    c->pstMethods[i].pstCode->u32CodeLength);
#endif
	    
	    /* read in code */
	    
	    // I want to malloc on a page boundary, inefficient, but useful 
	    // for debugging
#ifdef CLASSFILE_ALLOC_CODE_ON_BOUNDARY
	    {
	      int alloc_amt = ((c->pstMethods[i].pstCode->u32CodeLength / 
				PAGESIZE) + 1) * PAGESIZE;
	      c->pstMethods[i].pstCode->pbCode = (byte*) sys_malloc(alloc_amt);
	    }
#else
	    c->pstMethods[i].pstCode->pbCode = 
	      (byte*) sys_malloc(c->pstMethods[i].pstCode->u32CodeLength);
#endif
	    
	    assert(c->pstMethods[i].pstCode->pbCode);
	    
	    for (k = 0; k < c->pstMethods[i].pstCode->u32CodeLength; k++) {
	      c->pstMethods[i].pstCode->pbCode[k] = locgetu1();
	    }
	    
#ifdef CLASSFILE_TRACE
	    eprintf("\nCode for method:\n");
	    for (k = 0; k < c->pstMethods[i].pstCode->u32CodeLength; /* */) {
	      byte *addr = (byte *) &(c->pstMethods[i].pstCode->pbCode[k]);
	      eprintf("%4d ", k);
	      k += DISASS_PrintInstruction(addr, 
					   c->pstMethods[i].pstCode->pbCode);
	    }
	    eprintf("\n");
#endif
	    /* read in exception table Length and table */
	    c->pstMethods[i].pstCode->u16ExceptionTableLength = locgetu2();
	    
#ifdef CLASSFILE_TRACE
	    eprintf("	Exception table len: %d\n", 
		    c->pstMethods[i].pstCode->u16ExceptionTableLength);
#endif
	    
	    if (c->pstMethods[i].pstCode->u16ExceptionTableLength > 0) {
	      tExceptionInfo *etab = (tExceptionInfo*)
		sys_malloc(sizeof(tExceptionInfo) *
			   c->pstMethods[i].pstCode->u16ExceptionTableLength);
	      c->pstMethods[i].pstCode->pstExceptionTable = etab;
	      for (k = 0; 
		   k < c->pstMethods[i].pstCode->u16ExceptionTableLength;
		   k++) {
		uint16 u16CatchIdx;
		
		etab[k].u16StartPC = locgetu2();
		etab[k].u16EndPC = locgetu2();
		etab[k].u16HandlerPC = locgetu2();
		u16CatchIdx = locgetu2();
		/* if it's not zero, we check to see that it has already 
		   been loaded */
		if (u16CatchIdx) {
		  etab[k].pstCatchType =
		    CLASSFILE_FindOrLoad(env, 
					 CONSTGET_UidAsciz(c, u16CatchIdx), 
					 tuple);
		}
		else {
		  etab[k].pstCatchType = NULL;
		}
		
		
#ifdef CLASSFILE_TRACE
		eprintf("	Exception #%d\n", k);
		eprintf("			start_PC: %d\n", 
			etab[k].u16StartPC);
		eprintf("			end_PC: %d\n",
			etab[k].u16EndPC);
		eprintf("			handler_PC: %d\n", 
			etab[k].u16HandlerPC);
		eprintf("			CatchType: %s\n" ,
			etab[k].pstCatchType == NULL ? 
			"NULL" : etab[k].pstCatchType->uidName);
#endif
		
	      }
	    }
	    else {
	      c->pstMethods[i].pstCode->pstExceptionTable = NULL;
	    }
	    
	    /* read in attribute Length and attributes for this CODE
               attribute*/
	    u16AttributesCount1 = locgetu2();
	    
#ifdef CLASSFILE_TRACE
	    eprintf("	Code Attr count: %d\n", u16AttributesCount1);
#endif
	    
	    if (u16AttributesCount1 > 0) {
	      for (k = 0; k < u16AttributesCount1; k++) {
		tempu2 = locgetu2();
		
#ifdef CLASSFILE_TRACE
		eprintf("		Code Attribute# %d: %s\n",
			k, CONSTGET(c, tempu2));
#endif
		
		/* this must check to see if it's a line number table attr */
		if (AscizCmp(c, tempu2, "LineNumberTable") == 0) {
		  tLineNumberTable* tab = 
		    (tLineNumberTable*) sys_malloc(sizeof(tLineNumberTable));
		  c->pstMethods[i].pstCode->pstLineNumberTable = tab;
		    
		  tempu4 = locgetu4(); //Attrib length
		  
		  tab->u16LineNumberTableLength = locgetu2();
		  
#ifdef CLASSFILE_TRACE
		  eprintf("		      line-number table Length: %d\n", 
			  tab->u16LineNumberTableLength);
#endif
		  
		  if (tab->u16LineNumberTableLength > 0) {
		    tab->pstLineNumberTable = (tLineNumberInfo*)
		      sys_malloc(sizeof(tLineNumberInfo) * 
				 tab->u16LineNumberTableLength);
		    for (l = 0; l < tab->u16LineNumberTableLength; l++) {
		      tab->pstLineNumberTable[l].u16StartPC = locgetu2();
		      tab->pstLineNumberTable[l].u16LineNumber = locgetu2();
		      
#ifdef CLASSFILE_TRACE
		      eprintf("			        StartPC: %d\n", 
			      tab->pstLineNumberTable[l].u16StartPC);
		      eprintf("				line-number: %d\n", 
			      tab->pstLineNumberTable[l].u16LineNumber);
#endif
		    }
		  }
		  else {
		    c->pstMethods[i].pstCode->pstLineNumberTable->
		      pstLineNumberTable = NULL;
		    
#ifdef CLASSFILE_TRACE
		    eprintf("No Line-number table info");
#endif
		    
		  }
		} //end linenumbertable
		/* this must check to see if it's a local var table attr */
		else if (AscizCmp(c, tempu2, "LocalVariableTable") == 0) {
		  tLocalVariableTable* vtab = (tLocalVariableTable*)
		    sys_malloc(sizeof(tLocalVariableTable));
		  
		  c->pstMethods[i].pstCode->pstLocalVariableTable = vtab;
		  tempu4 = locgetu4();
		  
		  vtab->u16LocalVariableTableLength = locgetu2();
		  
		  if (vtab->u16LocalVariableTableLength)  {
		    tLocalVariableInfo* vitab = (tLocalVariableInfo*) 
		      sys_malloc(sizeof(tLocalVariableInfo) * 
				 vtab->u16LocalVariableTableLength);
		    
		    vtab->pstLocalVariableTable = vitab;
		    for (l = 0; l < vtab->u16LocalVariableTableLength; l++) {
		      vitab[l].u16StartPC = locgetu2();
		      vitab[l].u16Length = locgetu2();
		      vitab[l].u16NameIndex = locgetu2();
		      vitab[l].u16SignatureIndex = locgetu2();
		      vitab[l].u16Slot = locgetu2();
		      
#ifdef CLASSFILE_TRACE
		      eprintf("		Local Variable #%d:\n", l);
		      eprintf("			start_PC: %d\n", 
			      vitab[l].u16StartPC);
		      eprintf("			Length: %d\n",
			      vitab[l].u16Length);
		      eprintf("			name: %s\n", 
			      CONSTGET_UidAsciz(c, vitab[l].u16NameIndex));
		      eprintf("			sig: ", 
			      CONSTGET_UidAsciz(c, vitab[l].u16SignatureIndex));
		      eprintf("			slot: %d\n", 
			      vitab[l].u16Slot);
#endif
		    }
		  }
		  else {
		    c->pstMethods[i].pstCode->pstLocalVariableTable->
		      pstLocalVariableTable = NULL;
		  }
		} //end local var table
		else {
		  panic0("readMethodsInfo: check this code");
		  locskip(locgetu2()); /* skip past attribute */
		  
		} 
	      } //end for attribcount1
	    } //end if attribcount 1
	  } //end check for code attribute
	  else if (AscizCmp(c, tempu2, "Exceptions") == 0) {
	    int num_exceptions;
	    uint16 m;
	    int32 attrib_length = locgetu4();	    
	    
	    num_exceptions = locgetu2();
	    
	    attrib_length = attrib_length; //avoid compiler warning
	    
	    // eprintf("(tot length %i) Processing %i exceptions for %s.%s\n",
	    //         attrib_length, num_exceptions, c->uidName, 
	    //         c->pstMethods[i].uidName);
	    c->pstMethods[i].bHaveResolvedExceptions = 0;
	    c->pstMethods[i].u16NumExceptions = num_exceptions;
	    c->pstMethods[i].pstExceptions = 
	      (tClassLoaderTuple**) sys_malloc(sizeof(tClassLoaderTuple*) * 
					       num_exceptions);
	    assert(c->pstMethods[i].pstExceptions);
	    for (m = 0; m < num_exceptions; m++) {
	      c->pstMethods[i].pstExceptions[m] = 
		(tClassLoaderTuple*) ((int) locgetu2());
	      assert(((int) c->pstMethods[i].pstExceptions[m]) >= 0);
	      assert((uint16) ((int) c->pstMethods[i].pstExceptions[m]) < 
		     c->u16ConstPoolCount);
	      assert(CONSTTAG(c, (int) c->pstMethods[i].pstExceptions[m]) == 
		     CONSTANT_Class);
	    }
	    
	  }
	  else if (AscizCmp(c, tempu2, "Synthetic") == 0) {
	    int32 attrib_length = locgetu4();
	    locskip(attrib_length);
#ifdef CLASSFILE_TRACE
	    eprintf("Processing Synthetic attribute, length %i\n", 
		    attrib_length);
#endif
	  }
	  else if (AscizCmp(c, tempu2, "Deprecated") == 0)  {
	    int32 attrib_length = locgetu4();
	    locskip(attrib_length);
#ifdef CLASSFILE_TRACE
	    eprintf("Processing Deprecated attribute, length %i\n",
		    attrib_length);
#endif
	  }
	  else {
	    // It's an unknown attribute ...
	    int32 attrib_length = locgetu4();
	    locskip(attrib_length); /* skip past attribute */
#ifdef CLASSFILE_TRACE
	    eprintf("Skipping unknown '%s' attrib, length %li, "
		    "found in method %s\n",
		    CONSTGET_UidAsciz(c, tempu2), attrib_length,
		    c->pstMethods[i].uidName);
#endif
	  }
	} //end for all attributes
      } //if(attrib count0 > 0)
      
    } //for each method
    CalcMethodArgLen(c);
  } //if methods > 0
  else {
    c->pstMethods = NULL;
  }
  
  return 0;
}



/* Helper method for ClassCreate, sets up the constant pool 
 *
 * Takes a pointer to the tClass structure to fill in
 *
 * Takes a pointer to the file position, a pointer to the size of
 * constant pool, an index into the array and pointer to the array
 */


void setupConstantPool(JNIEnv* env, tClass* c, uint32* pi32FilePos, 
		       uint16* pu16ConstRealSize, int32* pi32ArrayPos, 
		       byte* pbArray)
     
{
  int i = 0, j = 0;
  tConstType Tag;
  byte*  tempu1p;
  uint32 tempu4;
  uint16 tempu2;
  byte u1Temp[STRING_TEMP_SIZE];  /* temporary storage during loading */


  /* We need to go through the entire constant pool to find out its real size,
     because a long and a double are supposed to take up two spaces and they
     really make life difficult. */

  // These are both set to 0
  *pi32FilePos = *pi32ArrayPos; /* remember position so we can return */

  *pu16ConstRealSize = 1;

  // Now loop through all the constant pool items. We ignore the first entry.
  for (i = 1; i < (c->u16ConstPoolCount - 1); i++) { //Jewel added -1
    Tag = locgetu1();     /* get type */
    // tempu2 is set to the amount to skip over

    /* for long or double add two, otherwise add one */
    // tempu2 is set to the amount to skip over
    if ((Tag == CONSTANT_Long) || (Tag == CONSTANT_Double)) {
      //5 and 6
      tempu2 = 8;
      *pu16ConstRealSize += 2;
      // and skip over the dummy entry? 
      i++;
    } 
    else {
      if (Tag == CONSTANT_Utf8) { // 1
        tempu2 = locgetu2();
      } 
      else if ((Tag == CONSTANT_Class) || 
	       (Tag == CONSTANT_String))  { // 8 and 7
        tempu2 = 2;
      } 
      else if ((Tag == CONSTANT_Fieldref) || 
	       (Tag == CONSTANT_Methodref) || 
	       (Tag == CONSTANT_InterfaceMethodref))  { // 9 10 and 11
	tempu2 = 4; // class_index and name_and_type_index
      }
      else if ((Tag == CONSTANT_Integer) || 
	       (Tag == CONSTANT_Float)) { // 3 and 4
	tempu2 = 4; // bytes
      } 
      else if ((Tag == CONSTANT_NameandType)) { // 12
	tempu2 = 4; // name_index, descriptor index
      } 
      else {
	tempu2 = 2; // 4;
	
	eprintf("Tag is 0 or %i , skipping over %i %i, pool count is %i (we're at %i)\n", 
		Tag, pbArray[*pi32ArrayPos], pbArray[*pi32ArrayPos + 1], 
		c->u16ConstPoolCount, i);
	panic0("What tag is this?");
      }
      *pu16ConstRealSize += 1;
    } 
    /* skip to next item */
    locskip(tempu2);
  } //end for
    
  /* move back to beginning of constant pool */
  locsetpos(*pi32FilePos);
  
  /* read in the values in the constant pool */
  if (c->u16ConstPoolCount > 0) {
    /* First increase the real size by one, to make space for the
       unused first element */
    c->pu32ConstPool = (uint32*) sys_malloc(sizeof(uint32) * 
					    (c->u16ConstPoolCount + 1));
    c->pbConstTags = (byte*) sys_malloc(sizeof(byte) * 
					(c->u16ConstPoolCount + 1));
    
    assert(c->pu32ConstPool);
    assert(c->pbConstTags);
    
    /* set unused elements to 0 for*/
    c->pu32ConstPool[0] = 0;
    c->pbConstTags[0] = 0;
    

    for (i = 1; i < c->u16ConstPoolCount; i++) { /* don't use 0th element */
      switch (Tag = locgetu1()) {
      case CONSTANT_Utf8:
	tempu2 = locgetu2();
	if ((tempu2 * sizeof(byte) + 1) > STRING_TEMP_SIZE) {
	  tempu1p = (byte*) sys_malloc(tempu2 * sizeof(byte) + 1);
	}
	else {
	  tempu1p = u1Temp;
	}

	for (j = 0; j < tempu2; j++) {
	  tempu1p[j] = locgetu1();
	}
	tempu1p[j] = 0; //NULL terminate the string
	
	c->pu32ConstPool[i] = (uint32) UID_GetUid(tempu1p);
	c->pbConstTags[i] = Tag;
	
#ifdef CLASSFILE_TRACE
	eprintf("CONSTANT POOL [%d] Asciz: %s\n", i, tempu1p);
#endif
	
	if (tempu1p != u1Temp) {
	  sys_free(tempu1p);
	}
	break;
	
	//removed CONSTANT_Unicode entry -- jewel
      case CONSTANT_Integer:
	tempu4 = locgetu4();
	c->pu32ConstPool[i] = tempu4;
	c->pbConstTags[i] = Tag;
#ifdef CLASSFILE_TRACE
	eprintf("CONSTANT POOL [%d] Integer: %i\n", i, tempu4);
#endif	 
	break;

      case CONSTANT_Float:
	{
	  union {int32 i; float f;} fconv;	  
	  fconv.i = locgetu4();
	  ((float*) c->pu32ConstPool)[i] = fconv.f;
	  c->pbConstTags[i] = Tag;
	  break;
	}
	
      case CONSTANT_Long:
	{
	  unsigned long long templ = locgetu8();
	  c->pu32ConstPool[i] = (uint32) (templ & 0xffffffff);
	  c->pbConstTags[i] = Tag;
	  c->pu32ConstPool[++i] = (uint32) ((templ >> 32) & 0xffffffff);
	  c->pbConstTags[i] = 0;
	  break;
	}
	
      case CONSTANT_Double:	
	{	
	  union {int32 i[2]; double d;} dconv;
	  unsigned long long longword;
	  jvalue conv;
	  longword = locgetu8();
	  conv.j = longword;
	  dconv.d = conv.d;	
	  
	  c->pbConstTags[i] = Tag;
	  c->pu32ConstPool[i++] = dconv.i[0];
	  c->pbConstTags[i] = 0;
	  c->pu32ConstPool[i] = dconv.i[1];
	  break;
	}

      case CONSTANT_Class:
	tempu2 = locgetu2();
	assert(tempu2 < c->u16ConstPoolCount);
	if (tempu2 < i) {
	  assert(CONSTTAG(c, tempu2) == CONSTANT_Utf8);
	}
	c->pu32ConstPool[i] = tempu2;
	c->pbConstTags[i] = Tag;
	break;

      case CONSTANT_String:
	tempu2 = locgetu2();
	assert(tempu2 < c->u16ConstPoolCount);
	c->pu32ConstPool[i] = tempu2;
	c->pbConstTags[i] = Tag;
	break;

      case CONSTANT_Fieldref:     /* these all have the same structure */
      case CONSTANT_Methodref:
      case CONSTANT_InterfaceMethodref:
	c->pbConstTags[i] = Tag;
	tempu2 = locgetu2();
	assert(tempu2 < c->u16ConstPoolCount);
	c->pu32ConstPool[i] = (((uint32) tempu2) << 16);
	tempu2 = locgetu2();
	assert(tempu2 < c->u16ConstPoolCount);
	c->pu32ConstPool[i] |= tempu2;
	break;

      case CONSTANT_NameandType:
	c->pbConstTags[i] = Tag;
	tempu2 = locgetu2();
	assert(tempu2 < c->u16ConstPoolCount);
	c->pu32ConstPool[i] = (((uint32) tempu2) << 16);
	tempu2 = locgetu2();
	assert(tempu2 < c->u16ConstPoolCount);
	c->pu32ConstPool[i] |= tempu2;
	break;

      default:
	panic("Illegal Tag in constant pool: %d (item no.%d)\n", Tag, i);
      } //end switch
    } //end for
  } //end const pool > 0
  else {
    c->pu32ConstPool = NULL;
    
#ifdef CLASSFILE_TRACE
    eprintf("no constant pool");
#endif
    
  }
  /* We need to go through the constant pool and replace the Class
     indices with Asciz pointers */
  for (i = 1; i < c->u16ConstPoolCount; i++) { 
    if (CONSTTAG(c, i) == CONSTANT_Class) {
      c->pu32ConstPool[i] = 
	(uint32) CONSTGET_UidAsciz(c, c->pu32ConstPool[i]);
    }
  }

}


/*
 * Another helper method for ClassCreate, this one processes the
 * superclass and interfaces for the class
 *
 * XXX shouldn't it also check the objects referenced by Exceptions?
 * (or does that come later?)
 *
 * Returns 1 if it couldn't find the superclass
 */
int buildClassInfo(JNIEnv* env, tClassLoaderTuple* tuple, char* pszName, 
		   uint32* pi32FilePos, uint16* pu16ConstRealSize, 
		   int32* pi32ArrayPos, byte* pbArray)
{
  uint16 u16ThisClass; //constant pool ref for this class's name
  uint16 u16SuperClass; //constant pool ref for this class's superclass's name
  int i = 0;
  uint16 tempu2;
  tClass* c = tuple->pstClass;
  
  /* read class access flags */
  c->u16AccessFlags = locgetu2();
  
  /* read this_class and super_class */
  u16ThisClass = locgetu2();
  u16SuperClass = locgetu2();

  // The strings have already been coverted to ascii equivalents
  tuple->uidName = CONSTGET_UidAsciz(c, u16ThisClass);
  if (pszName != NULL && strcmp(tuple->uidName, pszName) != 0) {
    return 2;
  }
  
  /* check if parent is loaded */
  if (u16SuperClass) { 
    /* only do it if it's not "Object", which has no parent */
    traceLoading("buildClassInfo(%s): loading super class %s\n", 
		 tuple->uidName, CONSTGET_UidAsciz(c, u16SuperClass));
    
    c->pstSuperClass = 
      CLASSFILE_FindOrLoad(env, CONSTGET_UidAsciz(c, u16SuperClass), tuple);
    
    if (!(c->pstSuperClass)) {
      return 1;
    }
  }
  else {
    // this is the java.lang.Object class
    c->pstSuperClass = NULL;
  }
      
#ifdef CLASSFILE_TRACE
  eprintf("Class Access: %d\n", c->u16AccessFlags);
  eprintf("This Class: %s\n", tuple->uidName);
  if (c->pstSuperClass) {
    eprintf("Super Class: %s\n", c->pstSuperClass->uidName);
  }
#endif

  /* read interfaces count */
  c->u16InterfacesCount = locgetu2();
  
  /* read interfaces and store their pointers */
  if (c->u16InterfacesCount > 0) {
    c->ppstInterfaces = (tClassLoaderTuple**) 
      sys_malloc(sizeof(tClassLoaderTuple*) * c->u16InterfacesCount);
    assert(c->ppstInterfaces);
    for (i = 0; i < c->u16InterfacesCount; i++) {
      tempu2 = locgetu2();
      /* check if interface is loaded */
      c->ppstInterfaces[i] = 
	CLASSFILE_FindOrLoad(env, CONSTGET_UidAsciz(c, tempu2), tuple);
      if (c->ppstInterfaces[i] == NULL) {
	return 3;
      }
    }
  }
  else {
    c->ppstInterfaces = NULL;
  }
  
  return 0;
}



int readFieldsInfo(JNIEnv* env, tClassLoaderTuple* tuple, uint32* pi32FilePos,
		   uint16* pu16ConstRealSize, int32* pi32ArrayPos, 
		   byte* pbArray) {
  tClass* c = tuple->pstClass;
  uint16 u16NumFields;
  uint32 tempu4;
  uint16 tempu2;
  uint16 u16AttributesCount0;
  tFieldInfo* pstFields = NULL;
  int i = 0, j = 0;
  
  /* read in fields count */
  u16NumFields = locgetu2();
  
#ifdef CLASSFILE_TRACE
  eprintf("Fields: %d\n", u16NumFields);
#endif
  
  /* read in fields */
  if (u16NumFields > 0) {
    pstFields = (tFieldInfo*) sys_malloc(sizeof(tFieldInfo) * u16NumFields);
    if (pstFields == NULL) {
      return -1;
    }
    
    for (i = 0; i < u16NumFields; i++) {
      pstFields[i].u16AccessFlags = locgetu2();
      tempu2 = locgetu2();
      pstFields[i].uidName = CONSTGET_UidAsciz(c, tempu2);
      tempu2 = locgetu2();
      pstFields[i].uidSignature = CONSTGET_UidAsciz(c, tempu2);
      pstFields[i].u16ConstValueIndex = 0; /* clear this here first */
      u16AttributesCount0 = locgetu2();
      
      
#ifdef CLASSFILE_TRACE
      eprintf( "Field no. %d\n", i);
      eprintf( "  access: %d\n", pstFields[i].u16AccessFlags);
      eprintf( "  name: %s", (Uid) pstFields[i].uidName);
      eprintf( "  sig: %s", (Uid) pstFields[i].uidSignature);
      eprintf( "  attr#: %d\n", u16AttributesCount0);
#endif
      
      if (u16AttributesCount0 > 0) {
	for (j = 0; j < u16AttributesCount0; j++) {
	  tempu2 = locgetu2();
#ifdef CLASSFILE_TRACE
	  eprintf("This field has an attribute %s\n", CONSTGET(c, tempu2));
#endif
	  /* this must look to see if it's a ContantValue attribute */
	  if (AscizCmp(c, tempu2,"ConstantValue") == 0) {
	    locgetu4(); //Get the length
	    
	    if (pstFields[i].u16AccessFlags & ACC_STATIC) {
	      pstFields[i].u16ConstValueIndex = locgetu2();
	    }
	    else {
	      pstFields[i].u16ConstValueIndex = 0;
	      locgetu2();
	    }
				
	    /* JP_MOD - ensure that the ACC_STATIC flag is set !!! */
	    
	    // I checked the VMSpec (4.7.2) and it says that
	    // ConstantValue attributes for non-static fields should
	    // be ignored
	    // Why? Jewel -- Why does it have to be static? 
	    // assert(pstFields[i].u16AccessFlags & ACC_STATIC);
	  }
	  else if (AscizCmp(c, tempu2, "Synthetic") == 0) {
	    // generated by a compiler
	    tempu4 = locgetu4(); //get the length
	    locskip(tempu4); /* skip past attribute */
	  }
	  else if (AscizCmp(c, tempu2, "Deprecated") == 0) {
	    tempu4 = locgetu4(); //get the length
	    locskip(tempu4); /* skip past attribute */
	  }
	  else {
	    eprintf("Unknown attribute for field %s\n", 
		    (char*) CONSTGET(c, tempu2));
	    locskip(locgetu4());
	  }
	}
      }
    }
  }
  /* we must do this even if there are no fields so that the field counts
     and similar data will be set to zero */
  if (BuildFieldTables(tuple, pstFields, u16NumFields) != 0) {
    return -1;
  }
  
  if (pstFields) {
    sys_free(pstFields);
  }
  return 0;
}



/* Read attribute sections from class data */

int readAttributesInfo(JNIEnv* env, tClass* c, uint32* pi32FilePos, 
		       uint16* pu16ConstRealSize, int32* pi32ArrayPos, 
		       byte* pbArray) 
{  
  uint16 u16AttributesCount0;
  int i = 0;
  uint16 u16Index1, u16Index2;
  uint16 tempu2;
  uint32 tempu4;
  
  /* read in attributes count ---> "SourceFileAttribute" is the only
     one recognized*/
  u16AttributesCount0 = locgetu2();

#ifdef CLASSFILE_TRACE
  eprintf("\n\nThere are %d Class Attributes\n", u16AttributesCount0);
#endif
  
  /* read in attributes */
  if (u16AttributesCount0 > 0) {
    // eprintf("There are %i attributes\n", u16AttributesCount0);
    for (i = 0; i < u16AttributesCount0; i++) {
      tempu2 = locgetu2();
      
      /* this must check to see if it's a pstSourceFile attribute */
      
#ifdef CLASSFILE_TRACE
      eprintf("Processing attribute %i (%i %s)\n", 
	      i, tempu2, (char *) CONSTGET(c, tempu2));
      eprintf("	Attribute# %d: %s\n", i, CONSTGET(c, tempu2));
#endif
      
      if (AscizCmp(c, tempu2, "SourceFile") == 0) {
	tempu4 = locgetu4();
	assert(tempu4 == 2);
	//Now get the source file index
	c->u16SourceFile = locgetu2();
	
#ifdef CLASSFILE_TRACE
	eprintf("	SourceFile: %s\n", 
		(char *) CONSTGET(c, c->u16SourceFile));
#endif
	
      }
      else if (AscizCmp(c, tempu2, "InnerClasses") == 0) {
	int j;

	tempu4 = locgetu4(); //Attribute length
	c->u16InnerClassesCount = locgetu2();
#ifdef CLASSFILE_TRACE
	eprintf("Inner classes for %s: length is %i, num classes %i\n", 
		(char *) CONSTGET(c, c->u16SourceFile), tempu4, 
		c->u16InnerClassesCount);
#endif
	c->pstInnerClasses = (tInnerClass *)
	  sys_malloc(sizeof(tInnerClass) * c->u16InnerClassesCount);
	assert(c->pstInnerClasses);

	for (j = 0; j < c->u16InnerClassesCount; j++) {
	  // We mustn't try to resolve the inner class at this point 
	  // because it will probably cause infinite recursion.
	  u16Index1 = locgetu2();
	  assert(u16Index1 < c->u16ConstPoolCount);
	  u16Index2 = locgetu2();
	  assert(u16Index2 < c->u16ConstPoolCount);

	  c->pstInnerClasses[j].u16InnerClassInfo = u16Index1;	 
	  c->pstInnerClasses[j].u16OuterClassInfo = u16Index2;
	  locskip(2);  // skip the class name
	  c->pstInnerClasses[j].u16AccessFlags = locgetu2();
	  c->pstInnerClasses[j].pstInnerClass = NULL;	 
	  c->pstInnerClasses[j].pstOuterClass = NULL;
	}
      }
      else if (AscizCmp(c, tempu2, "Deprecated") == 0) {
	tempu4 = locgetu4(); //Attribute length
	locskip(tempu4);
	// XXX maybe print out what is deprecated
      }
      else {
	tempu4 = locgetu4();
	locskip(tempu4);
#ifdef CLASSFILE_TRACE	
	eprintf("Unrecognized attribute, name %s\n", 
		(char *) CONSTGET(c, tempu2));
#endif
      }
    }
  }
  else {
    c->u16SourceFile = 0;
  }
  return 0;
}


/*
 * @doc FUNC
 * @func
 * This function goes through all the fields of the given class and
 * separates them into static variables and instance variables and works
 * out how much space is needed to store each of them. Space is then
 * allocated for the static fields in the class. Space for the instance
 * variables is allocated in the objects when they are created with a
 * 'new' instruction. This function is called on a class that has just
 * been loaded.
 *
 */
int BuildFieldTables (tClassLoaderTuple* tuple,             
		      /* @parm Class to build the tables for */
		      tFieldInfo* pstFields, 
		      /* @parm Pointer to array of field information */
		      uint16 u16NumFields    
		      /* @parm Number of fields in information array */
		      )
{
  tClass* c = tuple->pstClass;
  int i;
  int offset, currfield, field_size = 0, field_index;
  char signature;
  UTILS_stack *pstConstStack, *pstInstStack, *pstStatStack;
  

  /* get out if it's the Object class or there are no fields */
  if ((c->pstSuperClass == NULL) ||
      ((u16NumFields == 0) &&
       (c->pstSuperClass->pstClass->u16InstSize == 0) &&
       (c->pstSuperClass->pstClass->u16StatSize == 0))) {
    c->u16InstSize = 0;
    c->u16InstCount = 0;
    c->pstInstOffsets = NULL;
    c->u16StatSize = 0;
    c->u16StatCount = 0;
    c->pstStatOffsets = NULL;
    
    return 0;
  }
  
  pstConstStack = UTILS_Stack_New(20,10);
  pstInstStack = UTILS_Stack_New(20,10);
  pstStatStack = UTILS_Stack_New(20,10);
  
  /* count amounts for malloc */
  for (i = 0; i < u16NumFields; i++) {
    if (pstFields[i].u16AccessFlags & ACC_STATIC) {
      /* N.B. constants are forced to be static */
      UTILS_Stack_Push(pstStatStack, i);
    }
    else {
      UTILS_Stack_Push(pstInstStack, i);
    }
  }
  
#ifdef CLASSFILE_TRACE
  eprintf("pstFields: %d instance vars, %d statics\n",
	  pstInstStack->count, pstStatStack->count);
#endif
  
  if (pstInstStack->count == 0) {
    c->u16InstSize = c->pstSuperClass->pstClass->u16InstSize;
    c->u16InstCount = 0;
    c->pstInstOffsets = NULL;
  }
  else {
    c->pstInstOffsets = (tField*) sys_malloc(sizeof(tField) *
					     pstInstStack->count);
    //We will increment these later for each field
    c->u16InstSize = c->pstSuperClass->pstClass->u16InstSize; 
    c->u16InstCount = 0; 

    offset = c->u16InstSize;

    for (i = 0; pstInstStack->count > 0; i++) {
      c->pstInstOffsets[i].u16Offset = offset;
      currfield = UTILS_Stack_Pop(pstInstStack);
      c->u16InstCount++;
#ifdef CLASSPERSIST
      c->pstInstOffsets[i].pid = NULL;
#endif
      /* store pointer to string for name */
      c->pstInstOffsets[i].uidFieldName = pstFields[currfield].uidName;
      /* store pointer to sig */
      c->pstInstOffsets[i].uidFieldSig = pstFields[currfield].uidSignature;
      /* store access flags */
      c->pstInstOffsets[i].u16AccessFlags =
	pstFields[currfield].u16AccessFlags;
      /* need to store class pointer */
      c->pstInstOffsets[i].pstClass = tuple;
      c->pstInstOffsets[i].u16ConstValueIndex = 0;
#ifdef RESURRECT_QUICK
      c->pstInstOffsets[i].pi32Addr = 0;
#endif

#ifdef CLASSFILE_TRACE
      eprintf("Instance Field: %s (# %d offset: %d)\n",
	      c->pstInstOffsets[i].uidFieldName, i, offset);
#endif
      
      field_size = 0;
      switch (pstFields[currfield].uidSignature[0]) {
        /* 32 bits or fewer = 1 jint */
      case 'B':
      case 'C':
      case 'F':
      case 'I':
      case 'S':
      case 'L':
      case 'Z':
      case '[':
	field_size = 1;
	break;
	
        /* 64 bits = 2 jints */
      case 'D':
      case 'J':
	field_size = 2;
	break;
	
      default:
#ifdef CLASSFILE_TRACE
	eprintf("Build_Field_Tables error: unknown field signature\n");
#endif
	break;
      }
      offset += field_size;
      c->u16InstSize += field_size;
    }
    
#ifdef CLASSFILE_TRACE
    if (pstInstStack->count > 0) {
      eprintf("WARNING: instance stack not emptied\n");
    }
    eprintf("Instance Size: %d\n\n",c->u16InstSize);
#endif
  } //if there were instance vars

  if (pstStatStack->count == 0) {
    c->u16StatSize = 0; //c->pstSuperClass->pstClass->u16StatSize;
    c->u16StatCount = 0;
    c->pstStatOffsets = NULL;
  }
  else {
    // if there are static vars
    c->pstStatOffsets = (tField*) sys_malloc(sizeof(tField) * 
					     pstStatStack->count);
    if (c->pstStatOffsets == NULL) {
      return -1;
    }
    c->u16StatSize = 0; //c->pstSuperClass->pstClass->u16StatSize;
    c->u16StatCount = 0;
    offset = 0; //c->u16StatSize;
    
    for (i = 0; pstStatStack->count > 0; i++) {
      c->pstStatOffsets[i].u16Offset = offset;
      currfield = UTILS_Stack_Pop(pstStatStack);
      c->u16StatCount++;
      
#ifdef CLASSPERSIST
      c->pstStatOffsets[i].pid = NULL;
#endif
      /* store pointer to string for name */
      c->pstStatOffsets[i].uidFieldName = pstFields[currfield].uidName;
      /* store pointer to sig */
      c->pstStatOffsets[i].uidFieldSig = pstFields[currfield].uidSignature;
      /* store access flags */
      c->pstStatOffsets[i].u16AccessFlags =
	pstFields[currfield].u16AccessFlags;
      /* need to store class pointer */
      c->pstStatOffsets[i].pstClass = tuple; /*c;*/
      c->pstStatOffsets[i].u16ConstValueIndex = 0;
#ifdef RESURRECT_QUICK
      c->pstStatOffsets[i].pi32Addr = 0;
#endif
      
#ifdef CLASSFILE_TRACE
      eprintf("Static Field: %s (# %d offset: %d)\n",
	      c->pstStatOffsets[i].uidFieldName, i, offset);
#endif
      
      /* check for constant value attr */
      if (pstFields[currfield].u16ConstValueIndex) {
	UTILS_Stack_Push(pstConstStack, 
			 pstFields[currfield].u16ConstValueIndex);
	UTILS_Stack_Push(pstConstStack, i);
	c->pstStatOffsets[i].u16ConstValueIndex =
	  pstFields[currfield].u16ConstValueIndex;
#ifdef CLASSFILE_TRACE
        eprintf("CONSTANT FIELD: %s\n", c->pstStatOffsets[i].uidFieldName);
#endif
      }
      
      /* field signature */
      signature = (char) pstFields[currfield].uidSignature[0];
      switch (signature) {
        /* 32 bits or fewer = 1 jint */
      case 'B':
      case 'C':
      case 'F':
      case 'I':
      case 'S':
      case 'L':
      case 'Z':
      case '[':
	field_size = 1;
	break;
	
        /* 64 bits = 2 jints */
      case 'D':
      case 'J':
	field_size = 2;
	break;
	
      default:
#ifdef CLASSFILE_TRACE
	eprintf("Build_Field_Tables error: unknown field signature\n");
#endif
	break;
      }
      
      offset += field_size;
      c->u16StatSize += field_size;
    }
    
#ifdef CLASSFILE_TRACE
    if (pstStatStack->count > 0) {
      eprintf("WARNING: static's stack not emptied\n");
    }
    eprintf("Static Size: %d\n\n",c->u16StatSize);
#endif
    
    /* now that we have allocated space, we can load in constant values */
    /* we do this later for the tuple structure */
    while (pstConstStack->count > 0) {
      /* just empty the stack */
      i = UTILS_Stack_Pop(pstConstStack); // position in pstStatOffsets 
      field_index = UTILS_Stack_Pop(pstConstStack); // position in const pool 
    }  
  }
  
  UTILS_Stack_Destroy(pstConstStack);
  UTILS_Stack_Destroy(pstInstStack);
  UTILS_Stack_Destroy(pstStatStack);
  
  //  printk("BuildFieldTables for %s 5\n", c->uidName);
  return 0;
}


/*
 * @doc FUNC
 * @func
 * Build a virtual table which comprises the superclass's virtual table
 * overwritten and appended with the methods of this class. This is called
 * after reading in the classfile.
 */

static void BuildVirtualTable(tClass* c /* @parm The class whose virtual 
					   table is to be built */)
{
  uint16 u16Method;
  uint16 u16Interface;
  uint16 u16VTMethod;
  tClass* pstSuper;
  uint16 u16VTSize;
  
  sys_mutex_lock(classfileMutex);
  /* get parent's virtual table */
  if (c->pstSuperClass) {
    /* only do it if it's not "Object", which has no parent */
    pstSuper = c->pstSuperClass->pstClass;
    assert(pstSuper);
    u16VTSize = pstSuper->u16VTSize;
  }
  else {
    pstSuper = NULL;
    u16VTSize = 0;
  }
  
  /*
    We make an upper bound estimate of the size of the VT by adding the size
      of the superclass's VT to the total number of methods in this class. This
      is too much in general since:
        1) Some of these methods with the same sig will overwrite parent's
            methods.
        2) private, final and static methods are not put into the VT
   */
  c->ppstVT = (tMethod**) sys_malloc((u16VTSize + c->u16MethodsCount) *
				     sizeof(tMethod*));
  
  /* Now initialise this to the same as the parent's class. */
  if (pstSuper) {
    memcpy(c->ppstVT, pstSuper->ppstVT, 
	   pstSuper->u16VTSize * sizeof(tMethod*));
  }

  /* ... add all of this class's methods to the VT */
  for (u16Method = 0; u16Method < c->u16MethodsCount; u16Method++) {
    /* Only bother if this is an appropriate method.
       XXXX - CHECK THIS WITH LANG DEF*/
    if (!(c->pstMethods[u16Method].u16AccessFlags & ACC_STATIC)) {
      BOOL bFound = FALSE;
      
      /* Try to find this method in the VT */
      for (u16VTMethod = 0; u16VTMethod < u16VTSize; u16VTMethod++) {
        if (!uidcmp(c->pstMethods[u16Method].uidName, 
		    c->ppstVT[u16VTMethod]->uidName) &&
            !uidcmp(c->pstMethods[u16Method].uidSignature, 
		    c->ppstVT[u16VTMethod]->uidSignature)) {
          /* Bingo! We must overwrite the entry with this class's
             overriding method*/
          c->ppstVT[u16VTMethod] = &c->pstMethods[u16Method];
          c->pstMethods[u16Method].u16VTIndex = u16VTMethod;
          bFound = TRUE;
          break;
        }
      }
      
      /* ...otherwise we add this new method to the end of the VT */
      if (!bFound) {
        c->ppstVT[u16VTSize] = &c->pstMethods[u16Method];
        c->pstMethods[u16Method].u16VTIndex = u16VTSize;
        u16VTSize++;
      }
    }
  }
  /* ... shrink the VT */
  c->ppstVT = (tMethod**) sys_realloc(c->ppstVT, u16VTSize * sizeof(tMethod*));

  c->u16VTSize = u16VTSize;

  /* ... and add placeholders VT entries for any interface methods */
  /* Adding these placeholders solves a problem with resolving methods
     in the 'invokevirtual' instruction.  scc - 2003-02-20 */
  for (u16Interface = 0; 
       u16Interface < c->u16InterfacesCount; 
       u16Interface++) {
    AddInterfaceMethods(c, c->ppstInterfaces[u16Interface]->pstClass);
  }
  
  sys_mutex_unlock(classfileMutex);
}


static void AddInterfaceMethods(tClass *c, tClass *i)
{
  uint16 u16Method, u16VTMethod, u16Interface;

  /* ... check that the methods of interface i are in the VT for c */
  for (u16Method = 0; u16Method < i->u16MethodsCount; u16Method++) {
    BOOL bFound = FALSE;
    
    /* Try to find this method in the VT */
    for (u16VTMethod = 0; u16VTMethod < c->u16VTSize; u16VTMethod++) {
      if (!uidcmp(i->pstMethods[u16Method].uidName, 
		  c->ppstVT[u16VTMethod]->uidName) &&
	  !uidcmp(i->pstMethods[u16Method].uidSignature, 
		  c->ppstVT[u16VTMethod]->uidSignature)) {
	/* Bingo! */
	bFound = TRUE;
	break;
      }
    }
    
    /* ...otherwise we add a shadow copy of this interface method to
       the end of the VT */
    if (!bFound) {
      tMethod *shadow = (tMethod*) sys_malloc(sizeof(tMethod));

#ifdef CLASSPERSIST
      shadow->pid = i->pstMethods[u16Method].pid;
#endif
      shadow->u16AccessFlags = i->pstMethods[u16Method].u16AccessFlags;
      shadow->uidName = i->pstMethods[u16Method].uidName;
      shadow->uidSignature = i->pstMethods[u16Method].uidSignature;      
      shadow->pstCode = NULL;   
      shadow->pstClass = i->pstMethods[u16Method].pstClass;   
      shadow->u16ArgSize = i->pstMethods[u16Method].u16ArgSize;    
      shadow->u16RetSize = i->pstMethods[u16Method].u16RetSize; 
   
      shadow->u16NumExceptions = i->pstMethods[u16Method].u16NumExceptions;    
      shadow->pstExceptions = i->pstMethods[u16Method].pstExceptions;   
      shadow->bHaveResolvedExceptions = 
	i->pstMethods[u16Method].bHaveResolvedExceptions;   

      c->ppstVT = sys_realloc(c->ppstVT, 
			      sizeof(tMethod*) * (c->u16VTSize + 1));
      shadow->u16VTIndex = c->u16VTSize;
      c->ppstVT[c->u16VTSize] = shadow;
      c->u16VTSize++;
    }
  }

  /* recurse to deal with i's ancestor interfaces */
  for (u16Interface = 0; 
       u16Interface < i->u16InterfacesCount; 
       u16Interface++) {
    AddInterfaceMethods(c, i->ppstInterfaces[u16Interface]->pstClass);
  }

}


static int ClassHasFinalizer(tClassLoaderTuple *tuple)
{
  int i;
  tClass* c = tuple->pstClass;

  if (c->bHasFinalizer) {
    return 1;
  }

  for (i = 0; i < c->u16MethodsCount; i++) {
    if (strcmp(c->pstMethods[i].uidName, "finalize") == 0 &&
	strcmp(c->pstMethods[i].uidSignature, "()V") == 0 &&
	strcmp(tuple->uidName, pszObjectString) != 0) {
      return 1;
    }
  }

  if (c->pstSuperClass) {
    return ClassHasFinalizer(c->pstSuperClass);
  }
  else {
    // 'java.lang.Object' is not finalizable
    return 0;
  }
}


/*
 * @doc FUNC
 * @func
 * This function creates a tClass structure from the data given in an array
 * of bytes. This data should be in the Java classfile format.
 *
 * @rdesc Pointer to the created class structure.
 *
 * @ex Example of use |
 *
 * tClass* pstClass;
 * byte*   pbData;
 *
 * ... fill pbData with Java class file data ...
 * pstClass = CLASSFILE_ClassCreate(pbData);
 *
 * *errorCode may be one several values on return
 * LOADING_ERROR_NO_SUPERCLASS, LOADING_ERROR_BAD_CLASS_NAME etc
 * (see loading_errors.h)
 *
 */
tClassLoaderTuple* CLASSFILE_ClassCreate(JNIEnv* env,
					 byte* pbArray   
					 /* @parm Array of bytes that
                                            define the class */,
					 char* pszName,  
					 /* Expected name for this class */
					 jobject classLoader, 
					 /* Classloader that is loading this
                                            class (may be null) */
					 uint32* errorCode
					 /* An error code that is filled in error */
					 )
{
  int i;
  uint32 i32FilePos = 0;
  uint16 u16ConstRealSize = 0;
  int32 i32ArrayPos = 0;
  int result;
  tClass*  c;
  tClassLoaderTuple* tuple = CLASSFILE_NewTuple(env, pszName, classLoader);

  traceLoading("CLASSFILE_ClassCreate(%s, %s)", pszName,
	       classLoader ? DEREF(classLoader)->pstType->uidName : "NULL");

  c = tuple->pstClass;
  
  /* read magic number */
  if (getu4() != 0xcafebabeul) {
    eprintf("Class did not have correct magic number, %p\n", pbArray);
    *errorCode = LOADING_ERROR_BAD_MAGIC;
    panic0("bad magic number");
    return NULL;
  }
  
  /* read version number */
  c->u16MinorVersion = getu2();
  c->u16MajorVersion = getu2();
  
  /* read constant pool count */
  c->u16ConstPoolCount = getu2();
  
#ifdef CLASSFILE_TRACE
  eprintf("Constant pool count %d\n", c->u16ConstPoolCount);
#endif
  
  /* Go through constant pool and figure out strings and things */
  
  setupConstantPool(env, c, &i32FilePos, &u16ConstRealSize, 
		    &i32ArrayPos, pbArray);
  
  /* Build the class info (superclass, interfaces etc */
  
  result = buildClassInfo(env, tuple, pszName, &i32FilePos, 
			  &u16ConstRealSize, &i32ArrayPos, pbArray);
  
  if (result == 1) {	
    *errorCode = LOADING_ERROR_NO_SUPERCLASS;
    eprintf("Superclass missing for %s\n", tuple->uidName);
    return NULL; //Something went wrong, superclass missing?
  }
  else if (result == 2) {	
    *errorCode = LOADING_ERROR_BAD_CLASS_NAME;	
    return NULL; //Something went wrong, wrong name in class?
  } 
  else if (result == 3) {	
    *errorCode = LOADING_ERROR_NO_INTERFACE;	
    return NULL; //Something went wrong, interface not found
  }
  else if (result != 0) {
    panic("Unexpected result (%d) from buildClassInfo", result);
  }
  
  // Do the fields, methods and attributes 
  if (readFieldsInfo(env, tuple, &i32FilePos, &u16ConstRealSize,
		     &i32ArrayPos, pbArray) != 0) {
    *errorCode = LOADING_ERROR_BAD_FIELDS;
    return NULL;
  }
  if (readMethodsInfo(env, tuple, &i32FilePos, &u16ConstRealSize, 
		      &i32ArrayPos, pbArray) != 0) {
    *errorCode = LOADING_ERROR_BAD_METHODS;
    return NULL;
  }
  if (readAttributesInfo(env, c, &i32FilePos, &u16ConstRealSize, 
			 &i32ArrayPos, pbArray) != 0) {
    *errorCode = LOADING_ERROR_BAD_ATTRIBUTES;
    return NULL;
  }

  if (ClassHasFinalizer(tuple)) {
    c->bHasFinalizer = 1;
  }

  if (!(c->u16AccessFlags & ACC_INTERFACE)) {
    BuildVirtualTable(c);
  }
  
#ifdef QUICK2
  BuildClassTable(c);
#endif
  
  // deal with native methods 
  
  // find number of native methods 
  c->u16NumNatives = 0;
  for (i = 0; i < c->u16MethodsCount; i++) {
    if (c->pstMethods[i].u16AccessFlags & ACC_NATIVE) {
      c->u16NumNatives += 1;
    }
  }
  
#ifdef DEBUG_NATIVE
  eprintf("There are %i native methods for %s\n",
	  c->u16NumNatives, c->uidName);
#endif
  
  // don't need to do much if there are none 
  if (c->u16NumNatives == 0) {
    c->pstNativeMethods = NULL;
  }

  if (CLASSLOADER_TUPLE_InitializeConstants(tuple) != 0) {
    panic("CLASSLOADER_TUPLE_InitializeConstants failed");
    return NULL;
  }
  return tuple;
}


tClassLoaderTuple* CLASSFILE_ArrayClassCreate(JNIEnv* env, char* pszClassName,
					      tClassLoaderTuple* baseType)
{
  tClassLoaderTuple* tuple = CLASSFILE_NewTuple(env, pszClassName, NULL);

  traceLoading("CLASSFILE_ArrayClassCreate(%s, %s)", 
	       pszClassName, baseType ? baseType->uidName : "NULL");
  tuple->pstClass->pstSuperClass = 
    CLASSFILE_FindOrLoad(env, pszObjectString, NULL);

  if (baseType) {
    tuple->pstClass->u16AccessFlags = baseType->pstClass->u16AccessFlags & 
      (ACC_PUBLIC | ACC_PROTECTED | ACC_PRIVATE);
  }
  else {
    tuple->pstClass->u16AccessFlags = ACC_PUBLIC;
  }
  tuple->pstClass->u16AccessFlags |= ACC_FINAL;

  tuple->pstClass->u16InterfacesCount = 2;
  tuple->pstClass->ppstInterfaces = 
    (tClassLoaderTuple **) sys_malloc(sizeof(tClassLoaderTuple *) * 2);
  tuple->pstClass->ppstInterfaces[0] =
    CLASSFILE_FindOrLoad(env, pszCloneableString, NULL); 
  tuple->pstClass->ppstInterfaces[1] =
    CLASSFILE_FindOrLoad(env, pszSerializableString, NULL);
  
  if (baseType && baseType->classLoader) {
    tuple->classLoader = baseType->classLoader;
  }

  BuildVirtualTable(tuple->pstClass);
  
#ifdef QUICK2
  BuildClassTable(tuple->pstClass);
#endif

  return tuple;
}


/*
 * @doc FUNC
 * @func
 * This function creates and initialiises a tClassLoaderTuple structure
 */
tClassLoaderTuple* CLASSFILE_NewTuple(JNIEnv* env, char* pszName,  
				      jobject classLoader)

{
  tClass* c;
  tClassLoaderTuple* tuple;

  /* allocate memory for class file structure */
  c = (tClass*) sys_malloc(sizeof(tClass));
  assert(c);

  tuple = (tClassLoaderTuple*) sys_malloc(sizeof(tClassLoaderTuple));
  assert(tuple);

  tuple->pstClass = c;
  tuple->uidName = UID_GetUid(pszName);
  tuple->classLoader = classLoader;
  tuple->classObject = NULL;
  tuple->vmClassObject = NULL;
  tuple->pi32StatVars = NULL;
#ifdef DEBUG_OBJECT_FOR_GC
  tuple->magic1 = CMAGIC1;
  tuple->magic2 = CMAGIC2;
#endif

  /* set all the pointers for the tables to NULL beforehand */
  c->pu32ConstPool = NULL;
  c->pbConstTags = NULL;
  c->ppstInterfaces = NULL;
  c->ppstVT = NULL;
  c->u16InterfacesCount = 0;
  c->u16NumNatives = 0;
  c->u16MethodsCount = 0;
  c->u16SourceFile = 0;
  c->u16InstCount = 0;
  c->u16InstSize = 0;
  c->u16StatCount = 0;
  c->u16StatSize = 0;
  c->u16VTSize = 0;
  
  c->pstMethods = NULL;
  c->pstSuperClass = NULL;
  c->pstInstOffsets = NULL;
  c->pstNativeMethods = NULL;
  c->pstStatOffsets = NULL;
  
  c->u16InnerClassesCount = 0;
  c->pstInnerClasses = NULL;
  
#ifdef PERSIST
  c->persIndex = (int32) NULL;
#endif
  
  /* set class flags */
  c->bInitialised = 0;
  c->bHasFinalizer = 0;
  c->bIsInnerClass = 0;
  
  /* add the tuple to the CPLIST, even thogh the tClass hasn't been
     properly initialised yet! */
  traceLoading("Adding %s %p (%p) to the CPLIST", 
	       tuple->uidName, tuple, tuple->classLoader);
  CPTUPLELIST_Insert(tuple);
  
  assert(CPTUPLELIST_Find(tuple->uidName, tuple->classLoader) != NULL);
  
  return tuple;
}

