/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part
 * of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "config.h"

/*
 * @doc MODULE
 * @module classfil.c |
 *
 * This file defines the functions that are needed to load class files from
 * disk.  There are also functions that are used for finding a method or a
 * field in a class.
 *
 */ 

/*
 * @doc TOPIC
 * @topic Class loading |
 *
 * Classes need to be loaded from class files before they can be used by the
 * interpreter. A class is loaded from a file by the CLASSFILE_ClassLoad()
 * function. In order for a class to be useful once it is loaded, it needs
 * to be added to the class list. When a class is needed, it should be found
 * by calling CLASSFILE_FindOrLoad(). This will either find the class in the
 * class list or, if it is not in the list, load the class in with
 * CLASSFILE_LoadAndLink.
 *
 * A sequence of things happens when classes are loaded. First, a check is
 * done to see if the file really is a class file. The next step is to load
 * the constant pool. Once the constant pool is loaded, the class's superclass
 * can be checked to see if it has been loaded. If the class's superclass has
 * not been loaded then the function exits with an error message.
 *
 * After that, the class's interface references are loaded. Each interface
 * reference is checked to see if the interface class is loaded. The next
 * items to be loaded are the class's fields. The field information is loaded
 * into a temporary array. This temporary information is then passed to the
 * BuildFieldTables() function. The fields are sorted into static and instance
 * fields and stored in the class structure.
 *
 * Next the class's method information is loaded. If a method has exceptions
 * then a check is done to see if the relevant exception classes are loaded. If
 * they are not loaded then the procedure exits with an error message. When the
 * methods have been loaded, the virtual table is constructed. If quick2
 * instructions are being used then the class table is constructed as well.
 *
 * All the above is automatically done during the class loading process. After
 * a class has been loaded, it is initialised. This involves running its
 * \<clinit\> method if it has one and also initialising its native methods.
 *
 */

#include "vm/wrappers.h"

tMutex* classfileMutex;
extern tMutex* classTupleMutex;

/* Just sets up a mutex */
int CLASSFILE_Init()
{
  classfileMutex = sys_mutex_create(1); //recursive
  classTupleMutex = sys_mutex_create(1); //recursive
  CLASSLOADER_TUPLE_Init();
  return 0;
}

#ifdef KISSME_LINUX_USER
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <string.h>
#endif



#include "vm/uid.h"
#include "vm/classfil.h"
#include "vm/jni.h"
#include "vm/jni_data.h"
#include "vm/classfile_methods.h"
#include "vm/classfile_internal.h"
#include "vm/disass.h"
#include "vm/jutils.h"
#include "vm/cptuplelist.h"
#include "vm/classloader_tuple.h"
#include "vm/interp.h"
#include "vm/interp_methods.h"
#include "vm/global.h"

//Constants for errors returned when a class is loaded
#include "vm/loading_errors.h" 

#include "vm/natives.h"
#ifdef PERSIST
#include "vm/rpot.h"
#endif


#include "vm/jni.h"

// this can be globally used to get the type for object
extern tClassLoaderTuple* pstObjectType;
// this can be globally used to get the type for string 
extern tClassLoaderTuple* pstStringType;
// this can be globally used to get the type for class
extern tClassLoaderTuple* pstClassType; 



tClassLoaderTuple* CLASSFILE_LoadAndLink(JNIEnv* env, char* pszClassName,
					 uint32* errorCode, 
					 tClassLoaderTuple* loadingClass);


/* local prototypes ********************************************************/


#ifdef QUICK2
static void BuildClassTable(tClass* c);
#endif

#ifndef TEASEME
tClassPath CLASSFILE_BootClassPath;
tClassPath CLASSFILE_UserClassPath;
#endif


#include "vm/hash.h"

extern HASH_tstHashTable UID_sstUidHashTable; 

/*
 * We don't resolve a method's exceptions until something accesses
 * them. In one case, the Method reflection code
 */
int CLASSFILE_ResolveMethodExceptions(JNIEnv* env,
				      tMethod* method         
				      /* @parm Method whose caught exceptions
					 we want to resolve */
				      )
{
  tClassLoaderTuple *tuple = method->pstClass;
  sys_mutex_lock(classfileMutex);
  if (method->bHaveResolvedExceptions == 0) {
    int i;
    for (i = 0; i < method->u16NumExceptions; i++) {
      tClassLoaderTuple* ret;
      uint16 poolIndex = ((int) (method->pstExceptions[i])) & 0xffff;
      
      assert(CONSTTAG(tuple->pstClass, poolIndex) == CONSTANT_Class);
      /* We don't call ResolveClass here because we might still fail 
	 CLASSFILE_ResolveClass(env, tuple, poolIndex);	
				
	 We check if it is resolved already
      */
      if (CONSTTAG_GetRes(tuple->pstClass, poolIndex)) {
	ret = (tClassLoaderTuple*) CONSTGET(tuple->pstClass, poolIndex);
      }
      else {
	ret = CLASSFILE_FindOrLoad(env, 
				   CONSTGET_UidAsciz(tuple->pstClass,
						     poolIndex),
				   tuple);
      }
      
      if (ret == NULL) {
	(*env)->ThrowNew(env, 
			 (*env)->FindClass(env, 
					   "java/lang/NoClassDefFoundError"), 
			 CONSTGET_UidAsciz(tuple->pstClass, poolIndex));
	return -1;
      }
    }
    
    for (i = 0; i < method->u16NumExceptions; i++) {
      tClassLoaderTuple* ret;
      int poolIndex;
      assert(CONSTTAG(tuple->pstClass, 
		      (int32) method->pstExceptions[i]) == CONSTANT_Class);
      
      poolIndex = (int) method->pstExceptions[i];
      if (CONSTTAG_GetRes(tuple->pstClass, poolIndex)) {
	ret = (tClassLoaderTuple*) CONSTGET(tuple->pstClass, poolIndex);
      }
      else {
	ret = CLASSFILE_FindOrLoad(env, 
				   CONSTGET_UidAsciz(tuple->pstClass,
						     poolIndex), 
				   tuple);
      }
      method->pstExceptions[i] = ret;
    }
    method->bHaveResolvedExceptions = 1;
  }
  
  sys_mutex_unlock(classfileMutex);
  return 0;
  }
  

/*
 * @doc FUNC
 * @func
 * This function is used to resolve a class reference in a class's constant
 * pool.
 */
tClassLoaderTuple* CLASSFILE_ResolveClass(JNIEnv* env,
					  tClassLoaderTuple* tuple,      
					  /* @parm Class whose constant pool 
					     we want to resolve */
					  uint16 u16ClassIndex   
					  /* @parm Index of the class 
					     reference to resolve */
					  )
{
  tClass* c = tuple->pstClass;
  tClassLoaderTuple* pstTempClass;
  uint32 errorCode = 0;
  Uid pszClassName;
  
  sys_mutex_lock(classfileMutex);
#ifdef HASH_INTEGRITY_CHECK
  HASH_IntegrityCheck(&UID_sstUidHashTable);
#endif
  assert(CONSTTAG(c, u16ClassIndex) == CONSTANT_Class);
  
  /* Check that this entry has not already been resolved */
  if (CONSTTAG_GetRes(c, u16ClassIndex)) {
    tClassLoaderTuple* ret = (tClassLoaderTuple*) CONSTGET(c, u16ClassIndex);
    sys_mutex_unlock(classfileMutex);
    return ret;
  }
 
  /* Even though the class reference isn't resolved, the pool index for the
     class name should have been replaced by a pointer to the corresponding
     Uid.  This is done by setupConstantPool(). */
  pszClassName = CONSTGET_UidAsciz(c, u16ClassIndex);
#ifdef LOADING_VERBOSE
  traceLoading("ResolveClass %s", pszClassName);
#endif
  
  pstTempClass = 
    CLASSFILE_FindOrLoadWithError(env, pszClassName, tuple, &errorCode);
  /* Check that it was found */
  if (!pstTempClass) {
    if (CONSTTAG_GetRes(c, u16ClassIndex)) {
      panic("Class reference for %s was already resolved", pszClassName);
    }
    else {
      nonfatalError("CLASSFILE_ResolveClass: cannot find class %s "
                    "referenced from %s", 
		    pszClassName, tuple->uidName);
    }
    return NULL;
  }

  CONSTSET(c, u16ClassIndex, pstTempClass);
  CONSTTAG_SetRes(c, u16ClassIndex);
  sys_mutex_unlock(classfileMutex);
  return pstTempClass;
}


/*
 * @doc FUNC
 * @func
 * This function is used by CLASSFILE_ResolveFieldref to find a
 * (static) field in an interface and its superinterfaces.
 */
static tField* ResolveFieldrefHelper(JNIEnv* env,
				     tClassLoaderTuple* tuple,
				     Uid uidFieldName)
{
  int i;
  tClass *tempClass = tuple->pstClass;
  
  for (i = 0; i < tempClass->u16StatCount; i++) {
    if (uidcmp(uidFieldName,
	       tempClass->pstStatOffsets[i].uidFieldName) == 0) {
      return tempClass->pstStatOffsets + i;
    }
  }

  for (i = 0; i < tempClass->u16InterfacesCount; i++) {
    tField *field = ResolveFieldrefHelper(env, tempClass->ppstInterfaces[i],
					  uidFieldName);
    if (field != NULL) {
      return field;
    }
  }
  return NULL;
}


/*
 * @doc FUNC
 * @func
 * This function is used to resolve a field reference in a class's constant
 * pool.
 */
tField* CLASSFILE_ResolveFieldref(JNIEnv* env,
				  tClassLoaderTuple* tuple,       
				  /* @parm Class whose constant pool we 
				     want to resolve */
				  uint16 u16Index,
				  /* @parm Index of the field reference 
				     to resolve */
				  int iType 
				  /* @field 1 means static, 2 means instance, 
				     0 means either */
				  )
{
  tClass* c = tuple->pstClass;
  tClassLoaderTuple* tempTuple;
  uint16 u16NTIndex;
  Uid uidFieldName;
  
  sys_mutex_lock(classfileMutex);
  
  
  /* Check that this entry has not already been resolved, another
     thread could do it in between the check and getting here */
  if (CONSTTAG_GetRes(c, u16Index)) {
    tField* ret = (tField*) CONSTGET(c, u16Index);
    sys_mutex_unlock(classfileMutex);
    return ret;
  }
  assert(CONSTTAG(c, u16Index) == CONSTANT_Fieldref);
  
  tempTuple = 
    CONSTGET_Class(env, tuple, (CONSTGET(c, u16Index) & 0xffff0000UL) >> 16);

  /* Check that it was found */  
  if (tempTuple == NULL) {
    sys_mutex_unlock(classfileMutex);
    return NULL;
  }
  
  assert(tempTuple);
  u16NTIndex = CONSTGET(c, u16Index) & 0xffff;
  uidFieldName = 
    CONSTGET_UidAsciz(c, (CONSTGET(c, u16NTIndex) & 0xffff0000UL) >> 16);

#ifdef DEBUG
  if (!(iType == 0 || iType == 1 || iType == 2)) {
    panic("CLASSFILE_ResolveFieldRef: bad type value (%d)", iType);
  }
#endif
  
  //Search recursively upwards through the class hierarchy
  while (tempTuple != NULL) {
    tClass* tempClass = tempTuple->pstClass;

    if (iType == 0 || iType == 1) {
      int i;
      /* search static variables */
      for (i = 0; i < tempClass->u16StatCount; i++) {
	if (uidcmp(uidFieldName,
		   tempClass->pstStatOffsets[i].uidFieldName) == 0) {
	  CONSTSET(c, u16Index, tempClass->pstStatOffsets + i);
	  CONSTTAG_SetRes(c, u16Index);
	  sys_mutex_unlock(classfileMutex);
	  return tempClass->pstStatOffsets + i;
	}
      }
      /* recursively search this class's interfaces */
      for (i = 0; i < tempClass->u16InterfacesCount; i++) {
	CONSTSET(c, u16Index, 
		 ResolveFieldrefHelper(env, tempClass->ppstInterfaces[i],
				       uidFieldName));
	if (CONSTGET(c, u16Index) != 0) {
	  CONSTTAG_SetRes(c, u16Index);
	  sys_mutex_unlock(classfileMutex);
	  return (tField *) CONSTGET(c, u16Index);
	}
      }
    }
    
    if (iType == 0 || iType == 2) {
      int i;
      /* search instance variables */
      for (i = 0; i < tempClass->u16InstCount; i++) {
	if (uidcmp(uidFieldName, 
		   tempClass->pstInstOffsets[i].uidFieldName) == 0) {
	  CONSTSET(c, u16Index, tempClass->pstInstOffsets + i);
	  CONSTTAG_SetRes(c, u16Index);
	  sys_mutex_unlock(classfileMutex);
	  return tempClass->pstInstOffsets + i;
	}
      }
    }
    
    tempTuple = tempClass->pstSuperClass;
  }

  traceLoading("CLASSFILE_ResolveFieldRef: %sfield %s.%s not found", 
	       (iType == 0) ? "" : ((iType == 1) ? "static " : "instance "),
	       tuple->uidName, uidFieldName);
  
  sys_mutex_unlock(classfileMutex);
  return NULL;
}



/*
 * @doc FUNC
 * @func
 * This function is used to resolve a method reference in a class's 
 * constant pool.
 */
tMethod* CLASSFILE_ResolveMethodref(JNIEnv* env,
				    tClassLoaderTuple* tuple, 
				    /* @parm Class whose constant 
				       pool we want to resolve */
				    uint16 u16Index,
				    /* @parm Index of the method
				       reference to resolve */
				    int searchVTable
				    /* @param If non-zero, search 
				       the VTable */
				    )
{
  tClass* c = tuple->pstClass;
  uint16 u16ClassIndex;
  uint16 u16NTIndex;
  uint16 u16VTIndex;
  uint16 u16SigIndex;
  uint16 u16NameIndex;
  Uid    uidName;
  Uid    uidSig;
  tClassLoaderTuple* pstClassTemp, *pstOrigClass;
  tMethod* pstMethodTemp; 

  /* Quickly test to see if this index has already been resolved. */
  if (CONSTTAG_GetRes(c, u16Index)) {
    return (tMethod*) CONSTGET(c, u16Index);
  }

  sys_mutex_lock(classfileMutex);

  /* Retest inside the mutex ... */
  if (CONSTTAG_GetRes(c, u16Index)) {
    tMethod* ret = (tMethod*) CONSTGET(c, u16Index);
    sys_mutex_unlock(classfileMutex);
    return ret;
  }
  assert(CONSTTAG_GetRes(c, u16Index) == 0);

  /* get the class name reference */
  u16ClassIndex = (CONSTGET(c, u16Index) & 0xffff0000UL) >> 16;
  /* get the method name_and_signature index */
  u16NTIndex = CONSTGET(c, u16Index) & 0x0000ffffUL;
  
  assert(CONSTTAG(c, u16ClassIndex) == CONSTANT_Class);
  assert(CONSTTAG(c, u16NTIndex) == CONSTANT_NameandType);

  /* for method name */
  u16NameIndex = (CONSTGET(c, u16NTIndex) & 0xffff0000UL) >> 16;
  /* for method signature */
  u16SigIndex = CONSTGET(c, u16NTIndex) & 0x0000ffffUL;
  
  assert(CONSTTAG(c, u16NameIndex) == CONSTANT_Utf8);
  assert(CONSTTAG(c, u16SigIndex) == CONSTANT_Utf8);

  /* get method signature */
  uidSig = CONSTGET_UidAsciz(c, u16SigIndex);

  /* get method name */
  uidName = CONSTGET_UidAsciz(c, u16NameIndex);

  /* get the method's class pointer */
  pstOrigClass = CONSTGET_Class(env, tuple, u16ClassIndex);
  pstMethodTemp = NULL;
  
  if (searchVTable) {
    tClass* cc = pstOrigClass->pstClass;
    assert(cc->ppstVT);
    assert(cc->u16VTSize);
    for (u16VTIndex = 0; u16VTIndex < cc->u16VTSize; u16VTIndex++) {
      if (uidcmp(cc->ppstVT[u16VTIndex]->uidName, uidName) == 0 &&
	  uidcmp(cc->ppstVT[u16VTIndex]->uidSignature, uidSig) == 0) {
	pstMethodTemp = cc->ppstVT[u16VTIndex];
	break;
      }
    }
  }
  else {
    pstClassTemp = pstOrigClass;
    
    /* search the class and its superclasses for the method. */
    while (pstClassTemp != NULL) {
      pstMethodTemp = CLASSFILE_UidFindMethod(env, pstClassTemp, 
					      uidName, uidSig);
      if (pstMethodTemp) {
	break;
      }
      pstClassTemp = pstClassTemp->pstClass->pstSuperClass;
    }
  }
  
  if (!pstMethodTemp) {
    tOBREF pstExOb;
    char errMessage[4096];
    
    pstExOb = JNI_getJNIData(sys_current_thread())->pstException;
    if (pstExOb) {
      sys_mutex_unlock(classfileMutex);
      return NULL;
    }
    //throw an exception
#ifdef KISSME_LINUX_USER
    if (pstOrigClass) {
      traceLoading("Could not find method %s%s in class %s (or its ancestors)",
		   uidName, uidSig, pstOrigClass->uidName);
      sprintf(errMessage, 
	      "Could not find method %s%s in class %s (or its ancestors)", 
	      uidName, uidSig, pstOrigClass->uidName);
    }
    else {
      sprintf(errMessage, 
	      "Could not find method %s%s in unknown class",
	      uidName, uidSig);
    }
#else
    strcpy(errMessage, 
	   "Error encountered looking for method in class");
#endif
    traceLoading("CLASSFILE_ResolveMethodref: %s", errMessage);
    (*env)->Throw(env, 
		  INTERP_ExceptionObjectFromNameAndMessage(env,
							   "java/lang/NoSuchMethodError", 
							   errMessage));
    sys_mutex_unlock(classfileMutex);
    return NULL;
  }
  
  CONSTSET(c, u16Index, pstMethodTemp);
  CONSTTAG_SetRes(c, u16Index);
  sys_mutex_unlock(classfileMutex);
  return pstMethodTemp;
}


/*
 * @doc FUNC
 * @func
 * This function is used to resolve an interface method reference in a class's
 * constant pool.
 */
tMethod* CLASSFILE_ResolveMethodrefInterface(JNIEnv* env,
					     tClassLoaderTuple* tuple,
					     /* @parm Class whose constant
						pool we want to resolve */
					     uint16 u16Index, 
					     /* @parm Index of the field
						reference to resolve */
					     int32*  optop,
					     /* @parm Pointer to top of 
						operand stack */
					     uint16  u16ArgLen 
					     /* @parm Argument length */
					     )
{
  tClass* c = tuple->pstClass;
  uint16 u16ClassIndex;
  uint16 u16NTIndex;
  uint16 u16SigIndex;
  uint16 u16NameIndex;
  Uid    uidName;
  Uid    uidSig;
  tClassLoaderTuple* pstClassTemp;
  tMethod* pstMethodTemp;

#ifdef TEASEME
  tClassLoaderTuple* origClass;
  int iMarkAsIPC;
#endif

  sys_mutex_lock(classfileMutex);

#ifdef TEASEME
  iMarkAsIPC = 0;
#endif

  assert(CONSTTAG_GetRes(c, u16Index) == 0);

  /* get the class name reference */
  u16ClassIndex = (CONSTGET(c, u16Index) & 0xffff0000UL) >> 16;
  /* get the method name_and_signature index */
  u16NTIndex = CONSTGET(c, u16Index) & 0x0000ffffUL;

#ifdef TEASEME
  origClass = CONSTGET_Class(env, tuple, u16ClassIndex);
  eprintf("origClass is %s\n", origClass->uidName);
  if (strstr(origClass->uidName, "teaseme/system/ipc_interfaces") != NULL) {
    iMarkAsIPC = 1;
  }
#endif

  assert(CONSTTAG(c, u16ClassIndex) == CONSTANT_Class);
  assert(CONSTTAG(c, u16NTIndex) == CONSTANT_NameandType);
  
  /* for method name */
  u16NameIndex = (CONSTGET(c, u16NTIndex) & 0xffff0000UL) >> 16;
  /* for method signature */
  u16SigIndex = CONSTGET(c, u16NTIndex) & 0x0000ffffUL;

#ifdef DEBUG
  assert(CONSTTAG(c, u16NameIndex) == CONSTANT_Utf8);
  assert(CONSTTAG(c, u16SigIndex) == CONSTANT_Utf8);
#endif

  /* get name */
  uidName = CONSTGET_UidAsciz(c, u16NameIndex);
  /* get sig */
  uidSig = CONSTGET_UidAsciz(c, u16SigIndex);

  /* retrieve class pointer through handle on stack */
  assert(*(optop + 1 - u16ArgLen) != 0);
  pstClassTemp = DEREF((tOBREF) *(optop + 1 - u16ArgLen))->pstType; 
  pstMethodTemp = NULL;

#ifdef JP_DEBUG_TRACE
  printf("Invoke Interface on:  %s\n", pstClassTemp->uidName);
#endif

  /* recursive search from current class up its hierachy */
  while (pstMethodTemp == NULL && pstClassTemp != NULL) {
    pstMethodTemp = CLASSFILE_UidFindMethod(env, pstClassTemp, 
					    uidName, uidSig);
    if (pstMethodTemp == NULL) {
      /* check if we've still got more superclasses to check */
      if (pstClassTemp->pstClass->pstSuperClass == NULL) {
#ifdef DEBUG
        panic0("InvokeInterface: method reference not resolved");
#endif
        break;
      }
      /* go to superclass */
      pstClassTemp = pstClassTemp->pstClass->pstSuperClass;
    }
  }

  /* don't resolve for interfaces */
  CONSTSET(c, u16Index, pstMethodTemp);
  CONSTTAG_SetRes(c, u16Index);

#ifdef TEASEME
  if (iMarkAsIPC) {
    pstMethodTemp->u16AccessFlags |= ACC_IPC_METHOD;
  }
  eprintf("Returning method %p\n", pstMethodTemp);
#endif
  
  sys_mutex_unlock(classfileMutex);
  return pstMethodTemp;
}


/*
 * @doc FUNC
 * @func
 * This function takes a pointer to a tClass struct and a field name. The
 * function returns the field information of the specified instance field.
 *
 * @rdesc Returns a pointer to the specified field's tField information
 *        structure.
 *
 * @ex Example of use. pstClass is assumed to point to an existing class |
 *
 * tField* pstFieldInfo;
 *
 * pstFieldInfo = CLASSFILE_InstFieldOffset(pstClass, "myfield");
 */

tField* CLASSFILE_InstFieldOffset(JNIEnv* env,	
				  tClassLoaderTuple* tuple,       
				  /* @parm Pointer to the class the field 
				     is in */
				  char* pszFieldName    
				  /* @parm Name of the field to look for */
				  )
{
  Uid uidFieldName = UID_GetUid(pszFieldName);
  
#ifdef UID_WARNINGS
  if(uidFieldName == pszFieldName) {
    eprintf("Alert - CLASSFILE_StatFieldOffset called with Uid %s\n", 
	    pszFieldName);
  }
#endif /*UID_WARNINGS*/

  return CLASSFILE_UidInstFieldOffset(env, tuple, uidFieldName);
}


/*
 * @doc FUNC
 * @func
 * This function takes a pointer to a tClass struct and a field name. The
 * function returns the field information of the specified instance field. The
 * field name must be a Uid, so we can do quick matching.
 *
 * @rdesc Returns a pointer to the specified field's tField information
 *        structure.
 *
 * @ex Example of use. pstClass is assumed to point to an existing class and
 *     uidName is assumed to point to a Uid string |
 *
 * tField* pstFieldInfo;
 *
 * pstFieldInfo = CLASSFILE_UidInstFieldOffset(env, pstClass, uidName);
 */

tField* CLASSFILE_UidInstFieldOffset(JNIEnv* env,
				     tClassLoaderTuple* tuple,
				     /* @parm Pointer to the class the field
					is in */
				     Uid uidFieldName
				     /* @parm Name of the field to look for */
				     )
{
  int i;
  tClassLoaderTuple* currentClass;
  
  for (currentClass = tuple;
       currentClass; 
       currentClass = currentClass->pstClass->pstSuperClass) {
    for (i = 0; i < currentClass->pstClass->u16InstCount; i++) {
      if (uidcmp(currentClass->pstClass->pstInstOffsets[i].uidFieldName, 
		 uidFieldName) == 0) {
        return currentClass->pstClass->pstInstOffsets + i;
      }
    }
  }

  return NULL;
}


/*
 * @doc FUNC
 * @func
 * This function takes a pointer to a tClass struct and a field name. The
 * function returns the field information of the specified static field.
 *
 * @rdesc Returns a pointer to the specified field's tField information
 *        structure.
 *
 * @ex Example of use. pstClass is assumed to point to an existing class |
 *
 * tField* pstFieldInfo;
 *
 * pstFieldInfo = CLASSFILE_StatFieldOffset(pstClass, "mystaticfield");
 */
tField* CLASSFILE_StatFieldOffset(JNIEnv* env,
				  tClassLoaderTuple* tuple,
				  /* @parm Pointer to the class the field
				     is in */
				  char* pszFieldName
				  /* @parm Name of the field to look for */
				  )
{
  Uid uidFieldName = UID_GetUid(pszFieldName);
  
#ifdef UID_WARNINGS
  if (uidFieldName == pszFieldName) {
    eprintf("Alert - CLASSFILE_StatFieldOffset called with Uid %s\n",
	    pszFieldName);
  }
#endif /*UID_WARNINGS*/
  
  return CLASSFILE_UidStatFieldOffset(env, tuple, uidFieldName);
}


/*
 * @doc FUNC
 * @func
 * This function takes a pointer to a tClassLoaderTuple and a field name. The
 * function returns the field information of the specified static field.  The
 * field name must be a Uid, so we can do quick matching.
 *
 * @rdesc Returns a pointer to the specified field's tField information
 *        structure.
 *
 * @ex Example of use. pstClass is assumed to point to an existing class and
 *     uidName is assumed to point to a Uid string |
 *
 * tField* pstFieldInfo;
 *
 * pstFieldInfo = CLASSFILE_UidStatFieldOffset(pstClass, uidName);
 */
tField* CLASSFILE_UidStatFieldOffset(JNIEnv* env,
				     tClassLoaderTuple* tuple,          
				     /* @parm Pointer to the class the
					field is in */
				     Uid uidFieldName
				     /* @parm Name of the field to look for */
				     )
{
  int i;
  
  for (i = 0; i < tuple->pstClass->u16StatCount; i++) {
    if (uidcmp(tuple->pstClass->pstStatOffsets[i].uidFieldName,
	       uidFieldName) == 0) {
      return tuple->pstClass->pstStatOffsets + i;
    }
  }  
  return 0;
}


/*
 * @doc FUNC
 * @func
 * This function takes a class name and a field name. The function returns
 * the offset of the specified field from the beginning of the instance
 * field memory. This is used by native methods as a shortcut for searching
 * for field offsets
 *
 * @rdesc Returns the offset of the specified field from the beginning of
 *        an object of this class's instance field memory.
 */
int CLASSFILE_NativeInstOffset(JNIEnv* env,
			       tClassLoaderTuple* loadingClass,
			       char* pszClassName,     /* @parm Class Name */
			       char* pszFieldName      /* @parm Field Name */
			       )
{
  tField* pstFieldOffset;
  tClassLoaderTuple* pstTempClass = NULL;
  
  pstTempClass = CLASSFILE_FindOrLoad(env, pszClassName, loadingClass);
  assert(pstTempClass);

  /* find the offset */
  pstFieldOffset = CLASSFILE_InstFieldOffset(env, pstTempClass,pszFieldName);
  assert(pstFieldOffset);

  return pstFieldOffset->u16Offset;
}


/*
 * @doc FUNC
 * @func
 * This function takes a class name and a field name. The
 * function returns the offset of the specified field from the beginning of
 * the static field memory. This is used by native methods as a shortcut
 * for searching for field offsets
 *
 * @rdesc Returns the offset of the specified field from the beginning of
 *        the class's static field memory.
 */
int CLASSFILE_NativeStatOffset(JNIEnv* env,
			       tClassLoaderTuple* loadingClass,
			       char* pszClassName,   /* @parm Class Name */
			       char* pszFieldName    /* @parm Field Name */
			       )
{
  tField* pstFieldOffset;

  tClassLoaderTuple* pstTempClass =  NULL;
#ifdef LOADING_VERBOSE
  traceLoading("NativeStatOffset %s", pszClassName);
#endif
  pstTempClass = CLASSFILE_FindOrLoad(env, pszClassName, loadingClass);  
  assert(pstTempClass);

  /* find the offset */
  pstFieldOffset = CLASSFILE_StatFieldOffset(env, pstTempClass, pszFieldName);
  assert(pstFieldOffset);

  return pstFieldOffset->u16Offset;
}

/*
 * @doc FUNC
 * @func
 * This function takes a class pointer and a field name. The function returns
 * the offset of the specified field from the beginning of the instance
 * field memory. This is used by native methods as a shortcut for searching
 * for field offsets
 *
 * @rdesc Returns the offset of the specified field from the beginning of
 *        an object of this class's instance field memory.
 */
int CLASSFILE_NativeInstOffsetPtr(JNIEnv* env,
				  tClassLoaderTuple* pstClass,    
				  /* @parm Class pointer */
				  char*   pszFieldName 
				  /* @parm Field Name */
				  )
{
  tField* pstFieldOffset;
  
  /* find the offset */
  pstFieldOffset = CLASSFILE_InstFieldOffset(env, pstClass, pszFieldName);
  assert(pstFieldOffset);

  return pstFieldOffset->u16Offset;
}


/*
 * @doc FUNC
 * @func
 * This function takes a class pointer and a field name. The function returns
 * the offset of the specified field from the beginning of the static field
 * memory. This is used by native methods as a shortcut for searching for
 * field offsets
 *
 * @rdesc Returns the offset of the specified field from the beginning of
 *        an object of this class's instance field memory.
 */
int CLASSFILE_NativeStatOffsetPtr(JNIEnv* env,
				  tClassLoaderTuple* pstClass,   
				  /* @parm Class pointer */
				  char*   pszFieldName  
				  /* @parm Field Name */
				  )
{
  tField* pstFieldOffset;
  
  /* find the offset */
  pstFieldOffset = CLASSFILE_StatFieldOffset(env,pstClass, pszFieldName);
  assert(pstFieldOffset);

  return pstFieldOffset->u16Offset;
}


/*
 * @doc FUNC
 * @func
 * This function parses a method signature and returns the number of 32-bit
 * slots that the method's arguments will take up. A flag is passed to say
 * whether the method is static or not, since non static methods need an
 * extra slot for the "current object".  This function shouldn't need to
 * be called very often as it is run at class load time and used to store
 * the argument sizes of all methods in their tMethod structures.
 *
 * @rdesc Returns number of 32-bit slots that the method's arguments will use.
 *
 * @ex Example of use. pstMethod is assumed to point to a tMethod structure
 *     containing a static method. This example is quite pointless actually,
 *     since the argument size is stored in the tMethod structure as well;
 *     it's just used for demonstration. |
 *
 * uint16 u16ArgSize;
 *
 * u16ArgSize = CLASSFILE_ParseArgumentLength(pstMethod->uidSignature, 1);
 */
uint16 CLASSFILE_ParseArgumentLength(char* pszSignature,  
				     /* @parm Signature string to parse */
				     int   iIsStatic      
				     /* @parm 0 for non-static, 1 for static */
				     )
{
  int pos,len;
  uint16 jint_count=0;
  byte *Bytes;
  
  if (iIsStatic == 0) {/* i.e. not a static method */
    jint_count++;    /* i.e. first local var is handle to object */
  }

  assert(pszSignature);
  
#ifdef CLASSFILE_TRACE
  eprintf("Parsing: %s\n", pszSignature);
#endif

  len = strlen(pszSignature);
  Bytes = pszSignature;
  for (pos = 0; pos < len; pos++) {
    switch (Bytes[pos]) {
    case '(':
      break;

    case ')':
      goto LabelLoopEnd;  /* force a jump out of the loop */

    case 'B':
    case 'C':
    case 'F':
    case 'S':
    case 'Z':
    case 'I':
      jint_count++;
      break;

    case 'D':
    case 'J':
      jint_count += 2;
      break;

    case 'L':
      jint_count++;
      for (; Bytes[pos] != ';' && pos < len; pos++) { /* */ }
      break;

    case '[':
      jint_count++;
      while ((Bytes[pos] == '[' || 
	      (Bytes[pos] >= '0' && Bytes[pos] <= '9')) &&
	     pos < len) { /* multiple arrays with sizes*/
	pos++;
      }
      if (Bytes[pos] == 'L') { /* object signature */
	while (Bytes[pos] != ';') {
	  pos++;
	}
      }
      break;

#ifdef DEBUG
    default:
      eprintf("Error: Parse_Argument_Number: illegal signature");
#endif
    }
  }
  
 LabelLoopEnd:
#ifdef CLASSFILE_TRACE
  eprintf("int32 argument len: %d\n",jint_count);
#endif
  
  return jint_count;
}


/*
 * @doc FUNC
 * @func
 * This function parses a method signature and returns the number of arguments
 *
 * It is the same for static / non-static
 */
uint16 CLASSFILE_CalcNumArguments(char* pszSignature 
				  /* @parm Signature string to parse */
				  )
{
  int pos,len;
  uint16 jint_count=0;
  byte *Bytes;
  
  assert(pszSignature);
  
#ifdef CLASSFILE_TRACE
  eprintf("count Parsing: %s\n", pszSignature);
#endif

  len = strlen(pszSignature);
  Bytes = pszSignature;
  for (pos = 0; pos < len; pos++) {
    switch(Bytes[pos]) {
    case '(':
      break;
      
    case ')':
      goto LabelLoopEnd;  /* force a jump out of the loop */
      
    case 'B':
    case 'C':
    case 'F':
    case 'S':
    case 'Z':
    case 'I':
      jint_count++;
      break;
      
    case 'D':
    case 'J':
      jint_count++;
      break;
      
    case 'L':
      jint_count++;
      for ( ; Bytes[pos] != ';' && pos < len; pos++);
      break;
      
    case '[':
      jint_count++;
      while ((Bytes[pos] == '[' || 
	      (Bytes[pos] >= '0' && Bytes[pos] <= '9')) && 
	     pos < len) { /* multiple arrays with sizes*/
	pos++;
      }
      if (Bytes[pos] == 'L') {
	/* object signature */
	while (Bytes[pos] != ';') {
	  pos++;
	}
      }
      break;
      
    default:
      panic("illegal signature");
    }
  }
  
 LabelLoopEnd:
#ifdef CLASSFILE_TRACE
  eprintf("int32 argument len: %d\n", jint_count);
#endif
  
  return jint_count;
}


/*
 * @doc FUNC
 * @func
 * This function parses a method signature and returns the type of the
 * return type. 
 * 
 * It returns 0 for an integer (or equivalent 32 bit value), 1 for an address,
 * 2 for a double or long. 3 for void, -1 is failure.
 */

uint16 CLASSFILE_GetReturnType(char* pszSignature 
			       /* @parm Signature string to parse */
			       )
{
  int pos,len;
  byte *Bytes;
  
  assert(pszSignature);

#ifdef CLASSFILE_TRACE
  eprintf("count Parsing: %s\n", pszSignature);
#endif

  len = strlen(pszSignature);
  Bytes = pszSignature;

  // Skip to the first character after the ')'
  for (pos = 0; pos < len; pos++) {
    if (Bytes[pos] == ')') {
      break;
    }
  }
  pos++;

  switch(Bytes[pos]) {
  case 'B':
  case 'C':
  case 'F':
  case 'S':
  case 'Z':
  case 'I':
    return 0; //An integer type

  case 'D':
  case 'J':
    return 2; //A long/double type

  case 'L':
  case '[':
    return 1; //An address type

  case 'V':
    return 3;

  default:
    return -1; //failure
  } //end switch
}


/*
 * @doc FUNC
 * @func
 * This function parses a method signature and returns the number of 32-bit
 * slots that the method's return type will take up.  This function shouldn't
 * need to be called very often as it is run at class load time and used to
 * store the return sizes of all methods in their tMethod structures.
 *
 * @rdesc Returns number of 32-bit slots that the method's return type will use.
 *
 * @ex Example of use. pstMethod is assumed to point to a tMethod structure
 *     containing a method. This example is quite pointless actually,
 *     since the return size is stored in the tMethod structure as well;
 *     it's just used for demonstration. |
 *
 * uint16 u16RetSize;
 *
 * u16RetSize = CLASSFILE_ParseArgumentLength(pstMethod->uidSignature);
 */
uint16 CLASSFILE_ParseReturnLength(const char* pszSignature
				   /* @parm Signature string to parse */
				   )
{
  int pos, len;
  uint16 jint_count = 0;
  const byte *Bytes;

  assert(pszSignature);

  len = strlen(pszSignature);
  Bytes = pszSignature;
  for (pos = 0; (pos < len) && (Bytes[pos] != ')'); pos++) { /* */ }

  if (Bytes[pos] != ')') {
    panic0("ERROR parsing for return size");
  }

  pos++;
  switch (Bytes[pos]) {
  case 'V':
    jint_count = 0;
    break;
    
  case 'B':
  case 'C':
  case 'F':
  case 'S':
  case 'Z':
  case 'I':
    jint_count++;
    break;
    
  case 'D':
  case 'J':
    jint_count += 2;
    break;
    
  case 'L':
    jint_count++;
    for ( ; Bytes[pos] != ';' && pos < len; pos++) { /* */ }
    break;

  case '[':
    jint_count++;
    while ((Bytes[pos] == '[' ||
	    (Bytes[pos] >= '0' && Bytes[pos] <= '9')) && 
	   pos < len) { /* multiple arrays with sizes*/
      pos++;
    }
    if (Bytes[pos] == 'L') { /* object signature */
      while (Bytes[pos] != ';') {
	pos++;
      }
    }
    break;

  default:
    panic0("illegal signature");
    
  }
  return jint_count;
}


/*
 * @doc FUNC
 * @func
 * This function searches a class for a method that is specified by its
 * name and signature. It only searches for methods declared by the class
 * itself, not by superclasses. Use FindMethodInVT for that.
 *
 * @rdesc Returns a pointer to the tMethod structure for the specified method.
 *
 * @ex Example of use. pstClass is assumed to pointer to a loaded class. |
 *
 * tMethod* pstMethod;
 *
 * pstMethod = CLASSFILE_FindMethod(pstClass, "main", 
 *                                  "([Ljava/lang/String;)V");
 */
tMethod* CLASSFILE_FindMethod(JNIEnv* env,
			      tClassLoaderTuple* pstClass,
			      /* @parm Pointer to class to search */
			      char* pszName,
			      /* @parm Name of method to find */
			      char* pszSignature  
			      /* @parm Signature of method to find */
			      )
{
  Uid uidName;
  Uid uidSignature;
  
  /* Insert name into hash table */
  uidName = UID_GetUid(pszName);

#ifdef UID_WARNINGS
  if (uidName == pszName) {
    eprintf("Alert - CLASSFILE_FindMethod called with Uid %s\n", pszName);
  }
#endif /*UID_WARNINGS*/

  /* Insert signature into hash table */
  uidSignature = UID_GetUid(pszSignature);

#ifdef UID_WARNINGS
  if (uidSignature == pszSignature) {
    eprintf("Alert - CLASSFILE_FindMethod called with Uid %s\n", pszSignature);
  }
#endif /*UID_WARNINGS*/
  
  return CLASSFILE_UidFindMethod(env, pstClass, uidName, uidSignature);
}


/*
 * @doc FUNC
 * @func
 * This function searches a class for a method that is specified by its
 * name and signature. The name and signature must be Uid strings, eg
 * taken out the constant pool.
 *
 * @rdesc Returns a pointer to the tMethod structure for the specified method.
 *
 * @ex Example of use. pstClass is assumed to pointer to a loaded class and
 *     uidName and uidSig are assumed to point to Uid strings for the method's
 *     name and signature respectively. |
 *
 * tMethod* pstMethod;
 *
 * pstMethod = CLASSFILE_UidFindMethod(env, pstClass, uidName, uidSig);
 */
tMethod* CLASSFILE_UidFindMethod(JNIEnv* env,
				 tClassLoaderTuple* pstClass,   
				 /* @parm Pointer to class to search */
				 Uid uidName, 
				 /* @parm Name of method to find */
				 Uid uidSignature
				 /* @parm Signature of method to find */
				 )
{
  int i;
  
  for (i = 0; i < pstClass->pstClass->u16MethodsCount; i++) {
    if (uidcmp(pstClass->pstClass->pstMethods[i].uidName, uidName) == 0) {
      if (uidcmp(pstClass->pstClass->pstMethods[i].uidSignature, 
		 uidSignature) == 0) {
	return pstClass->pstClass->pstMethods + i;
      }
    }
  }
  return NULL;	/* not found */
}


/*
 * @doc FUNC
 * @func
 *
 * Searches a classes Virtual Table for a method identified by the name
 *  and signature
 *
 * @rdesc Returns a pointer to the tMethod structure for the specified method.
 *
 * @ex Example of use. pstClass is assumed to pointer to a loaded class. |
 *
 * tMethod* pstMethod;
 *
 * pstMethod = CLASSFILE_FindMethod(pstClass, "main", 
 *                                  "([Ljava/lang/String;)V");
 */
tMethod* CLASSFILE_FindMethodInVT(JNIEnv* env,
				  tClassLoaderTuple* pstClass,    
				  /* @parm Pointer to class to search */
				  char* pszName,
				  /* @parm Name of method to find */
				  char* pszSignature
				  /* @parm Signature of method to find */
				  )
{
  Uid uidName;
  Uid uidSignature;

  /* Insert name into hash table */
  uidName = UID_GetUid(pszName);

#ifdef UID_WARNINGS
  if (uidName == pszName) {
    eprintf("Alert - CLASSFILE_FindMethod called with Uid %s\n", pszName);
  }
#endif /*UID_WARNINGS*/
  
  /* Insert signature into hash table */
  uidSignature = UID_GetUid(pszSignature);
  
#ifdef UID_WARNINGS
  if (uidSignature == pszSignature) {
    eprintf("Alert - CLASSFILE_FindMethod called with Uid %s\n", pszSignature);
  }
#endif /*UID_WARNINGS*/
  
  return CLASSFILE_UidFindMethodInVT(env, pstClass, uidName, uidSignature);
}


tMethod* CLASSFILE_UidFindMethodInVT(JNIEnv* env,
				     tClassLoaderTuple* pstClass,      
				     /* @parm Pointer to class to search */
				     Uid uidName,       
				     /* @parm Name of method to find */
				     Uid uidSignature 
				     /* @parm Signature of method to find */
				     )
{
  int i;
  
  for (i = 0; i < pstClass->pstClass->u16VTSize; i++) {
    if (uidcmp(pstClass->pstClass->ppstVT[i]->uidName, uidName) == 0) {
      //The search does not have a return type in the signature
      char* bracket = strchr(uidSignature, ')');
      if (bracket != NULL) {
	int distance = bracket - uidSignature;
	if (strncmp(uidSignature, pstClass->pstClass->ppstVT[i]->uidSignature,
		    distance) == 0) {
	  return pstClass->pstClass->ppstVT[i];
	}
      }
    }
  }
  return NULL;	/* not found */
}

#ifdef TEASEME
#include "vm/teaseme_main.h"
#endif


/*
 * @doc FUNC
 * @func
 * This calls a class's \<clinit\> function if it has one. Its superclass
 * should be initialised already since it was initialised when it was loaded.
 *
 * Returns an exception object if one occurred
 */
tOBREF CLASSFILE_InitClass(JNIEnv* env,
			   tClassLoaderTuple* pstClass 
			   /* @parm Class structure of class to initialise */
			   )
{
  tMethod* pstMethod;
  
#ifdef DEBUG_TRACE
  eprintf("CLASSFILE_InitClass for %s\n", pstClass->uidName);
#endif 
  
#ifdef HASH_INTEGRITY_CHECK
  HASH_IntegrityCheck(&UID_sstUidHashTable);
#endif

  assert(pstClass->pstClass->bInitialised == 0);
  
  if (pstClass->pstClass->pstSuperClass) {
    if (pstClass->pstClass->pstSuperClass->pstClass->bInitialised == 0) {
      tOBREF hExOb = CLASSFILE_InitClass(env, 
					 pstClass->pstClass->pstSuperClass);
      if (hExOb) {
	/* an exception occurred while running the initializer 
	   now we throw a java.lang.ExceptionInInitializerError
	*/
	return hExOb;
      }
    }
  }
  
#ifdef HASH_INTEGRITY_CHECK
  HASH_IntegrityCheck( &UID_sstUidHashTable );
#endif
  
  /* initialise natives before calling <clinit> because it might use a
     native method */
  NATIVES_InitNatives(pstClass, 0);

#ifdef HASH_INTEGRITY_CHECK
  HASH_IntegrityCheck( &UID_sstUidHashTable );
#endif

  /* call the class initialising method */
  pstMethod = CLASSFILE_FindMethod(env, pstClass, "<clinit>", "()V");
  if (pstMethod) {
    tOBREF hExOb = INTERP_RunStaticMethodFromPtr(env, pstMethod, NULL);
    if (hExOb) {
      /* an exception occurred while running the initializer 
	 now we throw a java.lang.ExceptionInInitializerError
      */
      // XXX - well, why don't we???
      return hExOb;
    }
  }
  
  pstClass->pstClass->bInitialised = 1;
  
#ifdef HASH_INTEGRITY_CHECK
  HASH_IntegrityCheck(&UID_sstUidHashTable);
#endif
  return NULL;
}


#ifndef OLD_RESOLVE
tClassLoaderTuple* CONSTGET_Class(JNIEnv* env, tClassLoaderTuple* tuple, 
				  uint16 u16Index)
{
  tClass* c = tuple->pstClass;
  tClassLoaderTuple* ret;

  sys_mutex_lock(classfileMutex);
  assert(CONSTTAG(c, u16Index) == CONSTANT_Class);
  if (CONSTTAG_GetRes(c, u16Index)) {
    ret = (tClassLoaderTuple*) CONSTGET(c, u16Index);
  }
  else {
    ret = CLASSFILE_ResolveClass(env, tuple, u16Index);
  }
  sys_mutex_unlock(classfileMutex);
  return ret;
}

#endif


/**
 * @func
 * This function parses a method signature and returns the type of one of 
 * the arguments. 
 * 
 * It returns 0 for an integer (or equivalent 32 bit value), 1 for an 
 * address, 2 for a double or long. -1 is failure.
 */
int CLASSFILE_IsArgumentRef(char* pszSignature,
			    /* @parm Signature string to parse */
			    int index,
			    /* Index of argument to check */
			    int* slotOffset
			    /* Offset of this argument in 32-bit slots */
			    )
{
  int pos, len;
  uint16 jint_count = 0;
  int slotcount = 0;
  byte *Bytes;
  
  assert(pszSignature);

  len = strlen(pszSignature);
  Bytes = pszSignature;
  for (pos = 0; pos < len; pos++) {
    switch (Bytes[pos]) {
    case '(':
      break;

    case ')':
      goto LabelLoopEnd;  /* force a jump out of the loop */
      
    case 'B':
    case 'C':
    case 'F':
    case 'S':
    case 'Z':
    case 'I':
      jint_count++;
      slotcount++;
      if (jint_count == index) {
	*slotOffset = slotcount;
	return 0; //primitive
      }
      break;
      
    case 'D':
    case 'J':
      jint_count++;
      slotcount += 2;
      if (jint_count == index) {
	*slotOffset = slotcount;
	return 0; //primitive
      }
      break;
      
    case 'L':
      jint_count++;
      slotcount++;
      if (jint_count == index) {
	*slotOffset = slotcount;
	return 1; //non-primitive
      }
      for ( ; Bytes[pos] != ';' && pos < len; pos++) { /* */ }
      break;
      
    case '[':
      jint_count++;
      slotcount++;
      if (jint_count == index) {
	*slotOffset = slotcount;
	return 1; //non-primitive
      }
      while ((Bytes[pos] == '[' || 
	      (Bytes[pos] >= '0' && Bytes[pos] <= '9')) 
	     && pos < len) { /* multiple arrays with sizes*/
	pos++;
      }
      if (Bytes[pos] == 'L') {
	/* object signature */
	while (Bytes[pos] != ';') {
	  pos++;
	}
      }
      break;

    default:
      panic("illegal signature");
    }
  }
  
  
 LabelLoopEnd:
#ifdef CLASSFILE_TRACE
  eprintf("int32 argument len: %d\n", jint_count);
#endif
  
  return -1;
}

