#include "config.h"
/* 
 * @doc MODULE 
 * @module Class.c | 
 * 
 * This file contains native methods for both the Class and 
 * ClassLoader classes. 
 */ 
 
/* 
 * @doc TOPIC 
 * @topic Class Objects | 
 * 
 * When classes are used by Java programs, they are used in the form of Class 
 * objects. This is different to how they are represented internally in the 
 * virtual machine, so when a Class object is needed, we have to create one 
 * as a wrapper around the virtual machine's internal tClass structure. 
 * 
 * The Class object has a private field, classStruct, that stores a pointer to 
 * the tClass structure that the object represents. In turn, the tClass struct 
 * has a pointer, pstClassObject, that will point to the Class object that 
 * represents it if one exists. When a class object is made, it and its 
 * corresponding tClass struct are set to point to one another. When a class 
 * object is needed, a check is first done on the tClass struct to see if 
 * pstClassObject has been set. It it has been set then the Class object to 
 * which it points is used. If it is NULL then a new Class object must be 
 * created. 
 * 
 * Because Class objects are just normal objects, they might be garbage 
 * collected when they are no longer reachable. If this happens, the 
 * pstClassObject pointer in its corresponding tClass structure will be 
 * pointing to garbage (no pun intended). Because of this, the Class class has 
 * a native finalize() method that sets pstClassObject in its tClass struct to 
 * NULL. Then if a Class object for that class is needed again, a new one will 
 * be created. 
 * 
 */ 
 
#include <stdio.h> 
#include <string.h> 
 
#include "vm/jni.h" 
#include "vm/interp.h" 
#include "vm/interp_methods.h" 
#include "vm/cplist.h" 
#include "vm/jutils.h" 
#include "vm/classfil.h" 
#include "vm/classfile_methods.h" 
#include "vm/newobject.h"
 
#include "vm/global.h" 
 
#include "lib/indigenous/java.lang/VMClassLoader.h" 
#include "vm/uid.h" 

#include "vm/sys_linux_host/wrappers.h"

#include "lib/indigenous/java.lang/Class.h"
#include "lib/indigenous/java.lang/VMClass.h"
#include "lib/indigenous/java.io/VMObjectStreamClass.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"

extern tClassLoaderTuple* pstClassType; 
extern tOBREF OOMExceptionObject;
 
/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: public native String getName(); 
 *
 * Returns the name of this class. 
 */ 

// XXX - use of static buffer means that this method is not thread-safe
static char str_buffer[4096]; 
 
jobject Java_java_lang_VMClass_getName(JNIEnv* env, jobject vmClass) 
{ 
  tClassLoaderTuple* pstClass = CLASS_GetVMClassStruct(env, vmClass);
  int i; 

  strcpy(str_buffer, pstClass->uidName); 
  for (i = 0; i < strlen(str_buffer); i++) {
    if (str_buffer[i] == '/') {
      str_buffer[i] = '.'; 
    }
  }
  return INTERP_NewStringFromAsciz(env, str_buffer); 
} 
 
 
/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: public native boolean isInterface(); 
 * 
 * Returns true if this class is an interface class, false otherwise. 
 */  
jboolean Java_java_lang_VMClass_isInterface(JNIEnv* env, jobject vmClass) 
{ 
  return (CLASS_GetVMClassStruct(env, vmClass)->pstClass->u16AccessFlags &
	  ACC_INTERFACE) ? JNI_TRUE : JNI_FALSE; 
} 
 

/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: public native Class getSuperclass(); 
 * 
 * Returns the class object of this class's superclass. 
 */ 
jclass Java_java_lang_VMClass_getSuperclass(JNIEnv* env, jobject vmClass) 
{ 
  tOBREF pstSuperObject; 
  tClassLoaderTuple* ourClass;
  tClassLoaderTuple* pstSuperClass; 
  
  ourClass = CLASS_GetVMClassStruct(env, vmClass);
  pstSuperClass = ourClass->pstClass->pstSuperClass; 
  
  if (pstSuperClass == 0) {
    return NULL; 
  }
  else {
    //I hope this works for the void case too
    if (ourClass->pstClass->u16AccessFlags & (ACC_INTERFACE | ACC_PRIMITIVE)) {
      return NULL;
    }
    else {
      return pstSuperClass->classObject;
    }
  }
} 
 
 
/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: public native Class getInterfaces()[]; 
 * 
 * Returns an array containing the class structures of this class's 
 * interfaces. 
 */ 
jarray Java_java_lang_VMClass_getInterfaces(JNIEnv* env, jobject vmClass) 
{ 
  tOBREF pstTempObject; 
  tClassLoaderTuple* pstClass = CLASS_GetVMClassStruct(env, vmClass); 
  tClassLoaderTuple* pstInterface; 
  jarray pstInterfaceArray; 
  
  int i; 
 
  pstInterfaceArray = 
    (*env)->NewObjectArray(env, pstClass->pstClass->u16InterfacesCount, 
			   (*env)->FindClass(env, "java/lang/Class"),
			   NULL);
  for (i = 0; i < pstClass->pstClass->u16InterfacesCount; i++) { 
    (*env)->SetObjectArrayElement(env, pstInterfaceArray, i,
				  (pstClass->pstClass->
				   ppstInterfaces[i]->classObject));
  } 
  return pstInterfaceArray; 
} 

 
/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: public native ClassLoader getClassLoader(); 
 * 
 * Returns this class's classloader. 
 */ 
jobject Java_java_lang_VMClass_getClassLoader(JNIEnv* env, jobject vmClass) 
{ 
  return CLASS_GetVMClassStruct(env, vmClass)->classLoader;
} 
 
 
//	public native boolean isInstance(Object o); 
 
jboolean Java_java_lang_VMClass_isInstance(JNIEnv* env, jobject vmClass, 
					   jobject obj) 
{ 
  tClassLoaderTuple* desiredClass;
  tClassLoaderTuple* suppliedClass;

  desiredClass = CLASS_GetVMClassStruct(env, vmClass); 
 
  if (obj == NULL) {
    return JNI_FALSE; //NULL object is not an instanceof of anything?
  }

  suppliedClass = ODEREF(obj)->pstType;

  if (INTERP_CheckCast(env, desiredClass, suppliedClass) == 1) {
    return JNI_TRUE;
  }
  return JNI_FALSE;
} 
 
 
jboolean Java_java_lang_VMClass_isAssignableFrom(JNIEnv* env, jobject vmClass,
						 jclass clazz2) 
{ 
  jclass classObject = CLASS_GetVMClassStruct(env, vmClass)->classObject;
  return (*env)->IsAssignableFrom(env, classObject, clazz2); 
} 
 

/* 
 * Takes a java.lang.Class object and returns true if it represents a
 * primitive type 
 *
 * This could be made faster with a helper method from VMClassLoader
 * (another alternative is an ACC_PRIMITIVE check!)
 */ 
 
jboolean Java_java_lang_VMClass_isPrimitive(JNIEnv* env, jobject vmClass) 
{  
  tClassLoaderTuple* compClass = CLASS_GetVMClassStruct(env, vmClass);

  if (strcmp(compClass->uidName, "float") == 0 ||
      strcmp(compClass->uidName, "boolean") == 0 ||
      strcmp(compClass->uidName, "int") == 0 ||
      strcmp(compClass->uidName, "short") == 0 || 
      strcmp(compClass->uidName, "char") == 0 || 
      strcmp(compClass->uidName, "long") == 0 ||
      strcmp(compClass->uidName, "byte") == 0 || 
      strcmp(compClass->uidName, "double") == 0 || 
      strcmp(compClass->uidName, "void") == 0) { 
    return JNI_TRUE; 
  } 
  else {
    return JNI_FALSE; 
  }
} 

/*
 * Return 1 if vmClass represents an array type, zero if not.  (Returning
 * -1 means that this method is not supported)
 */
jint Java_java_lang_VMClass_isArray(JNIEnv* env, jobject vmClass) 
{  
  tClassLoaderTuple* tuple = CLASS_GetVMClassStruct(env, vmClass);
  if (tuple->uidName[0] == '[') {
    return 1;
  }
  else {
    return 0;
  }
}


jclass Java_java_lang_VMClass_getComponentType(JNIEnv* env, jobject vmClass)
{
  tClassLoaderTuple* tuple = CLASS_GetVMClassStruct(env, vmClass);

  if (tuple->uidName[0] != '[') {
    return NULL;
  }
  else {
    switch (tuple->uidName[1]) {
    case '[':
      return VMClass_forName(env, tuple->uidName + 1, tuple);
    case 'V':
      // XXX - should this ever happen?
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_VOID);
    case 'B':
      // XXX - if we were paranoid, we'd check that this is the last
      // character of the type string
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_BYTE);
    case 'C':
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_CHAR);
    case 'F':
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_FLOAT);
    case 'S':
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_SHORT);
    case 'Z':
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_BOOLEAN);
    case 'I':
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_INT);
    case 'J':
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_LONG);
    case 'D':
      return java_lang_VMClassLoader_getPrimitiveClassFromCode(env, T_DOUBLE);
    case 'L':
      {
	// We need to extract the base type's class name into a separate
	// string, because VMClass_forName will not accept a class name
	// in the form of a type string.
	jclass res;
        int len = strlen(tuple->uidName);
	char* pstBaseClassName;
	
	if (tuple->uidName[len - 1] != ';') {
	  panic("malformed type string %s", tuple->uidName);
	}
	pstBaseClassName = sys_malloc(len);
	strcpy(pstBaseClassName, tuple->uidName + 2);
	pstBaseClassName[len - 3] = '\000';
	res = VMClass_forName(env, pstBaseClassName, tuple);
	sys_free(pstBaseClassName);
	return res;
      }
    default : 
      panic("malformed type string %s", tuple->uidName);
      return NULL;
    }
  }
}

 
jint Java_java_lang_VMClass_getModifiers(JNIEnv* env, jobject vmClass) 
{ 
  tClassLoaderTuple* tuple;
  jint result;
  
  // Arrays have the access flags of their parent class.
  if (Java_java_lang_VMClass_isArray(env, vmClass)) {
    jclass baseClass = Java_java_lang_VMClass_getComponentType(env, vmClass);
    jobject baseVMClass = CLASS_GetVMClass(env, baseClass);

    result = Java_java_lang_VMClass_getModifiers(env, baseVMClass);
    // Arrays are always final and never an interface.
    // And also abstract (but that is actually a bug - see JDC Bug: 4208179)
    // It matters for serial version UID calculations :{
    result = (result | ACC_FINAL | ACC_ABSTRACT) & ~ACC_INTERFACE;
  }
  else {
    tuple = CLASS_GetVMClassStruct(env, vmClass);
    if (tuple->pstClass->bIsInnerClass) {
      // If this is an inner class, the declared access flags are hidden
      // in the pstInnerClasses array.
      int i;

      for (i = 0; i < tuple->pstClass->u16InnerClassesCount; i++) {
	if (tuple->pstClass->pstInnerClasses[i].pstInnerClass == tuple) {
          result = tuple->pstClass->pstInnerClasses[i].u16AccessFlags;
	  break;
	}
      }
      assert(i < tuple->pstClass->u16InnerClassesCount);
    }
    else {
      result = tuple->pstClass->u16AccessFlags;
    }
  }

  // synchronized is not a valid modifier but is sometimes set on a class
  // since it is also used for other things.
  result &= ~ACC_SYNCHRONISED;

  return result;
} 
 

jclass Java_java_lang_VMClass_getDeclaringClass(JNIEnv* env, jobject vmClass) 
{ 
  int i;
  tClassLoaderTuple* tuple = CLASS_GetVMClassStruct(env, vmClass);
  tClass *pstClass = tuple->pstClass;
  
  if (tuple->pstClass->bIsInnerClass) {
    for (i = 0; i < pstClass->u16InnerClassesCount; i++) {
      if (pstClass->pstInnerClasses[i].pstInnerClass == tuple) {
	return 
	  CLASS_GetClassFromStruct(env, 
				   pstClass->pstInnerClasses[i].pstOuterClass);
      }
    }
    panic("Couldn't find the declaring class for %s", tuple->uidName);
  }
  else {
    return NULL;
  }
} 
 

jobjectArray Java_java_lang_VMClass_getDeclaredClasses(JNIEnv* env, 
						       jobject vmClass,
						       jboolean publicOnly)
{ 
  // We make two passes.  In the first, we count the inner classes.
  // In the second, we copy the class objects to the array.
  int i, j;
  tClassLoaderTuple* tuple = CLASS_GetVMClassStruct(env, vmClass);
  tClass *pstClass;
  jobjectArray pstClassArray;

  assert(tuple);
  pstClass = tuple->pstClass;

  // Count the inner classes in this class 
  j = 0;
  for (i = 0; i < pstClass->u16InnerClassesCount; i++) {
    assert(pstClass->pstInnerClasses);
    assert(pstClass->pstInnerClasses[i].pstInnerClass);
    assert(pstClass->pstInnerClasses[i].pstOuterClass);
    if (pstClass->pstInnerClasses[i].pstOuterClass == tuple &&
	(!publicOnly || 
	 pstClass->pstInnerClasses[i].u16AccessFlags & ACC_PUBLIC)) {
      j++;
    }
  }

  // Allocate the result array
  pstClassArray = 
    (*env)->NewObjectArray(env, j, 
			   (*env)->FindClass(env, "java/lang/Class"),
			   NULL);
  if (pstClassArray == NULL) {
    // An OutOfMemoryException should have been thrown ...
    return NULL;
  }
  
  // Fill in the result array.
  if (j > 0) {
    j = 0;
    for (i = 0; i < pstClass->u16InnerClassesCount; i++) {
      if (pstClass->pstInnerClasses[i].pstOuterClass == tuple &&
	  (!publicOnly || 
	   pstClass->pstInnerClasses[i].u16AccessFlags & ACC_PUBLIC)) {
	jclass obj = 
	  CLASS_GetClassFromStruct(env, 
				   pstClass->pstInnerClasses[i].pstInnerClass);
	(*env)->SetObjectArrayElement(env, pstClassArray, j++, obj);
      }
    }
  }
  return pstClassArray;   
} 

/* 
 * @doc NATFUNC 
 * @func 
 * Implementation of: protected native void finalize(); 
 * 
 * This is called when the class is about to be garbage collected. It sets 
 * the pstClassObject field of this class's C class structure to NULL, 
 * indicating that it no longer has a class object. 
 */ 
void Java_java_lang_VMClass_finalize(JNIEnv* env, jobject vmClass) 
{ 
  tClassLoaderTuple* pstClass = CLASS_GetClassStruct(env, vmClass);

  pstClass->classObject = NULL;
  pstClass->vmClassObject = NULL;
} 

/* 
 * @doc FUNC 
 * @func 
 * Returns the class object representing the class of the specified object. A 
 * new class object is created if one does not already exist.  
 */ 
tOBREF CLASS_GetClass(JNIEnv* env, tOBREF pstObject) 
{ 
  // XXX ... the description is clearly wrong!!!
  return (tOBREF) DEREF(pstObject)->pstType->classObject; 
} 


/* 
 * @doc FUNC 
 * @func 
 * Returns the class object representing the class of the specified object. A 
 * new class object is created if one does not already exist. 
 */ 
tOBREF CLASS_GetClassFromStruct(JNIEnv* env, tClassLoaderTuple* pstClass) 
{ 
  return pstClass->classObject;
} 


jboolean 
  Java_java_io_VMObjectStreamClass_hasClassInitializer(JNIEnv *env, 
						       jobject dummy, 
						       jclass clazz) 
{
  tClassLoaderTuple* tuple;
  int i;

  if (clazz == NULL) {
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "class argument was null"));
    return JNI_FALSE;
  }

  tuple = CLASS_GetClassStruct(env, clazz);

  for (i = 0; i < tuple->pstClass->u16MethodsCount; i++) {
    if (strcmp(tuple->pstClass->pstMethods[i].uidName, "<clinit>") == 0) {
      return JNI_TRUE;
    }
  }
  return JNI_FALSE;
}
