#include "config.h"

#include <stdlib.h>
#include <stdio.h>

#include "vm/jni.h"

#include "lib/indigenous/java.lang/VMClass.h"
#include "lib/indigenous/java.lang/Class.h"
#include "lib/indigenous/java.lang/Class_Reflection.h"

#include "lib/indigenous/java.lang.reflect/Field.h"

#include "vm/classfile_methods.h"
#include "vm/classloader_tuple.h"
#include "vm/interp_methods.h"
#include "vm/newobject.h"


static int32 FIELD_getValue(jobject obj, tField* field)
{
  if (field->u16AccessFlags & ACC_STATIC) {
    return field->pstClass->pi32StatVars[field->u16Offset];
  }
  else {
    return ODEREF(obj)->pi32Vars[field->u16Offset];
  }
}

static jlong FIELD_getValue64(jobject obj, tField* field)
{
  if (field->u16AccessFlags & ACC_STATIC) {
    return *((jlong*) &(field->pstClass->pi32StatVars[field->u16Offset]));
  }
  else {
    return *((jlong*) &(ODEREF(obj)->pi32Vars[field->u16Offset]));
  }
}

static void FIELD_setValue(jobject obj, tField* field, int32 val)
{
  if (field->u16AccessFlags & ACC_STATIC) {
    field->pstClass->pi32StatVars[field->u16Offset] = val;
  }
  else {
    ODEREF(obj)->pi32Vars[field->u16Offset] = val;
  }
}

static void FIELD_setValue64(jobject obj, tField* field, jlong val)
{
  if (field->u16AccessFlags & ACC_STATIC) {
    *((jlong*) &(field->pstClass->pi32StatVars[field->u16Offset])) = val;
  }
  else {
    *((jlong*) &(ODEREF(obj)->pi32Vars[field->u16Offset])) = val;
  }
}


tField* FIELD_getInternalFieldStruct(JNIEnv* env, jobject fieldObject)
{
  tClassLoaderTuple* pstType;
  int slot;
  int isStatic;
  
  jfieldID decClassField;       
  jfieldID slotField;       
  jfieldID staticField;       
  jclass fieldClazz;
  jclass decClass;
  
  fieldClazz = CLASS_GetClass(env, fieldObject);
  decClassField = (*env)->GetFieldID(env, fieldClazz, "declaringClass",
				     "Ljava/lang/Class;");
  decClass = (*env)->GetObjectField(env, fieldObject, decClassField);
  pstType = CLASS_GetClassStruct(env, decClass);
  assert(pstType);
  
  slotField = (*env)->GetFieldID(env, fieldClazz, "slot", "I");
  
  assert(slotField);
  
  slot = (*env)->GetIntField(env, fieldObject, slotField);
  
  staticField = (*env)->GetFieldID(env, fieldClazz, "isstatic", "I");
  
  isStatic = (*env)->GetIntField(env, fieldObject, staticField);
  
  if (isStatic) {
    return pstType->pstClass->pstStatOffsets + slot;
  }
  else {
    return pstType->pstClass->pstInstOffsets + slot;
  }
}


jint Java_java_lang_reflect_Field_getModifiers(JNIEnv* env, jobject obj)
{
  tField* field = FIELD_getInternalFieldStruct(env, obj);
  return field->u16AccessFlags & 
    (ACC_PUBLIC | ACC_PRIVATE | ACC_PROTECTED | ACC_STATIC | ACC_FINAL | 
     ACC_VOLATILE | ACC_TRANSIENT);
  //XXX did I miss any?
}

jclass Java_java_lang_reflect_Field_getType(JNIEnv* env, jobject obj)
{
  tField* field = FIELD_getInternalFieldStruct(env, obj);
  return CLASS_GetClassFromSig(env, field->uidFieldSig, field->pstClass, 0);
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    get
 * Signature: (Ljava/lang/Object;)Ljava/lang/Object;
 */
JNIEXPORT jobject JNICALL Java_java_lang_reflect_Field_get(JNIEnv* env, 
							   jobject fieldObject, 
							   jobject obj) 
{
  jboolean isPrimitive = JNI_TRUE;
  tField* pstField; 
  tDConv dconv;
  tFConv fconv;
  
  pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  
  if (pstField->uidFieldSig[0] == 'L' || pstField->uidFieldSig[0] == '[') {
    isPrimitive = JNI_FALSE;
  }

  if (isPrimitive) {
    switch (pstField->uidFieldSig[0]) {
    case 'B':
      return INTERP_wrapPrimitiveByte(env, 
				      (jbyte) FIELD_getValue(obj, pstField));
    case 'C':
      return INTERP_wrapPrimitiveChar(env, 
				      (jchar) FIELD_getValue(obj, pstField));
    case 'F':
      fconv.i = FIELD_getValue(obj, pstField);
      return INTERP_wrapPrimitiveFloat(env, fconv.f);

    case 'S':
      return INTERP_wrapPrimitiveShort(env, 
				       (jshort) FIELD_getValue(obj, pstField));
    case 'Z':
      return INTERP_wrapPrimitiveBoolean(env,
					 (jboolean) FIELD_getValue(obj, 
								   pstField));
    case 'I':
      return INTERP_wrapPrimitiveInt(env,
				     (jint) FIELD_getValue(obj, pstField));
    case 'D':
      dconv.l = FIELD_getValue64(obj, pstField);
      return INTERP_wrapPrimitiveDouble(env, dconv.d);

    case 'J':
      return INTERP_wrapPrimitiveLong(env, 
				      (jlong) FIELD_getValue64(obj, pstField));
    default:
      panic("unknown primitive type %s in Field.get()",
	    pstField->uidFieldSig); 
      return NULL;
    }
  }
  else {
    if (pstField->u16AccessFlags & ACC_STATIC) {
      return (jobject) pstField->pstClass->pi32StatVars[pstField->u16Offset];
    }
    else {
      assert(INTERP_CheckCast(env, pstField->pstClass,
			      ODEREF(obj)->pstType) == 1);
      assert(obj != NULL);
      return (jobject) ODEREF(obj)->pi32Vars[pstField->u16Offset];
    }
  }
}


static int FIELD_getChecks(JNIEnv* env, tField* pstField, jobject obj)
{
  if ((pstField->u16AccessFlags & ACC_STATIC) == 0) {
    // Check that obj is non-NULL
    if (obj == NULL) {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "target object is null"));
      return 0;
    }
    
    // Check that obj an instance of the declaring class
    if (INTERP_CheckCast(env, pstField->pstClass, ODEREF(obj)->pstType) != 1) {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "target object is not compatible with this Field's declaring class"));
      return 0;
    }
  }

  // XXX - check access
  
  return 1;
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    getBoolean
 * Signature: (Ljava/lang/Object;)Z
 */
JNIEXPORT jboolean 
  JNICALL Java_java_lang_reflect_Field_getBoolean(JNIEnv* env,
						  jobject fieldObject,
						  jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return JNI_FALSE;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'Z':
    return (jboolean) FIELD_getValue(obj, pstField); 

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'boolean'"));
    return JNI_FALSE;
  }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    getByte
 * Signature: (Ljava/lang/Object;)B
 */
JNIEXPORT jbyte 
JNICALL Java_java_lang_reflect_Field_getByte(JNIEnv* env, jobject fieldObject, 
					     jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return 0;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'B':
    return (jbyte) FIELD_getValue(obj, pstField); 

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'byte'"));
    return 0;
  }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    getShort
 * Signature: (Ljava/lang/Object;)S
 */
JNIEXPORT jshort 
JNICALL Java_java_lang_reflect_Field_getShort(JNIEnv* env, jobject fieldObject,
					      jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return 0;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'B': 
  case 'S':
    return (jshort) FIELD_getValue(obj, pstField); 

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'short'"));
    return 0;
  }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    getChar
 * Signature: (Ljava/lang/Object;)C
 */
JNIEXPORT jchar 
JNICALL Java_java_lang_reflect_Field_getChar(JNIEnv* env, jobject fieldObject,
					     jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return 0;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'C':
    return (jchar) FIELD_getValue(obj, pstField); 

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'char'"));
    return 0;
  }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    getInt
 * Signature: (Ljava/lang/Object;)I
 */
JNIEXPORT jint 
JNICALL Java_java_lang_reflect_Field_getInt(JNIEnv* env, jobject fieldObject, 
					    jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return 0;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'B':  
  case 'S': 
  case 'C': 
  case 'I':
    return (jint) FIELD_getValue(obj, pstField); 

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'int'"));
    return 0; 
  }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    getLong
 * Signature: (Ljava/lang/Object;)J
 */
JNIEXPORT jlong 
JNICALL Java_java_lang_reflect_Field_getLong(JNIEnv* env, jobject fieldObject,
					     jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  jlong res;
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return 0;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'B':  
  case 'S': 
  case 'C': 
  case 'I':
    res = FIELD_getValue(obj, pstField);
    return res;

  case 'J':
    res = FIELD_getValue64(obj, pstField);
    return res;

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'long'"));
    return 0; 
  }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    getFloat
 * Signature: (Ljava/lang/Object;)F
 */
JNIEXPORT jfloat 
JNICALL Java_java_lang_reflect_Field_getFloat(JNIEnv* env, jobject fieldObject,
					      jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  tFConv fconv;
  jfloat res;
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return 0;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'B':  
  case 'S': 
  case 'C': 
  case 'I':
    res = FIELD_getValue(obj, pstField);
    return res;

  case 'F':
    fconv.i = FIELD_getValue(obj, pstField);
    return fconv.f;

  case 'J':
    res = FIELD_getValue64(obj, pstField);
    return res;

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'float'"));
    return 0; 
  }
}
 

/*
 * Class:     java_lang_reflect_Field
 * Method:    getDouble
 * Signature: (Ljava/lang/Object;)D
 */
JNIEXPORT jdouble 
JNICALL Java_java_lang_reflect_Field_getDouble(JNIEnv* env, jobject fieldObject,
					       jobject obj)
{
  tField* pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  jdouble res;

  tFConv fconv;
  tDConv dconv;
  
  if (!FIELD_getChecks(env, pstField, obj)) {
    // Checks failed.  An exception has been thrown.
    return 0;
  }
  
  // Extract / widen the field's value
  switch (pstField->uidFieldSig[0]) {
  case 'B':  
  case 'S': 
  case 'C': 
  case 'I':
    res = FIELD_getValue(obj, pstField);
    return res;

  case 'F':
    fconv.i = FIELD_getValue(obj, pstField);
    return fconv.f;

  case 'J':
    res = FIELD_getValue64(obj, pstField);
    return res;

  case 'D':
    dconv.l = FIELD_getValue64(obj, pstField);
    return dconv.d;

  default:
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "this Field's value cannot be widened to 'double'"));
    return 0; 
  }
}



//XXXXXXXXXXXXXXXXXXXXXXXXXX finish this XXXXXXXXXXXXXXXXXXXXXXXXXXXXX

/*
 * Class:     java_lang_reflect_Field
 * Method:    set
 * Signature: (Ljava/lang/Object;Ljava/lang/Object;)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_set(JNIEnv* env, 
							jobject fieldObject, 
							jobject obj, 
							jobject value)
{
  jboolean isPrimitive = JNI_TRUE;
  tField* pstField; 
  jclass fieldClass;
  
  pstField = FIELD_getInternalFieldStruct(env, fieldObject);
  
  if (pstField->uidFieldSig[0] == 'L' || pstField->uidFieldSig[0] == '[') {
    isPrimitive = JNI_FALSE;
  }
  
  //Check if it's an instance field
  if ((pstField->u16AccessFlags & ACC_STATIC) == 0) {
    if (obj == NULL) {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "null reference passed to Field.set for an instance field"));
      return;
    }
    //Check that it's an instance of the declaring class
    if (INTERP_CheckCast(env, pstField->pstClass, ODEREF(obj)->pstType) != 1) {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "Object passed to Field.set is not an instance of the Fields declaring class"));
      return;
    }
  }
  
  //XXX IllegalAccessException must be checked!
  
  if (pstField->u16AccessFlags & ACC_FINAL) {
    (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalAccessException", "Can't use Field.set on a final field"));
    return;
  }
  
  fieldClass = CLASS_GetClassFromSig(env, pstField->uidFieldSig, 
				     pstField->pstClass, 0);

  if (isPrimitive) {
    tClassLoaderTuple* suppliedClass;
    tClassLoaderTuple* targetClass;
    jclass wrapperClass;
    jvalue retval;
    
    if (value == NULL) {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/NullPointerException", "(Wrapped primitive) was NULL"));
      return;
    }
    
    wrapperClass = CLASS_GetWrapperClass(env, fieldClass);
    targetClass = CLASS_GetClassStruct(env, wrapperClass);
    suppliedClass = ODEREF(value)->pstType;
    if (suppliedClass != targetClass) {
      jclass targetClassObj =
	CLASSLOADER_TUPLE_MakeClassObject(env, targetClass);
      jclass suppliedClassObj = 
	CLASSLOADER_TUPLE_MakeClassObject(env, suppliedClass);
      jboolean check = 
	CLASS_CheckWrapperWidening(env, targetClassObj, suppliedClassObj);
      
      if (check == JNI_FALSE) {
	(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "Cannot widen wrapped primitive"));
	return;
      }
      else {
	if (CLASS_WidenResult(env, targetClassObj, suppliedClassObj,
			      &retval) == 0) {
	  //success
	}
	else {
	  (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Error occurred widening and unwrapping primitive"));    
	  return;
	}
      }
    }
    else {
      if (CLASS_UnwrapObject(env, value, &retval) != 0) {
	(*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/InternalError", "Error occurred unwrapping primitive"));
	return;
      }
    }
    
    switch (pstField->uidFieldSig[0]) {
    case 'B':
      FIELD_setValue(obj, pstField, retval.b);
      return;

    case 'C':
      FIELD_setValue(obj, pstField, retval.c);
      return;

    case 'Z':
      FIELD_setValue(obj, pstField, retval.z);
      return;

    case 'I':
      FIELD_setValue(obj, pstField, retval.i);
      return;

    case 'F':
      FIELD_setValue(obj, pstField, retval.f);
      return;

    case 'S':
      FIELD_setValue(obj, pstField, retval.s);
      return;

    case 'J':
      FIELD_setValue64(obj, pstField, retval.j);
      return;

    case 'D':
      FIELD_setValue64(obj, pstField, retval.d);
      return;

    default:
      panic("Field_set: not implemented for type %s", pstField->uidFieldSig);
    }
  }
  else {
    //check that supplied value is subtype of field type
    tClassLoaderTuple* suppliedClass;
    tClassLoaderTuple* targetClass;
    
    //skip the type check and just do the assignment
    if (value == NULL) {
      FIELD_setValue(obj, pstField, (uint32)value);
      return;
    }
    
    targetClass = CLASS_GetClassStruct(env, fieldClass);
    suppliedClass = ODEREF(value)->pstType;
    if (INTERP_CheckCast(env, targetClass, suppliedClass) != 1) {
      (*env)->Throw(env, INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalArgumentException", "Argument to field.set of incorrect type for this field"));
      return;
    }
    FIELD_setValue(obj, pstField, (uint32)value);
  }
  
  return;
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setBoolean
 * Signature: (Ljava/lang/Object;Z)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setBoolean(JNIEnv* env, 
							       jobject fieldObject, 
							       jobject obj, 
							       jboolean value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "Z");
      (*env)->SetStaticBooleanField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "Z");
      (*env)->SetBooleanField(env, obj, fid, value);
    }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setByte
 * Signature: (Ljava/lang/Object;B)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setByte(JNIEnv* env,
							    jobject fieldObject,
							    jobject obj,
							    jbyte value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "B");
      (*env)->SetStaticByteField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "B");
      (*env)->SetByteField(env, obj, fid, value);
    }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setShort
 * Signature: (Ljava/lang/Object;S)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setShort(JNIEnv *env, 
							     jobject fieldObject, 
							     jobject obj,
							     jshort value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "S");
      (*env)->SetStaticShortField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "S");
      (*env)->SetShortField(env, obj, fid, value);
    }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setChar
 * Signature: (Ljava/lang/Object;C)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setChar(JNIEnv *env, 
							    jobject fieldObject,
							    jobject obj,
							    jchar value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "C");
      (*env)->SetStaticCharField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "C");
      (*env)->SetCharField(env, obj, fid, value);
    }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setInt
 * Signature: (Ljava/lang/Object;I)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setInt(JNIEnv *env, 
							   jobject fieldObject, 
							   jobject obj, 
							   jint value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "I");
      (*env)->SetStaticIntField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "I");
      (*env)->SetIntField(env, obj, fid, value);
    }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setLong
 * Signature: (Ljava/lang/Object;J)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setLong(JNIEnv *env,
							    jobject fieldObject,
							    jobject obj,
							    jlong value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "J");
      (*env)->SetStaticLongField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "J");
      (*env)->SetLongField(env, obj, fid, value);
    }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setFloat
 * Signature: (Ljava/lang/Object;F)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setFloat(JNIEnv *env, 
							     jobject fieldObject, 
							     jobject obj, 
							     jfloat value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "F");
      (*env)->SetStaticFloatField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "F");
      (*env)->SetFloatField(env, obj, fid, value);
    }
}


/*
 * Class:     java_lang_reflect_Field
 * Method:    setDouble
 * Signature: (Ljava/lang/Object;D)V
 */
JNIEXPORT void JNICALL Java_java_lang_reflect_Field_setDouble(JNIEnv *env,
							      jobject fieldObject,
							      jobject obj, 
							      jdouble value)
{
  jclass fclass = (*env)->GetObjectClass(env, fieldObject);
  jfieldID nfid = (*env)->GetFieldID(env, fclass, "name", "Ljava/lang/String;");
  jstring jname = (jstring)(*env)->GetObjectField(env, fieldObject, nfid);
  const char* name =  (*env)->GetStringUTFChars(env, jname, NULL);
  if (obj == NULL)
    {
      jfieldID cfid = (*env)->GetFieldID(env, fclass, "declaringClass",
			"Ljava/lang/Class;");
      jclass clazz = (jclass)(*env)->GetObjectField(env, fieldObject, cfid);
      jfieldID fid = (*env)->GetStaticFieldID(env, clazz, name, "D");
      (*env)->SetStaticDoubleField(env, clazz, fid, value);
    }
  else
    {
      jclass oclass = (*env)->GetObjectClass(env, obj);
      jfieldID fid = (*env)->GetFieldID(env, oclass, name, "D");
      (*env)->SetDoubleField(env, obj, fid, value);
    }
}
