/* VMClassLoader.java -- Reference implementation of native interface
   required by ClassLoader
   Copyright (C) 1998, 2001, 2002 Free Software Foundation

This file is part of GNU Classpath.

GNU Classpath is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU Classpath is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU Classpath; see the file COPYING.  If not, write to the
Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA.

Linking this library statically or dynamically with other modules is
making a combined work based on this library.  Thus, the terms and
conditions of the GNU General Public License cover the whole
combination.

As a special exception, the copyright holders of this library give you
permission to link this library with independent modules to produce an
executable, regardless of the license terms of these independent
modules, and to copy and distribute the resulting executable under
terms of your choice, provided that you also meet, for each linked
independent module, the terms and conditions of the license of that
module.  An independent module is a module which is not derived from
or based on this library.  If you modify this library, you may extend
this exception to your version of the library, but you are not
obligated to do so.  If you do not wish to do so, delete this
exception statement from your version. */

package java.lang;

import java.security.ProtectionDomain;
import java.net.URL;
import java.io.IOException;
import java.net.MalformedURLException;
import java.util.Enumeration;
import java.util.Map;
import java.util.HashMap;
import java.util.Vector;
import java.util.StringTokenizer;
import java.util.NoSuchElementException;
import java.lang.reflect.Constructor;
import java.lang.InternalError;
import java.lang.DefaultClassLoader; 

import gnu.java.util.EmptyEnumeration;

import kissme.vm.loaders.FileClassLoader;
import kissme.vm.loaders.JarClassLoader;
import kissme.vm.loaders.ZipClassLoader;


/**
 * java.lang.VMClassLoader is a package-private helper for VMs to implement
 * on behalf of java.lang.ClassLoader.
 *
 * @author John Keiser
 * @author Mark Wielaard <mark@klomp.org>
 * @author Eric Blake <ebb9@email.byu.edu>
 */
final class VMClassLoader
{

  private static class ResourceEnumeration implements Enumeration 
  {
    private String[] urlStrings;
    private int next;

    public ResourceEnumeration(String[] urls) 
    {
      urlStrings = urls;
    }

    public boolean hasMoreElements() 
    {
      return next < urlStrings.length;
    }

    public Object nextElement() 
    {
      if (next == urlStrings.length) 
	throw new NoSuchElementException("no more resources");
      else 
	try 
	  {
	    return new URL(urlStrings[next++]);
	  }
	catch (MalformedURLException ex) 
	  {
	    InternalError ex2 = 
	      new InternalError("The VM has returned a bad resource URL: " +
				urlStrings[next - 1]);
	    ex2.initCause(ex);
	    throw ex2;
	  }
    }
  }

  private static String[] bootClasspathComponents;

  /**
   * Helper to define a class using a string of bytes. This assumes that
   * the security checks have already been performed, if necessary.
   * <strong>This method will be removed in a future version of GNU
   * Classpath</strong>.
   *
   * @param name the name to give the class, or null if unknown
   * @param data the data representing the classfile, in classfile format
   * @param offset the offset into the data where the classfile starts
   * @param len the length of the classfile data in the array
   * @return the class that was defined
   * @throws ClassFormatError if data is not in proper classfile format
   * @deprecated Implement
   * {@link #defineClass(ClassLoader, String, byte[], int, int, ProtectionDomain)}
   *   instead.
   */
  static final native Class defineClass(ClassLoader cl, String name,
                                        byte[] data, int offset, int len)
    throws ClassFormatError;

  /**
   * Helper to define a class using a string of bytes. This assumes that
   * the security checks have already been performed, if necessary.
   *
   * <strong>For backward compatibility, this just ignores the protection
   * domain; that is the wrong behavior, and you should directly implement
   * this method natively if you can.</strong>
   *
   * @param name the name to give the class, or null if unknown
   * @param data the data representing the classfile, in classfile format
   * @param offset the offset into the data where the classfile starts
   * @param len the length of the classfile data in the array
   * @param pd the protection domain
   * @return the class that was defined
   * @throws ClassFormatError if data is not in proper classfile format
   */
  static final Class defineClass(ClassLoader cl, String name,
                                 byte[] data, int offset, int len,
                                 ProtectionDomain pd)
    throws ClassFormatError
  {
    return defineClass(cl, name, data, offset, len);
  }

  /**
   * Helper to resolve all references to other classes from this class.
   *
   * @param c the class to resolve
   */
  static final native void resolveClass(Class c);

  /**
   * Helper to load a class from the bootstrap class loader.
   *
   * @param name the class name to load
   * @param resolve whether to resolve it
   * @return the class, loaded by the bootstrap classloader or null
   * if the class wasn't found. Returning null is equivalent to throwing
   * a ClassNotFoundException (but a possible performance optimization).
   */
  static final native Class loadClass(String name, boolean resolve)
    throws ClassNotFoundException;

  private static String[] getBootClasspathComponents()
  {
    if (bootClasspathComponents == null) 
      {
	String classpath = System.getProperty("java.boot.class.path");
	if (classpath == null) 
	  {
	    throw new InternalError("java.boot.class.path is null");
	  }
	String pathSeparator = System.getProperty("path.separator");
	StringTokenizer st = 
	  new StringTokenizer(classpath, pathSeparator, false);
	int nosComponents = st.countTokens();
	
	st = new StringTokenizer(classpath, pathSeparator, false);
	bootClasspathComponents = new String[nosComponents];
	for (int i = 0; i < nosComponents; i++) 
	  {
	    bootClasspathComponents[i] = st.nextToken();
	  }
      }
    return bootClasspathComponents;
  }

  /**
   * Helper to find a resource from the bootstrap classpath.
   *
   * @param name the resource to find
   * @return the URL to the resource
   */
  static URL getResource(String name) 
  {
    String urlString = null;
    String[] components = getBootClasspathComponents();
    
    for (int i = 0; i < components.length; i++) 
      {
	String component = components[i];
	if (component.endsWith(".jar"))
	  urlString = JarClassLoader.getResourceUrl(name, component);
	else if (component.endsWith(".zip")) 
	  urlString = ZipClassLoader.getResourceUrl(name, component);
	else
	  urlString = FileClassLoader.getResourceUrl(name, component);

	if (urlString != null)
	  break;
      }
    
    if (urlString == null) 
      return null;

    try 
      {
	return new URL(urlString);
      }
    catch (MalformedURLException ex) 
      {
	InternalError ex2 = 
	  new InternalError("Loader returned a bad resource URL: " + urlString);
	ex2.initCause(ex);
	throw ex2;
      }
  }
  

  /**
   * Helper to find all instances of a resource from the bootstrap classpath.
   *
   * @param name the resource to find
   * @return an enumeration of resources
   * @throws IOException if one occurs
   */
  static Enumeration getResources(String name) throws IOException
  {
    String urlString = null;
    String[] components = getBootClasspathComponents();
    Vector urls = new Vector();
    
    for (int i = 0; i < components.length; i++) 
      {
	String component = components[i];
	if (component.endsWith(".jar"))
	  urlString = JarClassLoader.getResourceUrl(name, component);
	else if (component.endsWith(".zip")) 
	  urlString = ZipClassLoader.getResourceUrl(name, component);
	else
	  urlString = FileClassLoader.getResourceUrl(name, component);

	// XXX - need to check that the resource can be accessed by
	// the class loader.  The spec says that innaccessible resources
	// should not be returned
	if (urlString != null)
	  try 
	    {
	      urls.addElement(new URL(urlString));
	    }
	  catch (MalformedURLException ex) 
	    {
	      InternalError ex2 = 
		new InternalError("Loader returned a bad resource URL: " + 
				  urlString);
	      ex2.initCause(ex);
	      throw ex2;
	    }
      }
    return urls.elements();
  }
  

  /**
   * Helper to get a package from the bootstrap class loader.  The default
   * implementation of returning null may be adequate, or you may decide
   * that this needs some native help.
   *
   * @param name the name to find
   * @return the named package, if it exists
   */
  static Package getPackage(String name)
  {
    return null;
  }

  /**
   * Helper to get all packages from the bootstrap class loader.  The default
   * implementation of returning an empty array may be adequate, or you may
   * decide that this needs some native help.
   *
   * @return all named packages, if any exist
   */
  static Package[] getPackages()
  {
    return new Package[0];
  }

  /**
   * Helper for java.lang.Integer, Byte, etc to get the TYPE class
   * at initialization time. The type code is one of the chars that
   * represents the primitive type as in JNI.
   *
   * <ul>
   * <li>'Z' - boolean</li>
   * <li>'B' - byte</li>
   * <li>'C' - char</li>
   * <li>'D' - double</li>
   * <li>'F' - float</li>
   * <li>'I' - int</li>
   * <li>'J' - long</li>
   * <li>'S' - short</li>
   * <li>'V' - void</li>
   * </ul>
   *
   * Note that this is currently a java version that converts the type code
   * to a string and calls the native <code>getPrimitiveClass(String)</code>
   * method for backwards compatibility with VMs that used old versions of
   * GNU Classpath. Please replace this method with a native method
   * <code>final static native Class getPrimitiveClass(char type);</code>
   * if your VM supports it. <strong>The java version of this method and
   * the String version of this method will disappear in a future version
   * of GNU Classpath</strong>.
   *
   * @param type the primitive type
   * @return a "bogus" class representing the primitive type
   */
  final static native Class getPrimitiveClass(char type);

  /**
   * The system default for assertion status. This is used for all system
   * classes (those with a null ClassLoader), as well as the initial value for
   * every ClassLoader's default assertion status.
   *
   * XXX - Not implemented yet; this requires native help.
   *
   * @return the system-wide default assertion status
   */
  static final boolean defaultAssertionStatus()
  {
    return true;
  }

  /**
   * The system default for package assertion status. This is used for all
   * ClassLoader's packageAssertionStatus defaults. It must be a map of
   * package names to Boolean.TRUE or Boolean.FALSE, with the unnamed package
   * represented as a null key.
   *
   * XXX - Not implemented yet; this requires native help.
   *
   * @return a (read-only) map for the default packageAssertionStatus
   */
  static final Map packageAssertionStatus()
  {
    return new HashMap();
  }

  /**
   * The system default for class assertion status. This is used for all
   * ClassLoader's classAssertionStatus defaults. It must be a map of
   * class names to Boolean.TRUE or Boolean.FALSE
   *
   * XXX - Not implemented yet; this requires native help.
   *
   * @return a (read-only) map for the default classAssertionStatus
   */
  static final Map classAssertionStatus()
  {
    return new HashMap();
  }

  static ClassLoader getSystemClassLoader()
  {
    // This method is called as the initialization of systemClassLoader,
    // so if there is a null value, this is the first call and we must check
    // for java.system.class.loader.
    final String DEFAULT_CLASS_NAME = "java.lang.DefaultClassLoader";
    String loader = System.getProperty("java.system.class.loader",
				       DEFAULT_CLASS_NAME);
    try
      {
	// Give the new system class loader a null parent.
	Constructor c = Class.forName(loader).getConstructor
	  ( new Class[] { ClassLoader.class } );
	return (ClassLoader) c.newInstance(new Object[1]);
      }
    catch (Exception e)
      {
	try
	  {
	    // Fallback to kissme.java.lang.DefaultClassLoader.
	    System.err.println("Instantiating the system classloader "
			       + loader + " has failed, trying "
			       + DEFAULT_CLASS_NAME);
	    e.printStackTrace();
	    return new DefaultClassLoader(null);
	  }
	catch (Exception e1)
	  {
            System.err.println("Instantiating the system classloader "
			       + DEFAULT_CLASS_NAME + " also has failed");
	    e1.printStackTrace();
	    throw (Error) new InternalError
	      ("System class loader could not be found: " + e1)
	      .initCause(e1);
	  }
      }
 
  }
}
