/*
* audio_filters.h -- audio filters
* Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Project Includes
#include <gnome.h>
#include "audio_filters.h"
#include "page_magick.h"

// C Includes

#include <string.h>
#include <stdio.h>

/** Class for simple pass through of audio.
*/

class AudioFilterKeep : public AudioFilter
{
public:
	char *GetDescription( ) const
	{
		return _( "No Change" );
	}

	void GetFrame( int16_t **buffer, int frequency, int channels, int samples, double position, double frame_delta )
	{}
}
;

/** Silences the audio sample.
*/

class AudioFilterSilence : public AudioFilter
{
public:
	char *GetDescription( ) const
	{
		return _( "Silence" );
	}

	void GetFrame( int16_t **buffer, int frequency, int channels, int samples, double position, double frame_delta )
	{
		for ( int i = 0; i < channels; i ++ )
			for ( int j = 0; j < samples; j ++ )
				buffer[ i ][ j ] = 0;
	}
};

/** Fade out.
*/

class AudioFilterFadeOut : public AudioFilter
{
public:
	char *GetDescription( ) const
	{
		return _( "Fade Out" );
	}

	void GetFrame( int16_t **buffer, int frequency, int channels, int samples, double position, double frame_delta )
	{
		for ( int i = 0; i < channels; i ++ )
			for ( int j = 0; j < samples; j ++ )
				buffer[ i ][ j ] = ( int ) ( buffer[ i ][ j ] * ( 1 - position ) );
	}
};

/** Fade In.
*/

class AudioFilterFadeIn : public AudioFilter
{
public:
	char *GetDescription( ) const
	{
		return _( "Fade In" );
	}

	void GetFrame( int16_t **buffer, int frequency, int channels, int samples, double position, double frame_delta )
	{
		for ( int i = 0; i < channels; i ++ )
			for ( int j = 0; j < samples; j ++ )
				buffer[ i ][ j ] = ( int ) ( buffer[ i ][ j ] * position );
	}
};

/** Callback for selection change.
*/

static void
on_optionmenu_selected ( GtkMenuItem *menu_item, gpointer user_data )
{
	( ( GDKAudioFilterRepository * ) user_data ) ->SelectionChange();
}

/** Constructor for the audio filter repository.
 
  	Registers an instance of each known filter for later GUI exposure via the Initialise method.
*/

GDKAudioFilterRepository::GDKAudioFilterRepository() : selected_filter( NULL ), menu( NULL ), container( NULL )
{
	printf( ">> audio filter repository created\n" );
	// Register an instance of each object (adapting raw filters to GDK filters where necessary)
	Register( new GDKAudioFilterAdapter( new AudioFilterKeep() ) );
	Register( new GDKAudioFilterAdapter( new AudioFilterSilence() ) );
	Register( new GDKAudioFilterAdapter( new AudioFilterFadeIn() ) );
	Register( new GDKAudioFilterAdapter( new AudioFilterFadeOut() ) );
}

/** Destructor for the image repository - clears all the registered filters
*/

GDKAudioFilterRepository::~GDKAudioFilterRepository()
{
	printf( ">> audio filter repository destroyed\n" );
	// Remove the filters in the repository
	for ( unsigned int index = 0; index < filters.size(); index ++ )
		delete filters[ index ];
}

/** Register an audio filter.
*/

void GDKAudioFilterRepository::Register( GDKAudioFilter *filter )
{
	printf( ">>> Audio Filter: %s\n", filter->GetDescription() );
	filters.push_back( filter );
}

/** Initialise the option menu with the current list of registered filters.
*/

void GDKAudioFilterRepository::Initialise( GtkOptionMenu *menu, GtkBin *container )
{
	// Store these for future reference
	this->menu = menu;
	this->container = container;

	// Add the filters to the menu
	GtkMenu *menu_new = GTK_MENU( gtk_menu_new( ) );
	for ( unsigned int index = 0; index < filters.size(); index ++ )
	{
		GtkWidget *item = gtk_menu_item_new_with_label( filters[ index ] ->GetDescription( ) );
		gtk_widget_show( item );
		gtk_menu_append( menu_new, item );
		g_signal_connect( G_OBJECT( item ), "activate", G_CALLBACK( on_optionmenu_selected ), this );
	}
	gtk_menu_set_active( menu_new, 0 );
	gtk_option_menu_set_menu( menu, GTK_WIDGET( menu_new ) );

	// Register the selected items widgets
	SelectionChange();
}

/** Get the currently selected audio filter.
*/

GDKAudioFilter *GDKAudioFilterRepository::Get( ) const
{
	GtkMenu * filterMenu = GTK_MENU( gtk_option_menu_get_menu( menu ) );
	GtkWidget *active_item = gtk_menu_get_active( filterMenu );
	return filters[ g_list_index( GTK_MENU_SHELL( filterMenu ) ->children, active_item ) ];
}

/** Handle attach/detach widgets on last selected/selected items.
*/

void GDKAudioFilterRepository::SelectionChange( )
{
	// Detach the selected filters widgets
	if ( selected_filter != NULL )
		selected_filter->DetachWidgets( container );

	// Get the new selection
	selected_filter = Get();

	// Inform the main page of the change (should be interface driven)
	if ( common != NULL && common->getPageMagick( ) != NULL )
		common->getPageMagick( ) ->RefreshStatus( );

	// Attach the new filters widgets
	if ( selected_filter != NULL )
		selected_filter->AttachWidgets( container );
}

