/*
 *  Widget-independent player class
 *  Copyright (C) 2002-2003 Tim Jansen <tim@tjansen.de>
 *  Based on gst-player's play.c
 *  Copyright (C) 1999,2000,2001,2002 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000,2001,2002 Wim Taymans <wtay@chello.be>
 *                               2002 Steve Baker <steve@stevebaker.org>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "play.h"
#include "kde/gst/gstreamer.h"
#include "kde/gst/element.h"
#include "kde/gst/elementfactory.h"
#include "kde/gst/bin.h"
#include "kde/gst/pipeline.h"
#include "kde/gst/thread.h"
#include "kde/gst/pad.h"
#include "kde/gst/clock.h"
#include "kde/gst/event.h"
#include "kde/gst/dparamsmooth.h"
#include "kde/gst/dparammanager.h"

#include <qthread.h>
#include <qvaluelist.h>

using namespace KDE::GST;
using namespace KDE::GST::DP;
using namespace KDE::GSTPlay;

const int VideoSizeEventType = 784001;
class VideoSizeEvent : public QCustomEvent
{
	int m_width, m_height;
public:
	VideoSizeEvent(int w, int h) : 
		QCustomEvent(VideoSizeEventType),
		m_width(w),
		m_height(h)
	{};
	int width() { return m_width; }
	int height() { return m_height; }
};

const int XidEventType = 784002;
class XidEvent : public QCustomEvent
{
	int m_xid;
public:
	XidEvent(int xid) : 
		QCustomEvent(XidEventType),
		m_xid(xid)
	{};
	int xid() { return m_xid; }
};

class KDE::GSTPlay::PlayPrivate {
public:
	bool (Play::* setupPipeline)();
	bool (Play::* setAutoplugger)(KDE::GST::Element *autoplugger);
	bool (Play::* setVideoSink)(KDE::GST::Element *videosink);
	bool (Play::* setAudioSink)(KDE::GST::Element *audiosink);

	QTimer timer200ms;

	/* core elements */
	KDE::GST::Bin *pipeline;
	KDE::GST::Element *volume;
	KDE::GST::Element *source;
	KDE::GST::Element *autoplugger;
	KDE::GST::Element *videoSink;
	KDE::GST::Element *audioSink;
	KDE::GST::Element *colorspace;

	QValueList<KDE::GST::Element*> seekableElements;

	DParamManager *volDpman;
	DParam *volDparam;
	
	QMap<QString,KDE::GST::Element*> otherElements;

	QMutex audioBinMutex, videoBinMutex;

	bool needStreamLength;
 	int timeSeconds;
	long long seekTime;
 	long long timeNanos;
 	long long lengthNanos;

	PlayPrivate() :
		pipeline(0),
		volume(0),
		source(0),
		autoplugger(0),
		videoSink(0),
		audioSink(0),
		colorspace(0),
		needStreamLength(false),
		timeSeconds(0),
		seekTime(0),
		timeNanos(0LL),
		lengthNanos(0LL) {
	}
};


Play::Play(PipeType pipeType, QObject *parent, const char *name) :
	QObject(parent, name)
{
	GStreamer::init(0, 0);

	d = new PlayPrivate();

	switch (pipeType){
	case PIPE_VIDEO:
		d->setupPipeline = &Play::setupForVideoNormal;
		d->setAutoplugger = &Play::setAutoForVideoNormal;
		d->setVideoSink = &Play::setVideoForVideoNormal;
		d->setAudioSink = &Play::setAudioForVideoNormal;
		break;
	case PIPE_AUDIO_THREADED:
		d->setupPipeline = &Play::setupForAudioThreaded;
		d->setAutoplugger = &Play::setAutoForAudioThreaded;
		d->setVideoSink = 0;
		d->setAudioSink = &Play::setAudioForAudioThreaded;
		break;
	case PIPE_AUDIO_BUFFER_THREADED:
		d->setupPipeline = &Play::setupForAudioBufferThreaded;
		d->setAutoplugger = &Play::setAutoForAudioBufferThreaded;
		d->setVideoSink = 0;
		d->setAudioSink = &Play::setAudioForAudioBufferThreaded;
		break;
	default:
		Q_ASSERT(false);
	}

	/* connect timers */
	connect(&d->timer200ms, SIGNAL(timeout()), SLOT(slotTick()));
	connect(&d->timer200ms, SIGNAL(timeout()), SLOT(slotGetLength()));

	/* init pipeline */
	if (!(this->*(d->setupPipeline))()) {
		qDebug("couldn't init pipeline");
		return;
	}

	if (d->pipeline) {
		/* connect to pipeline events */
		connect(d->pipeline, 
			SIGNAL(stateChange(KDE::GST::Element::State,KDE::GST::Element::State)), 
			SLOT(slotStateChange(KDE::GST::Element::State,KDE::GST::Element::State)));
	}

	if (d->volume){
		d->volDpman =  DParamManager::getManager(d->volume);
		d->volDparam = new DParam(DParam::Float);
  
		if (!d->volDpman->attachDParam("volume", d->volDparam)){
			qDebug("could not attach dparam to volume element\n");
		}
		d->volDpman->setMode("asynchronous");
		setVolume(0.9);
	}
}

Play::~Play()
{
	setState(Element::STATE_READY);
	delete d;
}

void Play::slotStateChange(Element::State old, Element::State state)
{
	if (d->pipeline){
		switch (state) {
		case Element::STATE_PLAYING:
			d->timer200ms.start(200);
			break;
		default:
			d->timer200ms.stop();
			break;
		}
	}	

	emit stateChange(old, state);
}

void Play::callbackVideoHaveXid(void *, int xid, Play *play)
{
	QThread::postEvent(play, new XidEvent(xid));
}

void Play::callbackVideoHaveSize(void *, int width, int height, Play *play)
{
	QThread::postEvent(play, new VideoSizeEvent(width, height));
}

void Play::customEvent(QCustomEvent *e) {
	if (e->type() == VideoSizeEventType) {
		int w = ((VideoSizeEvent*)e)->width();
		int h = ((VideoSizeEvent*)e)->height();
		emit haveVideoSize(w, h);
	}
	else if (e->type() == XidEventType) {
		int x = ((XidEvent*)e)->xid();
		emit haveXId(x);
	}
}

void Play::callbackBinPreIterate(Bin*, void *m)
{
	QMutex *mutex = (QMutex*) m;
	mutex->lock();
}

void Play::callbackBinPostIterate(Bin*, void *m)
{
	QMutex *mutex = (QMutex*) m;
	mutex->unlock();
}

void Play::slotGetLength()
{
	if (!d->needStreamLength)
		return; 

	long long value;
	Format format = FORMAT_TIME;
	bool queryWorked = false;

	if (d->audioSink){
		d->audioBinMutex.lock();
		queryWorked = d->audioSink->query(QUERY_TOTAL, 
						  &format, 
						  &value);
		d->audioBinMutex.unlock();
	}
	else if (d->videoSink){
		d->videoBinMutex.lock();
		queryWorked = d->videoSink->query(QUERY_TOTAL, 
						  &format, 
						  &value);
		d->videoBinMutex.unlock();
	}
	if (queryWorked) {
		d->lengthNanos = value;
		d->needStreamLength = false;
		emit streamLength(value);
	}
}

void Play::slotTick()
{
	int secs;
	Clock *clock = d->pipeline->getClock();
	d->timeNanos = clock->getTime();
	secs = (int) (d->timeNanos / Clock::SECOND);
	if (secs != d->timeSeconds){
		d->timeSeconds = secs;
		emit timeTick(d->timeNanos);
	}
}

void Play::seekToTime(long long timeNanos)
{
	Event *sEvent;
	bool seekWorked = true;

	if (timeNanos < 0LL)
		d->seekTime = 0LL;
	else if (timeNanos > d->lengthNanos)
		d->seekTime = d->lengthNanos;
	else
		d->seekTime = timeNanos;

	Element::State prevState = d->pipeline->getState();
	d->pipeline->setState(Element::STATE_PAUSED);

	sEvent = new Event ((Event::SeekType) (FORMAT_TIME |
					       Event::SEEK_METHOD_SET |
					       Event::SEEK_FLAG_FLUSH), 
			    d->seekTime);

	QValueList<KDE::GST::Element*>::iterator it = d->seekableElements.begin();
	while (it != d->seekableElements.end()) {
		sEvent->ref();
		seekWorked &= (*it)->sendEvent(sEvent);
		it++;
	}
	delete sEvent;

	d->pipeline->setState(prevState);
}

void Play::needNewVideoWindow()
{
	if (d->videoSink)
		d->videoSink->setBool("need_new_window", true);
}

bool Play::setVideoSink(Element *videoSink)
{
	if (getState() == Element::STATE_PLAYING)
		setState(Element::STATE_READY);

	if (d->setVideoSink)
		return (this->*(d->setVideoSink))(videoSink);

	/* if there is no setVideoSink func, fail quietly */
	return false;
}

bool Play::setAudioSink(Element *audioSink)
{
	if (getState() == Element::STATE_PLAYING)
	        setState(Element::STATE_READY);

	if (d->setAudioSink)
		return (this->*(d->setAudioSink))(audioSink);

	/* if there is no setAudioSink func, fail quietly */
	return false;
}

Element::State Play::setState(Element::State state)
{
	return (Element::State) d->pipeline->setState(state);
}

Element::State Play::getState()
{
	return d->pipeline->getState();
}

bool Play::setLocation(const QString &location)
{
	Element::State currentState;
	if (location.isNull())
		return false;

	currentState = getState();
	if (currentState != Element::STATE_READY)
		setState(Element::STATE_READY);

	if (d->setAutoplugger) {
		if (! (this->*(d->setAutoplugger))(ElementFactory::make("spider", 
									"autoplugger"))){
			qDebug("couldn't replace autoplugger");
			return false;
		}
	}

	d->source->setString("location", location);

	/* reset time/length values */
	d->timeSeconds = 0;
	d->lengthNanos = 0LL;
	d->timeNanos = 0LL;
	emit streamLength(0LL);
	emit timeTick(0LL);
	d->needStreamLength = true;

	return true;
}

QString Play::getLocation()
{
	return d->source->getString("location");
}


void Play::setVolume(float volume)
{
	d->volDparam->setFloat("value_float", volume);
}

float Play::getVolume()
{
	return d->volDparam->getFloat("value_float");
}

void Play::setMute(bool mute)
{
	d->volume->setBool("mute", mute);
}
	
bool Play::getMute()
{
	return d->volume->getBool("mute");
}

/*  
 *  GST_PLAY_PIPE_AUDIO_THREADED
 *  { filesrc ! spider ! volume ! osssink }
 */
bool Play::setupForAudioThreaded()
{
	d->pipeline = new Thread("mainPipeline");
	Q_ASSERT(d->pipeline);

	/* create source element */
	d->source = ElementFactory::make("filesrc", "source");
	Q_ASSERT(d->source);
	d->pipeline->add(d->source);
	
	/* create audio elements */
	d->volume = ElementFactory::make("volume", "volume");
	if (!d->volume)
		qDebug("You need the volume element to use this program.\n");

	d->audioSink = ElementFactory::make("osssink", "playAudio");
	if (!d->audioSink)
		qDebug("You need the osssink element to use this program.\n");
	connect(d->audioSink, SIGNAL(eos()), SIGNAL(streamEnd()));

	d->pipeline->add(d->volume, d->audioSink);
	d->volume->link(d->audioSink);

	d->seekableElements.append(d->audioSink);
	
	// iterate function must be 
	d->pipeline->setPreIterateFunction(&callbackBinPreIterate, 
					  &d->audioBinMutex);
	d->pipeline->setPostIterateFunction(&callbackBinPostIterate, 
					   &d->audioBinMutex);

	return true;
}

bool Play::setAudioForAudioThreaded(Element *audioSink)
{
	if (d->audioSink)
	{
		d->volume->unlink(d->audioSink);
		d->pipeline->remove(d->audioSink);
	}

	d->audioSink = audioSink;
	d->pipeline->add(audioSink);
	d->volume->link(audioSink);

	return true;
}

bool Play::setAutoForAudioThreaded(Element *autoplugger) {

	if (d->autoplugger){
		/* we need to remove the existing autoplugger before creating a new one */
		d->autoplugger->unlink(d->volume);
		d->autoplugger->unlink(d->source);
		d->pipeline->remove(d->autoplugger);
	}
	
	d->autoplugger = autoplugger;
	if (!d->autoplugger)
		return false;

	d->pipeline->add(autoplugger);
	d->source->link(autoplugger);
	autoplugger->link(d->volume);
	return true;
}

/*  
 *  GST_PLAY_PIPE_AUDIO_BUFFER_THREADED
 *  { filesrc ! spider ! { queue ! volume ! osssink }
 */
bool Play::setupForAudioBufferThreaded()
{
	Thread *audioBin, *workThread;
	Element *audioQueue, *videoQueue;

	d->pipeline = new Pipeline("mainPipeline");
	Q_ASSERT(d->pipeline);

	workThread = new Thread("workThread");
	Q_ASSERT(workThread);
	d->otherElements["workThread"] = workThread;
	
	d->pipeline->add(workThread);

	/* create source element */
	d->source = ElementFactory::make("filesrc", "source");
	Q_ASSERT(d->source);
	workThread->add(d->source);
	
	d->volume = ElementFactory::make("volume", "volume");
	if (!d->volume) {
		qDebug("You need the volume element to use this program.\n");
		return false;
	}

	d->audioSink = ElementFactory::make("osssink", "playAudio");
	Q_ASSERT(d->audioSink);
	connect(d->audioSink, SIGNAL(eos()), SIGNAL(streamEnd()));

	audioQueue = ElementFactory::make("queue", "audioQueue");
	Q_ASSERT(audioQueue);
	d->otherElements["audioQueue"] = audioQueue;
	
	audioBin = new Thread("audioBin");
	Q_ASSERT(audioBin);
	d->otherElements["audioBin"]= audioBin;

	audioBin->setPreIterateFunction(&callbackBinPreIterate, 
					&d->audioBinMutex);
	audioBin->setPostIterateFunction(&callbackBinPostIterate, 
					 &d->audioBinMutex);

	audioBin->add(audioQueue, d->volume, d->audioSink);
	audioQueue->link(d->volume, d->audioSink);
	audioBin->addGhostPad(audioQueue->getPad("sink"), "sink");

	workThread->add(audioBin);

	d->seekableElements.append(d->audioSink);

	return true;
}

bool Play::setAudioForAudioBufferThreaded(Element *audioSink)
{
	Bin *audioBin;
	audioBin = (Bin*) d->otherElements["audioBin"];
	
	if (d->audioSink)
	{
		d->volume->unlink(d->audioSink);
		audioBin->remove(d->audioSink);
	}

	d->audioSink = audioSink;
	audioBin->add(audioSink);
	d->volume->link(audioSink);

	return true;
}

bool Play::setAutoForAudioBufferThreaded(Element *autoplugger)
{
	Element *audioBin;
	Thread *workThread;

	audioBin = d->otherElements["audioBin"];
	workThread = (Thread*)d->otherElements["workThread"];

	if (d->autoplugger){
		/* we need to remove the existing autoplugger before creating a new one */
		d->autoplugger->unlink(audioBin);
		d->autoplugger->unlink(d->source);

		workThread->remove(d->autoplugger);
	}
	
	d->autoplugger = autoplugger;
	if (!d->autoplugger)
		return false;

	workThread->add(autoplugger);
	d->source->link(autoplugger);
	autoplugger->link(audioBin);

	return true;
}

/*
 * GST_PLAY_PIPE_VIDEO
 * { filesrc ! spider ! { queue ! volume ! osssink }
 * spider0.src2 ! { queue ! colorspace ! xvideosink } }
 */

bool Play::setupForVideoNormal()
{
	Thread *audioBin, *workThread, *videoBin;
	Element *audioQueue, *videoQueue;

	d->pipeline = new Pipeline("mainPipeline");
	Q_ASSERT(d->pipeline);

	workThread = new Thread("workThread");
	Q_ASSERT(workThread);
	d->otherElements["workThread"] = workThread;
	
	d->pipeline->add(workThread);

	/* create source element */
	d->source = ElementFactory::make("filesrc", "source");
	Q_ASSERT(d->source);
	workThread->add(d->source);
	
	d->volume = ElementFactory::make("volume", "volume");
	if (!d->volume) {
		qDebug("You need the volume element to use this program.\n");
		return false;
	}

	d->audioSink = ElementFactory::make("osssink", "playAudio");
	Q_ASSERT(d->audioSink);
	connect(d->audioSink, SIGNAL(eos()), SIGNAL(streamEnd()));

	audioQueue = ElementFactory::make("queue", "audioQueue");
	Q_ASSERT(audioQueue);
	d->otherElements["audioQueue"] = audioQueue;
	
	audioBin = new Thread("audioBin");
	Q_ASSERT(audioBin);
	d->otherElements["audioBin"]= audioBin;

	audioBin->setPreIterateFunction(&callbackBinPreIterate, 
					&d->audioBinMutex);
	audioBin->setPostIterateFunction(&callbackBinPostIterate, 
					 &d->audioBinMutex);

	audioBin->add(audioQueue, d->volume, d->audioSink);
	audioQueue->link(d->volume, d->audioSink);
	audioBin->addGhostPad(audioQueue->getPad("sink"), "sink");

	workThread->add(audioBin);

	/* create video elements */
	d->videoSink = ElementFactory::make("xvideosink", "show");
	Q_ASSERT(d->videoSink);
        d->videoSink->setBool("toplevel", false);
	d->videoSink->registerRawSignal("have_xid",
				     (void*)callbackVideoHaveXid, this);
	d->videoSink->registerRawSignal("have_size",
				     (void*)callbackVideoHaveSize, this);
	
	videoQueue = ElementFactory::make("queue", "videoQueue");
	Q_ASSERT(videoQueue);

	d->otherElements["videoQueue"] = videoQueue;

	videoBin = new Thread("videoBin");
	Q_ASSERT(videoBin);
	d->otherElements["videoBin"] = videoBin;

	videoBin->add(videoQueue, d->videoSink);
	
	d->colorspace = ElementFactory::make("colorspace", "colorspace");
	if (!d->colorspace) 
	{
		qDebug("could not create the 'colorspace' element, doing without");
		videoQueue->link(d->videoSink);
	}
	else 
	{
		videoBin->add(d->colorspace);
		videoQueue->link(d->colorspace, d->videoSink);
	}

	videoBin->setPreIterateFunction(&callbackBinPreIterate, 
					&d->videoBinMutex);
	videoBin->setPostIterateFunction(&callbackBinPostIterate, 
					 &d->videoBinMutex);
	
	videoBin->addGhostPad(videoQueue->getPad("sink"),
			      "sink");
	workThread->add(videoBin);

	d->seekableElements.append(d->videoSink);
	d->seekableElements.append(d->audioSink);

	return true;
}

bool Play::setAutoForVideoNormal(Element *autoplugger){

	Element *audioBin, *videoBin;
	Thread *workThread;

	audioBin = d->otherElements["audioBin"];
	videoBin = d->otherElements["videoBin"];
	workThread = (Thread*)d->otherElements["workThread"];

	if (d->autoplugger){
		/* we need to remove the existing autoplugger before creating a new one */
		d->autoplugger->unlink(audioBin);
		d->autoplugger->unlink(d->source);
		d->autoplugger->unlink(videoBin);

		workThread->remove(d->autoplugger);
	}
	
	d->autoplugger = autoplugger;
	if (!d->autoplugger)
		return false;

	workThread->add(autoplugger);
	d->source->link(autoplugger);
	autoplugger->link(audioBin);
	autoplugger->link(videoBin);

	return true;
}

bool Play::setVideoForVideoNormal(Element *videoSink)
{
	Element *videoMate;
	if (d->colorspace){
		qDebug("reattach to colorspace\n");
		videoMate = d->colorspace;
	}
	else {
		qDebug("reattach to videoQueue\n");
		videoMate = d->otherElements["videoQueue"];
	}

	if (d->videoSink){
		videoMate->unlink(d->videoSink);
		d->pipeline->remove(d->videoSink);
	}
	d->videoSink = videoSink;
	d->pipeline->add(videoSink);
	videoMate->link(videoSink);

	return true;
}

bool Play::setAudioForVideoNormal(Element *audioSink)
{
	Bin *audioBin;
	audioBin = (Bin*) d->otherElements["audioBin"];
	
	if (d->audioSink)
	{
		d->volume->unlink(d->audioSink);
		audioBin->remove(d->audioSink);
	}

	d->audioSink = audioSink;
	audioBin->add(audioSink);
	d->volume->link(audioSink);

	return true;
}

#include "play.moc"
