/*
 *  Wrapper base class for GObjects
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_WRAPPER_H
#define KDE_GST_WRAPPER_H

#include <qobject.h>
#include <qmap.h>

struct _GObject;


namespace KDE {
namespace GST {
	class Wrapper;
	typedef Wrapper* WrapObjectFactory(void *gobject);

	class CallbackManager;

/**
 * Wrapper is a base class for wrapping GObjects and allows
 * you to access some of its basic features.
 * 
 * @short Wrapper for GObject
 * @see SimpleWrapper
 */
	class Wrapper : public QObject {
	Q_OBJECT
	private:
		static QMap<QString,WrapObjectFactory*> classMap;
	        static QMap<void*,Wrapper*> instanceMap;

		void removeFromMap();
		void addToMap();
		static Wrapper* getFromMap(void *real);
		static Wrapper* createWrapObject(void *real);

		friend class GStreamer;
		static Wrapper* wrapperFactory(void *real);

		/**
		 * Registers a new GObject and its wrapper
		 */
		static void registerType(const QString &name, 
					 WrapObjectFactory* f);

		void *reserved;

	protected:
		/**
		 * Pointer to the GObject.
		 */
	        void *m_real;

		/**
		 * Registers the wrappers when created without
		 * real object.
		 */
		void registerReal(); 

		/** 
		 * Private class to manage callback wrappers and 
		 * their data. Implementations should inherit
		 * callbackmanager.h
		 */ 
		CallbackManager *m_cbm;

		/**
		 * Returns the callback manager, creates a new one
		 * if neccessary.
		 */
		CallbackManager *getCBManager();		

		/**
		 * Maps registered signal callbacks -> handler ids
		 */
		QMap<QString, unsigned long> *m_signalMap;

		void *d; // reserved
		
	public:
/**
 * Creates a new Wrapper that wrapps the given GObject. Usually
 * you really dont want to call this, use @ref #wrap instead.
 * You must never create a object of this class on the stack, always 
 * use new.
 * @param real the GObject to be wrapped. Can be null, you need to 
 *             call @ref #registerReal later when you set m_real.
 */
	        Wrapper(void *real);
	        virtual ~Wrapper();

/**
 * Creates a new Wrapper that wrapps the given GObject. Unlike the
 * constructor this returns the appropriate sub-class for the 
 * GObject, so when you give a GstElement you will get a KDE::GST::Element 
 * instead of a KDE::GST::Wrapper. And if there is already a wrapper
 * object it will be re-used.
 * @param real the GObject to be wrapped
 * @return the Wrapper (or derivative)
 */
		static Wrapper* wrap(void *real);

/**
 * Returns the original GStreamer object that is wrapped. You need to cast
 * it to the appropriate GObject.
 * @return the wrapped object
 */
		void *realObject() const;		

/**
 * Deletes the wrapper object without deleting the real GStreamer object. 
 * All callbacks will be unregistered, expect those registered with
 * @ref #registerRawCallback.
 * @param weakUnref internal, always use true unless you know what you are 
 *                  doing
 * @return the real GStreamer object
 */
		void *disassociate(bool weakUnref = true);
	

/**
 * Allows you to register a callback for the signal with the given name.
 * The signal that you get is a raw GObject callback, with the original 
 * arguments. You may want to @ref #wrap them before you use them. 
 * Most wrapped GStreamer objects have convenience functions for signals,
 * and usually you should only need this function when you are registering
 * the signal of a custom Element.
 * The callback will be called in the Element's @ref Thread, so you can not use
 * any Qt/KDE functionality unless it is thread-safe.
 *
 * @param name the name of the callback
 * @param callback the function that will be called when an error occurred.
 *                 The exact type of the function depends on the signal. 
 * @param cookie a pointer that will be passed to the callback function
 * @return true if registration was successful
 */
		bool registerRawSignal(const QString &name,
				       void *callback,
				       void *cookie);

/**
 * Disconnects the callback for the StateChange event.
 *
 * @param name the name of the callback
 * @param callback the callback function
 * @param cookie a pointer that will be passed to the callback function
 */
		void unregisterRawSignal(const QString &name,
					 void *callback,
					 void *cookie);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setChar(const QString &name, char value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setUChar(const QString &name, unsigned char value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setBool(const QString &name, bool value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setInt(const QString &name, int value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setUInt(const QString &name, unsigned int value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setLong(const QString &name, long value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setULong(const QString &name, unsigned long value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setInt64(const QString &name, long long value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setUInt64(const QString &name, unsigned long long value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setFloat(const QString &name, float value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setDouble(const QString &name, double value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setString(const QString &name, const QString &value);

/**
 * Sets the property of the object.
 * @param name name of the property
 * @param value the value to be set
 */
		void setPointer(const QString &name, void *pointer);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		char getChar(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		unsigned char getUChar(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		int getInt(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		unsigned int getUInt(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		long getLong(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		unsigned long getULong(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		bool getBool(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		float getFloat(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		double getDouble(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
		QString getString(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
	        long long getInt64(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
	        unsigned long long getUInt64(const QString &name);

/**
 * Retrieves the property of the object.
 * @param name the name of the property
 * @return the value of the property
 */
	        void *getPointer(const QString &name);
	};
}
}

#endif
