/*
 *  Wrapper base class for GstType
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_TYPE_H
#define KDE_GST_TYPE_H

#include <kde/gst/simplewrapper.h>
#include <qvaluevector.h>

namespace KDE {
namespace GST {

/**
 * Types exist to try to make sure data everyone is talking about the
 * right kind of data.  They aid quite a bit in autoplugging, in fact make it
 * possible.  Each well-formed type includes a function (typefind) that will
 * take one or more buffers and determine whether or not it is indeed a
 * stream of that type, and possible a metadata to go with it.  It may
 * provide related metadata structure IDs (and must if it provides metadata
 * from the typefind function).
 *
 * The point of returning an existing MIME type is a result of the goal of
 * unifying types enough to guarantee that, for instance, all MP3 decoders
 * will work interchangably.  If MP3 decoder A says "MIME type 'audio/mpeg'
 * with extensions 'mpeg3'" and decoder B says "MIME type 'audio/mpeg' with
 * extensions 'mp3'", we don't want to have two types defined, possibly with
 * two typefind functions. If we did, it's not obvious which of the two would
 * be tried first (luck) and if both would really identify streams as mp3
 * correctly in all cases.  And whichever wins, we're stuck using the
 * associated decoder to play that stream.  We lose the choice between any
 * valid mp3 decoder, and thus the whole point of the type system.
 * 
 * @short Identifies the data
 */
	class Type : public SimpleWrapper {
	private:
		void *reserved;
			
	public:
/**
 * Creates a new Type that wrapps the given GstType.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstType to be wrapped.
 */
	        Type(void *real);
	        virtual ~Type();

/**
 * Creates a new Type that wrapps the given object. Unlike the
 * constructor this re-used already existing objects.
 * @param real the object to be wrapped
 * @return the Type
 */
		static Type *wrap(void *real);

/**
 * Returns the type's id.
 *
 * @return the id
 */
		unsigned short id() const;

/**
 * Returns the type's mime type.
 *
 * @return the mime
 */
		QString mime() const;

/**
 * Returns the type's supported extensions.
 *
 * @return the extensions, space-delimited
 */
		QString extensions() const;

/**
 * Find the type id of a given mime type.
 *
 * @param mime the mime type
 * @return the type id
 */
		static unsigned short findByMime(const QString &mime);

/**
 * Find the type id of a given extention.
 *
 * @param ext the extension to find
 * Returns: the type id
 */
		static unsigned short findByExt(const QString &ext);

/**
 * Find the type of a given id.
 *
 * @param id the id to find
 * @return the type (or NULL if not found)
 */
		static Type* findById(unsigned short id);

/**
 * Return a list of all registered types.
 *
 * @return a vector with pointers to all types (shallow copy)
 */
		static QValueVector<Type*> getList();
	};
}
}

#endif
