/*
 *  Wrapper/interface class for GstEvent
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_EVENT_H
#define KDE_GST_EVENT_H

#include <kde/gst/simplewrapper.h>
#include <kde/gst/format.h>

#define KDE_GST_SEEK_METHOD_SHIFT	16
#define KDE_GST_SEEK_FLAGS_SHIFT	20

namespace KDE {
namespace GST {
	class Object;

/**
 * The event classes are used to construct and query events.
 *
 * @short Event definitions.
 * @see Pad
 */
	class Event : public SimpleWrapper {
	private:
		void *reserved;
			
	public:
		enum EventType {
			EVENT_UNKNOWN,
			EVENT_EOS,
			EVENT_FLUSH,
			EVENT_EMPTY,
			EVENT_DISCONTINUOUS,
			EVENT_NEW_MEDIA,
			EVENT_QOS,
			EVENT_SEEK,
			EVENT_FILLER,
			EVENT_SIZE
		};

		enum SeekType {
			SEEK_METHOD_CUR		= (1 << KDE_GST_SEEK_METHOD_SHIFT),
			SEEK_METHOD_SET		= (2 << KDE_GST_SEEK_METHOD_SHIFT),
			SEEK_METHOD_END		= (3 << KDE_GST_SEEK_METHOD_SHIFT),
			SEEK_FLAG_FLUSH		= (1 << (KDE_GST_SEEK_FLAGS_SHIFT + 0)),
			SEEK_FLAG_ACCURATE	= (1 << (KDE_GST_SEEK_FLAGS_SHIFT + 1)),
			SEEK_FLAG_KEY_UNIT	= (1 << (KDE_GST_SEEK_FLAGS_SHIFT + 2))
		};

		enum SeekAccuracy {
			GST_SEEK_CERTAIN,
			GST_SEEK_FUZZY
		};

/**
 * Creates a new Event that wrapps the given GstEvent.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstEvent to be wrapped.
 */
	        Event(void *real);
	        virtual ~Event();

/**
 * Creates a new Event that wrapps the given object. Unlike the
 * constructor this re-used already existing objects.
 * @param real the object to be wrapped
 * @return the Event
 */
		static Event *wrap(void *real);

/**
 * Allocate a new event of the given type.
 *
 * @param type The type of the new event
 */
		Event(EventType type);

/**
 * Copy the event
 *
 * @return A copy of the event.
 */
		Event* copy();

/**
 * Unreference the object, delete it if no references are left.
 */
		void unref();

/**
 * Increase the object's reference count by 1.
 * @return this object
 */
		Event *ref();

/**
 * Increase the object's reference count by @p count.
 * @return this object
 */
		Event *ref(int count);

/**
 * Allocate a new seek event with the given parameters.
 *
 * @param type The type of the seek event
 * @param offset The offset of the seek
 */
		Event(SeekType type, long long offset);

/**
 * Allocate a new segment seek event with the given parameters. 
 *
 * @param type The type of the seek event
 * @param start The start offset of the seek
 * @param stop The stop offset of the seek
 */
		Event(SeekType type, long long start, long long stop);

/**
 * Allocate a new discontinuous event with the geven format/value pairs.
 * 
 * @param newMedia: A flag indicating a new media type starts
 * @param format1: The first format of the discont value
 * @param format2: The second format of the discont value, or FORMAT_NONE
 * @param format3: The third format of the discont value, or FORMAT_NONE
 * @param format4: The 4th format of the discont value, or FORMAT_NONE
 * @param format5: The 5th format of the discont value, or FORMAT_NONE
 */
		Event(bool newMedia, 
		      Format format1,
		      Format format2,
		      Format format3,
		      Format format4,
		      Format format5);


/**
 * Get the value for the given format in the discont event.
 *
 * @param format The format of the discont value
 * @param value A pointer to the value
 * @return TRUE if the discont event caries the specified format/value pair.
 */
		bool discontGetValue(Format format, long long *value);

/**
 * Return the type of the event.
 *
 * @return the event's type
 */
		EventType type();

/**
 * Return the source of the event.
 *
 * @return the event's source
 */
		Object* source();

/**
 * Return the timestamp of the event
 *
 * @return the event's timestamp
 */
		unsigned long long timestamp();

/**
 * Return the type of the seek event.
 *
 * @return the seek event's type (undefined if not a seek event)
 */
		SeekType seekType();

/**
 * Return the format of the seek event.
 *
 * @return the seek event's format (undefined if not a seek event)
 */
		int seekFormat();

/**
 * Return the method of the seek event. (see SeekType)
 *
 * @return the seek event's method (undefined if not a seek event)
 */
		SeekType seekMethod();

/**
 * Return the flags of the seek event. (see SeekType)
 *
 * @return the seek event's flags (undefined if not a seek event)
 */
		SeekType seekFlags();

/**
 * Return the offset of the seek event.
 *
 * @return the seek event's offset (undefined if not a seek event)
 */
		long long seekOffset();

/**
 * Return the accuracy of the seek event.
 *
 * @return the seek event's accuracy (undefined if not a seek event)
 */
		SeekAccuracy seekAccuracy();
	};
}
}

#endif
