/*
 *  Interface/wrapper for GStreamer GstElement
 *  Copyright (C) 2002-2003 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_ELEMENT_H
#define KDE_GST_ELEMENT_H

#include <kde/gst/object.h>
#include <kde/gst/clock.h>
#include <kde/gst/pad.h>
#include <kde/gst/query.h>
#include <kde/gst/format.h>

#include <qvaluevector.h>

namespace KDE {
namespace GST {
	class Bin;
	class ElementFactory;
	class PadTemplate;
	class Caps;
	class Event;

/**
 * Element is the base class needed to construct an element that can be
 * used in a GStreamer @ref Pipeline.  As such, it is not a functional entity, 
 * and cannot do anything when placed in a pipeline.
 * 
 * The name of a Element can be get with @ref #getName() and set with
 * @ref #setName(). 
 * 
 * All elements have pads (of the type @ref Pad). These pads kubj to pads on
 * other elements.  Buffers flow between these linked pads.
 * A Element has a list of @ref Pad structures for all their input (or sink)
 * and output (or source) pads.  
 * Core and plug-in writers can add and remove pads with @ref #addPad()
 * and @ref #removePad().
 * Application writers can manipulate ghost pads (copies of real pads inside a 
 * bin) @ref #addGhostPad() and @ref #removeGhostPad().
 * A pad of an element can be retrieved by name with @ref #getPad().
 * A list of all pads can be retrieved with @ref #getPadList().
 * 
 * Elements can be linked through their pads.
 * If the link is straightforward, use the @ref #link() 
 * convenience function to link two or more elements.
 * Use @ref #linkFiltered() to link two elements constrained by
 * a specified set of @ref Caps.
 * For finer control, use @ref #linkPads() 
 * to specify the pads to link on each element by name.
 *
 * Each element has a state (see @ref ElementState).  You can get and set the 
 * state of an element with @ref #getState() and @ref #setState().  
 * You can wait for an element to change it's state with @ref #waitStateChange().
 * To get a string representation of a @ref ElementState, use 
 * @ref #stateGetName().
 * 
 * You can get and set a @ref Clock on an element using @ref #getClock()
 * and @ref #setClock(). You can wait for the clock to reach a given
 * @ref ClockTime using @ref #clockWait().
 * 
 * @short Base class for all pipeline elements
 * @see ElementFactory
 * @see Pad
 * @see Pipeline
 */
	class Element : public Object {
	Q_OBJECT
	private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		void *reserved;	

	protected:
		void connectNotify(const char *signal);
		void customEvent(QCustomEvent *);

	public:

	enum State {
		STATE_VOID_PENDING        = 0,
		STATE_NULL                = (1 << 0),
		STATE_READY               = (1 << 1),
		STATE_PAUSED              = (1 << 2),
		STATE_PLAYING             = (1 << 3)
	};

	enum Flags {
		/* element is complex (for some def.) and generally require a cothread */
		FLAG_COMPLEX		= Object::FLAG_LAST,
		/* input and output pads aren't directly coupled to each other
		   examples: queues, multi-output async readers, etc. */
		FLAG_DECOUPLED,
		/* this element should be placed in a thread if at all possible */
		FLAG_THREAD_SUGGESTED,
		/* this element, for some reason, has a loop function that performs
		 * an infinite loop without calls to element_yield () */
		FLAG_INFINITE_LOOP,
		/* there is a new loopfunction ready for placement */
		FLAG_NEW_LOOPFUNC,
		/* if this element can handle events */
		FLAG_EVENT_AWARE,
		/* use threadsafe property get/set implementation */
		FLAG_USE_THREADSAFE_PROPERTIES,

		/* private flags that can be used by the scheduler */
		FLAG_SCHEDULER_PRIVATE1,
		FLAG_SCHEDULER_PRIVATE2,
		
		/* use some padding for future expansion */
		FLAG_LAST		= Object::FLAG_LAST + 16
	};

/**
 * Creates a new Element that wrapps the given GstElement. Usually
 * you really don't want to call this, use @ref #wrap instead.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstElement to be wrapped
 */
	        Element(void *real);
	        virtual ~Element();

/**
 * Creates a new Element that wrapps the given GstElement. Unlike the
 * Element constructor this returns the appropriate sub-class for the 
 * GstElement, so when you give a GstBin you will get a KDE::GST::Bin
 * instead of a KDE::GST::Element. And if there is already a wrapper
 * object it will be re-used. 
 *
 * @param real the GstElement to be wrapped
 * @return the wrapped Element (or derivative)
 */
		static Element* wrap(void *real);

/**
 * Retrieves the factory that was used to create this element.
 *
 * @return the factory used for creating this element
 */
		ElementFactory* getFactory();

/**
 * Add a pad (link point) to the element, setting the parent of the
 * pad to the element (and thus adding a reference).
 *
 * @param pad pad to add
 */
		void addPad(Pad *pad);

/**
 * Remove a pad (link point) from the element.
 *
 * @param pad pad to remove
 */
		void removePad(Pad *pad);

/**
 * Create a ghost pad from the given pad, and add it to the list of pads
 * for this element.
 * 
 * @param pad pad from which the new ghost pad will be created
 * @param name name of the new ghost pad
 * @return the added ghost pad or NULL, if no ghost pad was created.
 */
		Pad* addGhostPad(Pad *pad, const QString &name);

/**
 * Removes a ghost pad from an element.
 *
 * @param pad ghost pad to remove
 */
		void removeGhostPad(Pad *pad);

/**
 * Retrieve a pad from the element by name.
 *
 * @param name name of pad to retrieve
 * @return requested pad if found, otherwise NULL.
 */
		Pad* getPad(const QString &name);

/**
 * Retrieve a pad from the element by name. This version only retrieves
 * already-existing (i.e. 'static') pads.
 *
 * @param name name of pad to retrieve
 * @return requested pad if found, otherwise NULL.
 */
		Pad *getStaticPad(const QString &name);

/**
 * Retrieve a pad from the element by name. This version only retrieves
 * request pads.
 *
 * @param name name of pad to retrieve
 * @return requested pad if found, otherwise NULL.
 */
		Pad* getRequestPad(const QString &name);

/**
 * Make the element free the previously requested pad as obtained
 * with getRequestPad().
 *
 * @param pad The pad to release
 */
		void releaseRequestPad(Pad *pad);

/**
 * Retrieve a list of the pads associated with the element. 
 * The list is a shallow copy of the internal list.
 *
 * @return QValueVector with pointers to the element's pads
 */
		QValueVector<Pad*> getPadList();

/**
 * Retrieve a padtemplate from this element with the
 * given name.
 *
 * @param name the name of the padtemplate to get.
 * @return the padtemplate with the given name. No unreferencing is necessary.
 */
		PadTemplate* getPadTemplate(const QString &name);

/**
 * Retrieve a list of the padtemplates associated with the element.
 * The list is a shallow copy of the internal list.
 *
 * @return QValueVector with pointers to the element's padtemplates
 */
		QValueVector<PadTemplate*> getPadTemplateList();

/**
 * Add a padtemplate to the element's class. This is useful if you have derived a custom
 * bin and wish to provide an on-request pad at runtime. Plugin writers should use
 * ElementFactory::addPadTemplate instead.
 * TODO: find better way to implement this
 * 
 * @param templ padtemplate to add
 */
		void classAddPadTemplate (PadTemplate *templ);

/**
 * Chain together two or more of elements.
 * Links the source to the destination element, the other
 * direction will not be tried.
 * The functions looks for existing pads and request pads that aren't
 * linked yet. If multiple links are possible, only one is
 * established.
 *
 * @param elem1 element containing first destination pad
 * @param elem2 element containing second destination pad (can be 0)
 * @param elem3 element containing third destination pad (can be 0)
 * @param elem4 element containing fourth destination pad (can be 0)
 * @param elem5 element containing fifth destination pad (can be 0)
 * @return TRUE if the elements could be linked.
 */
		bool link(Element *elem1, 
			  Element *elem2 = 0,
			  Element *elem3 = 0,
			  Element *elem4 = 0,
			  Element *elem5 = 0);
		
/**
 * Links this source to the destination element using the filtercaps.
 * The link must be from source to destination, the other
 * direction will not be tried.
 * The functions looks for existing pads that aren't linked yet. 
 * It will use request pads if possible. But both pads will not be requested.
 * If multiple links are possible, only one is established.
 *
 * @param dest the element containing destination pad
 * @param filtercaps the caps to use as filter
 * @return TRUE if the elements could be linked.
 */
		bool linkFiltered(Element *dest, Caps *filtercaps);

/**
 * Link the two named pads of the source and destination elements.
 * Side effect is that if one of the pads has no parent, it becomes a
 * child of the parent of the other element. If they have different
 * parents, the linking fails.
 *
 * @param srcpadname name of pad in source element
 * @param dest element containing destination pad
 * @param destpadname name of pad in destination element
 * @param filtercaps the caps to use as a filter, 0 to ignore
 * @return TRUE if the pads could be linked.
 */
		bool linkPads(const QString &srcpadname,
			      Element *dest, 
			      const QString &destpadname, 
			      Caps *filtercaps = 0);

/**
 * Unlinks all pads linking the chain of elements in the 
 * direction src -> dest.
 * @param elem1 sink element
 * @param elem2 sink element (can be 0)
 * @param elem3 sink element (can be 0)
 * @param elem4 sink element (can be 0)
 * @param elem5 sink element (can be 0)
 */
		void unlink(Element *elem1, 
			    Element *elem2 = 0,
			    Element *elem3 = 0,
			    Element *elem4 = 0,
			    Element *elem5 = 0);
		

/**
 * Unlink the two named pads of the source and destination elements.
 *
 * @param srcpadname name of pad in source element
 * @param dest element containing destination pad
 * @param destpadname name of pad in destination element
 */
		void unlinkPads(const QString &srcpadname,
				Element *dest, const QString &destpadname);

/**
 * Looks for an unlinked pad to which the given pad can link to.
 * It is not guaranteed that linking the pads will work, though
 * it should work in most cases.
 *
 * @param pad the pad to find a compatible one for
 * @return the pad to which a link can be made
 */
		Pad* getCompatiblePad(Pad *pad);

/**
 * Looks for an unlinked pad to which the given pad can link to.
 * It is not guaranteed that linking the pads will work, though
 * it should work in most cases.
 *
 * @param pad the pad to find a compatible one for
 * @param filtercaps the caps to use as a filter
 * @return the pad to which a link can be made
 */
		Pad* getCompatiblePadFiltered(Pad *pad, Caps *filtercaps);

/**
 * Generate a padtemplate for this element compatible with the given
 * template, ie able to link to it.
 *
 * @param compattempl a template to find a compatible template for
 * @return the padtemplate. No unreferencing is necessary.
 */
		PadTemplate* getCompatiblePadTemplate(PadTemplate *compattempl);

/**
 * Sets the state of the element. This function will try to set the
 * requested state by going through all the intermediary states and calling
 * the class's state change function for each.
 *
 * @param state the element's new ElementState
 * @return whether or not the state was successfully set 
 *         (using ElementStateReturn).
 */
		int setState(State state);

/**
 * Gets the state of the element. 
 *
 * @return The ElementState of the element
 */
		State getState();

/**
 * Gets a string representing the given state.
 *
 * @param state 
 * @return a string with the statename.
 */
		static QString stateGetName(State state);

/**
 * Waits and blocks until the element changed its state.
 */
		void waitStateChange();

/**
 * Perform the actions needed to bring the element in the EOS state.
 */
		void setEos();

/**
 * Request the scheduler of this element to interrupt the execution of
 * this element and scheduler another one.
 *
 * @return a boolean indicating that the child should exit its chain/loop/get
 * function ASAP, depending on the scheduler implementation.
 */
		bool interrupt();

/**
 * Request a yield operation for the child. The scheduler will typically
 * give control to another element.
 */
		void yield();

/**
 * Instruct the element to release all the locks it is holding, such as
 * blocking reads, waiting for the clock, ...
 *
 * @return TRUE if the locks could be released.
 */
		bool releaseLocks();

/**
 * Gets the clock of the element.
 *
 * @return the @ref Clock of the element.
 */
		Clock* getClock();

/**
 * Sets the clock for the element.
 *
 * @param clock the @ref Clock to set for the element
 */
		void setClock(Clock *clock);

/**
 * Waits for a specific time on the clock.
 *
 * @param id the @ref Clock to use
 * @param jitter The difference between requested time and actual time
 * @return the @ref Clock::ClockReturn result of the wait operation
 */
		Clock::ClockReturn clockWait(Clock::ID id, 
					     long long *jitter);
/**
 * Gets the managing bin (a pipeline or a thread, for example) of an element.
 * @return the @ref Bin, or NULL on failure
 **/
		Bin* getManagingBin();

/**
 * Perform a query on the given element. If the format is set
 * to FORMAT_DEFAULT and this function returns TRUE, the 
 * format pointer will hold the default format.
 * For element that don't implement a query handler, this function
 * forwards the query to a random linked sinkpad of this element.
 * 
 * @param type The query type
 * @param format a pointer to hold the format of the result
 * @param value a pointer to the value of the result
 * @return TRUE if the query could be performed.
 */
		bool query(QueryType type,
			   Format *format, long long *value);

/**
 * Sends an event to an element. If the element doesn't 
 * implement an event handler, the event will be forwarded
 * to a random sinkpad.
 * 
 * @param event The event to send to the object.
 * @return TRUE if the event was handled.
 */
		bool sendEvent(Event *event);

/**
 * Connects a callback for the eos (End-Of-Stream) event. The callback
 * will be called in the Element's @ref Thread, so you can not use
 * any Qt/KDE functionality unless it is thread-safe.
 * You can register an unlimited number of callbacks for the event.
 *
 * @param callback the function that will be called when the stream
 *                 ends
 * @param cookie a pointer that will be passed to the callback function
 * @see #unregisterEosCallback()
 * @see #eos()
 */
		void registerEosCallback(void (*callback)(Element *e, 
							  void *cookie),
					 void *cookie);

/**
 * Disconnects the callback for the eos (End-Of-Stream) event.
 *
 * @param callback the callback function
 * @see #registerEosCallback()
 */
		void unregisterEosCallback(void (*callback)(Element *e, 
							    void *cookie),
					   void *cookie);

/**
 * Connects a callback for the StateChange event. The callback
 * will be called in the Element's @ref Thread, so you can not use
 * any Qt/KDE functionality unless it is thread-safe.
 *
 * @param callback the function that will be called when the stream
 *                 state changes
 * @param cookie a pointer that will be passed to the callback function
 * @see #unregisterStateChangeCallback()
 * @see #stateChange()
 */
		void registerStateChangeCallback(void (*callback)(Element *e, 
								  int oldState,
								  int newState,
								  void *cookie),
						 void *cookie);

/**
 * Disconnects the callback for the StateChange event.
 *
 * @param callback the callback function
 * @see #registerStateChangeCallback()
 */
		void unregisterStateChangeCallback(void (*callback)(Element *e, 
								    int oldState,
								    int newState,
								    void *cookie),
						   void *cookie);

/**
 * Connects a callback for the Error event. The callback
 * will be called in the Element's @ref Thread, so you can not use
 * any Qt/KDE functionality unless it is thread-safe.
 *
 * @param callback the function that will be called when an error occurred
 * @param cookie a pointer that will be passed to the callback function
 * @see #unregisterErrorCallback()
 * @see #error()
 */
		void registerErrorCallback(void (*callback)(Element *e, 
						       const char *error,
						       void *cookie),
					   void *cookie);

/**
 * Disconnects the callback for the StateChange event.
 *
 * @param callback the callback function
 * @see #registerErrorCallback()
 */
		void unregisterErrorCallback(
				     void (*callback)(Element *e, 
						      const char *error,
						      void *cookie),
				     void *cookie);
signals:
/**
 * Emitted for End-Of-Stream events in Qt's event loop.
 * @see #registerEosCallback
 * @see #unregisterEosCallback
 */
		void eos();

/**
 * Emitted when the Elements state changed. Called from Qt's 
 * event loop.
 * @param oldState the elements old state
 * @param newState the elements new state
 * @see #registerStateChangeCallback
 * @see #unregisterStateChangeCallback
 */
		void stateChange(KDE::GST::Element::State oldState,
				 KDE::GST::Element::State newState);

/**
 * Emitted when an error occurred. Called from Qt's event loop.
 * @param msg an error text (not i18n'd!)
 * @see #registerErrorCallback
 * @see #unregisterErrorCallback
 */
		void error(QString msg);
	};
}
}

#endif
