/*
 *  Interface/wrapper for GStreamer GstClock
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_CLOCK_H
#define KDE_GST_CLOCK_H

#include <kde/gst/object.h>

namespace KDE {
namespace GST {
/**
 * GStreamer uses a global clock to synchronise the plugins in a pipeline.
 * Different clock implementations are possible by implementing this abstract
 * base class.
 * 
 * The clock time is always measured in nanoseconds, with the clock being set
 * to 0 when the pipeline goes to READY. Usually all renderers sync to the global
 * clock so that the clock is always a good measure of the time in the pipeline.
 *
 * @short Abstract class for global clocks
 * @see SystemClock
 */
	class Clock : public Object {
	Q_OBJECT
	private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		void *reserved;
		
	public:

	enum ClockReturn {
		CLOCK_STOPPED 	  = 0,
		CLOCK_TIMEOUT 	  = 1,
		CLOCK_EARLY 	  = 2,
		CLOCK_ERROR 	  = 3,
		CLOCK_UNSUPPORTED = 4
	};

	static const unsigned long long SECOND  = 1000000000LL;
	static const unsigned long long MSECOND = SECOND/1000LL;
	static const unsigned long long USECOND = SECOND/1000000LL;
	static const unsigned long long NSECOND = SECOND/1000000000LL;

	static const long long TIME_NONE = (long long)-1;

	/**
	 * ID for timers.
	 */
	typedef void *ID;

/**
 * Creates a new Clock that wrapps the given GstElement. Usually
 * you really don't want to call this, use @ref #wrap instead.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstClock to be wrapped
 */
	        Clock(void *real);
	        virtual ~Clock();

/**
 * Creates a new Clock that wrapps the given GstClock. Unlike the
 * Clock constructor this returns the appropriate sub-class for the 
 * GstClock, and it also re-uses existing wrapper objects.
 *
 * @param real the GstClock to be wrapped
 * @return the wrapped Clock (or derivative)
 */
		static Clock* wrap(void *real);

/**
 * Sets the speed on the given clock. 1.0 is the default 
 * speed.
 *
 * @param speed the speed to set on the clock
 */
		void setSpeed(double speed);

/**
 * Gets the speed of the given clock.
 *
 * @return the speed of the clock
 */
		double getSpeed();

/**
 * Set the accuracy of the clock.
 * @param resolution The resolution to set
 */
		void setResolution(long long resolution);

/**
 * Get the accuracy of the clock.
 *
 * @return the resolution of the clock in microseconds.
 */
		long long getResolution();
		
/**
 * Activates or deactivates the clock based on the active parameter.
 * As soon as the clock is activated, the time will start ticking.
 *
 * @param active flag indicating if the clock should be activated (TRUE) 
 *               or deactivated
 */
		void setActive(bool active);

/**
 * Checks if the given clock is active.
 * 
 * @return TRUE if the clock is active.
 */
		bool isActive();

/**
 * Reset the clock to time 0.
 */
		void reset();

/**
 * Notifies the clock of a discontinuity in time.
 *
 * @param time The new time
 * @return TRUE if the clock was updated. It is possible that
 *   the clock was not updated by this call because only the first
 *   discontinuitity in the pipeline is honoured.
 */
		bool handleDiscont(long long time);

/**
 * Gets the current time of the given clock. The time is always
 * monotonically increasing.
 * 
 * @return the time of the clock in nanoseconds.
 */
		long long getTime();

/**
 * Get the clockid of the next event.
 *
 * @return a clockid or 0 is no event is pending.
 */
		ID getNextId();

/**
 * Get an ID from the given clock to trigger a single shot 
 * notification at the requested time.
 *
 * @param time the requested time
 * @return An id that can be used to request the time notification.
 */
		ID newSingleShotId(unsigned long long time);

/**
 * Get an ID from the given clock to trigger a periodic notification.
 * The periodeic notifications will be start at time @p startTime and
 * will then be fired with the given @p interval.
 *
 * @param startTime the requested start time
 * @param interval the requested interval
 * @return An id that can be used to request the time notification.
 */
		ID newPeriodicId(unsigned long long startTime,
				 unsigned long long interval);

/**
 * Get the time of the clock ID.
 *
 * @param id The clockid to query
 * @return the time of the given clock id
 */
		static unsigned long long idGetTime(ID id);

/**
 * Perform a blocking wait on the given ID. The jitter arg can be
 * 0.
 *
 * @param id The clockid to wait on
 * @param jitter A pointer that will contain the jitter
 * @return the result of the blocking wait.
 */
		static ClockReturn idWait(ID id, long long *jitter);

/**
 * Cancel an outstanding async notification request with the given ID.
 * 
 * @param id The id to unschedule
 */
		static void idUnschedule(ID id);

/**
 * Unlock the givan ClockID.
 *
 * @param id The clockid to unlock
 */
		static void idUnlock(ID id);

/**
 * Free the resources held by the given id.
 *
 * @param id The clockid to free
 */
		static void idFree(ID id);
	};
}
}

#endif
