/*
 *  Wrapper/interface class for GstAutoplug
 *  Copyright (C) 2002 Tim Jansen <tim@tjansen.de>
 *  API Documentation
 *  Copyright (C) 1999,2000 Erik Walthinsen <omega@cse.ogi.edu>
 *                     2000 Wim Taymans <wtay@chello.be>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef KDE_GST_AUTOPLUG_H
#define KDE_GST_AUTOPLUG_H

#include <kde/gst/object.h>

namespace KDE {
namespace GST {
	class Element;
	class Caps;

/**
 * Autoplug is an abstract class that is used for constructing and 
 * connecting elements. Two types of autopluggers exist: renderer ones and non
 * renderer ones. The renderer autopluggers will not have any src pads while 
 * the non renderer ones do.
 * 
 * You first need to create a suitable autoplugger with 
 * AutoplugFactory::make(). (see @ref AutoplugFactory).
 * The name of the autoplugger must be one of the registered autopluggers 
 * (see GstStaticAutoplug and GstStaticAutoplugRender).
 *
 * If the autoplugger supports the RENDERER API, use 
 * @ref #toRenderers call to create a bin that connectes the src caps to 
 * the specified rende rer elements. You can then add the bin to a pipeline 
 * and run it.
 *
 * <pre>
 *    Autoplug *autoplug;
 *    Element  *element;
 *    Element  *sink;
 *
 *    // create a static autoplugger 
 *    autoplug = AutoplugFactory::make("staticrender");
 *
 *    // create an osssink 
 *    sink = ElementFactory::make("osssink", "our_sink");
 * 
 *    // create an element that can play audio/mp3 through osssink
 *    element = autoplug->toRenderers(new Caps("sink_audio_caps",
 *					       "audio/mp3",
 *					       0),
 * 				      sink);
 *
 *    // add the element to a bin and connect the sink pad
 *    ...
 * </pre>
 *
 * If the autoplugger supports the CAPS API, use @ref Autoplug#toCaps() 
 * call to connect the src caps to the destination caps. The created bin 
 * will have src pads compatible with the provided sink caps.
 *
 * <pre>
 *    Autoplug *autoplug;
 *    Element  *element;
 *
 *    // create a static autoplugger
 *    autoplug = AutoplugFactory::make("static");
 *
 *    // create an element that converts audio/mp3 to audio/raw 
 *    element = autoplug->toCaps(new Caps("sink_audio_caps",
 *				          "audio/mp3",
 *				          NULL),
 *   				 new Caps("src_audio_caps",
 *				          "audio/raw",
 *				          NULL));
 *
 *    // add the element to a bin and connect the src/sink pads 
 *    ...
 * </pre>
 *
 * Optionally you can get a notification when a new object is added to the 
 * created pipeline with @ref Wrapper::registerRawSignal() for the 
 * "new_object" signal.
 *
 * Use the regular delete to destroy the autoplugger.
 *
 * @short Automatically create and connect elements
 * @see AutoplugFactory
 */
	class Autoplug : public Object {
	Q_OBJECT
	private:
                friend class GStreamer;
                static Wrapper* wrapperFactory(void *real);
		void *reserved;
		
	public:

	  enum AutoplugFlags {
	    AUTOPLUG_TO_CAPS 	= Object::FLAG_LAST,
	    AUTOPLUG_TO_RENDERER,
	    AUTOPLUG_FLAG_LAST	= Object::FLAG_LAST + 8
	  };

/**
 * Creates a new Autoplug that wrapps the given GstAutoplug.
 * You must not create a object of this class on the stack, always 
 * use new.
 * @param real the GstAutoplug to be wrapped.
 */
	        Autoplug(void *real);
	        virtual ~Autoplug();

/**
 * Creates a new Autoplug that wrapps the given object. Unlike the
 * constructor this re-used already existing objects.
 * @param real the object to be wrapped
 * @return the Autoplug
 */
		static Autoplug *wrap(void *real);

/**
 * Perform the autoplugging procedure on the given autoplugger. 
 * The src caps will be connected to the sink caps.
 * 
 * @param srccaps The source cpabilities
 * @param sinkcaps1 The target capabilities
 * @param sinkcaps2 more target capabilities (can be 0)
 * @param sinkcaps3 more target capabilities (can be 0)
 * @param sinkcaps4 more target capabilities (can be 0)
 * @param sinkcaps5 more target capabilities (can be 0)
 * @return A new Element that connects the src caps to the sink caps.
 */
		Element* toCaps(Caps *srccaps, 
				Caps *sinkcaps1, 
				Caps *sinkcaps2 = 0,
				Caps *sinkcaps3 = 0,
				Caps *sinkcaps4 = 0,
				Caps *sinkcaps5 = 0);

/**
 * Perform the autoplugging procedure on the given autoplugger. 
 * The src caps will be connected to the target elements.
 * 
 * @param srccaps The source capabilities
 * @param target1 The target element 
 * @param target2 more target elements (can be 0)
 * @param target3 more target elements (can be 0)
 * @param target4 more target elements (can be 0)
 * @param target5 more target elements (can be 0)
 * @return A new Element that connects the src caps to the target elements.
 */
		Element* toRenderers(Caps *srccaps, 
				     Element *target1,
				     Element *target2 = 0,
				     Element *target3 = 0,
				     Element *target4 = 0,
				     Element *target5 = 0);
	};
}
}
#endif
