 /*
 * Multi-threaded GUI player for audio+video. 
 * Demonstrates Qt event loop integration and seeking.
 * Based on spider_seek.c example from gst-plugins.
 */

#include "guiseeker.h"

#include <kde/gst/gstreamer.h>
#include <kde/gst/element.h>
#include <kde/gst/elementfactory.h>
#include <kde/gst/event.h>
#include <kde/gst/bin.h>
#include <kde/gst/pipeline.h>
#include <kde/gst/thread.h>

#include <qapplication.h>
#include <qthread.h>

using namespace KDE::GST;

GuiSeeker::GuiSeeker(QApplication *a) :
	QWidget(0,0),
	m_app(a),
        m_timer(this),
	m_xembed(0),
	m_pipeline(0),
	m_audiosink(0),
	m_videosink(0) {

	QBoxLayout *topLayout = new QVBoxLayout(this, 8);
	QBoxLayout *buttonLayout = new QHBoxLayout(topLayout, 8);
	QBoxLayout *sliderLayout = new QHBoxLayout(topLayout, 8);
	m_videoLayout = new QHBoxLayout(topLayout, 8);
	
	m_playButton = new QPushButton("Play", this);
	connect(m_playButton, SIGNAL(clicked()), SLOT(play()));
	m_playButton->setEnabled(true);
	buttonLayout->addWidget(m_playButton);

	m_pauseButton = new QPushButton("Pause", this);
	connect(m_pauseButton, SIGNAL(clicked()), SLOT(pause()));
	m_pauseButton->setEnabled(false);
	buttonLayout->addWidget(m_pauseButton);

	m_stopButton = new QPushButton("Stop", this);
	connect(m_stopButton, SIGNAL(clicked()), SLOT(stop()));
	m_stopButton->setEnabled(false);
	buttonLayout->addWidget(m_stopButton);

	m_slider = new QSlider(0, 1000, 50, 0, Qt::Horizontal, this);
	connect(m_slider, SIGNAL(sliderPressed()), SLOT(pause()));
	connect(m_slider, SIGNAL(sliderReleased()), SLOT(seek()));
	m_slider->setEnabled(false);
	sliderLayout->addWidget(m_slider);

	connect(&m_timer, SIGNAL(timeout()), SLOT(updateSlider()));
}

GuiSeeker::~GuiSeeker() {
	if (m_pipeline)
		delete m_pipeline;
}

/* XVideoSink's have_xid callback (called from video thread) */
void GuiSeeker::haveXid(void*, int xid, GuiSeeker *player) {
		if (player->m_xembed) 
			return;

		player->m_app->lock();
		player->m_xembed = new QXEmbed(player);
		player->m_videoLayout->addWidget(player->m_xembed);
		player->m_xembed->embed(xid);
		player->m_xembed->show();
		player->m_app->unlock();
}

/* XVideoSink's have_size callback (called from video thread) */
void GuiSeeker::haveSize(void*, int width, int height, void *cookie) {
	QThread::postEvent((GuiSeeker*)cookie, 
			   new VideoSizeEvent(width, height));
}

/* build a {filesrc ! spider ! {queue ! osssink } spider0.src0! {queue ! xvideosink } */
void GuiSeeker::makeSpiderPipeline (const QString &location) {
	Element *src, *decoder, *a_queue, *v_queue;
	Thread *a_thread, *v_thread;
	
	m_pipeline = new Thread("app");
	
	src = ElementFactory::make("filesrc", "src");
	
        decoder = ElementFactory::make("spider", "decoder");
	a_thread = new Thread("a_thread");
	a_queue = ElementFactory::make("queue", "a_queue");
	m_audiosink = ElementFactory::make("osssink", "a_sink");
	
	v_thread = new Thread("v_thread");
	v_queue = ElementFactory::make("queue", "v_queue");
	m_videosink = ElementFactory::make("xvideosink", "v_sink");

	Q_ASSERT(src);
	Q_ASSERT(decoder);
	Q_ASSERT(a_queue);
	Q_ASSERT(v_queue);
	Q_ASSERT(m_audiosink);
	Q_ASSERT(m_videosink);
	
	src->setString("location", location);
	connect(m_audiosink, SIGNAL(eos()), SLOT(stop()));
        m_videosink->setBool("toplevel", false);
	m_videosink->registerRawSignal("have_xid", (void*)haveXid, this);
	m_videosink->registerRawSignal("have_size", (void*)haveSize, this);
	
	m_pipeline->add(src, decoder);
	a_thread->add(a_queue, m_audiosink);
	v_thread->add(v_queue, m_videosink);
	m_pipeline->add(a_thread, v_thread);

	src->link(decoder);
	v_queue->link(m_videosink);
	decoder->link(v_queue);
	a_queue->link(m_audiosink);
	decoder->link(a_queue);
}

/* Read position from QSlider and seek... (after slider has been moved) */
void GuiSeeker::seek()
{
	long long real = m_slider->value() * m_duration / 1000;
	bool res;
	Event *sEvent;
	
	sEvent = new Event ((Event::SeekType) (FORMAT_TIME |
					       Event::SEEK_METHOD_SET |
					       Event::SEEK_FLAG_FLUSH), 
			    real);
	
	res = m_audiosink->sendEvent(sEvent);
	if (res)
		res = m_videosink->sendEvent(sEvent);
	delete sEvent;
	
	play();
}

void GuiSeeker::play()
{
	if (m_pipeline->getState() != Element::STATE_PLAYING) {
		m_pipeline->setState(Element::STATE_PLAYING);
		m_timer.start(UPDATE_INTERVAL);
	}
	m_playButton->setEnabled(false);
	m_pauseButton->setEnabled(true);
	m_stopButton->setEnabled(true);
	m_slider->setEnabled(true);
}

void GuiSeeker::pause()
{
	if (m_pipeline->getState() != Element::STATE_PAUSED) {
		m_pipeline->setState(Element::STATE_PAUSED);
		m_timer.stop();
	}
	m_playButton->setEnabled(true);
	m_pauseButton->setEnabled(false);
	m_stopButton->setEnabled(true);
	m_slider->setEnabled(true);
}

void GuiSeeker::stop()
{
	if (m_pipeline->getState() != Element::STATE_READY) {
		m_pipeline->setState(Element::STATE_READY);
		m_timer.stop();
	}
	m_playButton->setEnabled(true);
	m_pauseButton->setEnabled(false);
	m_stopButton->setEnabled(false);
	m_slider->setValue(0);
	m_slider->setEnabled(false);
}

/* Called every UPDATE_INTERVAL ms using a QTimer to set the slider */
void GuiSeeker::updateSlider() 
{
	unsigned long long position = 0;
	Format format = FORMAT_TIME;
	Clock *clock = m_pipeline->getClock();
	Q_ASSERT(clock);
	
	if (m_videosink)
		m_videosink->query(QUERY_TOTAL, &format, 
				   (long long*) &m_duration);
	position = clock->getTime();
	
	if (m_duration > 0)
		m_slider->setValue((int)(position * 1000.0 / m_duration));
}

void GuiSeeker::customEvent(QCustomEvent *e) {
	if (e->type() == VideoSizeEventType) {
		int w = ((VideoSizeEvent*)e)->width();
		int h = ((VideoSizeEvent*)e)->height();
		QSize s = sizeHint();
		resize((s.width() > w) ? s.width() : w,
		       s.height() + h);
	}
}


int main(int argc, char **argv)
{
	if (argc != 2) {
		qDebug("usage: %s <filename>\n", argv[0]);
		return -1;
	}
	
	GStreamer::init(&argc, &argv);

	QApplication a(argc, argv);

	GuiSeeker player(&a);
	player.setCaption("QGst Gui Seeking Example");
	player.makeSpiderPipeline(argv[1]);
	a.setMainWidget(&player);
	player.show();

	return a.exec();
}

#include "guiseeker.moc"
