\ pde1.4th
\
\ Numerical Solution of Electrostatics Boundary-Value Problems.
\ Solve Laplace's Equation in 2 Dimensions:
\
\	D_xx u(x,y) + D_yy u(x,y) = 0
\
\ Copyright (c) 2003 Krishna Myneni, Creative Consulting for Research 
\ and Education
\
\ Provided under the terms of the GNU General Public License.
\
\ This program demonstrates a method of solving one kind of a partial 
\ differential equation (PDE) for a function u(x,y), a function
\ of the two variables x and y. In Laplace's Equation above, 
\ D_xx represents taking the second partial derivative with respect to 
\ x of u(x,y), and D_yy the second partial derivative w.r.t. y. This 
\ equation holds for the electrostatic potential u(x,y) inside
\ a charge-free two dimensional region. If we know the values of
\ u(x,y) along a boundary enclosing the region, Laplace's equation
\ may be solved to obtain the values of u(x,y) at all interior points
\ of the region. 
\
\ In this demonstration, we can setup two different bounding regions:
\
\ 1) a hollow rectangular box with voltages defined on the edges,
\
\ 2) a hollow circular region with the top half boundary at one voltage,
\      and the bottom half boundary at a second voltage.
\
\ Very thin insulators are assumed to be separating the regions which 
\ are at different potentials on the bounding region.
\ 
\ Laplace's equation is solved by an iterative application of the 
\ "mean value theorem for the electrostatic potential" (see 
\ "Classical Electrodynamics", 2nd ed, by J.D. Jackson) to each grid 
\ point inside the boundary until the solution converges. For more 
\ information on solving PDEs and boundary value problems, 
\ see "Partial differential equations for engineers and scientists", 
\ by Stanley J. Farlow, 1982, Dover. The method of solving Laplace's 
\ equation used in this example is known as Liebmann's method.
\
\ If your system has installed the "R" package for statistical computing 
\ (see www.r-project.org) and ghostview, you can generate and
\ view a contour plot of the equipotential lines for the solution.
\
\ K. Myneni, 1998-10-23
\
\
include ans-words
include matrix

: fmat_copy ( a1 a2 -- | copy fmatrix a1 into a2)
    over mat_size@ * dfloats cell+ cell+ cmove ;

\ Create a floating pt matrix to hold the grid values

64 constant GRIDSIZE
GRIDSIZE dup fmatrix grid
GRIDSIZE dup fmatrix last_grid	\ copy of last grid values for convergence test

\ Rectangular Region Boundary Values

100e  FCONSTANT  TOP_EDGE	\ Top edge at   100.0 V
0e    FCONSTANT  RIGHT_EDGE	\ Right edge at   0.0 V
0e    FCONSTANT  BOTTOM_EDGE	\ Bottom edge at  0.0 V
50e   FCONSTANT  LEFT_EDGE	\ Left edge at   50.0 V

: inside_rectangle? ( row col -- flag | inside rectangular boundary?)
    dup 1 > swap GRIDSIZE < AND swap
    dup 1 > swap GRIDSIZE < AND AND
;

: set_rectangular_bvs ( -- | setup the rectangular boundary values)
    GRIDSIZE 1+ 1 do  TOP_EDGE    1 i grid fmat! loop
    GRIDSIZE 1+ 1 do  RIGHT_EDGE  i GRIDSIZE grid fmat! loop
    GRIDSIZE 1+ 1 do  BOTTOM_EDGE GRIDSIZE i grid fmat! loop
    GRIDSIZE 1+ 1 do  LEFT_EDGE   i 1 grid fmat! loop
;

: init_rectangular_grid ( -- | set up the starting grid values )
    set_rectangular_bvs
    TOP_EDGE BOTTOM_EDGE RIGHT_EDGE LEFT_EDGE f+ f+ f+ 4e f/
    GRIDSIZE 1+ 1 do
      GRIDSIZE 1+ 1 do
        j i inside_rectangle? IF fdup j i grid fmat! THEN
      loop
    loop fdrop ;

\ Circular Region Boundary Values

100e  FCONSTANT  TOP_HALF	\ Top half of boundary region at 100. V
0e    FCONSTANT  BOTTOM_HALF    \ Bottom half at 0.0 V
GRIDSIZE 2- 2/ CONSTANT RADIUS  \ Radius of boundary region

: inside_circle? ( row col -- flag | inside circular boundary? )
     GRIDSIZE 2/ - dup * swap 
     GRIDSIZE 2/ - dup * + s>f fsqrt fround>s
     RADIUS < ;
 
: set_circular_bvs ( -- | setup the circular boundary region )
    GRIDSIZE 1+ 1 do
      GRIDSIZE 1+ 1 do
        j i inside_circle? 0= IF
	  j GRIDSIZE 2/ < IF TOP_HALF ELSE BOTTOM_HALF THEN
	  j i grid fmat!
	THEN 
      LOOP
    LOOP ;

: init_circular_grid ( -- | set starting values of the grid)
    set_circular_bvs
    TOP_HALF BOTTOM_HALF f+ 2e f/
    GRIDSIZE 1+ 1 do
      GRIDSIZE 1+ 1 do
        j i inside_circle? IF fdup j i grid fmat! THEN
      loop
    loop fdrop ;
	    
defer inside?

: circ ( -- | use the two semi-circle boundary values )
    grid fmat_zero
    ['] inside_circle? is inside? 
    init_circular_grid ;

: rect ( -- | use rectangular boundary values )
    grid fmat_zero
    ['] inside_rectangle? is inside?
    init_rectangular_grid ;


: nearest@ ( i j -- f1 f2 f3 f4 | fetch the nearest neighbor grid values )
    2>R
    2R@ 1- 1 MAX grid fmat@         \ fetch left nearest neighbor
    2R@ 1+ GRIDSIZE MIN grid fmat@  \ fetch right nearest neighbor
    2R@ SWAP 1- 1 MAX SWAP grid fmat@  \ fetch up nearest neighbor
    2R> SWAP 1+ GRIDSIZE MIN SWAP grid fmat@ \ fetch down nearest neighbor
;	    	  

\ Apply the mean value theorem once to each of the interior grid values:
\   Replace each grid value with the average of the four nearest
\   neighbor values.

: iterate ( -- ) 
	GRIDSIZE 1+ 1 ?do
	  GRIDSIZE 1+ 1 ?do
	    j i inside? IF
	      j i nearest@	\ fetch four nearest neighbors
	      f+ f+ f+ 4e f/	\ take average of the four values
	      j i grid fmat!	\ store at this position
	    THEN
	  loop
	loop
;



fvariable tol	\ tolerance for solution
1e-3 tol f!

: converged? ( -- flag | test for convergence between current and last grid)
    GRIDSIZE 1+ 1 do
      GRIDSIZE 1+ 1 do
        j i inside? IF j i grid fmat@ j i last_grid fmat@ f-
	               fabs tol f@ f> IF FALSE unloop unloop EXIT THEN
		    THEN
      loop
    loop TRUE ;


\ Iterate until the solution converges to the specified tolerance 
\ at all interior points.

: solve ( -- )
	begin
          grid last_grid fmat_copy
	  iterate
	  converged?
	until
;



: grid_minmax ( -- fmin fmax | find min and max of grid values )
	1 1 grid fmat@ fdup
	GRIDSIZE 1+ 1 do
	  GRIDSIZE 1+ 1 do
	    j i grid fmat@ fswap fover fmax 2>r fmin 2r>
	  loop
	loop
;

: display_grid ( -- | display the grid values as a character map )
	grid_minmax
	fover f- 
	15e fswap f/	\ scale factor to scale grid value from 0 to 15
	fswap

	GRIDSIZE 1+ 1 ?do
	  GRIDSIZE 1+ 1 ?do
	    fover fover
	    j i grid fmat@ fswap f- f*
	    fround>s dup 9 >
	    if 55 + else 48 + then emit
	  loop
	  cr
	loop

	fdrop fdrop
;


\ Optional script code for generating a contour plot with "R"  
\ (see www.r-project.org). Also requires "ghostview".
\
\ To use, redirect the R output to a file, e.g.
\
\	>file pde1.txt R-output console
\
\ Then use R to generate the encapsulated postscript (eps) output:
\
\	R --vanilla < pde1.txt > /dev/null
\
\ Finally, to view, use ghostview to look at the R-generated eps file
\
\	gv pde1.eps
\
\ Under Linux, this procedure is automated with the word "PLOT"
\ 
variable nout

: R-output ( -- | generate graphics output; redirect to a file and use R)
     ." x <- seq(1:" GRIDSIZE . [char] ) emit CR
     ." y <- seq(1:" GRIDSIZE . [char] ) emit CR
     ." z <- matrix(data = " CR
     ." c( "
     0 nout !
     1 GRIDSIZE do		\ reverse order for row output
       GRIDSIZE 1+ 1 do
         j i grid fmat@ f. [char] , emit
	 1 nout +!
	 nout @ 8 mod 0= if CR then
       loop
     -1 +loop
     [char] ) emit [char] , emit CR
     ." nrow = " GRIDSIZE . [char] , emit
     ." ncol = " GRIDSIZE . [char] ) emit CR
     ." postscript(" [char] " emit ." pde1.eps" [char] " emit 
     [char] , emit ."  horizontal=FALSE" [char] , emit
     ."  height = 6" [char] , emit ."  width = 6" [char] ) emit CR
     ." contour(x,y,z)" CR
     ." dev.off()" CR
;

: plot ( -- | generate and show a contour plot of the grid values )
     ." Generating contour plot ... please wait" CR
     S" >file pde1.txt R-output console" evaluate
     c" R --vanilla < pde1.txt > /dev/null" system drop
     c" gv pde1.eps &" system drop ;
  

rect
CR CR
.( Numerical Solution of Electrostatics Boundary-Value Problems ) CR 
GRIDSIZE dup 3 .r char x emit .    
.( grid has been setup. Type: ) CR CR
.(    rect           to use the rectangular boundary values) CR
.(    circ           to use the circular boundary values) CR
.(    solve          to find the solution) CR
.(    display_grid   to view grid as a character map) CR
.(    plot           to view contour plot [Linux only]) CR
CR
    