/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999-2000 Espen Sand, espensa@online.no
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

// kate: space-indent on; indent-width 2; replace-tabs on;

#include "toplevel.h"
#include "dialog.h"
#include "draglabel.h"
#include "hexerror.h"

#include <q3ptrlist.h>
#include <qsignalmapper.h>
#include <QTimer>

#include <ktoggleaction.h>
#include <kaction.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>
#include <kstatusbar.h>
#include <kmenu.h>
#include <kstandardaction.h>
#include <kdebug.h>
#include <kurl.h>
#include <kxmlguifactory.h>
#include <ktoolbar.h>
#include <ktoolbarspaceraction.h>
#include <kicon.h>
#include <kactioncollection.h>

Q3PtrList<KHexEdit> KHexEdit::mWindowList;

KHexEdit::KHexEdit( void )
  : KXmlGuiWindow(), mStartupOffset(0),
  mIsModified(0)
{
  setObjectName( "toplevel" );
  mWindowList.append( this );

  //
  // Create main widget and register it.
  //
  mManager = new CHexManagerWidget( this );
  setCentralWidget( mManager );

  //
  // I use an eventfilter to catch all drag/drop events for the
  // area outside the editor window. It will only accept KURLDrag.
  // (The editor window will catch KURLDrag, QTextDrag and CHexDrag)
  //
  installEventFilter( this );
  setAcceptDrops(true);

  //
  // Prepare menus and status bar
  //
  mAction.bookmarkMapper = new QSignalMapper(this);
  connect(mAction.bookmarkMapper, SIGNAL(mapped(int)), editor(), SLOT(gotoBookmark(int)));
  setupActions();
  setupStatusBar();

  connect( hexView(), SIGNAL( cursorChanged( SCursorState & ) ),
	   this, SLOT( cursorChanged( SCursorState & ) ) );
  connect( hexView(), SIGNAL( editMode( CHexBuffer::EEditMode ) ),
	   this, SLOT( editMode( CHexBuffer::EEditMode ) ) );
  connect( hexView(), SIGNAL( encodingChanged( const SEncodeState &)),
	   this, SLOT( encodingChanged( const SEncodeState & )) );
  connect( hexView(), SIGNAL( textWidth( uint ) ),
	   this, SLOT( textWidth( uint ) ) );
  connect( hexView(), SIGNAL( fileState( SFileState & ) ),
	   this, SLOT( fileState( SFileState & ) ) );
  connect( hexView(), SIGNAL( layoutChanged( const SDisplayLayout & ) ),
	   this, SLOT( layoutChanged( const SDisplayLayout & ) ) );
  connect( hexView(), SIGNAL( inputModeChanged( const SDisplayInputMode & ) ),
	   this, SLOT( inputModeChanged( const SDisplayInputMode & ) ) );
  connect( hexView(), SIGNAL( bookmarkChanged( Q3PtrList<SCursorOffset> &)),
	   this, SLOT( bookmarkChanged( Q3PtrList<SCursorOffset> & ) ) );
  connect( hexView(), SIGNAL( fileName( const QString &, bool ) ),
	   this, SLOT( fileActive( const QString &, bool ) ) );
  connect( hexView(), SIGNAL( fileRename( const QString &, const QString & )),
	   this, SLOT( fileRename( const QString &, const QString & ) ) );
  connect( hexView(), SIGNAL( fileClosed( const QString & ) ),
	   this, SLOT( fileClosed( const QString & ) ) );
  connect( editor(), SIGNAL( errorLoadFile( const QString & ) ),
	   this, SLOT( removeRecentFile( const QString & ) ) );
  connect( editor(), SIGNAL( operationChanged( bool ) ),
	   this, SLOT( operationChanged( bool ) ) );
  connect( editor(), SIGNAL( removeRecentFiles() ),
	   this, SLOT( removeRecentFiles() ) );
  connect( mManager, SIGNAL( conversionClosed() ),
	   this, SLOT(conversionClosed()) );
  connect( mManager, SIGNAL( searchBarClosed() ),
	   this, SLOT(searchBarClosed()) );

  //
  // Read configuration from file and set the default editor size.
  // Open files if this is the first toplevel window.
  //
  readConfiguration();
  initialize( mWindowList.count() == 1 ? true : false );
  setAutoSaveSettings();
}


KHexEdit::~KHexEdit( void )
{
  delete mAction.bookmarkMapper;
  delete mManager;
}


void KHexEdit::setupActions( void )
{
  KStandardAction::openNew( editor(), SLOT(newFile()), actionCollection() );
  KStandardAction::open( editor(), SLOT(open()), actionCollection() );
  KStandardAction::keyBindings(guiFactory(), SLOT(configureShortcuts()), actionCollection());

  mAction.insert = actionCollection()->addAction( "insert_file" );
  mAction.insert->setText( i18n("&Insert...") );
  connect(mAction.insert, SIGNAL(triggered(bool) ), editor(), SLOT(insertFile()));
  mAction.insert->setShortcut(Qt::CTRL+Qt::Key_I);
  mAction.openRecent = KStandardAction::openRecent( this, SLOT( slotFileOpenRecent( const KUrl& ) ), actionCollection() );
  mAction.save       = KStandardAction::save( editor(), SLOT(save()), actionCollection() );
  mAction.saveAs     = KStandardAction::saveAs( editor(), SLOT(saveAs()), actionCollection() );
  mAction.revert     = KStandardAction::revert( editor(), SLOT(reload()), actionCollection() );
  //mAction.revert     = KStandardAction::revert( this, SLOT(resizeTest()), actionCollection() );

  mAction.close      = KStandardAction::close( editor(), SLOT(close()), actionCollection() );
  mAction.print      = KStandardAction::print( editor(), SLOT(print()), actionCollection() );
  mAction.exportData = actionCollection()->addAction( "export" );
  mAction.exportData->setText( i18n("E&xport...") );
  connect( mAction.exportData, SIGNAL(triggered(bool) ), editor(), SLOT(exportDialog()) );
  mAction.cancel     = actionCollection()->addAction( "cancel" );
  mAction.cancel->setIcon( KIcon("process-stop") );
  mAction.cancel->setText( i18n("&Cancel Operation") );
  connect( mAction.cancel, SIGNAL(triggered(bool)), editor(), SLOT(stop()) );
  mAction.readOnly   = actionCollection()->add<KToggleAction>( "read_only" );
  mAction.readOnly->setText( i18n("&Read Only") );
  connect( mAction.readOnly, SIGNAL(triggered(bool)), editor(), SLOT(toggleWriteProtection()) );
  mAction.resizeLock = actionCollection()->add<KToggleAction>( "resize_lock" );
  mAction.resizeLock->setText( i18n("&Allow Resize") );
  connect( mAction.resizeLock, SIGNAL(triggered(bool)), editor(), SLOT(toggleResizeLock()) );
  mAction.newWindow = actionCollection()->addAction( "new_window" );
  mAction.newWindow->setText( i18n("N&ew Window") );
  connect(mAction.newWindow, SIGNAL(triggered(bool) ), SLOT(newWindow()));
  mAction.closeWindow = actionCollection()->addAction( "close_window" );
  mAction.closeWindow->setText( i18n("Close &Window") );
  connect(mAction.closeWindow, SIGNAL(triggered(bool) ), SLOT(closeWindow()));
  mAction.quit       = KStandardAction::quit( this, SLOT(closeProgram()), actionCollection() );

  mAction.undo       = KStandardAction::undo(  editor(), SLOT(undo()), actionCollection() );
  mAction.redo       = KStandardAction::redo(  editor(), SLOT(redo()), actionCollection() );
  mAction.cut        = KStandardAction::cut(  editor(), SLOT(cut()), actionCollection() );
  mAction.copy       = KStandardAction::copy(  editor(), SLOT(copy()), actionCollection() );
  mAction.paste      = KStandardAction::paste(  editor(), SLOT(paste()), actionCollection() );
  mAction.selectAll  = KStandardAction::selectAll(editor(), SLOT(selectAll()),actionCollection() );
  mAction.unselect   = KStandardAction::deselect(editor(), SLOT(unselect()), actionCollection());
  mAction.find       = KStandardAction::find( editor(), SLOT(find()), actionCollection() );
  mAction.findNext   = KStandardAction::findNext(editor(), SLOT(findNext()), actionCollection() );
  mAction.findPrev = KStandardAction::findPrev(editor(),SLOT(findPrevious()),actionCollection() );
  mAction.replace    = KStandardAction::replace( editor(), SLOT(replace()), actionCollection() );
  mAction.gotoOffset = actionCollection()->addAction( "goto_offset" );
  mAction.gotoOffset->setText( i18n("&Goto Offset...") );
  connect(mAction.gotoOffset, SIGNAL(triggered(bool) ), editor(), SLOT(gotoOffset()));
  mAction.gotoOffset->setShortcut(Qt::CTRL+Qt::Key_G);
  mAction.insertPattern = actionCollection()->addAction( "insert_pattern" );
  mAction.insertPattern->setText( i18n("&Insert Pattern...") );
  connect(mAction.insertPattern, SIGNAL(triggered(bool) ), editor(), SLOT(insertPattern()));
  mAction.insertPattern->setShortcut(Qt::CTRL+Qt::Key_Insert);

  mAction.copyAsText = actionCollection()->addAction( "copy_as_text" );
  mAction.copyAsText->setText( i18n("Copy as &Text") );
  connect(mAction.copyAsText, SIGNAL(triggered(bool) ), editor(), SLOT(copyText()));
  mAction.pasteToNewFile = actionCollection()->addAction( "paste_into_new_file" );
  mAction.pasteToNewFile->setText( i18n("Paste into New &File") );
  connect(mAction.pasteToNewFile, SIGNAL(triggered(bool) ), editor(), SLOT(pasteNewFile()));
  mAction.pasteToNewWindow = actionCollection()->addAction( "paste_into_new_window" );
  mAction.pasteToNewWindow->setText( i18n("Paste into New &Window") );
  connect(mAction.pasteToNewWindow, SIGNAL(triggered(bool) ), SLOT(pasteNewWindow()));

  QActionGroup *displayMode = new QActionGroup( this );
  mAction.hexadecimal = actionCollection()->add<KToggleAction>( "mode_hex" );
  mAction.hexadecimal->setText( i18n("&Hexadecimal") );
  connect( mAction.hexadecimal, SIGNAL(triggered(bool) ), SLOT(setDisplayMode()) );
  displayMode->addAction(mAction.hexadecimal);
  mAction.decimal     = actionCollection()->add<KToggleAction>( "mode_dec" );
  mAction.decimal->setText( i18n("&Decimal") );
  connect( mAction.decimal, SIGNAL(triggered(bool) ), SLOT(setDisplayMode()) );
  displayMode->addAction(mAction.decimal);
  mAction.octal       = actionCollection()->add<KToggleAction>( "mode_oct" );
  mAction.octal->setText( i18n("&Octal") );
  connect( mAction.octal, SIGNAL(triggered(bool) ), SLOT(setDisplayMode()) );
  displayMode->addAction(mAction.octal);
  mAction.binary      = actionCollection()->add<KToggleAction>( "mode_bin" );
  mAction.binary->setText( i18n("&Binary") );
  connect( mAction.binary, SIGNAL(triggered(bool) ), SLOT(setDisplayMode()) );
  displayMode->addAction(mAction.binary);
  mAction.textOnly    = actionCollection()->add<KToggleAction>( "mode_text" );
  mAction.textOnly->setText( i18n("&Text") );
  connect( mAction.textOnly, SIGNAL(triggered(bool) ), SLOT(setDisplayMode()) );
	displayMode->addAction(mAction.textOnly);

	displayMode->setExclusive( true );

  mAction.showOffsetColumn = actionCollection()->add<KToggleAction>( "show_offset_column" );
  mAction.showOffsetColumn->setText( i18n("Show O&ffset Column") );
  connect( mAction.showOffsetColumn, SIGNAL(triggered(bool) ), editor(), SLOT(toggleOffsetColumnVisibility()) );
  mAction.showTextColumn = actionCollection()->add<KToggleAction>( "show_text_field" );
  mAction.showTextColumn->setText( i18n("Show Te&xt Field") );
  connect( mAction.showTextColumn, SIGNAL(triggered(bool) ), editor(), SLOT(toggleTextColumnVisibility()) );
  mAction.offsetAsDecimal = actionCollection()->add<KToggleAction>( "offset_as_decimal" );
  mAction.offsetAsDecimal->setText( i18n("Off&set as Decimal") );
  connect( mAction.offsetAsDecimal, SIGNAL(triggered(bool) ), editor(), SLOT(toggleOffsetAsDecimal()) );
  mAction.dataUppercase = actionCollection()->add<KToggleAction>( "upper_case_data" );
  mAction.dataUppercase->setText( i18n("&Upper Case (Data)") );
  connect( mAction.dataUppercase, SIGNAL(triggered(bool) ), editor(), SLOT(toggleDataUppercase()) );
  mAction.offsetUppercase = actionCollection()->add<KToggleAction>( "upper_case_offset" );
  mAction.offsetUppercase->setText( i18n("Upper &Case (Offset)") );
  connect( mAction.offsetUppercase, SIGNAL(triggered(bool) ), editor(), SLOT(toggleOffsetUppercase()) );

  QActionGroup *encodingMode = new QActionGroup( this );
  encodingMode->setExclusive(true);
  mAction.defaultEncoding = actionCollection()->add<KToggleAction>( "enc_default" );
  mAction.defaultEncoding->setText( i18nc("&Default encoding", "&Default") );
  connect( mAction.defaultEncoding, SIGNAL(triggered(bool) ), SLOT(setEncoding()) );
  encodingMode->addAction(mAction.defaultEncoding);
  mAction.usAsciiEncoding = actionCollection()->add<KToggleAction>( "enc_ascii" );
  mAction.usAsciiEncoding->setText( i18n("US-&ASCII (7 bit)") );
  connect( mAction.usAsciiEncoding, SIGNAL(triggered(bool) ), SLOT(setEncoding()) );
  encodingMode->addAction(mAction.usAsciiEncoding);
  mAction.ebcdicEncoding  = actionCollection()->add<KToggleAction>( "enc_ebcdic" );
  mAction.ebcdicEncoding->setText( i18n("&EBCDIC") );
  connect( mAction.ebcdicEncoding, SIGNAL(triggered(bool) ), SLOT(setEncoding()) );
  encodingMode->addAction(mAction.ebcdicEncoding);
//   mAction.customEncoding  = new KAction( i18n("&Custom..."),
//     0, editor(), SLOT( encoding()), actionCollection(), "enc_custom" );

  mAction.strings = actionCollection()->addAction( "extract_strings" );
  mAction.strings->setText( i18n("&Extract Strings...") );
  connect(mAction.strings, SIGNAL(triggered(bool) ), editor(), SLOT(strings()));
//   mAction.recordViewer = new KAction( i18n("&Record Viewer"), 0,
//     editor(), SLOT(recordView()), actionCollection(), "record_viewer" );
  mAction.filter = actionCollection()->addAction( "binary_filter" );
  mAction.filter->setText( i18n("&Binary Filter...") );
  connect(mAction.filter, SIGNAL(triggered(bool) ), editor(), SLOT(filter()));
  mAction.characterTable = actionCollection()->addAction( "char_table" );
  mAction.characterTable->setText( i18n("&Character Table") );
  connect(mAction.characterTable, SIGNAL(triggered(bool) ), editor(), SLOT(chart()));
  mAction.converter = actionCollection()->addAction( "converter" );
  mAction.converter->setText( i18n("C&onverter") );
  connect(mAction.converter, SIGNAL(triggered(bool) ), editor(), SLOT(converter()));
  mAction.statistics = actionCollection()->addAction( "statistics" );
  mAction.statistics->setText( i18n("&Statistics") );
  connect(mAction.statistics, SIGNAL(triggered(bool) ), editor(), SLOT(statistics()));

  mAction.addBookmark = KStandardAction::addBookmark( editor(), SLOT(addBookmark()), actionCollection() );
  mAction.replaceBookmark = actionCollection()->addAction( "replace_bookmark");
  mAction.replaceBookmark->setText( i18n("&Replace Bookmark") );
  connect(mAction.replaceBookmark, SIGNAL(triggered(bool) ), editor(), SLOT(replaceBookmark()));
  mAction.replaceBookmark->setShortcut(Qt::CTRL+Qt::Key_E);
  mAction.removeBookmark = actionCollection()->addAction( "remove_bookmark" );
  mAction.removeBookmark->setText( i18n("R&emove Bookmark") );
  mAction.removeBookmark->setShortcut(Qt::CTRL+Qt::Key_U);
  connect(mAction.removeBookmark, SIGNAL(triggered(bool) ), editor(), SLOT(removeBookmark()));
  mAction.removeAllBookmark = actionCollection()->addAction( "remove_all_bookmarks" );
  mAction.removeAllBookmark->setText( i18n("Re&move All") );
  connect(mAction.removeAllBookmark, SIGNAL(triggered(bool) ), editor(), SLOT(removeAllBookmark()));
  mAction.nextBookmark = actionCollection()->addAction( "next_bookmark" );
  mAction.nextBookmark->setText( i18n("Goto &Next Bookmark") );
  mAction.nextBookmark->setShortcut(Qt::ALT+Qt::Key_Down);
  connect(mAction.nextBookmark, SIGNAL(triggered(bool) ), editor(), SLOT(gotoNextBookmark()));
  mAction.prevBookmark = actionCollection()->addAction( "prev_bookmark" );
  mAction.prevBookmark->setText( i18n("Goto &Previous Bookmark") );
  mAction.prevBookmark->setShortcut(Qt::ALT+Qt::Key_Up);
  connect(mAction.prevBookmark, SIGNAL(triggered(bool) ), editor(), SLOT(gotoPrevBookmark()));

  createStandardStatusBarAction();
  setStandardToolBarMenuEnabled(true);
  mAction.showFullPath  = actionCollection()->add<KToggleAction>( "show_full_path" );
  mAction.showFullPath->setText( i18n("Show F&ull Path") );
  connect(mAction.showFullPath, SIGNAL(triggered(bool) ), SLOT(showFullPath()));

  QActionGroup *editorTab = new QActionGroup( this );
  editorTab->setExclusive(true);
  mAction.tabHide = actionCollection()->add<KToggleAction>( "doctab_hide" );
  mAction.tabHide->setText( i18n("&Hide") );
  connect(mAction.tabHide, SIGNAL(triggered(bool) ), SLOT(showDocumentTabs()));
  editorTab->addAction(mAction.tabHide);
  mAction.tabShowAboveEditor = actionCollection()->add<KToggleAction>( "doctab_above" );
  mAction.tabShowAboveEditor->setText( i18n("&Above Editor") );
  connect(mAction.tabShowAboveEditor, SIGNAL(triggered(bool) ), SLOT(showDocumentTabs()));
  editorTab->addAction(mAction.tabShowAboveEditor);
  mAction.tabShowBelowEditor = actionCollection()->add<KToggleAction>( "doctab_below" );
  mAction.tabShowBelowEditor->setText( i18n("&Below Editor") );
  connect(mAction.tabShowBelowEditor, SIGNAL(triggered(bool) ), SLOT(showDocumentTabs()));
  editorTab->addAction(mAction.tabShowBelowEditor);

  QActionGroup *conversionField = new QActionGroup( this );
  conversionField->setExclusive(true);
  mAction.conversionHide  = actionCollection()->add<KToggleAction>( "conversion_field_hide");
  mAction.conversionHide->setText( i18n("&Hide") );
  connect(mAction.conversionHide, SIGNAL(triggered(bool) ), SLOT(showConversionField()));
  conversionField->addAction(mAction.conversionHide);
  mAction.conversionFloat = actionCollection()->add<KToggleAction>( "conversion_field_float");
  mAction.conversionFloat->setText( i18n("&Floating") );
  connect(mAction.conversionFloat, SIGNAL(triggered(bool) ), SLOT(showConversionField()));
  conversionField->addAction(mAction.conversionFloat);
  mAction.conversionEmbed = actionCollection()->add<KToggleAction>( "conversion_field_embed");
  mAction.conversionEmbed->setText( i18n("&Embed in Main Window") );
  connect(mAction.conversionEmbed, SIGNAL(triggered(bool) ), SLOT(showConversionField()));
  conversionField->addAction(mAction.conversionEmbed);

  QActionGroup *searchBar = new QActionGroup( this );
  searchBar->setExclusive( true );
  mAction.searchHide = actionCollection()->add<KToggleAction>( "searchbar_hide" );
  mAction.searchHide->setText( i18n("&Hide") );
  connect(mAction.searchHide, SIGNAL(triggered(bool) ), SLOT(showConversionField()));
  searchBar->addAction(mAction.searchHide);
  mAction.searchShowAboveEditor = actionCollection()->add<KToggleAction>( "searchbar_above" );
  mAction.searchShowAboveEditor->setText( i18n("&Above Editor") );
  connect(mAction.searchShowAboveEditor, SIGNAL(triggered(bool) ), SLOT(showConversionField()));
  searchBar->addAction(mAction.searchShowAboveEditor);
  mAction.searchShowBelowEditor = actionCollection()->add<KToggleAction>( "searchbar_below" );
  mAction.searchShowBelowEditor->setText( i18n("&Below Editor") );
  connect(mAction.searchShowBelowEditor, SIGNAL(triggered(bool) ), SLOT(showConversionField()));
  searchBar->addAction(mAction.searchShowBelowEditor);

  KStandardAction::saveOptions(this, SLOT(writeConfiguration()), actionCollection());
  KStandardAction::preferences(editor(),SLOT(options()),actionCollection() );
//   mAction.favorites = new KAction( i18n("P&rofiles..."), 0,
//     editor(), SLOT(favorites()), actionCollection(), "favorites" );

  KStandardAction::help( this, SLOT(appHelpActivated()), actionCollection() );

  mDragLabel = new CDragLabel(this);
  mDragLabel->setPixmap( UserIcon( "hexdrag" ) );
  mDragLabel->setDragMask( UserIcon( "hexmask" ) );
  mDragLabel->setEnabled( false ); // Enabled once we open a document
  mDragLabel->setToolTip( i18n("Drag document") );
  QAction *act = actionCollection()->addAction( "drag_document");
  act->setText( i18n("Drag Document") );
  qobject_cast<KAction*>( act )->setDefaultWidget( mDragLabel );

  createGUI("khexeditui.rc");

  Q3PopupMenu *popup = (Q3PopupMenu *)factory()->container("editor_popup", this);
  hexView()->setContextMenu( popup );

  toolBar(0)->addAction( new KToolBarSpacerAction( this ) );
  mWriteProtectButton = new KAction( KIcon("system-lock-screen"), i18n("Toggle write protection"), this );
  connect( mWriteProtectButton, SIGNAL(triggered()), editor(), SLOT(toggleWriteProtection()) );
  toolBar(0)->addAction( mWriteProtectButton );

  Q3PopupMenu *documentMenu = (Q3PopupMenu *)factory()->container("documents", this);
  hexView()->setDocumentMenu(documentMenu);
  connect(documentMenu, SIGNAL(activated(int)), SLOT(documentMenuCB(int)));

}


void KHexEdit::setupStatusBar( void )
{
  CStatusBarProgress *progress = new CStatusBarProgress( statusBar() );
  statusBar()->addWidget( progress, 10 );
  connect( progress, SIGNAL(pressed()), editor(), SLOT(stop()) );
  connect( editor(), SIGNAL( setProgress( int ) ),
	   progress, SLOT( setValue( int ) ) );
  connect( editor(), SIGNAL( setProgress( int, int ) ),
	   progress, SLOT( setValue( int, int ) ) );
  connect( editor(), SIGNAL( enableProgressText( bool ) ),
	   progress, SLOT( setTextEnabled( bool ) ) );
  connect( editor(), SIGNAL( setProgressText( const QString & ) ),
	   progress, SLOT( setText( const QString & ) ) );

  statusBar()->insertFixedItem( i18n("Selection: 0000:0000 0000:0000"),
				status_Selection );
  statusBar()->insertFixedItem( i18n("M"), status_Modified );
  statusBar()->insertFixedItem( i18n("OVR"), status_Ovr );
  statusBar()->insertFixedItem( i18n("Size: FFFFFFFFFF"), status_Size );
  statusBar()->insertFixedItem( i18n("Offset: FFFFFFFFFF-F"), status_Offset );
  statusBar()->insertFixedItem( i18n("FFF"), status_Layout );
  statusBar()->insertFixedItem( i18n("RW"), status_WriteProtect );

  statusBar()->setItemAlignment( status_Selection, Qt::AlignLeft|Qt::AlignVCenter );
  statusBar()->setItemAlignment( status_Size, Qt::AlignLeft|Qt::AlignVCenter );
  statusBar()->setItemAlignment( status_Offset, Qt::AlignLeft|Qt::AlignVCenter );

  //
  // Some good default strings we can use after the fields have got
  // their fixed width.
  //
  statusBar()->changeItem( i18n("Offset:"), status_Offset );
  statusBar()->changeItem( i18n("Size:"), status_Size );
  statusBar()->changeItem( "", status_Modified );
  statusBar()->changeItem( "", status_Selection );
  connect( statusBar(),SIGNAL(pressed(int)),this,SLOT(statusBarPressed(int)) );
}


void KHexEdit::removeRecentFiles( void )
{
  mAction.openRecent->clear();
}


void KHexEdit::initialize( bool openFiles )
{
  mUndoState = CHexBuffer::UndoOk; // Reset in function below
  setUndoState( 0 );

  mSelectionOffset = 0;
  mSelectionSize   = 1;
  mSelectionAsHexadecimal = true;
  setSelectionState( 0, 0 );

  operationChanged( false );

  editor()->initialize();

  mAction.showFullPath->setChecked( mShowFullPath );
  showFullPath();

  if ( openFiles && editor()->openFile() != SDisplayMisc::none )
  {
    // Code modified from kdelibs/kdeui/kactionclasses.cpp KRecentFilesAction::loadEntries

    KConfigGroup config( KGlobal::config(), "RecentFiles" );

    // read file list
    unsigned int maxItems = mAction.openRecent->maxItems();
    if ( editor()->openFile() == SDisplayMisc::mostRecent )
        maxItems = 1;
    for( unsigned int i = 1 ; i <= maxItems ; i++ )
    {
        const QString key = QString( "File%1" ).arg( i );
        const QString value = config.readPathEntry( key );

        if (!value.isEmpty())
        {
            mStartupFileList.append( value );
            mStartupOffsetList.append( "0" ); // ### TODO find a way to still have kept offsets
        }
    }
  }

  //
  // Open file(s) after editor has been displayed.
  // Main reason: If anything goes wrong and an error dialog box
  // is displayed, then the editor will not be visible until the error
  // is confirmed and the (modal) dialog is closed.
  //
  QTimer::singleShot( 100, this, SLOT(delayedStartupOpen()) );
}


void KHexEdit::delayedStartupOpen( void )
{
  open( mStartupFileList, mStartupOffsetList );
  mStartupFileList.clear();
  mStartupOffsetList.clear();
}


void KHexEdit::statusBarPressed( int id )
{
  if( id == status_WriteProtect )
  {
    editor()->toggleWriteProtection();
  }
  else if( id == status_Ovr )
  {
    editor()->toggleInsertMode();
  }
  else if( id == status_Offset )
  {
    editor()->toggleOffsetAsDecimal();
  }
  else if( id == status_Selection )
  {
    if( mSelectionSize > 0 )
    {
      mSelectionAsHexadecimal = mSelectionAsHexadecimal == true ? false : true;
      setSelectionText( mSelectionOffset, mSelectionSize );
    }
  }

}



void KHexEdit::open( QStringList &fileList, QStringList &offsetList )
{
  //
  // We load in reverse to allow the first document in
  // the list to become the active one.
  //

  uint offset;
  for( int i = fileList.count(); i>0; i-- )
  {
    if( i <= offsetList.count() )
    {
      offset = offsetList.at(i-1).toUInt(0,16);
    }
    else
    {
      offset = 0;
    }
    editor()->open( fileList.at(i-1), true, offset );
  }
}

void KHexEdit::addRecentFile( const QString &fileName )
{
  if( fileName.isEmpty() == true )
  {
    return;
  }

  if( fileName.contains( i18n( "Untitled" ), Qt::CaseInsensitive ) )
  {
    return;
  }

  const KUrl url( fileName );
  kDebug(1501) << " adding recent " << fileName << " => " << url.prettyUrl() ;
  mAction.openRecent->addUrl( url );

}


void KHexEdit::removeRecentFile( const QString &fileName )
{
  if( fileName.isEmpty() == true )
  {
    return;
  }

  if( fileName.contains( i18n( "Untitled" ), Qt::CaseInsensitive ) )
  {
    return;
  }

  const KUrl url( fileName );
  mAction.openRecent->removeUrl( url );

}


void KHexEdit::renameRecentFile(const QString &curName, const QString &newName)
{
  if( curName.contains( i18n( "Untitled" ), Qt::CaseInsensitive ) )
  {
    addRecentFile( newName );
  }
  else
  {
      mAction.openRecent->removeUrl( curName );
      mAction.openRecent->addUrl( newName );
  }
}



void KHexEdit::slotFileOpenRecent( const KUrl& url )
{
  if ( url.isLocalFile() )
  {
    editor()->open( url.path(), false, 0 );
  }
  else
  {
    // ### TODO: support network transparency
    KMessageBox::error( this, i18n("Non local recent file: %1", url.prettyUrl() ) );
  }
}


KHexEdit *KHexEdit::newWindow( void )
{
  KHexEdit *hexEdit = new KHexEdit;
  if( hexEdit == 0 )
  {
    QString msg = i18n( "Can not create new window.\n" );
    msg += hexError( Err_NoMemory );
    KMessageBox::error( topLevelWidget(), msg );
    return(0);
  }
  hexEdit->show();
  return(hexEdit);
}



void KHexEdit::pasteNewWindow( void )
{
  KHexEdit *app = newWindow();
  if( app != 0 )
  {
    app->editor()->pasteNewFile();
  }
}



void KHexEdit::closeWindow( void )
{
  close();
}


void KHexEdit::closeProgram( void )
{
  if( mWindowList.count() > 1 )
  {
    for( KHexEdit *w = mWindowList.first(); w != 0; w = mWindowList.next() )
    {
      if( w->editor()->modified() == true )
      {
	QString msg = i18n(""
	  "There are windows with unsaved modified documents. "
	  "If you quit now, these modifications will be lost.");
	int reply = KMessageBox::warningContinueCancel( topLevelWidget(), msg, QString(), KStandardGuiItem::quit() );
	if( reply == KMessageBox::Continue )
	{
	  break;
	}
	else
	{
	  return;
	}
      }
    }


    //
    // Ths will write current configuration to disk.
    //
    queryExit();

    //
    // Make sure every widget is destroyed. The reason why I do this
    // (besides it is the clean way to do it) is to make sure any
    // configuration updates in the various destructors are triggered.
    //
    mWindowList.setAutoDelete(true);
    while( mWindowList.first() )
    {
      mWindowList.removeRef( mWindowList.first() );
    }

    kapp->quit();
  }
  else
  {
    closeWindow();
  }

}


bool KHexEdit::queryClose( void )
{
  bool success = editor()->closeAll();
  if( success == true )
  {
    mWindowList.removeRef( this );
  }

  return( success );
}


//
// We will always return true here.
//
bool KHexEdit::queryExit( void )
{
  writeConfiguration();
  return( true );
}

void KHexEdit::readConfiguration( void )
{
  KConfig &config = *KGlobal::config();
  readConfiguration( config );
}


void KHexEdit::writeConfiguration( void )
{
  KConfig &config = *KGlobal::config();
  writeConfiguration( config );
}


void KHexEdit::writeConfiguration( KConfig &config )
{
  KConfigGroup cg( &config, "General Options" );
  saveMainWindowSettings( cg );
  cg.writeEntry("ShowFullPath",  mShowFullPath );
  cg.writeEntry("ConversionPosition", (int)mManager->conversionPosition());
  cg.writeEntry("TabBarPosition", (int)mManager->tabBarPosition());
  cg.writeEntry("SearchBarPosition", (int)mManager->searchBarPosition());

  if ( editor()->discardRecentFiles() )
      mAction.openRecent->clear();
  mAction.openRecent->saveEntries( cg );

  converter()->writeConfiguration( config );
  editor()->writeConfiguration( config );

  cg.sync();
}


void KHexEdit::readConfiguration( KConfig &config )
{
  KConfigGroup cg( &config, "General Options" );
  applyMainWindowSettings(cg );

  mShowFullPath   = cg.readEntry( "ShowFullPath", true );
  int position    = cg.readEntry( "ConversionPosition",
					 int(CHexManagerWidget::Embed) );

  mAction.conversionHide->blockSignals(true);
  mAction.conversionFloat->blockSignals(true);
  mAction.conversionEmbed->blockSignals(true);
  mAction.conversionHide->setChecked(
    position == CHexManagerWidget::Hide );
  mAction.conversionFloat->setChecked(
    position == CHexManagerWidget::Float );
  mAction.conversionEmbed->setChecked(
    position == CHexManagerWidget::Embed );
  mAction.conversionHide->blockSignals(false);
  mAction.conversionFloat->blockSignals(false);
  mAction.conversionEmbed->blockSignals(false);
  showConversionField();

  position = cg.readEntry( "TabBarPosition", int(CHexManagerWidget::Hide) );
  mAction.tabHide->blockSignals(true);
  mAction.tabShowAboveEditor->blockSignals(true);
  mAction.tabShowBelowEditor->blockSignals(true);
  mAction.tabHide->setChecked(
    position==CHexManagerWidget::Hide );
  mAction.tabShowAboveEditor->setChecked(
    position==CHexManagerWidget::AboveEditor );
  mAction.tabShowBelowEditor->setChecked(
    position==CHexManagerWidget::BelowEditor );
  mAction.tabHide->blockSignals(false);
  mAction.tabShowAboveEditor->blockSignals(false);
  mAction.tabShowBelowEditor->blockSignals(false);
  showDocumentTabs();

  position = cg.readEntry( "SearchBarPosition",
                           int(CHexManagerWidget::BelowEditor) );
  mAction.searchHide->blockSignals(true);
  mAction.searchShowAboveEditor->blockSignals(true);
  mAction.searchShowBelowEditor->blockSignals(true);
  mAction.searchHide->setChecked(
    position==CHexManagerWidget::Hide );
  mAction.searchShowAboveEditor->setChecked(
    position==CHexManagerWidget::AboveEditor );
  mAction.searchShowBelowEditor->setChecked(
    position==CHexManagerWidget::BelowEditor );
  mAction.searchHide->blockSignals(false);
  mAction.searchShowAboveEditor->blockSignals(false);
  mAction.searchShowBelowEditor->blockSignals(false);
  showSearchBar();

  mAction.openRecent->loadEntries( cg );

  converter()->readConfiguration( config );
  editor()->readConfiguration( config );
}



void KHexEdit::operationChanged( bool state )
{
  mAction.cancel->setEnabled( state );
}



void KHexEdit::cursorChanged( SCursorState &state )
{
  QString offset( i18n("Offset:") );

  if( state.valid == false )
  {
    statusBar()->changeItem( offset, status_Offset );
  }
  else
  {
    SDisplayLayout &layout = editor()->layout();

    if( layout.offsetMode == SDisplayLayout::hexadecimal )
    {
      if( layout.offsetUpperCase == false )
      {
	offset += QString().sprintf( " %04x:%04x-%u", state.offset>>16,
				     state.offset&0x0000FFFF, state.cell );
      }
      else
      {
	offset += QString().sprintf( " %04X:%04X-%u", state.offset>>16,
				     state.offset&0x0000FFFF, state.cell );
      }
    }
    else
    {
      offset += QString().sprintf( " %010u-%u", state.offset, state.cell );
    }
    statusBar()->changeItem( offset, status_Offset );
  }

  setUndoState( state.undoState );
  setSelectionState( state.selectionOffset, state.selectionSize );
}




void KHexEdit::fileState( SFileState &state )
{
  if( state.valid == true )
  {
    statusBar()->changeItem( i18n("Size: %1", state.size ), status_Size);
    statusBar()->changeItem( state.modified ? "!" : "", status_Modified);

    if( mIsModified != state.modified )
    {
      mIsModified = state.modified;
      setupCaption( hexView()->url() );
    }

  }
  else
  {
    statusBar()->changeItem( "", status_Size );
    statusBar()->changeItem( "", status_Modified );
  }
}



void KHexEdit::layoutChanged( const SDisplayLayout &layout )
{
  KToggleAction *radioAction;

  if( layout.primaryMode == SDisplayLayout::hexadecimal )
  {
    radioAction = mAction.hexadecimal;
  }
  else if( layout.primaryMode == SDisplayLayout::decimal )
  {
    radioAction = mAction.decimal;
  }
  else if( layout.primaryMode == SDisplayLayout::octal )
  {
    radioAction = mAction.octal;
  }
  else if( layout.primaryMode == SDisplayLayout::binary )
  {
    radioAction = mAction.binary;
  }
  else
  {
    radioAction = mAction.textOnly;
  }
  radioAction->blockSignals(true);
  radioAction->setChecked( true );
  radioAction->blockSignals(false);

  mAction.showOffsetColumn->blockSignals(true);
  mAction.showTextColumn->blockSignals(true);
  mAction.offsetAsDecimal->blockSignals(true);
  mAction.dataUppercase->blockSignals(true);
  mAction.offsetUppercase->blockSignals(true);

  mAction.showOffsetColumn->setChecked( layout.offsetVisible );
  mAction.showTextColumn->setEnabled(
    layout.primaryMode != SDisplayLayout::textOnly );
  mAction.showTextColumn->setChecked(
    layout.secondaryMode != SDisplayLayout::hide );
  mAction.offsetAsDecimal->setChecked(
    layout.offsetMode != SDisplayLayout::hexadecimal);
  mAction.dataUppercase->setChecked( layout.primaryUpperCase );
  mAction.offsetUppercase->setChecked( layout.offsetUpperCase );

  mAction.showOffsetColumn->blockSignals(false);
  mAction.showTextColumn->blockSignals(false);
  mAction.offsetAsDecimal->blockSignals(false);
  mAction.dataUppercase->blockSignals(false);
  mAction.offsetUppercase->blockSignals(false);

  if( layout.primaryMode == SDisplayLayout::hexadecimal )
  {
    statusBar()->changeItem( i18n("Hex"), status_Layout );
  }
  else if( layout.primaryMode == SDisplayLayout::decimal )
  {
    statusBar()->changeItem( i18n("Dec"), status_Layout );
  }
  else if( layout.primaryMode == SDisplayLayout::octal )
  {
    statusBar()->changeItem( i18n("Oct"), status_Layout );
  }
  else if( layout.primaryMode == SDisplayLayout::binary )
  {
    statusBar()->changeItem( i18n("Bin"), status_Layout );
  }
  else if( layout.primaryMode == SDisplayLayout::textOnly )
  {
    statusBar()->changeItem( i18n("Txt"), status_Layout );
  }
}


void KHexEdit::inputModeChanged( const SDisplayInputMode &mode )
{
  mAction.readOnly->blockSignals(true);
  mAction.resizeLock->blockSignals(true);
  mAction.readOnly->setChecked( mode.readOnly );
  mAction.resizeLock->setChecked( mode.allowResize );
  mAction.readOnly->blockSignals(false);
  mAction.resizeLock->blockSignals(false);

  statusBar()->changeItem( mode.readOnly == true ?
			   i18n("R") : i18n("RW"), status_WriteProtect );

  if( mode.readOnly == false )
  {
    mWriteProtectButton->setIcon(KIcon("edit"));
  }
  else
  {
    mWriteProtectButton->setIcon(KIcon("system-lock-screen"));
  }
}


void KHexEdit::bookmarkChanged( Q3PtrList<SCursorOffset> &list )
{
  unplugActionList("bookmark_list");
  qDeleteAll( mAction.bookmarkList );
  mAction.bookmarkList.clear();

  if( list.count() == 0 )
  {
    return;
  }

  QAction *sep = new QAction(this);
  sep->setSeparator(true);
  mAction.bookmarkList.append(sep);

  QString text, offset;
  int i=0;

  for( SCursorOffset *p=list.first(); p!=0; p=list.next(), i++ )
  {
    offset.sprintf("%04X:%04X", p->offset>>16, p->offset&0x0000FFFF );
    text = i18n("Offset: %1", offset);
    QAction *action = actionCollection()->addAction(text.toLatin1());
    action->setText(text);
    connect(action, SIGNAL(triggered(bool) ), mAction.bookmarkMapper, SLOT(map()));
    int key = acceleratorNumKey( i );
    if( key > 0 )
    {
      action->setShortcuts(KShortcut(Qt::ALT+key));
    }
    mAction.bookmarkMapper->setMapping(action, i);
    mAction.bookmarkList.append(action);
  }
  plugActionList("bookmark_list", mAction.bookmarkList);
}


void KHexEdit::setupCaption( const QString &url )
{
  KUrl u(url);
  if( mShowFullPath == true )
  {
    setCaption( u.prettyUrl(), hexView()->modified() );
  }
  else
  {
    setCaption( u.fileName(), hexView()->modified() );
  }
}


void KHexEdit::fileActive( const QString &url, bool onDisk )
{
  setupCaption( url );
  mDragLabel->setUrl( url );
  mDragLabel->setEnabled( onDisk );

  mAction.revert->setEnabled( onDisk );

  if( url.isNull() == false )
  {
    addDocument( url );
    addRecentFile( url );
    setTickedDocument( url );
  }
}


void KHexEdit::fileRename( const QString &curName, const QString &newName )
{
  if( newName.isNull() == true )
  {
    return;
  }

  renameDocument( curName, newName );
  renameRecentFile( curName, newName );
  setupCaption( newName );
  //setCaption( newName );
}


void KHexEdit::fileClosed( const QString &url )
{
  removeDocument( url );
}


void KHexEdit::editMode( CHexBuffer::EEditMode editMode )
{
  statusBar()->changeItem( editMode == CHexBuffer::EditReplace ?
			   i18n("OVR") : i18n("INS"), status_Ovr );
}


void KHexEdit::encodingChanged( const SEncodeState &encodeState )
{
  KToggleAction *radioAction;

  if( encodeState.mode == CConversion::cnvDefault )
  {
    radioAction = mAction.defaultEncoding;
  }
  else if( encodeState.mode == CConversion::cnvUsAscii )
  {
    radioAction = mAction.usAsciiEncoding;
  }
  else if( encodeState.mode == CConversion::cnvEbcdic )
  {
    radioAction = mAction.ebcdicEncoding;
  }
  else
  {
    return;
  }

  radioAction->blockSignals(true);
  radioAction->setChecked( true );
  radioAction->blockSignals(false);

  if( mSelectionSize == 0 )
  {
    statusBar()->changeItem( i18n("Encoding: %1", encodeState.name),
			     status_Selection );
  }
}


void KHexEdit::textWidth( uint )
{
  resize( mManager->preferredWidth(), height() );
}


void KHexEdit::setDisplayMode( void )
{
  if( mAction.hexadecimal->isChecked() == true )
  {
    editor()->setHexadecimalMode();
  }
  else if( mAction.decimal->isChecked() == true )
  {
    editor()->setDecimalMode();
  }
  else if( mAction.octal->isChecked() == true )
  {
    editor()->setOctalMode();
  }
  else if( mAction.binary->isChecked() == true )
  {
    editor()->setBinaryMode();
  }
  else if( mAction.textOnly->isChecked() == true )
  {
    editor()->setTextMode();
  }
}


void KHexEdit::showFullPath( void )
{
  mShowFullPath = mAction.showFullPath->isChecked();
  setupCaption( hexView()->url() );
}


void KHexEdit::showDocumentTabs( void )
{
  if( mAction.tabHide->isChecked() == true )
  {
    mManager->setTabBarPosition( CHexManagerWidget::HideItem );
  }
  else if( mAction.tabShowAboveEditor->isChecked() == true )
  {
    mManager->setTabBarPosition( CHexManagerWidget::AboveEditor );
  }
  else if( mAction.tabShowBelowEditor->isChecked() == true )
  {
    mManager->setTabBarPosition( CHexManagerWidget::BelowEditor );
  }
}


void KHexEdit::showConversionField( void )
{
  if( mAction.conversionHide->isChecked() == true )
  {
    mManager->setConversionVisibility( CHexManagerWidget::Hide );
  }
  else if( mAction.conversionFloat->isChecked() == true )
  {
    mManager->setConversionVisibility( CHexManagerWidget::Float );
  }
  else if( mAction.conversionEmbed->isChecked() == true )
  {
    mManager->setConversionVisibility( CHexManagerWidget::Embed );
    if( width() < mManager->preferredWidth() )
    {
      resize( mManager->preferredWidth(), height() );
    }
  }
}


void KHexEdit::showSearchBar( void )
{
  if( mAction.searchHide->isChecked() == true )
  {
    mManager->setSearchBarPosition( CHexManagerWidget::HideItem );
  }
  else if( mAction.searchShowAboveEditor->isChecked() == true )
  {
    mManager->setSearchBarPosition( CHexManagerWidget::AboveEditor );
  }
  else if( mAction.searchShowBelowEditor->isChecked() == true )
  {
    mManager->setSearchBarPosition( CHexManagerWidget::BelowEditor );
  }
}


void KHexEdit::setEncoding( void )
{
  if( mAction.defaultEncoding->isChecked() == true )
  {
    editor()->encode( CConversion::cnvDefault );
  }
  else if( mAction.usAsciiEncoding->isChecked() == true )
  {
    editor()->encode( CConversion::cnvUsAscii );
  }
  else if( mAction.ebcdicEncoding->isChecked() == true )
  {
    editor()->encode( CConversion::cnvEbcdic );
  }
}


void KHexEdit::setUndoState( uint undoState )
{
  if( mUndoState != undoState )
  {
    mUndoState = undoState;
    mAction.undo->setEnabled( mUndoState & CHexBuffer::UndoOk );
    mAction.redo->setEnabled( mUndoState & CHexBuffer::RedoOk );
  }
}


void KHexEdit::setSelectionState( uint selectionOffset, uint selectionSize )
{
  if( mSelectionSize != selectionSize )
  {
    mSelectionOffset = selectionOffset;
    mSelectionSize   = selectionSize;

    bool valid = mSelectionSize > 0 ? true : false;
    mAction.copy->setEnabled( valid );
    mAction.cut->setEnabled( valid );
    mAction.copyAsText->setEnabled( valid );

    setSelectionText( selectionOffset, selectionSize );
  }
}


void KHexEdit::setSelectionText( uint selectionOffset, uint selectionSize )
{
  if( selectionSize > 0 )
  {
    QString selection = i18n("Selection:");
    if( mSelectionAsHexadecimal == true )
    {
      selection += QString().sprintf( " %04x:%04x %04x:%04x",
        selectionOffset>>16, selectionOffset&0x0000FFFF,
        selectionSize>>16, selectionSize&0x0000FFFF );
    }
    else
    {
      selection += QString().sprintf( " %08u %08u",
        selectionOffset, selectionSize);
    }
    statusBar()->changeItem( selection, status_Selection );
  }
  else
  {
    statusBar()->changeItem(
      i18n("Encoding: %1", hexView()->encoding().name), status_Selection);
  }
}


void KHexEdit::documentMenuCB( int index )
{
  if( index < mDocumentList.count() )
  {
    editor()->open( mDocumentList[index], false, 0 );
  }
}


void KHexEdit::addDocument( const QString &fileName )
{
  uint documentCount = mDocumentList.count();
  for( uint i=0; i < documentCount; i++ )
  {
    if( fileName == mDocumentList[i] )
    {
      return;
    }
  }

  mDocumentList.append( fileName );

  Q3PopupMenu *documentMenu = (Q3PopupMenu *)factory()->container("documents", this);
  documentMenu->insertItem( fileName, documentCount, documentCount );
  int key = acceleratorNumKey( documentCount );
  if( key > 0 )
  {
    documentMenu->setAccel( Qt::CTRL+key, documentCount );
  }
}


void KHexEdit::removeDocument( const QString &fileName )
{
  QStringList::Iterator it;
  for( it = mDocumentList.begin(); it != mDocumentList.end(); it++ )
  {
    if( *it == fileName )
    {
      Q3PopupMenu *documentMenu = (Q3PopupMenu *)factory()->container("documents", this);

      documentMenu->removeItemAt( mDocumentList.indexOf(*it) );
      mDocumentList.erase( it );

      for( uint i=0; i < mDocumentList.count(); i++ )
      {
	documentMenu->setId( i, i );
	int key = acceleratorNumKey( i );
	if( key > 0 )
	{
	  documentMenu->setAccel( Qt::CTRL+key, i );
	}
      }
      return;

    }
  }
}


void KHexEdit::renameDocument( const QString &curName, const QString &newName )
{
  QStringList::Iterator it;
  for( it = mDocumentList.begin(); it != mDocumentList.end(); it++ )
  {
    if( *it == curName )
    {
      Q3PopupMenu *documentMenu = (Q3PopupMenu *)factory()->container("documents", this);
      documentMenu->changeItem( mDocumentList.indexOf(*it), newName );
      mDocumentList.insert( it, newName );
      mDocumentList.erase( it );
      return;
    }
  }
}


void KHexEdit::setTickedDocument( const QString &fileName )
{
  uint documentCount = mDocumentList.count();
  Q3PopupMenu *documentMenu = (Q3PopupMenu *)factory()->container("documents", this);
  for( uint i=0; i < documentCount; i++ )
  {
    documentMenu->setItemChecked( i, fileName == mDocumentList[i] );
  }
}


void KHexEdit::conversionClosed( void )
{
  mAction.conversionHide->setChecked(true);
}


void KHexEdit::searchBarClosed( void )
{
  mAction.searchHide->setChecked(true);
}


bool KHexEdit::eventFilter( QObject *o, QEvent *event )
{
  if( event->type() == QEvent::DragEnter )
  {
    QDragEnterEvent *e = (QDragEnterEvent*)event;
    KUrl::List list = KUrl::List::fromMimeData( e->mimeData() );
    if( !list.isEmpty() )
    {
      e->accept();
      hexView()->setDropHighlight( true );
    }
    return( true );
  }
  else if( event->type() == QEvent::Drop )
  {
    QDropEvent *e = (QDropEvent*)event;
    hexView()->setDropHighlight( false );

    KUrl::List list = KUrl::List::fromMimeData( e->mimeData() );
    if( !list.isEmpty() )
    {
      QStringList offset;
      QStringList urlList = list.toStringList();
      open( urlList, offset );
      return( true );
    }
    return( true );
  }
  else if( event->type() == QEvent::DragLeave )
  {
    hexView()->setDropHighlight( false );
    return( true );
  }

  return KXmlGuiWindow::eventFilter( o, event );
}


int KHexEdit::acceleratorNumKey( uint index )
{
  static int keys[9] =
  {
    Qt::Key_1, Qt::Key_2, Qt::Key_3, Qt::Key_4, Qt::Key_5, Qt::Key_6, Qt::Key_7, Qt::Key_8, Qt::Key_9
  };
  return( index >= 9 ? -1 : keys[index] );
}


#include <sys/time.h>
#include <unistd.h>
#include <kglobal.h>

//
// This function is used to test kwin performance
//
void KHexEdit::resizeTest()
{
  struct timeval t1, t2;
  gettimeofday( &t1, 0 );

  int loop=400;

  for( int i=0; i<loop; i+=4 )
  {
    resize( i+400, i+400 );
    //kapp->processOneEvent();
  }

  gettimeofday( &t2, 0 );
  uint last = (t2.tv_sec-t1.tv_sec) * 1000000 + (t2.tv_usec - t1.tv_usec);

  kDebug(1501) << "Duration: " << (float)last/ 1000000.0 <<  endl;
  kDebug(1501) << "Duration/loop: " << (float)last/ (1000000.0*(float)loop) ;
}
#include "toplevel.moc"
