/*
 *   khexedit - Versatile hex editor
 *   Copyright (C) 1999  Espen Sand, espensa@online.no
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "printdialogpage.h"

#include <QLayout>
#include <q3buttongroup.h>
#include <QSpinBox>
#include <QLabel>
#include <QCheckBox>
#include <KComboBox>

#include <klocale.h>
#include <kconfiggroup.h>


LayoutDialogPage::LayoutDialogPage( QWidget *parent )
 : KPrintDialogPage( parent )
{
  mConfig = 0;
  setTitle( i18n( "Page Layout" ) );
  setupLayoutPage();

  readConfiguration();
}


LayoutDialogPage::~LayoutDialogPage( void )
{
  writeConfiguration();

  delete mConfig; mConfig = 0;
}


void LayoutDialogPage::setupLayoutPage( void )
{
  QString text;
  QVBoxLayout *topLayout = new QVBoxLayout( this );
  topLayout->setSpacing( 6 /*KDialog::mSpacingSize*/ );
  topLayout->setMargin( 0 );

  text = i18n("Margins [millimeter]");
  Q3ButtonGroup *group = new Q3ButtonGroup( text, this );
  topLayout->addWidget( group );

  QGridLayout *gbox = new QGridLayout( group );
  gbox->setSpacing( 6 );
  gbox->addItem( new QSpacerItem( 0, group->fontMetrics().height() ), 0, 0 );
  gbox->setColumnStretch( 5, 10 );

  QString name[4];
  int i;

  name[0] = i18n("&Top:");
  name[1] = i18n("&Bottom:");
  name[2] = i18n("&Left:");
  name[3] = i18n("&Right:");

  for( i=0; i<4; i++ )
  {
    mLayout.marginSpin[i] = new QSpinBox( group );
    mLayout.marginSpin[i]->setFixedHeight(
      mLayout.marginSpin[i]->sizeHint().height() );
    mLayout.marginSpin[i]->setMinimumWidth(
      mLayout.marginSpin[i]->fontMetrics().width("M")*10 );
    mLayout.marginSpin[i]->setRange( 0, INT_MAX );

    QLabel *label = new QLabel( name[i], group );
    label->setBuddy( mLayout.marginSpin[i] );
    label->setFixedHeight( mLayout.marginSpin[i]->sizeHint().height() );
    label->setFixedWidth( label->sizeHint().width() );

    if( i < 2 )
    {
      gbox->addWidget( label, i+1, 0, Qt::AlignLeft );
      gbox->addWidget( mLayout.marginSpin[i], i+1, 1, Qt::AlignLeft );
    }
    else
    {
      gbox->addWidget( label, i-1, 3, Qt::AlignLeft );
      gbox->addWidget( mLayout.marginSpin[i], i-1, 4, Qt::AlignLeft );
    }
  }

  text = i18n("Draw h&eader above text");
  mLayout.headerCheck = new QCheckBox( text, this );
  mLayout.headerCheck->setFixedSize( mLayout.headerCheck->sizeHint() );
  connect( mLayout.headerCheck, SIGNAL( toggled(bool)),
	   SLOT( slotDrawHeader(bool)));
  topLayout->addWidget( mLayout.headerCheck, 0, Qt::AlignLeft );

  gbox = new QGridLayout();
  topLayout->addLayout( gbox );

  gbox->setColumnStretch ( 5, 10 );
  gbox->addItem( new QSpacerItem( 6 /*KDialog::mSpacingSize*/, 0 ), 0, 1 );
  gbox->addItem( new QSpacerItem( 6 /*KDialog::mSpacingSize*/, 0 ), 0, 3 );
  gbox->addItem( new QSpacerItem( 0, 6 /*KDialog::mSpacingSize*/ ), 2, 0 );

  name[0] = i18n("Left:");
  name[1] = i18n("Center:");
  name[2] = i18n("Right:");
  name[3] = i18n("Border:");

  QStringList textList;
  textList.append(i18n("None"));
  textList.append(i18n("Date & Time"));
  textList.append(i18n("Page Number"));
  textList.append(i18n("Filename"));

  QStringList lineList;
  lineList.append(i18n("None"));
  lineList.append(i18n("Single Line"));
  lineList.append(i18n("Rectangle"));

  for( i=0; i<4; i++ )
  {
    mLayout.headerCombo[i] = new KComboBox( this );
    mLayout.headerCombo[i]->setFixedHeight(
      mLayout.headerCombo[i]->sizeHint().height() );
    mLayout.headerCombo[i]->setMinimumWidth(
      mLayout.headerCombo[i]->fontMetrics().width("M")*10 );

    mLayout.headerLabel[i] = new QLabel( name[i], this );
    mLayout.headerLabel[i]->setBuddy( mLayout.headerCombo[i] );
    mLayout.headerLabel[i]->setFixedHeight(
      mLayout.headerLabel[i]->sizeHint().height() );
    mLayout.headerLabel[i]->setFixedWidth(
      mLayout.headerLabel[i]->sizeHint().width() );

    if( i<3 )
    {
      mLayout.headerCombo[i]->addItems( textList );
      gbox->addWidget( mLayout.headerLabel[i], 0, i*2, Qt::AlignLeft );
      gbox->addWidget( mLayout.headerCombo[i], 1, i*2, Qt::AlignLeft );
    }
    else
    {
      mLayout.headerCombo[i]->addItems( lineList );
      gbox->addWidget( mLayout.headerLabel[i], 3, 0, Qt::AlignLeft );
      gbox->addWidget( mLayout.headerCombo[i], 4, 0, Qt::AlignLeft );
    }
  }


  text = i18n("Draw &footer below text");
  mLayout.footerCheck = new QCheckBox( text, this );
  mLayout.footerCheck->setFixedSize( mLayout.footerCheck->sizeHint() );
  connect( mLayout.footerCheck, SIGNAL( toggled(bool)),
	   SLOT( slotDrawFooter(bool)));
  topLayout->addWidget( mLayout.footerCheck, 0, Qt::AlignLeft );

  gbox = new QGridLayout( );
  topLayout->addLayout( gbox );

  gbox->setColumnStretch ( 5, 10 );
  gbox->addItem( new QSpacerItem( 6 /*KDialog::mSpacingSize*/, 0 ), 0, 1 );
  gbox->addItem( new QSpacerItem( 6 /*KDialog::mSpacingSize*/, 0 ), 0, 3 );
  gbox->addItem( new QSpacerItem( 0, 6 /*KDialog::mSpacingSize*/ ), 2, 0 );

  for( i=0; i<4; i++ )
  {
    mLayout.footerCombo[i] = new KComboBox( this );
    mLayout.footerCombo[i]->setFixedHeight(
      mLayout.footerCombo[i]->sizeHint().height() );
    mLayout.footerCombo[i]->setMinimumWidth(
      mLayout.footerCombo[i]->fontMetrics().width("M")*10 );

    mLayout.footerLabel[i] = new QLabel( name[i], this );
    mLayout.footerLabel[i]->setBuddy( mLayout.footerCombo[i] );
    mLayout.footerLabel[i]->setFixedHeight(
      mLayout.footerLabel[i]->sizeHint().height() );
    mLayout.footerLabel[i]->setFixedWidth(
      mLayout.footerLabel[i]->sizeHint().width() );

    if( i<3 )
    {
      mLayout.footerCombo[i]->addItems( textList );
      gbox->addWidget( mLayout.footerLabel[i], 0, i*2, Qt::AlignLeft );
      gbox->addWidget( mLayout.footerCombo[i], 1, i*2, Qt::AlignLeft );
    }
    else
    {
      mLayout.footerCombo[i]->addItems( lineList );
      gbox->addWidget( mLayout.footerLabel[i], 3, 0, Qt::AlignLeft );
      gbox->addWidget( mLayout.footerCombo[i], 4, 0, Qt::AlignLeft );
    }
  }

  for( i=0; i<4; i++ )
  {
    mLayout.marginSpin[i]->setValue( 15 );
  }
  mLayout.headerCheck->setChecked( true );
  slotDrawHeader( true );
  mLayout.footerCheck->setChecked( true );
  slotDrawFooter( true );

  topLayout->addStretch(10);
}


void LayoutDialogPage::readConfiguration( void )
{
  if( mConfig != 0 ) { return; }

  mConfig = new KConfig( QLatin1String("hexprinter"), KConfig::OnlyLocal );
  if( mConfig == 0 ) { return; }

  KConfigGroup gr = mConfig->group( "PageLayout" );
  int val = gr.readEntry( "MarginTop", 15 );
  mLayout.marginSpin[0]->setValue( val );
  val = gr.readEntry( "MarginBottom", 15 );
  mLayout.marginSpin[1]->setValue( val );
  val = gr.readEntry( "MarginLeft", 15 );
  mLayout.marginSpin[2]->setValue( val );
  val = gr.readEntry( "MarginRight", 15 );
  mLayout.marginSpin[3]->setValue( val );
  bool state = gr.readEntry( "DrawHeader", true );
  mLayout.headerCheck->setChecked( state );
  slotDrawHeader( state );
  state = gr.readEntry( "DrawFooter", true );
  mLayout.footerCheck->setChecked( state );
  slotDrawFooter( state );

  mLayout.headerCombo[0]->setCurrentIndex(
    headerTextIndex( gr.readEntry("HeaderLeft","DateTime")));
  mLayout.headerCombo[1]->setCurrentIndex(
    headerTextIndex( gr.readEntry("HeaderCenter")));
  mLayout.headerCombo[2]->setCurrentIndex(
    headerTextIndex( gr.readEntry("HeaderRight","FileName")));
  mLayout.headerCombo[3]->setCurrentIndex(
    headerLineIndex( gr.readEntry("HeaderLine","SingleLine")));

  mLayout.footerCombo[0]->setCurrentIndex(
    headerTextIndex( gr.readEntry("FooterLeft")));
  mLayout.footerCombo[1]->setCurrentIndex(
    headerTextIndex( gr.readEntry("FooterCenter","PageNumber")));
  mLayout.footerCombo[2]->setCurrentIndex(
    headerTextIndex( gr.readEntry("FooterRight")));
  mLayout.footerCombo[3]->setCurrentIndex(
    headerLineIndex( gr.readEntry("FooterLine","SingleLine")));

}


void LayoutDialogPage::writeConfiguration( void )
{
  if( mConfig == 0 )
  {
    return;
  }

  KConfigGroup gr = mConfig->group( "PageLayout" );
  gr.writeEntry( "MarginTop", mLayout.marginSpin[0]->value() );
  gr.writeEntry( "MarginBottom", mLayout.marginSpin[1]->value() );
  gr.writeEntry( "MarginLeft", mLayout.marginSpin[2]->value() );
  gr.writeEntry( "MarginRight", mLayout.marginSpin[3]->value() );
  gr.writeEntry( "DrawHeader", mLayout.headerCheck->isChecked() );
  gr.writeEntry( "DrawFooter", mLayout.footerCheck->isChecked() );

  gr.writeEntry( "HeaderLeft",
    headerText( mLayout.headerCombo[0]->currentIndex() ) );
  gr.writeEntry( "HeaderCenter",
    headerText( mLayout.headerCombo[1]->currentIndex() ) );
  gr.writeEntry( "HeaderRight",
    headerText( mLayout.headerCombo[2]->currentIndex() ) );
  gr.writeEntry( "HeaderLine",
    headerLine( mLayout.headerCombo[3]->currentIndex() ) );

  gr.writeEntry( "FooterLeft",
    headerText( mLayout.footerCombo[0]->currentIndex() ) );
  gr.writeEntry( "FooterCenter",
    headerText( mLayout.footerCombo[1]->currentIndex() ) );
  gr.writeEntry( "FooterRight",
    headerText( mLayout.footerCombo[2]->currentIndex() ) );
  gr.writeEntry( "FooterLine",
    headerLine( mLayout.footerCombo[3]->currentIndex() ) );

  mConfig->sync();
}


QString LayoutDialogPage::headerText( uint index )
{
  static const QString text[4] = {"None","DateTime","PageNumber","FileName"};
  return( text[ index > 3 ? 0 : index ] );
}


int LayoutDialogPage::headerTextIndex( const QString & headerText )
{
  static const QString text[4] = {"None","DateTime","PageNumber","FileName"};
  if( headerText != 0 )
  {
    for( int i=0; i<4; i++ )
    {
      if( headerText == text[i] ) { return( i ); }
    }
  }
  return( 0 );
}


QString LayoutDialogPage::headerLine( uint index )
{
  static const QString text[3] = {"None","SingleLine","Rectangle"};
  return( text[ index > 2 ? 0 : index ] );
}


int LayoutDialogPage::headerLineIndex( const QString & headerLine )
{
  static const QString text[3] = {"None","SingleLine","Rectangle"};
  if( headerLine != 0 )
  {
    for( int i=0; i<3; i++ )
    {
      if( headerLine == text[i] ) { return( i ); }
    }
  }
  return( 0 );
}


void LayoutDialogPage::slotDrawHeader( bool state )
{
  for( int i=0; i<4; i++ )
  {
    mLayout.headerLabel[i]->setEnabled( state );
    mLayout.headerCombo[i]->setEnabled( state );
  }
}


void LayoutDialogPage::slotDrawFooter( bool state )
{
  for( int i=0; i<4; i++ )
  {
    mLayout.footerLabel[i]->setEnabled( state );
    mLayout.footerCombo[i]->setEnabled( state );
  }
}


void LayoutDialogPage::getOptions( QMap<QString,QString>& opts, bool /*incldef*/ )
{
   opts[ "kde-khexedit-topmarginmm" ]    = QString::number( mLayout.marginSpin[0]->value() );
   opts[ "kde-khexedit-bottommarginmm" ] = QString::number( mLayout.marginSpin[1]->value() );
   opts[ "kde-khexedit-leftmarginmm" ]   = QString::number( mLayout.marginSpin[2]->value() );
   opts[ "kde-khexedit-rightmarginmm" ]  = QString::number( mLayout.marginSpin[3]->value() );

   opts[ "kde-khexedit-headercheck" ]  = (mLayout.headerCheck->isChecked() ? "true" : "false");
   opts[ "kde-khexedit-headercombo0" ] = QString::number( mLayout.headerCombo[0]->currentIndex() );
   opts[ "kde-khexedit-headercombo1" ] = QString::number( mLayout.headerCombo[1]->currentIndex() );
   opts[ "kde-khexedit-headercombo2" ] = QString::number( mLayout.headerCombo[2]->currentIndex() );
   opts[ "kde-khexedit-headercombo3" ] = QString::number( mLayout.headerCombo[3]->currentIndex() );

   opts[ "kde-khexedit-footercheck" ]  = (mLayout.footerCheck->isChecked() ? "true" : "false");
   opts[ "kde-khexedit-footercombo0" ] = QString::number( mLayout.footerCombo[0]->currentIndex() );
   opts[ "kde-khexedit-footercombo1" ] = QString::number( mLayout.footerCombo[1]->currentIndex() );
   opts[ "kde-khexedit-footercombo2" ] = QString::number( mLayout.footerCombo[2]->currentIndex() );
   opts[ "kde-khexedit-footercombo3" ] = QString::number( mLayout.footerCombo[3]->currentIndex() );
}


#include "printdialogpage.moc"
