/***************************************************************************
 *                                                                         *
 *   copyright (C) 2005 by Michael Buesch                                  *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   Based on a patch by                                                   *
 *   copyright (C) 2004 Martin Preuss <martin@libchipcard.de>              *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include "masterkey/smartkey/libchipcard2/chipcard2interface.h"
#include "pwmexception.h"


Chipcard2Interface::Chipcard2Interface()
 : client (0)
 , card (0)
{
}

SmartKey_backend::Status Chipcard2Interface::write(const QByteArray &data)
{
	SmartKey_backend::Status ret;

	ret = ct_open();
	if (ret != SmartKey_backend::status_ok) {
		printDebug("Chipcard2Interface: ct_open() failed");
		goto out;
	}
	ret = ct_write(data);
	if (ret != SmartKey_backend::status_ok)
		printDebug("Chipcard2Interface: ct_write() failed");
	ct_close();
out:
	return ret;
}

SmartKey_backend::Status Chipcard2Interface::read(QByteArray *data)
{
	SmartKey_backend::Status ret;

	ret = ct_open();
	if (ret != SmartKey_backend::status_ok) {
		printDebug("Chipcard2Interface: ct_open() failed");
		goto out;
	}
	ret = ct_read(data);
	if (ret != SmartKey_backend::status_ok)
		printDebug("Chipcard2Interface: ct_read() failed");
	ct_close();
out:
	return ret;
}

SmartKey_backend::Status Chipcard2Interface::erase()
{
	SmartKey_backend::Status ret;

	ret = ct_open();
	if (ret != SmartKey_backend::status_ok) {
		printDebug("Chipcard2Interface: ct_open() failed");
		goto out;
	}
	ret = ct_erase();
	if (ret != SmartKey_backend::status_ok)
		printDebug("Chipcard2Interface: ct_erase() failed");
	ct_close();
out:
	return ret;
}

SmartKey_backend::Status Chipcard2Interface::ct_open()
{
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;
	int i;
	LC_CLIENT_RESULT lcerr;

	showOpenWindow(true);

	client = LC_Client_new(PACKAGE_NAME, PACKAGE_VER, 0);
	if (LC_Client_ReadConfigFile(client, 0)) {
		printWarn("Chipcard2Interface: Could not read config");
		ret = SmartKey_backend::status_errOpen;
		goto err_freeclient;
	}

	// wait for the user to insert the card
	lcerr = LC_Client_StartWait(client, 0, 0);
	if (lcerr != LC_Client_ResultOk) {
		printDebug("Chipcard2Interface: LC_Client_StartWait() failed");
		ret = SmartKey_backend::status_errOpen;
		goto err_freeclient;
	}
	for (i = 0; i < CARD_OPEN_TIMEOUT; ++i) {
		if (isUserCancel()) {
			LC_Client_StopWait(client);
			ret = SmartKey_backend::status_cancel;
			goto err_freeclient;
		}
		card = LC_Client_WaitForNextCard(client, 1);
		if (card) {
			// found a card.
			const char *cardtype;
			cardtype = LC_Card_GetCardType(card);
			PWM_ASSERT(cardtype);
			if (strcasecmp(cardtype, "memory")) {
				printDebug("Chipcard2Interface: Not a memory card");
				LC_Client_StopWait(client);
				//FIXME: extra error message for this:
				ret = SmartKey_backend::status_errOpen;
				goto err_freecard;
			}
			break;
		}
		openWindowProgress();
	}
	LC_Client_StopWait(client);

	if (!card) {
		printDebug("Chipcard2Interface: Card insert timeout");
		ret = SmartKey_backend::status_errTimeout;
		goto err_freeclient;
	}

	// extend card (use it as a memory card)
	i = LC_MemoryCard_ExtendCard(card);
	if (i) {
		printDebug("Chipcard2Interface: Could not extend card as memory card");
		ret = SmartKey_backend::status_errOpen;
		goto err_freecard;
	}

	// open card (actually take the card)
	lcerr = LC_Card_Open(card);
	if (lcerr != LC_Client_ResultOk) {
		printDebug("Chipcard2Interface: Could not open card");
		ret = SmartKey_backend::status_errOpen;
		goto err_freecard;
	}
	// :)
out:
	showOpenWindow(false);
	return ret;

err_freecard:
	LC_Card_free(card);
	card = 0;
err_freeclient:
	LC_Client_free(client);
	client = 0;
	goto out;
}

void Chipcard2Interface::ct_close()
{
	PWM_ASSERT(client);
	PWM_ASSERT(card);
	LC_Card_Close(card);
	LC_Card_free(card);
	card = 0;
	LC_Client_free(client);
	client = 0;
}

SmartKey_backend::Status Chipcard2Interface::ct_write(const QByteArray &data)
{
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;
	unsigned int cardSize;
	LC_CLIENT_RESULT lcerr;

	PWM_ASSERT(client);
	PWM_ASSERT(card);
	showAccessWindow(true);
	cardSize = LC_MemoryCard_GetCapacity(card);
	if (cardSize < data.size() + CARD_BEGIN_OFFSET) {
		printDebug("Chipcard2Interface: Card too small");
		ret = SmartKey_backend::status_err2small;
		goto out;
	}
	lcerr = LC_MemoryCard_WriteBinary(card,
					  CARD_BEGIN_OFFSET,
					  data.data(),
					  data.size());
	if (lcerr != LC_Client_ResultOk) {
		printDebug("Chipcard2Interface: LC_MemoryCard_WriteBinary() failed");
		ret = SmartKey_backend::status_errWrite;
		goto out;
	}
out:
	showAccessWindow(false);
	return ret;
}

SmartKey_backend::Status Chipcard2Interface::ct_read(QByteArray *data)
{
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;
	unsigned int cardSize;
	GWEN_BUFFER *gwenBuf;
	LC_CLIENT_RESULT lcerr;

	PWM_ASSERT(client);
	PWM_ASSERT(card);
	showAccessWindow(true);
	cardSize = LC_MemoryCard_GetCapacity(card);
	gwenBuf = GWEN_Buffer_new(0, cardSize - CARD_BEGIN_OFFSET, 0, 1);
	if (!gwenBuf) {
		printDebug("Chipcard2Interface: OOM");
		ret = SmartKey_backend::status_errNoMem;
		goto out;
	}
	lcerr = LC_MemoryCard_ReadBinary(card,
					 CARD_BEGIN_OFFSET,
					 cardSize - CARD_BEGIN_OFFSET,
					 gwenBuf);
	if (lcerr != LC_Client_ResultOk) {
		printDebug("Chipcard2Interface: LC_MemoryCard_ReadBinary() failed");
		ret = SmartKey_backend::status_errRead;
		goto out_freegwen;
	}
	data->duplicate(GWEN_Buffer_GetStart(gwenBuf), GWEN_Buffer_GetUsedBytes(gwenBuf));
out_freegwen:
	GWEN_Buffer_free(gwenBuf);
out:
	showAccessWindow(false);
	return ret;
}

SmartKey_backend::Status Chipcard2Interface::ct_erase()
{
	SmartKey_backend::Status ret = SmartKey_backend::status_ok;
	unsigned int i = 0, cardSize;
	const unsigned int byteCntEachLoop = 255;
	unsigned int bytesToWrite = byteCntEachLoop;
	char nulBuf[byteCntEachLoop];
	LC_CLIENT_RESULT lcerr;

	PWM_ASSERT(client);
	PWM_ASSERT(card);
	showAccessWindow(true);
	memset(nulBuf, 0, byteCntEachLoop);
	cardSize = LC_MemoryCard_GetCapacity(card);
	while (i < cardSize) {
		if ((cardSize - i) < byteCntEachLoop)
			bytesToWrite = cardSize - i;
		lcerr = LC_MemoryCard_WriteBinary(card,
						  i,
						  nulBuf,
						  bytesToWrite);
		if (lcerr != LC_Client_ResultOk) {
			printDebug("Chipcard2Interface: LC_Client_WriteBinary() failed");
			ret = SmartKey_backend::status_errWrite;
			goto out;
		}
		i += byteCntEachLoop;
	}
out:
	showAccessWindow(false);
	return ret;
}
