/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004, 2005  by Michael Buesch                           *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#ifndef PWMANAGER_IPC_H_
#define PWMANAGER_IPC_H_

#include <qcstring.h>
#include <qvaluelist.h>
#include <qmutex.h>
#include <qtimer.h>

#define IPC_DEFAULT_LOOKUP_FREQ		100 /* msec */


/** Message Queue based thread communication.
  * This is the Host-end of the queue, which receives
  * the messages.
  */
class ThreadIpcHost : public QObject
{
	Q_OBJECT

	friend class ThreadIpcClient;

public:
	ThreadIpcHost(int lookupFreq = IPC_DEFAULT_LOOKUP_FREQ,
		      bool start = false);
	~ThreadIpcHost();

public:
	/** Start the queue lookup.
	  * You will only receive messages, if the queue lookup
	  * is started.
	  */
	void startLookup()
		{ timer.start(freq); }
	/** Stop the queue lookup. */
	void stopLookup();

signals:
	/** This signal is emitted as soon as a new messages arrives
	  * on the queue.
	  * @param msg This is the message itself.
	  */
	void receivedMessage(const QByteArray &msg);

protected:
	/** Used by ThreadIpcClient to queue a new message. */
	void queueMessage(const QByteArray &msg);

protected slots:
	/** queue lookup. This is triggered by a timer. */
	void queueLookup();

protected:
	/** The message queue. */
	QValueList<QByteArray> mq;
	/** lock for the message queue. */
	QMutex mqLock;
	/** message queue lookup timer. */
	QTimer timer;
	/** Lookup timer frequency. */
	int freq;
};


/** Message Queue based thread communication.
  * This is the Client-end of the queue, which sends
  * the messages.
  */
class ThreadIpcClient
{
public:
	ThreadIpcClient(ThreadIpcHost &_host)
		 : host (&_host)
		{ }

	/** Send a new message through the Message Queue to the host.
	  * @param msg The message data.
	  */
	void sendMessage(const QByteArray &msg)
		{ host->queueMessage(msg); }

protected:
	/** Pointer to the host.
	  * Be careful when manipulating data in the host. We are
	  * in another thread here!
	  */
	ThreadIpcHost *host;
};

#endif // PWMANAGER_IPC_H_
