/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "backendjobs.h"

#include <QMetaType>

#include <kurl.h>
#include <kdebug.h>
#include <kio/global.h>
#include <kblog/blogger1.h>
#include <kblog/metaweblog.h>
#include <kblog/gdata.h>
#include <kblog/movabletype.h>
#include <kblog/wordpressbuggy.h>
#include <kblog/blogpost.h>
#include <kblog/blogmedia.h>
#include <klocalizedstring.h>
#include <kmessagebox.h>

#include "backend.h"
#include "itemsmanager.h"
#include "kbloggerpost.h"
#include "kbloggermedia.h"

#define APPLICATION_NAME "KBlogger"
#define APPLICATION_VER "0.8-wip"

Q_DECLARE_METATYPE(KBlogger::KBloggerPost*);
Q_DECLARE_METATYPE(KBlog::BlogMedia*);
namespace KBlogger
{

///backendJobs
BackendJobs::BackendJobs( KBloggerBlog kbBlog,
                          BlogActions action,
                          QVariant arg,
                          QWidget *WaitWidgetParent,
                          QObject* parent ):
        KJob ( parent ),
        mAction(action),
        mArg(arg),
        mKBlogInfo(kbBlog),
        mKBlog(0),
        mBackend( Backend::self() ),
        mItemsManager( ItemsManager::self() ),
        mWaitWidgetParent(WaitWidgetParent)
{
    kDebug();
    switch (action) {
    case LIST_RECENT_POSTS: {
        mWaitMessage = i18n("Listing recent posts");
    }
    break;

    case FETCH_POST: {
        mWaitMessage = i18n("Fetch Post");
    }
    break;

    case MODIFY_POST: {
        mWaitMessage = i18n("modifying Post");
    }
    break;

    case CREATE_POST: {
        mWaitMessage = i18n("Creating Post");
    }
    break;

    case REMOVE_POST: {
        mWaitMessage = i18n("Removing Post");
    }
    break;

    case FETCH_CATEGORIES: { //METAWEBLOG
        mWaitMessage = i18n("Fetching Categories");
    }
    break;

    case CREATE_MEDIA: {  //METAWEBLOG
        mWaitMessage = i18n("Creating Media");
    }
    break;

    }//switch
}

BackendJobs::~BackendJobs()
{
    kDebug();
}

void BackendJobs::start ()
{
    kDebug();

    createBackend(&mKBlogInfo);
    if (!mKBlog) {
        kError() << "mKBlog is NULL" << endl;
        return;
    }

    //Default connections
    //A. XML Parsing Error
    connect(mKBlog, SIGNAL( error( KBlog::Blog::ErrorType, const QString& ) ),
            mBackend, SLOT( slotError( KBlog::Blog::ErrorType, const QString&) )  );
    connect(mKBlog, SIGNAL( error( KBlog::Blog::ErrorType, const QString& ) ),
            this, SLOT( slotExitError( KBlog::Blog::ErrorType, const QString&) )  );

    //B. Post Error
    connect(mKBlog, SIGNAL( errorPost( KBlog::Blog::ErrorType,
                                       const QString &,
                                       KBlog::BlogPost* ) ),
            mBackend, SLOT( slotError( KBlog::Blog::ErrorType, const QString&) )  );

    connect(mKBlog, SIGNAL( errorPost(KBlog::Blog::ErrorType,
                                      const QString &,
                                      KBlog::BlogPost* )),
            this, SLOT( slotExitError( KBlog::Blog::ErrorType, const QString&) )  );

    //C. Comment Error
    connect(mKBlog, SIGNAL( errorComment( KBlog::Blog::ErrorType,
                                          const QString &,
                                          KBlog::BlogPost* ) ),
            mBackend, SLOT( slotError( KBlog::Blog::ErrorType, const QString&) )  );

    connect(mKBlog, SIGNAL( errorMedia(KBlog::Blog::ErrorType,
                                       const QString &,
                                       KBlog::BlogMedia* )),
            this, SLOT( slotExitError( KBlog::Blog::ErrorType, const QString&) )  );

    //D. Media Error
    void errorMedia( KBlog::Blog::ErrorType type,
                     const QString &errorMessage, KBlog::BlogMedia* media );
    connect(mKBlog, SIGNAL( errorMedia( KBlog::Blog::ErrorType,
                                        const QString &,
                                        KBlog::BlogMedia* ) ),
            mBackend, SLOT( slotError( KBlog::Blog::ErrorType, const QString&) )  );

    connect(mKBlog, SIGNAL( errorMedia(KBlog::Blog::ErrorType,
                                       const QString &,
                                       KBlog::BlogMedia* )),
            this, SLOT( slotExitError( KBlog::Blog::ErrorType, const QString&) )  );

    //Specific connections
    switch (mAction) {
    case LIST_RECENT_POSTS: {
        int postsToDownload;
        postsToDownload = mArg.toInt();
        connect(mKBlog, SIGNAL( listedRecentPosts( const QList<KBlog::BlogPost>& ) ),
                this, SLOT( slotListedRecentPosts( const QList<KBlog::BlogPost>& ) ) );
        kDebug() << "Running " << mKBlog
        << "->listRecentPosts(" << postsToDownload << ")" << endl;
        mKBlog->listRecentPosts( postsToDownload );
    }
    break;

    case FETCH_POST: {
        KBloggerPost *post;
        post = mArg.value<KBlogger::KBloggerPost*>();
        Q_ASSERT(post);
        connect(mKBlog, SIGNAL( fetchedPost( KBlog::BlogPost *) ),
                this, SLOT( slotFetchedPost( KBlog::BlogPost *)) );
        kDebug() << "Running " << mKBlog << "->fetchPost()" << endl;
        mKBlog->fetchPost(post);
    }
    break;

    case MODIFY_POST: {
        KBloggerPost *post;
//         qRegisterMetaType<KBlog::BlogPost*>("KBlog::BlogPost*");
        if ( ! mArg.canConvert<KBlogger::KBloggerPost *>() ) {
            kError() << "Impossible convert QVariant to BlogPost";
            return;
        }
        post = mArg.value<KBlogger::KBloggerPost*>();
        Q_ASSERT(post);

        if ( post->hasPendingMedia() ) {
            KMessageBox::sorry( 0 , i18n("This post cannot be uploaded. Upload its media first"));
            return;
        }

        connect(mKBlog, SIGNAL( modifiedPost( KBlog::BlogPost *) ),
                this, SLOT( slotModifiedPost( KBlog::BlogPost *) ));
        kDebug() << "Running " << mKBlog << "->modifyPost()" << endl;
        mKBlog->modifyPost( post );
    }
    break;

    case CREATE_POST: {
        KBloggerPost *post;
//         qRegisterMetaType<KBlog::BlogPost*>("KBlog::BlogPost*");
        if ( ! mArg.canConvert<KBlogger::KBloggerPost*>() ) {
            kError() << "Impossible convert QVariant to BlogPost";
            return;
        }
        post = mArg.value<KBlogger::KBloggerPost*>();

        Q_ASSERT(post);

        if ( post->hasPendingMedia() ) {
            KMessageBox::sorry( 0 , i18n("This post cannot be uploaded. Upload its media first"));
            slotExitError(KBlog::Blog::Other, i18n("This post cannot be uploaded. Upload its media first"));
            return;
        }

        connect(mKBlog, SIGNAL( createdPost( KBlog::BlogPost *) ),
                this, SLOT( slotCreatedPost( KBlog::BlogPost *) ));
        kDebug() << "Running " << mKBlog << "->createPost()" << endl;
        kDebug() << "Post DateTime" << post->creationDateTime().dateTime().toString() << endl;
        mKBlog->createPost( post );
    }
    break;

    case REMOVE_POST: {
        KBlog::BlogPost *post;
//         qRegisterMetaType<KBlog::BlogPost*>("KBlog::BlogPost*");
        if ( ! mArg.canConvert<KBlogger::KBloggerPost*>() ) {
            kError() << "impossible convert QVariant to BlogPost";
            slotExitError(KBlog::Blog::Other, i18n("This post cannot be uploaded. Upload its media first"));
            return;
        }
        post = mArg.value<KBlogger::KBloggerPost*>();
        Q_ASSERT(post);
        connect(mKBlog, SIGNAL( removedPost( KBlog::BlogPost *) ),
                this, SLOT( slotRemovedPost( KBlog::BlogPost *) ));
        kDebug() << "Running " << mKBlog << "->removedPost()" << endl;
        mKBlog->removePost( post );
    }
    break;

    case FETCH_CATEGORIES: { //METAWEBLOG
        KBlog::MetaWeblog *MWBlog = qobject_cast<KBlog::MetaWeblog*>(mKBlog);
        connect(MWBlog, SIGNAL( listedCategories( const QList<QMap<QString, QString> >&  )),
                this, SLOT( slotListedCategories( const QList<QMap<QString, QString> >&  ) ));
        kDebug() << "Running " << MWBlog << "->listCategories" << endl;

        MWBlog->listCategories();
    }
    break;

    case CREATE_MEDIA: { //METAWEBLOG
        KBlog::BlogMedia *media;
        KBlog::MetaWeblog *MWBlog = qobject_cast<KBlog::MetaWeblog*>(mKBlog); // FIXME not possible for GDATA
        media = mArg.value<KBlog::BlogMedia*>();
        if (!media) {
            kError() << "media is NULL" << endl;
            return;
        }
        if(!MWBlog) { // FIXME do not crash on GDATA for the moment
            kError() << "MWBlog is NULL: casting has not worked, this should NEVER happen, has the gui allowed using GDATA?" << endl;
            return;
        }
        connect(MWBlog, SIGNAL( createdMedia( KBlog::BlogMedia* ) ),
                this, SLOT( slotCreatedMedia( KBlog::BlogMedia* ) ));
        kDebug() << "Running " << MWBlog << "->createMedia" << endl;

        MWBlog->createMedia(media);
    }
    break;

    }

    // TODO why do it again? remove if there are no issues, otherwise fix it...
//     createBackend(&mKBlogInfo);
    if (!mKBlog) {
        kError() << "mKBlog is NULL" << endl;
        return;
    }

}

const QString& BackendJobs::getWaitMessage()
{
    kDebug() << "backendJobs::getWaitMessage message" << mWaitMessage << endl;
    return mWaitMessage;
};

///Private Slots
void BackendJobs::createBackend(KBloggerBlog *kbBlog)
{
    //TODO ReUse & Store mBackends!
    // i don' t know if this would be of large benefit. creating the backends is not 
    // a that big task, but you have to keep the momory for stored backends
    // then. memory <-> processing time
    kDebug();
    if (! kbBlog ) {
        kError() << "kbBlog is NULL" << endl;
        return;
    }
    delete mKBlog;
    kDebug() << "\t kbBlog blogname=" << kbBlog->url() << endl;
    switch (kbBlog->api()) {
    case Backend::BLOGGER_API: {
        mKBlog = new KBlog::Blogger1( KUrl() , 0 );
    }
    break;
    case Backend::METAWEBLOG_API: {
        mKBlog = new KBlog::MetaWeblog( KUrl() , 0 );
    }
    break;
    case Backend::MOVABLETYPE_API: {
        mKBlog = new KBlog::MovableType( KUrl() , 0 );
    }
    break;
    case Backend::WORDPRESSBUGGY_API: {
        mKBlog = new KBlog::WordpressBuggy( KUrl() , 0 );
    }
    break;
    case Backend::GDATA_API: {
        mKBlog = new KBlog::GData( KUrl() , 0 );
    }
    break;
    default:
        kError() << "Unknow API Type" << endl;
    }

     mKBlog->setUrl( kbBlog->url() );
     mKBlog->setBlogId ( kbBlog->blogId() );
     mKBlog->setUsername ( kbBlog->username() );
     mKBlog->setPassword ( kbBlog->password() );


     if (mKBlog) {
        kDebug() << "Set up Backend:" << endl
        << "Url=" << mKBlog->url() << endl
        << "BlogID=" << mKBlog->blogId() << endl
        << "Username=" << mKBlog->username() << endl
        << "Password=" << mKBlog->password() << endl;
    }
}

void BackendJobs::slotExitSuccess()
{
    kDebug();
    setError(0);
    emitResult();
}

void BackendJobs::slotExitError( KBlog::Blog::ErrorType type, const QString& message )
{
    kDebug();
    setError( static_cast<int>(type) );
    setErrorText( message );
    emitResult();
}

void  BackendJobs::slotListedRecentPosts( const QList<KBlog::BlogPost>& posts)
{
    kDebug();

    for (int i = 0; i < posts.size(); ++i) {
        KBloggerPost *post = new KBloggerPost(&posts.at(i));
        //post->setDeleted( false );
        post->setBlogName( mKBlogInfo.blogname() );
        if ( post->title().isEmpty() ) {
            QString title = i18n("Untitled post of %1", KDateTime::currentUtcDateTime().toString()) ;
            post->setTitle(title);
        }
        kDebug() << "blogname=" << mKBlogInfo.blogname()
        << " title=" << post->title()
        << " status=" << post->status();
        post->setPrivate( false ); //NOTE: Metaweblog don't return published var! :(
        mItemsManager->addToSentList(post);
    }
    slotExitSuccess();
    mItemsManager->getUncachedMediaInThePosts( mKBlogInfo.blogname() );
}

void BackendJobs::slotListedCategories( const QList< QMap<QString, QString> >& categories )
{
    kDebug();
    mBackend->categoryRetrieved(categories);
    slotExitSuccess();
}

void BackendJobs::slotCreatedPost( KBlog::BlogPost *post)
{
    kDebug();
    KBloggerPost *kbPost = static_cast<KBloggerPost*>(post);
    if (!kbPost) {
        kError() << "kbpost is NULL" << endl;
        return;
    }
    kDebug() << "blogname=" << mKBlogInfo.blogname() << endl;
    kbPost->setBlogName(mKBlogInfo.blogname());
    mItemsManager->moveDraftToSentList(kbPost);

    slotExitSuccess();
    /*
     KLocalizedString successMessage;
     successMessage = ki18n("Post %1 uploaded with success");
     QString successString = successMessage.subs( id ).toString();
     KMessageBox::information( 0 , successString , i18n("KBlogger"));
    */
}

void BackendJobs::slotRemovedPost( KBlog::BlogPost * post)
{
    kDebug();
    KBloggerPost *kbPost = static_cast<KBloggerPost*>(post);
    if (!kbPost) {
        kError() << "kbpost is NULL" << endl;
        return;
    }
    kDebug() << "blogname=" << mKBlogInfo.blogname() << endl;
    kbPost->setBlogName(mKBlogInfo.blogname());

    mItemsManager->removeTrashedPost(kbPost); //TEST IT

    slotExitSuccess();
}

void BackendJobs::slotFetchedPost( KBlog::BlogPost *post)
{
    kDebug();
    KBloggerPost *kbPost = static_cast<KBloggerPost*>(post);
    if (!kbPost) {
        kError() << "kbpost is NULL" << endl;
        return;
    }
    kDebug() << "blogname=" << mKBlogInfo.blogname() << endl;
    kbPost->setBlogName(mKBlogInfo.blogname());
    //TODO
    slotExitSuccess();
}

void BackendJobs::slotModifiedPost( KBlog::BlogPost *post)
{
    kDebug();
    KBloggerPost *kbPost = static_cast<KBloggerPost*>(post);
    if (!kbPost) {
        kError() << "kbpost is NULL" << endl;
        return;
    }
    kDebug() << "blogname=" << mKBlogInfo.blogname() << endl;
    kbPost->setBlogName(mKBlogInfo.blogname());
    mItemsManager->moveDraftToSentList(kbPost);
    slotExitSuccess();
}

void BackendJobs::slotCreatedMedia( KBlog::BlogMedia* media)
{
    kDebug();
    KBloggerMedia *kbMedia = static_cast<KBloggerMedia*>(media);
    if (!kbMedia) {
        kError() << "kbMedia is NULL" << endl;
        return;
    }
    kDebug() << "blogname=" << mKBlogInfo.blogname() << endl;
    kbMedia->setBlogname(mKBlogInfo.blogname());
    mItemsManager->addMedia(kbMedia);
    slotExitSuccess();
}

}

#include "backendjobs.moc"
