{-
    Kaya - My favourite toy language.
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU General
    Public Licence. See COPYING for licence.
-}

module TAC where

-- This module implements Three Address Code (sort of). 
-- Exports compileAll, which compiles well typed expressions to
-- bytecode (using fcompile).
-- Also some helpers for generating raw C.

-- Big fat hairy FIXME: fcompile needs to return a monad, otherwise
-- we don't get proper error handling!

import Language
import Debug.Trace
import List
import System.Random

data CompileResult = RawCode String  -- Raw C, eg for #includes
		   | ByteCode (Name,[TAC]) -- Function name and bytecode 
		   | ExternDef Name
   deriving Show

type Var = Name
type Tmp = Int

data TAC = DECLARE Var
	 | DECLAREARG Var
	 | DECLAREQUICK Var
	 | TMPINT Tmp
	 | TMPREAL Tmp
	 | TMPVAL Var
	 | ARRAY Var
	 | SET Var Int Var -- Int is number of indices to get off stack
	 | TOINDEX
	 | SETTOP
	 | ADDTOP
	 | SUBTOP
	 | MULTOP
	 | DIVTOP
	 | MKARRAY Int
	 | USETMP Var -- Flag that Var was used, do nothing
	 | DEREF
	 | TMPSET Tmp Int
	 | RTMPSET Tmp Double
	 | CALL Var
	 | CALLNAME Name
	 | CALLTOP
	 | TAILCALL Var
	 | TAILCALLNAME Name
	 | TAILCALLTOP
	 | CLOSURE Name Int
	 | FOREIGNCALL String String Type [Type] -- name, library, type, args
	 | MKCON Int Int -- Create a constructor, tag and arity
	 | MKEXCEPT -- Create an exception from top two stack items
	 | EQEXCEPT -- Check two exceptions for equality
	 | NEEXCEPT -- Check two exceptions for inequality
	 | EQSTRING -- Check two strings for equality
	 | NESTRING -- Check two stringss for inequality
	 | GETFUN Var Var
	 | GETVAL Tmp
	 | GETRVAL Tmp
	 | PUSHGETVAL PushItem Tmp
	 | PUSHGETRVAL PushItem Tmp
	 | GETINDEX
	 | PROJARG Int Int -- Project first arg, check tag is second arg.
	 | INFIX Tmp Op Tmp Tmp
	 | REALINFIX Tmp Op Tmp Tmp
	 | REALINFIXBOOL Op Tmp Tmp
	 | INTPOWER Tmp Tmp Tmp
	 | REALPOWER Tmp Tmp Tmp
	 | UNARY Tmp UnOp Tmp
	 | REALUNARY Tmp UnOp Tmp
	 | APPEND
	 | PRINTINT
	 | PRINTSTR
	 | PRINTEXC
	 | NEWLINE
	 | LABEL Name
	 | JUMP Name
	 | JFALSE Name
	 | JTRUE Name
	 | JTFALSE Tmp Name
	 | JTTRUE Tmp Name
	 | TRY Name -- label to jump to on failure
	 | TRIED -- Pop last item from try stack
	 | THROW -- Throw an exception
	 | RESTORE -- Restore stack state to before the exception
	 | CASE [[TAC]] -- Get tag from top stack item.
	 | PROJ Var Int -- Project an argument from top stack item into Var
	 | PUSHGLOBAL String Int
	 | CREATEGLOBAL String Int
	 | PUSH PushItem
	 | PUSH2 PushItem PushItem
	 | PUSH3 PushItem PushItem PushItem
	 | PUSH4 PushItem PushItem PushItem PushItem
	 | PUSHSETTOP PushItem
	 | RETURN
	 | SETVAL Var Int
	 | SETINT Var Tmp
	 | SETVAR Var Var
	 | GETLENGTH
	 | POP Var
	 | POPARG Var
	 | POPINDEX Var
	 | DISCARD
	 | STR2INT
	 | INT2STR
	 | REAL2STR
	 | BOOL2STR
	 | STR2REAL
	 | STR2CHR
	 | CHR2STR
	 | INT2REAL
	 | REAL2INT
	 | VMPTR
	 | DUMMY
	 | ERROR
   deriving Show

data PushItem = NAME Name
	      | VAL Int
	      | RVAL Double
	      | STR String
	      | INT Tmp
	      | REAL Tmp
	      | VAR Var
   deriving Show

loc i = "l" ++ show i
tmp i = "t" ++ show i
tmpval i = UN ("tv" ++ show i)

type Local = Int

comp :: Monad m => Program -> Expr Name -> Name -> m [TAC]
comp ds e mod = do code <- fcompile ds e mod
		   return $ peephole $ decls $ code

-- Metadata for compilation of an expression
data CompData = CD {
		    modulename :: Name,
		    locs :: [Var],
		    tmpvar :: Int,
		    breaklabel :: Maybe Name,
		    tries :: Int,
		    alldecls :: Program
		   }

fcompile :: Monad m => Program -> Expr Name -> Name -> m [TAC]
fcompile ds exp mod = do
   (code, tmps) <- fcompile' exp (CD mod [] 0 Nothing 0 ds) 0
   return code

-- fcompile' needs to keep track of:
--   local variable names ([Var])
--   labels used so far (first int, also returned)
--   temporary variables in the current block (second int)

fcompile' :: Monad m => Expr Name -> CompData -> Int -> m ([TAC], Int)
fcompile' (Global n) _ lab = return ([PUSH (NAME n)], lab)
fcompile' (Loc l) cd lab = return ([PUSH (VAR ((locs cd)!!l))], lab)
fcompile' (GVar i) cd lab = return ([PUSHGLOBAL (show (modulename cd)) i], lab)
fcompile' (GConst c) _ lab = return (ccompile c, lab)
fcompile' (Lambda args sc) (CD mod vars tmp b tr ds) lab 
    = do (code,lab') <- fcompile' sc (CD mod (vars++(map fst args)) tmp b tr ds) lab
	 let acode = acompile args
	 return (acode ++ code, lab')
fcompile' (Closure _ _ _) cd lab = fail "Encountered a closure. Something is probably broken."
fcompile' (Bind n t v sc) (CD mod vars tmp b tr ds) lab 
    = do (vcode, lab') <- fcompile' v (CD mod vars tmp b tr ds) lab
         (sccode, lab'') <- fcompile' sc (CD mod (vars++[n]) tmp b tr ds) lab'
         return ((vcode ++ (POP n):sccode), lab'')
fcompile' (Declare f l n t sc) (CD mod vars tmp b tr ds) lab
    = fcompile' sc (CD mod (vars++[n]) tmp b tr ds) lab
fcompile' (Return v) cd lab
    = do (vcode, lab') <- fcompile' v cd lab
	 return (vcode ++ (take (tries cd) (repeat TRIED)) ++ [RETURN], lab')
fcompile' (VoidReturn) cd lab
    = return ([RETURN], lab)
fcompile' (Assign l v) cd lab
     = do (code, lab') <- fcompile' v cd lab
	  (fpcode, lab'') <- findplace l lab'
	  return (code ++ fpcode ++ [SETTOP], lab'')
  where findplace (AName i) lab = return ([PUSH (VAR ((locs cd)!!i))], lab)
	findplace (AGlob i) lab = return ([PUSHGLOBAL (show (modulename cd)) i], lab)
	findplace (AIndex l i) lab 
            = do (fpcode,lab') <- findplace l lab
		 (code,lab'') <- fcompile' i cd lab'
		 return (fpcode ++ code ++ [TOINDEX], lab'')
	findplace (AField l f a t) lab
            = do (fpcode,lab') <- findplace l lab
		 return (fpcode ++ [PROJARG a t], lab')
fcompile' (AssignOp op l v) cd lab
     = do (code, lab') <- fcompile' v cd lab
	  (fpcode, lab'') <- findplace l lab'
	  return (fpcode ++ code ++ [doToTop op], lab'')
  where doToTop Plus = ADDTOP
	doToTop Minus = SUBTOP
	doToTop Times = MULTOP
	doToTop Divide = DIVTOP
--	   (code ++ [POP (MN ("tmp",0))] ++ fpcode, lab'')
{-
  where findplace (AName i) j lab = ([SET (vars!!i) j (MN ("tmp",0))], lab)
	findplace (AIndex l i) j lab 
	    = let (code, lab') = fcompile' i vars lab tmp
		  (fpcode, lab'') = findplace l (j+1) lab' in
		  (code ++ fpcode, lab'')
	findplace (AField l f a t) j lab = error "Broken" -}
	findplace (AName i) lab = return ([PUSH (VAR ((locs cd)!!i))], lab)
	findplace (AGlob i) lab = return ([PUSHGLOBAL (show (modulename cd)) i], lab)
	findplace (AIndex l i) lab 
            = do (fpcode,lab') <- findplace l lab
		 (code,lab'') <- fcompile' i cd lab'
		 return (fpcode ++ code ++ [TOINDEX], lab'')
	findplace (AField l f a t) lab
            = do (fpcode,lab') <- findplace l lab
		 return (fpcode ++ [PROJARG a t], lab')


--fcompile' (IndexAssign l i v) vars lab
--     = fcompile' v vars lab ++ fcompile' i vars lab ++ [POPINDEX (vars!!l)]
fcompile' (Seq x y) cd lab 
    = do (xcode, lab') <- fcompile' x cd lab 
	 (ycode, lab'') <- fcompile' y cd lab'
	 return (xcode ++ ycode, lab'')
fcompile' (Apply f as) cd lab 
    = do (pcode, lab') <- pushargs as cd lab
	 (ccode, lab'') <- call f cd lab'
	 return (pcode ++ ccode, lab')
fcompile' (Partial f as i) cd lab 
    = do (pcode, lab') <- pushargs as cd lab
	 (ccode, lab'') <- closure f (length as) cd lab'
	 return (pcode ++ ccode, lab')
fcompile' (Foreign ty f as) cd lab = foreigncomp ty f as cd lab
fcompile' (While c sc) cd@(CD mod vars tmp b tr ds) lab
     = do let cd' = (CD mod vars tmp (Just end) tr ds)
	  (ccode, nlab) <- fcompile' c cd lab'
	  (sccode, nlab') <- fcompile' sc cd' nlab
	  return ([LABEL start] ++ ccode ++
		  [JFALSE end] ++ sccode ++ [JUMP start, LABEL end], nlab')
  where start = MN ("l",lab)
	end = MN ("l",(lab+1))
	lab' = lab+2
fcompile' (DoWhile sc c) cd@(CD mod vars tmp b tr ds) lab
     = do let cd' = (CD mod vars tmp (Just end) tr ds)
          (ccode, nlab) <- fcompile' c cd lab'
	  (sccode, nlab') <- fcompile' sc cd' nlab
	  return ([LABEL start] ++ sccode ++ ccode ++
		  [JTRUE start,LABEL end], nlab')
  where start = MN ("l",lab)
	end = MN ("l",(lab+1))
	lab' = lab+2
fcompile' (For x y l ar e) (CD mod vars tmp b tr ds) lab
     = -- Start of loop, set l to first element of ar
       do let cd' = (CD mod vars' tmp (Just end) tr ds)
          (xcode, labs1) <- fcompile' (Assign (AName x) (GConst (Num 0))) cd' lab'
	  (ycode, labs2) <- fcompile' (Assign (AName y) ar) cd' labs1
-- Check if x equals the length of the array
	  (testcode, labs3) <- fcompile' 
	       (Infix Equal (Loc x) (Apply (Global sizefn) [(Loc y)])) cd' labs2 
	  (acode, labs4) <- fcompile' (Assign l (Index (Loc y) (Loc x))) cd' labs3
       -- Compile the body
	  (bodycode, labs5) <- fcompile' e cd' labs4 
       -- Point to the next element of ar, jump to start
	  (inccode, labs6) <- fcompile' 
	       (Assign (AName x) (Infix Plus (Loc x) (GConst (Num 1)))) 
	       cd' labs5
	  return (xcode ++ ycode ++ [LABEL start] ++ testcode ++ [JTRUE end] ++
		  acode ++ bodycode ++ inccode ++ 
		  [JUMP start,LABEL end], labs6)
  where	start = MN ("l", lab)
	end = MN ("l",lab+1)
	lcx = MN ("i", lab+2)
	lcy = MN ("i", lab+3)
	vars' = vars ++ [lcx,lcy]
	lab' = lab+4
	sizefn = NS (UN "Builtins") (UN "size")
fcompile' (TryCatch tr ca (Loc err) fin) cd@(CD mod vars tmp b trys ds) lab = 
        do let cd' = (CD mod vars tmp b (trys+1) ds)
	   (trcode, lab'') <- fcompile' tr cd' lab'
	   (cacode, lab''') <- fcompile' ca cd lab''
	   (fincode, lab'''') <- fcompile' fin cd lab'''
	   let tcode = ((TRY catchcode):trcode) ++ [TRIED, JUMP okay] ++
		    ((LABEL catchcode):(PUSH (VAR ((locs cd)!!err))):SETTOP:
		    RESTORE:TRIED:cacode) ++ fincode ++ [LABEL okay]
	   return (tcode, lab'''')
   where catchcode = MN ("l",lab)
	 okay = MN ("l",lab+1)
	 lab' = lab+2
fcompile' (Throw err) cd lab = 
        do (errcode, lab') <- fcompile' err cd lab
	   return (errcode ++ [THROW], lab')
fcompile' (Except err code) cd lab = 
        do (errcode, lab') <- fcompile' err cd lab
	   (ccode, lab'') <- fcompile' code cd lab'
	   return (errcode ++ ccode ++ [MKEXCEPT], lab'')
fcompile' (Break f l) cd lab = case (breaklabel cd) of
			    Nothing -> fail $ f++":"++show l++":"++
				       "Can't break out of function - use 'return'"
			    Just l -> return ([JUMP l],lab)
fcompile' (PrintNum x) cd lab
     = do (code, lab') <- fcompile' x cd lab
	  return (code ++ [PRINTINT], lab')
fcompile' (PrintStr x) cd lab
     = do (code, lab') <- fcompile' x cd lab
	  return (code ++ [PRINTSTR], lab')
fcompile' (PrintExc x) cd lab
     = do (code, lab') <- fcompile' x cd lab
	  return (code ++ [PRINTEXC], lab')
fcompile' (If c t e) cd lab
     = do (ccode, lab'') <- fcompile' c cd lab'
	  (tcode, lab''') <- fcompile' t cd lab''
	  (ecode, lab'''') <- fcompile' e cd lab'''
	  return (ccode ++ [JFALSE false] ++ tcode ++
		  [JUMP end,LABEL false] ++ ecode ++ [LABEL end], lab'''')
  where false = MN ("l",lab)
	end = MN ("l",lab+1)
	lab' = lab+2
fcompile' (Infix op x y) cd lab = icompile (Infix op x y) cd lab
fcompile' (RealInfix op x y) cd lab = rcompile (RealInfix op x y) cd lab
fcompile' (CmpExcept op x y) cd lab = 
   do (xcode, lab') <- fcompile' x cd lab
      (ycode, lab'') <- fcompile' y cd lab'
      return (xcode ++ ycode ++ cmpcode op, lab'')
   where cmpcode Equal = [EQEXCEPT]
	 cmpcode NEqual = [NEEXCEPT]
fcompile' (CmpStr op x y) cd lab = 
   do (xcode, lab') <- fcompile' x cd lab
      (ycode, lab'') <- fcompile' y cd lab'
      return (xcode ++ ycode ++ cmpcode op, lab'')
   where cmpcode Equal = [EQSTRING]
	 cmpcode NEqual = [NESTRING]
fcompile' (Append x y) cd lab = 
   do (xcode, lab') <- fcompile' x cd lab
      (ycode, lab'') <- fcompile' y cd lab'
      return (xcode ++ ycode ++ [APPEND], lab'')
fcompile' (Unary op x) cd lab = icompile (Unary op x) cd lab
fcompile' (RealUnary op x) cd lab = rcompile (RealUnary op x) cd lab
fcompile' (Coerce t1 t2 v) cd lab = cocompile t1 t2 v cd lab
fcompile' (Case v alts) cd lab = 
   do (vcode,lab') <- fcompile' v cd lab
      (acode,lab'') <- altcompile alts cd lab'
      return (vcode ++ acode, lab'')

fcompile' (Index v i) cd lab = 
   do (vcode, lab') <- fcompile' v cd lab
      (icode, lab'') <- fcompile' i cd lab'
      return (vcode ++ icode ++ [GETINDEX], lab'')
fcompile' (Field v n a t) cd lab =
   do (vcode, lab') <- fcompile' v cd lab
      return (vcode ++ [PROJARG a t], lab')
fcompile' (ArrayInit xs) cd lab =
   do (xscode, lab') <- pushargs xs cd lab
      return (xscode ++ [MKARRAY (length xs)], lab')

--fcompile' (GConst (Num i)) cd lab
--   where i = getFunID n (alldecls cd)
    
fcompile' (Error str) cd lab = return ([PUSH (STR str), ERROR], lab)
fcompile' Noop cd lab = return ([], lab)
fcompile' VMPtr cd lab = return ([VMPTR],lab)
fcompile' (Metavar f l i) cd lab = fail $ f++":"++show l++":Can't compile a metavariable"
fcompile' x _ lab
    = return (trace ("Warning, unknown parse tree entry " ++ show x) $ ([],lab))

acompile xs = map POPARG (map fst xs)

ccompile (Num x) = [PUSH (VAL x)]
ccompile (Re x) = [PUSH (RVAL x)]
ccompile (Ch x) = [PUSH (VAL (fromEnum x))]
ccompile (Bo True) = [PUSH (VAL 1)]
ccompile (Bo False) = [PUSH (VAL 0)]
ccompile (Str str) = [PUSH (STR str)]
ccompile (Exc str i) = [PUSH (STR str), PUSH (VAL i), MKEXCEPT]

pushargs :: Monad m => [Expr Name] -> CompData -> Int -> m ([TAC],Int)
pushargs [] _ lab = return ([], lab)
pushargs (x:xs) cd lab 
    = do (pcode, lab') <- pushargs xs cd lab
	 (xcode, lab'') <- fcompile' x cd lab
	 return (pcode ++ xcode, lab'')

call (Global n) cd lab = return ([CALLNAME n],lab)
call (Loc i) cd lab = return ([CALL ((locs cd)!!i)],lab)
call x cd lab = do (ccode,lab') <- fcompile' x cd lab
		   return ((ccode ++ [CALLTOP]),lab')

closure (Global n) i cd lab = return ([CLOSURE n i],lab)
{-
closure (Loc i) cd lab = ([CALL ((locs cd)!!i)],lab)
closure x cd lab = let (ccode,lab') = fcompile' x cd lab in
		    ((ccode ++ [CALLTOP]),lab')
-}

icompile :: Monad m => Expr Name -> CompData -> Int -> m ([TAC], Int)
icompile e cd@(CD mod vars tmp b tr ds)  lab = 
     do (ecode, lab') <- icomp' e lab tmp
	return (ecode ++ [PUSH (INT tmp)], lab') 
  where
 -- icomp compiles an expression and puts the result of its evaluation in <tmp>
 -- Shortcut special case
     icomp' (Infix OpAndBool x y) lab tmp =
          do (cx,lab') <- icomp' x nextlab (tmp+1)
             (cy,lab'') <- icomp' y lab' (tmp+2)
	     return (cx ++ (JTFALSE (tmp+1) shortcut1):cy ++ 
		     [INFIX tmp OpAndBool (tmp+1) (tmp+2), JUMP shortcut2,
		      LABEL shortcut1, TMPSET tmp 0,
	              LABEL shortcut2], lab'')
        where nextlab = lab+2
	      shortcut1 = MN ("l",lab)
	      shortcut2 = MN ("l",lab+1)
 -- Shortcut special case
     icomp' (Infix OpOrBool x y) lab tmp =
          do (cx,lab') <- icomp' x nextlab (tmp+1)
	     (cy,lab'') <- icomp' y lab' (tmp+2)
	     return (cx ++ (JTTRUE (tmp+1) shortcut1):cy ++ 
		     [INFIX tmp OpOrBool (tmp+1) (tmp+2), JUMP shortcut2,
		      LABEL shortcut1, TMPSET tmp 1,
	              LABEL shortcut2], lab'')
        where nextlab = lab+2
	      shortcut1 = MN ("l",lab)
	      shortcut2 = MN ("l",lab+1)
 -- Power operator
     icomp' (Infix Power x y) lab tmp =
          do (cx,lab') <- icomp' x lab (tmp+1)
	     (cy,lab'') <- icomp' y lab' (tmp+2)
	     return (cx ++ cy ++ [INTPOWER tmp (tmp+1) (tmp+2)], lab'')
 -- General case
     icomp' (Infix op x y) lab tmp = 
          do (cx,lab') <- icomp' x lab (tmp+1)
	     (cy,lab'') <- icomp' y lab' (tmp+2)
	     return (cx ++ cy ++ [INFIX tmp op (tmp+1) (tmp+2)], lab'')
     icomp' (Unary op x) lab tmp =
	  do (cx,lab') <- icomp' x lab (tmp+1)
	     return (cx ++ [UNARY tmp op (tmp+1)], lab')
     icomp' (GConst (Num x)) lab tmp = return ([TMPSET tmp x], lab)
     icomp' x lab tmp = 
	  do (xcode, lab') <- fcompile' x (CD mod vars (tmp+1) b tr ds) lab
	     return (xcode ++ [GETVAL tmp], lab')

rcompile :: Monad m => Expr Name -> CompData -> Int -> m ([TAC], Int)
rcompile e cd@(CD mod vars tmp b tr ds)  lab = 
     do (ecode, lab', pushit) <- rcomp' e lab (tmp+10000) -- TMP HACK!
	if pushit 
	    then return (ecode ++ [PUSH (REAL (tmp+10000))], lab')
	    else return (ecode, lab')
  where
 -- rcomp compiles an expression and puts the result of its evaluation in <tmp>
 -- Power operator
     rcomp' (RealInfix Power x y) lab tmp =
          do (cx,lab',_) <- rcomp' x lab (tmp+1)
	     (cy,lab'',_) <- rcomp' y lab' (tmp+2)
	     return (cx ++ cy ++ [REALPOWER tmp (tmp+1) (tmp+2)], lab'', True)
     rcomp' (RealInfix op x y) lab tmp
       | op `elem` boolops = -- Put result in boolean, so ignore tmp
          do (cx,lab',_) <- rcomp' x lab (tmp+1)
	     (cy,lab'',_) <- rcomp' y lab' (tmp+2)
	     return (cx ++ cy ++ [REALINFIXBOOL op (tmp+1) (tmp+2)], lab'', False)
       | otherwise = 
          do (cx,lab',_) <- rcomp' x lab (tmp+1)
	     (cy,lab'',_) <- rcomp' y lab' (tmp+2)
	     return (cx ++ cy ++ [REALINFIX tmp op (tmp+1) (tmp+2)], lab'', True)
     rcomp' (RealUnary op x) lab tmp =
	  do (cx,lab',_) <- rcomp' x lab (tmp+1)
	     return (cx ++ [REALUNARY tmp op (tmp+1)], lab', True)
     rcomp' (GConst (Re x)) lab tmp = return ([RTMPSET tmp x], lab, True)
     rcomp' x lab tmp = 
	 do (xcode, lab') <- fcompile' x (CD mod vars (tmp+1-10000) b tr ds) lab
	    return (xcode ++ [GETRVAL tmp], lab', True)

cocompile :: Monad m => Type -> Type -> Expr Name -> CompData -> Int -> 
	                m ([TAC], Int)
cocompile (Prim StringType) (Prim Number) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [STR2INT], lab')
cocompile (Prim Number) (Prim StringType) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [INT2STR], lab')
cocompile (Prim RealNum) (Prim StringType) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [REAL2STR], lab')
cocompile (Prim StringType) (Prim RealNum) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [STR2REAL], lab')
cocompile (Prim StringType) (Prim Character) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [STR2CHR], lab')
cocompile (Prim Character) (Prim StringType) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [CHR2STR], lab')
cocompile (Prim Boolean) (Prim StringType) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [BOOL2STR], lab')
cocompile (Prim Number) (Prim RealNum) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [INT2REAL], lab')
cocompile (Prim RealNum) (Prim Number) v cd lab =
    do (vcode, lab') <- fcompile' v cd lab
       return (vcode ++ [REAL2INT], lab')
cocompile (Prim Character) (Prim Number) v cd lab =
    fcompile' v cd lab -- No conversion needed
cocompile (Prim Number) (Prim Character) v cd lab =
    fcompile' v cd lab -- No conversion needed
cocompile _ _ v _ _ = fail "Internal error, can't coerce"

altcompile :: Monad m => [CaseAlt Name] -> CompData -> Int -> m ([TAC], Int)
altcompile alts cd lab = do (codes, lab') <- mkcase sortalts lab
			    return ([CASE codes], lab') 
  where mkcase [] lab = return ([], lab)
	mkcase (x:xs) lab = do (xcode, lab') <- ac x lab
			       (codes, lab'') <- mkcase xs lab'
			       return (xcode:codes,lab'')

        sortalts = insertdefault (sort alts)
	ac (Alt t _ as r) lab = 
	    do (rcode, lab') <- (fcompile' r cd lab)
	       return (bindargs 0 as ++ (DISCARD:rcode), lab')
	bindargs arg [] = []
	bindargs arg ((Loc i):xs) = (PROJ ((locs cd)!!i) arg):(bindargs (arg+1) xs)
	insertdefault xs@((Alt t tot args x):alts) = idef 0 tot xs
	idef next tot all@((alt@(Alt t _ _ _)):alts)
              | next == t = alt:(idef (next+1) tot alts)
	      | next-1 == t = error "Overlapping cases"
	      | otherwise = (errorcase next tot):(idef (next+1) tot all)
	idef next tot []
	      | next == tot = []
	      | otherwise = (errorcase next tot):(idef (next+1) tot [])
	errorcase t tot = Alt t tot [] (Throw (GConst (Exc "Missing case" (127))))

decls :: [TAC] -> [TAC]
decls c = add_decls (find_decls c ([],[],[],[],[])) c

add_decls (ts,ts',vs,vs',as) c 
    = (map TMPINT (removedups ts)) ++
      (map TMPREAL (removedups ts')) ++
      (map DECLARE (removedups (removeothers vs' vs))) ++ 
      (map DECLAREARG (removedups vs')) ++
      (map ARRAY (removedups as)) ++ c

removedups [] = []
removedups (x:xs) | x `elem` xs = removedups xs
		  | otherwise = x:(removedups xs)

removeothers vs [] = []
removeothers vs (x:xs) | x `elem` vs = removeothers vs xs
		       | otherwise = x:(removeothers vs xs)

find_decls :: [TAC] -> ([Int],[Int],[Var],[Var],[Var]) -> 
	               ([Int],[Int],[Var],[Var],[Var])
find_decls [] (ts,ts',vs,vs',as) = (ts,ts',vs,vs',as)
find_decls ((SET x i y):xs) (ts,ts',vs,vs',as) 
    = find_decls xs (ts,ts',x:y:vs,vs',as)
--find_decls ((USETMP v):xs) (ts,ts',vs,vs',as) 
--    = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((TMPSET x i):xs) (ts,ts',vs,vs',as) 
    = find_decls xs (x:ts,ts',vs,vs',as)
find_decls ((RTMPSET x i):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,x:ts',vs,vs',as)
find_decls ((CALL v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((INFIX t1 _ t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (t1:t2:t3:ts,ts',vs,vs',as)
find_decls ((INTPOWER t1 t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (t1:t2:t3:ts,ts',vs,vs',as)
find_decls ((UNARY t1 _ t2):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (t1:t2:ts,ts',vs,vs',as)
find_decls ((REALINFIX t1 _ t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t1:t2:t3:ts',vs,vs',as)
find_decls ((REALINFIXBOOL _ t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t2:t3:ts',vs,vs',as)
find_decls ((REALPOWER t1 t2 t3):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t1:t2:t3:ts',vs,vs',as)
find_decls ((REALUNARY t1 _ t2):xs) (ts,ts',vs,vs',as) 
   = find_decls xs (ts,t1:t2:ts',vs,vs',as)
find_decls ((PUSH (VAR v)):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((POP v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls ((POPARG v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',vs,v:vs',as)
find_decls ((USETMP v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',vs,v:vs',as)
find_decls ((POPINDEX v):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',v:as)
find_decls ((CASE alts):xs) (ts,ts',vs,vs',as) = find_decls xs 
					    (fd_alts alts (ts,ts',vs,vs',as))
   where fd_alts [] acc = acc
	 fd_alts (xs:xss) acc = find_decls xs (fd_alts xss acc)
find_decls ((PROJ v i):xs) (ts,ts',vs,vs',as) = find_decls xs (ts,ts',v:vs,vs',as)
find_decls (_:xs) acc = find_decls xs acc


-- compileDecls :: [ConDecl] -> [CompileResult]
-- compileDecls cs = cd 0 cs
--   where cd i [] = []
-- 	cd i ((Con n t):xs) = (ByteCode (n,[MKCON i (numargs t)])):
-- 			          (cd (i+1) xs)
-- 	numargs (Fn _ ts t) = length ts
-- 	numargs _ = 0

compileAll :: Monad m => Program -> Name -> m [CompileResult]
compileAll ds n = ca' ds n
  where
    ca' [] _ = return []
    ca' (CInclude str:xs) mod = do
        rest <- ca' xs mod
	return $ (RawCode ("extern \"C\" {\n#include <"++str++">\n}")):rest
    ca' ((FunBind (_,_,n,_,_,Defined e) _):xs) mod = do
        rest <- ca' xs mod
	code <- comp ds e mod
	return $ (ByteCode (n,code)):rest
    ca' ((FunBind (_,_,n,_,_,Unbound) _):xs) mod = do
        rest <- ca' xs mod
	return $ (ExternDef n):rest
    ca' ((FunBind (_,_,n,_,_,DataCon t i True) _):xs) mod = do
        rest <- ca' xs mod    
	return $ (ByteCode (n,[MKCON t i])):rest
    ca' ((FunBind (_,_,n,_,_,DataCon t i False) _):xs) mod = do
        rest <- ca' xs mod    
	return $ (ExternDef n):rest
--ca' ((DataDecl n tys cs):xs) = (compileDecls cs) ++ (ca' xs)
    ca' (x:xs) mod = ca' xs mod

peephole :: [TAC] -> [TAC]
peephole [] = []
-- Add argument declaration.
--peephole ((POPARG x):xs) = (DECLAREARG x):(POPARG x):(peephole xs)
--peephole ((PUSHVAL x):(POP v):xs) = (SETVAL v x):(peephole xs)
peephole ((PUSH (INT x)):(POP v):xs) = (SETINT v x):(peephole xs)
peephole ((PUSH (VAR x)):(POP v):xs) = (SETVAR v x):(peephole xs)
peephole ((PUSH x):SETTOP:xs) = (PUSHSETTOP x):(peephole xs)
peephole ((PUSH (INT x)):(JFALSE l):xs) = (JTFALSE x l):(peephole xs)
peephole ((PUSH (INT x)):(JTRUE l):xs) = (JTTRUE x l):(peephole xs)
peephole ((PUSH x):(GETVAL t):xs) = (PUSHGETVAL x t):(peephole xs)
peephole ((PUSH x):(GETRVAL t):xs) = (PUSHGETRVAL x t):(peephole xs)
{-
peephole ((PUSH x):(PUSH y):(PUSH z):(PUSH w):xs) 
    = (PUSH4 x y z w):(peephole xs)
peephole ((PUSH x):(PUSH y):(PUSH z):xs) = (PUSH3 x y z):(peephole xs)
peephole ((PUSH x):(PUSH y):xs) = (PUSH2 x y):(peephole xs)
-}

peephole ((SET v 0 x):xs) = (SETVAR v x):(peephole xs)
peephole (USETMP _:xs) = peephole xs
peephole (RETURN:[]) = []
peephole (CALL n:RETURN:xs) = (TAILCALL n):RETURN:xs
peephole (CALLNAME n:RETURN:xs) = (TAILCALLNAME n):RETURN:xs
peephole (CALLTOP:RETURN:xs) = TAILCALLTOP:RETURN:xs
peephole (x:xs) = x:(peephole xs)

foreigncomp :: Monad m => Type -> Name -> [(Expr Name,Type)] -> CompData -> 
	       Int -> m ([TAC], Int)
foreigncomp ty n as cd lab = 
    do (pcode, lab') <- pushargs (map fst as) cd lab
       return (pcode ++ foreigncall ty n (map snd as), lab')

foreigncall :: Type -> Name -> [Type] -> [TAC]
{- old way
foreigncall ty (UN n) args = [USETMP (tmpval n) | n <- [0..(length args)-1]]
			     ++ [RAWC ((popvals 0 args)++"\n\t"++
			        (conv ty)++"("++ n ++
			       "("++stackconv 0 args++")))")]
-}		     
foreigncall ty (UN n) args = [USETMP (tmpval n) | n <- [0..(length args)-1]]
			     ++ [FOREIGNCALL n "nolibsyet" ty args]

{-   where conv (Prim Void) = "VOID("
	 conv (Prim Number) = "PUSH(MKINT"
	 conv (Prim RealNum) = "PUSH(MKREAL"
	 conv (Prim Boolean) = "PUSH(MKINT"
	 conv (Prim Character) = "PUSH(MKCHAR"
	 conv (Prim StringType) = "PUSH(MKSTR"
	 conv (Prim File) = "PUSH(MKINT"
	 conv (Prim Pointer) = "PUSH(MKINT"
	 conv (TyVar _) = "PUSH(" -- Enough rope to hang yourself with!
	 conv (Array _) = "PUSH(MKARRAYVAL"
	 conv t = error $ "Can't deal with that type in foreign calls" ++ show t

         popvals n [] = ""
	 popvals n (x:xs) = show (tmpval n) ++ 
			    " = vm->doPop(); " ++ popvals (n+1) xs

         stackconv n [] = ""
	 stackconv n [x] = stackconv' n x
	 stackconv n (x:xs) = stackconv' n x ++ "," ++ stackconv (n+1) xs
	 stackconv' n (Prim Number) = show (tmpval n) ++ "->getInt()"
	 stackconv' n (Prim RealNum) = show (tmpval n) ++ "->getReal()"
	 stackconv' n (Prim Boolean) = show (tmpval n) ++ "->getInt()"
	 stackconv' n (Prim Character) = show (tmpval n) ++ "->getInt()"
	 stackconv' n (Prim StringType) 
	     = show (tmpval n) ++ "->getString()->getVal()"
	 stackconv' n (Prim File) 
	     = "(FILE*)("++show (tmpval n) ++ "->getRaw())"
	 stackconv' n (Prim Pointer) = show (tmpval n) ++ "->getRaw()"
	 stackconv' n (Array _) = show (tmpval n) ++ "->getArray()"
	 stackconv' n (TyVar _) = show (tmpval n)
	 
	 stackconv' _ t = error $ "Can't deal with that type (" ++ show t ++ ") in foreign calls"
-}

type FunMap = [(Name,Int)]

mkfnmap :: Program -> CompileResult
mkfnmap p = let fm = mkf' 0 p
		mkfm = mkfcode fm in
		RawCode $ "void _my_initfunmap(VMState* vm) {\n " ++ mkfm ++ "}"

mkf' :: Int -> Program -> FunMap
mkf' i [] = []
-- Add any functions which take one argument and return nothing.
--mkf' i ((FunBind (_,_,n,Fn _ [x] (Prim Void),_,_) _):xs) = (n,i):(mkf' (i+1) xs)
-- No, we now add all functions
mkf' i ((FunBind (_,_,n,Fn _ _ _,_,_) _):xs) = (n,i):(mkf' (i+1) xs)

mkf' i (x:xs) = mkf' (i+1) xs

mkfcode :: FunMap -> String
mkfcode [] = ""
mkfcode ((n,i):xs) = "\taddToFunMap("++show i++", "++show n++");\n" ++
		     mkfcode xs

mkdeskey :: Int -> IO CompileResult
mkdeskey 0 = do seed <- randomIO :: IO Int
		mkdeskey (seed+1)
mkdeskey seed = do (key, seed') <- mkkey 24 seed
		   return 
		      (RawCode $ "char* kaya_secret = \"" ++ key ++ "\";")
--	      return $ RawCode $ "char* kaya_secret = \"\\242\\052\\233\\025\\198\\096\\164\\249\\003\\060\\103\\245\\221\\110\\124\\061\\148\\090\\015\\149\\200\\240\\052\\020\";"

mkivec :: Int -> IO CompileResult
mkivec seed = do (key, seed') <- mkkey 8 seed
		 return 
		   (RawCode $ "char* kaya_ivec = \"" ++ key ++ "\";")
--	    return $ RawCode $ "char* kaya_ivec = \"\\208\\079\\091\\021\\123\\197\\092\\156\";"

dornd seed = seed*1103515245+12345

mkkey 0 seed = return ("", seed)
mkkey n seed = do let seed' = dornd seed
		  (rest,seed'') <- mkkey (n-1) seed'
		  return ("\\" ++ show3d (seed' `mod` 256) ++ rest, seed'')
  where show3d n | n < 10 = "00"++show n
		 | n < 100 = "0"++show n
		 | otherwise = show n

showtac :: [CompileResult] -> String
showtac [] = ""
showtac ((ByteCode (n,b)):xs) = showuser n ++ ":\n" ++ showbc 1 b ++ 
				"\n\n" ++ showtac xs
  where showbc i [] = ""
	showbc i ((CASE bcs):xs) = (take i (repeat '\t')) ++ 
				  "CASE\n" ++ showcase (i+1) 0 bcs
	showbc i (x:xs) = (take i (repeat '\t'))++show x ++ "\n" ++ showbc i xs
	showcase i c [] = ""
	showcase i c (bc:bcs) = (take (i-1) (repeat '\t')) ++ show c ++ 
				":\n" ++ showbc i bc ++ showcase i (c+1) bcs
showtac (x:xs) = showtac xs
