#include <assert.h>
#if HAVE_STDINT_H
#include <stdint.h>
#else
typedef unsigned short uint16_t;
#endif

#include "autosocket.h"

#include "libcompat.h"

/* This is a simplified version of getaddrinfo.  It does not support address
   families other than PF_INET. */

/*#if !HAVE_INET_ATON
int inet_aton(const char *cp, struct in_addr *inp)
{
    unsigned long rval = inet_addr(cp);
 
    if(rval == -1)
	return -1;
    inp->S_un.S_addr = rval;
    return 0;
}
#endif*/

int getaddrinfo(const char *node, const char *service,
		const struct addrinfo *hints, struct addrinfo **res)
{
    struct addrinfo default_hints = {
	.ai_flags = 0,
	.ai_family = PF_UNSPEC,
	.ai_socktype = 0,
	.ai_protocol = 0,
    };
    const struct in_addr *const *addrs, *const *addr;
    struct {
	uint16_t port;
	int socktype;
	int protocol;
    } services[3];
    int i;

    if(!(node || service))
	return EAI_NONAME;

    if(!hints)
	hints = &default_hints;

    switch(hints->ai_family) {
    case PF_UNSPEC:
    case PF_INET:
	break;
    default:
	return EAI_FAMILY;
    }

    switch(hints->ai_socktype) {
    case 0:
    case SOCK_STREAM:
    case SOCK_DGRAM:
	break;
    default:
	return EAI_SOCKTYPE;
    }
    switch(hints->ai_protocol) {
    case 0:
    case IPPROTO_TCP:
    case IPPROTO_UDP:
	break;
    default:
	return EAI_SOCKTYPE;
    }
    if(hints->ai_flags)
	return EAI_BADFLAGS;

    if(service) {
	struct servent *se;

	i = 0;

	services[i].socktype = SOCK_STREAM;
	services[i].protocol = IPPROTO_TCP;
	switch(hints->ai_socktype) {
	case 0:
	case SOCK_STREAM:
	    if(hints->ai_protocol && hints->ai_protocol != IPPROTO_TCP)
		break;
	    se = getservbyname(service, "tcp");
	    if(se) {
		services[i].port = se->s_port;
		i++;
	    } else {
		services[i].port = htons(atoi(service));
		if(services[i].port)
		    i++;
	    }
	    break;
	}

	services[i].socktype = SOCK_DGRAM;
	services[i].protocol = IPPROTO_UDP;
	switch(hints->ai_socktype) {
	case 0:
	case SOCK_DGRAM:
	    if(hints->ai_protocol && hints->ai_protocol != IPPROTO_UDP)
		break;
	    se = getservbyname(service, "udp");
	    if(se) {
		services[i].port = se->s_port;
		i++;
	    } else {
		services[i].port = htons(atoi(service));
		if(services[i].port)
		    i++;
	    }
	    break;
	}

	if(i == 0)
	    return EAI_SERVICE;
	services[i].port = 0;
    }

    if(node) {
	struct hostent *he;

	he = gethostbyname(node);
	if(!he) switch(h_errno) {
	case HOST_NOT_FOUND:
	    return EAI_NONAME;
	case NO_ADDRESS:
#if NO_DATA != NO_ADDRESS
	case NO_DATA:
#endif
	    return EAI_NODATA;
	case NO_RECOVERY:
	    return EAI_FAIL;
	case TRY_AGAIN:
	    return EAI_AGAIN;
	default:
	    abort();
	}

	if(he->h_addrtype != AF_INET)
	    return EAI_ADDRFAMILY;

	addrs = (const struct in_addr **)he->h_addr_list;
    } else {
	static const struct in_addr addr_any = {INADDR_ANY};
	static const struct in_addr *const no_addrs[] = {&addr_any, NULL};

	assert(service);
	addrs = no_addrs;
    }

    *res = NULL;
    for(addr = addrs; *addr; addr++) {
	for(i = 0; services[i].port; i++) {
	    struct sockaddr_in *sa = malloc(sizeof *sa);

	    if(!sa)
		return EAI_MEMORY;	/* FIXME */
	    *sa = (struct sockaddr_in) {
		.sin_family = AF_INET,
		.sin_port = services[i].port,
		.sin_addr = **addr,
	    };
	    *res = malloc(sizeof **res);
	    if(!*res)
		return EAI_MEMORY;	/* FIXME */
	    **res = (struct addrinfo) {
		.ai_family = AF_INET,
		.ai_socktype = services[i].socktype,
		.ai_protocol = services[i].protocol,
		.ai_addrlen = sizeof *sa,
		.ai_addr = (struct sockaddr *)sa,
		.ai_next = NULL,
	    };
	    res = &((*res)->ai_next);
	}
    }
    assert(!*res);

    return 0;
}
