/***************************************************************************
                         vertexmodifier.cpp  -  description
                            -------------------
   begin                : Sat Apr 21 2001
   copyright            : (C) 2001 by Jon Anderson
   email                : janderson@onelink.com
***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <cassert>
#include "vertexmodifier.h"
#include "Entities/mesh.h"
#include "Entities/vertex.h"

#include <i3d.h>
#include <config.h>
#include <i3dworkspace.h>
#include "modetoolbar.h"

#include <qcolordialog.h>
#include <qcolor.h>

/*
 * Macros used for modifiying vertexes in an undoable manner.
 * they should be used like:
 *
 * BEGIN( "Connecting vertices...", 2 );
 * getSubObjectParent() -> connectVerts( getSubObjectList();
 * END();
 *
 */

#define BEGIN(c,x) \
  if( assertSelected(x) && assertSameParent() ) { \
    setStatus( c ); \
    Mesh *m = getSubObjectParent(); \
    vector<int> zzlist = getSubObjectList(); \
    CheckPointCmd *chc = new CheckPointCmd( m );

#define END() \
    chc -> save(); \
    setStatusDone(); \
  } \
  updateViews();

#define SUBOBJECTS() zzlist

static
float snap( float pos, int interval )
{

   int x;
   int sign;

   //find and save the sign, and make the pos positive.
   if ( pos > 0 )
      sign = 1;
   else
      sign = -1;

   //convert to positive number.
   pos *= sign;

   int a = (int)(( pos * 10)+0.5);

   x = a / interval;

   if ( x == 0 ) {
   	x = 1;
   }
   	
   float result = x * interval;

   result = result / 10;

   result *= sign;

   return result;
}


int VertexModifier::TYPE = IControl::getUID();

VertexModifier::VertexModifier() : SubObjectModifier( Vertex::TYPE )
{

   QPopupMenu * axis = new QPopupMenu();

   axis -> insertItem( "X", this, SLOT( slotGridSnapX() ) );
   axis -> insertItem( "Y", this, SLOT( slotGridSnapY() ) );
   axis -> insertItem( "Z", this, SLOT( slotGridSnapZ() ) );
   axis -> insertItem( "All", this, SLOT( slotGridSnapAll() ) );



   setName( "&Vertices" );
   m_popup -> insertSeparator();
   m_popup -> insertItem( "Snap", this, SLOT( slotSnap() ) );
   m_popup -> insertItem( "Weld", this, SLOT( slotWeld() ) );
   m_popup -> insertItem( "Grid Snap", axis );

   // Added new menu items
   m_popup -> insertSeparator();
   m_popup -> insertItem( "Hide Selected", this, SLOT( slotHideSelected() ) );
   m_popup -> insertItem( "Show All", this, SLOT( slotUnhideAll() ) );

   m_popup -> insertSeparator();
   m_popup -> insertItem( "Set Animatable", this, SLOT(slotAnimatable() ) );
   m_popup -> insertItem( "Set Static", this, SLOT(slotStatic() ) );

   m_popup -> insertSeparator();
   m_popup -> insertItem( "Connect", this, SLOT( slotConnect() ) );
   m_popup -> insertItem( "Loop", this, SLOT( slotLoop() ) );
   m_popup -> insertSeparator();
   m_popup -> insertItem( "Bevel", this, SLOT( slotBevel() ) );
   m_popup -> insertItem( "Extrude", this, SLOT( slotExtrude() ) );
   m_popup -> insertItem( "New Face", this, SLOT( slotCreateFace() ) );
   m_popup -> insertItem( "Delete", this, SLOT( slotDelete() ) );
   m_popup -> insertSeparator();
   m_popup -> insertItem( "Set Color", this, SLOT( slotSetColor() ) );
   m_popup -> insertItem( "Reset Color", this, SLOT( slotResetColor() ) );


}

VertexModifier::~VertexModifier()
{
}

/**
  * Checks the current selection.  If vertices or controlpoints
  * are present who do not belong to a mesh, it disables the
  * options only available for meshes.
  */
void VertexModifier::selectionEvent()
{

}

void VertexModifier::activate()
{

   clearSelection();
   I3D::getWorkspace() -> setCurrentControl( this );
   I3D::setModeMsg( " Vertex Mode " );
   SelectMode::set( Vertex::TYPE );
   //change to select mode immediately.
   ModeToolbar::setMode( ModeToolbar::MODE_SELECT );

   updateViews();

}

void VertexModifier::deactivate()
{
}

void VertexModifier::slotSnap()
{
   if ( assertSelected( 2 ) )
   {

      setStatus( "Snapping vertices..." );

      TransactionCommand *tc = new TransactionCommand();

      Vector4 center;
      Vector4 movePos;
      Vector4 temp;
      int n = 0;

      SelectionIterator it = m_selection -> begin();

      while ( it != m_selection -> end() )
      {

         static_cast<Vertex *>( *it ) -> getTransformedPosition( &temp );
         center += temp;
         n++;
         ++it;

      }

      center /= n;

      it = m_selection -> begin();

      while ( it != m_selection -> end() )
      {

         Vertex * v = static_cast<Vertex *>( *it );
         v -> getTransformedPosition( &temp );
         movePos = center - temp;

         TransformCmd *c = new TransformCmd( v );
         c -> begin();
         v -> move( movePos.x, movePos.y, movePos.z );
         c -> end();
         tc -> addCommand( c );
         ++it;

      }

      tc->save();

      setStatusDone();
   }

   updateViews();

}

void VertexModifier::slotWeld()
{
   BEGIN( "Welding selected vertices...", 2 );
   getSubObjectParent() -> weldVerts( SUBOBJECTS() );
   END();
   clearSelection();
}

void VertexModifier::slotGridSnapX()
{
   Vector3 a( 1, 0, 0 );
   gridSnap( a );
}

void VertexModifier::slotGridSnapY()
{
   Vector3 a( 0, 1, 0 );
   gridSnap( a );
}

void VertexModifier::slotGridSnapZ()
{
   Vector3 a( 0, 0, 1 );
   gridSnap( a );
}
void VertexModifier::slotGridSnapAll()
{
   Vector3 a( 1, 1, 1 );
   gridSnap( a );
}

void VertexModifier::gridSnap( Vector3 &axis )
{
   if( assertSelected( 1 ) )
   {
      Config *cfg = I3D::getConfig();
      int interval = cfg -> getInt( "Display", "GridSpacing" );
      Vector4 p;
      Matrix44 m;
      Matrix44 m_inv;

      setStatus( "Snapping to grid..." );
      //use the start and end transform functions of SubObjectModifier to
      //take care of UNDO for us.
      startTransform();

      SelectionIterator it = m_selection -> begin();

      while( it != m_selection -> end() )
      {
         Vertex * v = static_cast<Vertex *>( *it );
         v -> getCompleteMatrix( &m );
         v -> getPosition( &p );

         m_inv = m;
         m_inv.Invert();

         p = m * p;
				 p.Dump();
				
         if(axis.x > 0 )
            p.x = snap( p.x, interval );
         if(axis.y > 0 )
            p.y = snap( p.y, interval );
         if(axis.z > 0 )
            p.z = snap( p.z, interval );

         p = m_inv * p;

         v -> setPosition( p );

         ++it;
      }

      endTransform();
      setStatusDone();
   }
   updateViews();
}

void VertexModifier::slotExtrude()
{
   Vector4 p( 0, 0, 0, 1 );
   BEGIN( "Extruding selected vertices...", 1 );
   getSubObjectParent() -> extrudeVerts( SUBOBJECTS(), p );
   END();

   slotMoveMode();

}

void VertexModifier::slotConnect()
{

   BEGIN( "Connecting selected vertices...", 2 );
   getSubObjectParent() -> connectVerts( SUBOBJECTS() );
   END();

}

void VertexModifier::slotLoop()
{

   BEGIN( "Looping selected vertices...", 2 );
   getSubObjectParent() -> loopVerts( SUBOBJECTS() );
   END();

}


void VertexModifier::slotBevel()
{

   BEGIN( "Beveling selected vertices...", 1 );
   getSubObjectParent() -> bevelVerts( SUBOBJECTS(), 1 );
   END();

   clearSelection();
}

void VertexModifier::slotCreateFace()
{
   BEGIN( "Creating face from selected vertices...", 3 );
   getSubObjectParent() -> createNewFace( SUBOBJECTS() );
   END();
}

void VertexModifier::slotDelete()
{
   BEGIN( "Deleting selected vertices...", 1 );
   getSubObjectParent() -> deleteVerts( SUBOBJECTS() );
   END();

   clearSelection();
}

void VertexModifier::slotSetColor()
{
   bool ok = false;
   QRgb c = QColorDialog::getRgba( qRgba( 255, 255, 255, 255 ), &ok , 0, 0 );

   if ( ! ok )
      return ;

   BEGIN( "Setting Color of vertices...", 1 );

   getSubObjectParent() -> setVertexColors( SUBOBJECTS(),
         ( float ) qRed( c ) / 255,
         ( float ) qGreen( c ) / 255,
         ( float ) qBlue( c ) / 255,
         ( float ) qAlpha( c ) / 255 );

   END();
}

void VertexModifier::slotResetColor()
{
   BEGIN( "Resetting Color of vertices...", 1 );
   getSubObjectParent() -> resetVertexColors( SUBOBJECTS() );
   END();
}

void VertexModifier::slotStatic()
{
   BEGIN( "Adding animation to vertices...", 1 );
   getSubObjectParent() -> setAnimatableVertices( SUBOBJECTS(), false );
   END();
}
void VertexModifier::slotAnimatable()
{
   BEGIN( "Adding animation to vertices...", 1 );
   getSubObjectParent() -> setAnimatableVertices( SUBOBJECTS(), true );
   END();
}
/** Hides currently selected cp's. */
void VertexModifier::slotHideSelected(){
    if ( assertSelected(1) ) {
        setStatus( "Hiding selected..." );
        SelectionIterator it = m_selection -> begin();

        while ( it != m_selection -> end() )
        {
            static_cast<SubObject *>( *it ) -> setVisible( false );
            ++it;
        }

        clearSelection();                 //
        setStatusDone();
   }

   updateViews();
}
/** No descriptions */
void VertexModifier::slotUnhideAll(){
   setStatus( "Showing All..." );
   vector<Selectable *> *list;

   list = ObjectDB::getInstance() ->getList();
   assert( list );

   for ( int i = 0; i < ( int ) list->size(); i++ )
   {
        Object * tmp = static_cast< Object * >( ( *list ) [ i ] );
        vector< SubObject *> * vsub = (tmp -> getVerts());
        vector<SubObject *>::iterator it = vsub->begin();
        while ( it != vsub->end() ){
            static_cast<SubObject *>( *it )->setVisible(true);
            it++;
        }
   }

   updateViews();
   setStatusDone();
}
