#include <stdio.h>

#include <lua.h>
#include <lauxlib.h>
#include <lualib.h>

#include "imapfilter.h"


static int ifcore_login(lua_State * lua);
static int ifcore_logout(lua_State * lua);
static int ifcore_status(lua_State * lua);
static int ifcore_select(lua_State * lua);
static int ifcore_close(lua_State * lua);
static int ifcore_search(lua_State * lua);
static int ifcore_fetch(lua_State * lua);
static int ifcore_store(lua_State * lua);
static int ifcore_copy(lua_State * lua);
static int ifcore_append(lua_State * lua);


/* Lua imapfilter core library functions. */
static const luaL_reg ifcorelib[] = {
	{"login", ifcore_login},
	{"logout", ifcore_logout},
	{"status", ifcore_status},
	{"select", ifcore_select},
	{"close", ifcore_close},
	{"search", ifcore_search},
	{"fetch", ifcore_fetch},
	{"store", ifcore_store},
	{"copy", ifcore_copy},
	{"append", ifcore_append},
	{NULL, NULL}
};


/*
 * Core function to login to the server.
 */
static int
ifcore_login(lua_State * lua)
{
	const char *s, *u, *p;
	int r;

	if (lua_gettop(lua) != 1)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);

	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	if (!(p = get_table_string("password")))
		luaL_error(lua, "no password specified");

	r = request_login(s, u, p, (unsigned int)(get_table_number("port")),
	    get_table_string("ssl"));

	lua_pop(lua, 1);

	lua_pushboolean(lua, (r == RESPONSE_OK || r == RESPONSE_PREAUTH));

	return 1;
}


/*
 * Core function to logout from the server.
 */
static int
ifcore_logout(lua_State * lua)
{
	const char *s, *u;
	int r;

	if (lua_gettop(lua) != 1)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);

	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");

	r = request_logout(s, u);

	lua_pop(lua, 1);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	return 1;
}


/*
 * Core function to get the status of a mailbox.
 */
static int
ifcore_status(lua_State * lua)
{
	const char *s, *u;
	int r;
	int exists, recent, unseen;

	if (lua_gettop(lua) != 2)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);
	luaL_checktype(lua, 2, LUA_TSTRING);

	lua_pushvalue(lua, 1);
	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	lua_pop(lua, 1);

	r = request_status(s, u, lua_tostring(lua, 2), &exists, &recent, &unseen);

	lua_pop(lua, 2);

	lua_pushboolean(lua, (r == RESPONSE_OK));
	lua_pushnumber(lua, (lua_Number) (exists));
	lua_pushnumber(lua, (lua_Number) (recent));
	lua_pushnumber(lua, (lua_Number) (unseen));

	return 4;
}


/*
 * Core function to select a mailbox.
 */
static int
ifcore_select(lua_State * lua)
{
	const char *s, *u;
	int r;

	if (lua_gettop(lua) != 2)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);
	luaL_checktype(lua, 2, LUA_TSTRING);

	lua_pushvalue(lua, 1);
	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	lua_pop(lua, 1);

	r = request_select(s, u, lua_tostring(lua, 2));

	lua_pop(lua, 2);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	return 1;
}


/*
 * Core function to close a mailbox.
 */
static int
ifcore_close(lua_State * lua)
{
	const char *s, *u;
	int r;

	if (lua_gettop(lua) != 1)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);

	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");

	r = request_close(s, u);

	lua_pop(lua, 1);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	return 1;
}


/*
 * Core function to search the messages of a mailbox.
 */
static int
ifcore_search(lua_State * lua)
{
	char *m;
	const char *s, *u;
	int r;

	m = NULL;

	if (lua_gettop(lua) != 2)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);
	luaL_checktype(lua, 2, LUA_TSTRING);

	lua_pushvalue(lua, 1);
	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	lua_pop(lua, 1);

	r = request_search(s, u, lua_tostring(lua, 2), &m);

	lua_pop(lua, 2);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	if (!m)
		return 1;

	lua_pushstring(lua, m);

	xfree(m);

	return 2;
}


/*
 * Core function to fetch message items.
 */
static int
ifcore_fetch(lua_State * lua)
{
	const char *s, *u;
	int r;
	char *f;

	if (lua_gettop(lua) != 3)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);
	luaL_checktype(lua, 2, LUA_TSTRING);
	luaL_checktype(lua, 3, LUA_TSTRING);

	lua_pushvalue(lua, 1);
	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	lua_pop(lua, 1);

	r = request_fetch(s, u, lua_tostring(lua, 2), lua_tostring(lua, 3), &f);

	lua_pop(lua, 3);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	if (!f)
		return 1;

	lua_pushstring(lua, f);

	return 2;
}


/*
 * Core function to change message flags.
 */
static int
ifcore_store(lua_State * lua)
{
	const char *s, *u;
	int r;

	if (lua_gettop(lua) != 4)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);
	luaL_checktype(lua, 2, LUA_TSTRING);
	luaL_checktype(lua, 3, LUA_TSTRING);
	luaL_checktype(lua, 4, LUA_TSTRING);

	lua_pushvalue(lua, 1);
	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	lua_pop(lua, 1);

	r = request_store(s, u, lua_tostring(lua, 2), lua_tostring(lua, 3),
	    lua_tostring(lua, 4));

	lua_pop(lua, 4);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	return 1;
}


/*
 * Core function to copy messages between mailboxes.
 */
static int
ifcore_copy(lua_State * lua)
{
	const char *s, *u;
	int r;

	if (lua_gettop(lua) != 3)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);
	luaL_checktype(lua, 2, LUA_TSTRING);
	luaL_checktype(lua, 3, LUA_TSTRING);

	lua_pushvalue(lua, 1);
	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	lua_pop(lua, 1);

	r = request_copy(s, u, lua_tostring(lua, 2), lua_tostring(lua, 3));

	lua_pop(lua, 3);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	return 1;
}


/*
 * Core function to append messages to a mailbox.
 */
static int
ifcore_append(lua_State * lua)
{
	const char *s, *u;
	int r;

	if (lua_gettop(lua) != 5)
		luaL_error(lua, "wrong number of arguments");

	luaL_checktype(lua, 1, LUA_TTABLE);
	luaL_checktype(lua, 2, LUA_TSTRING);
	luaL_checktype(lua, 3, LUA_TSTRING);
	luaL_checktype(lua, 4, LUA_TSTRING);
	luaL_checktype(lua, 5, LUA_TSTRING);

	lua_pushvalue(lua, 1);
	if (!(s = get_table_string("server")))
		luaL_error(lua, "no mail server specified");
	if (!(u = get_table_string("username")))
		luaL_error(lua, "no username specified");
	lua_pop(lua, 1);

	r = request_append(s, u, lua_tostring(lua, 2), lua_tostring(lua, 3),
	    lua_tostring(lua, 4), lua_tostring(lua, 5));

	lua_pop(lua, 5);

	lua_pushboolean(lua, (r == RESPONSE_OK));

	return 1;
}


/*
 * Open imapfilter core library.
 */
LUALIB_API int
luaopen_ifcore(lua_State * lua)
{

	luaL_openlib(lua, "ifcore", ifcorelib, 0);
	return 1;
}
