#
# Copyright 2009 Canonical Ltd.
#
# Written by:
#     Gustavo Niemeyer <gustavo.niemeyer@canonical.com>
#     Sidnei da Silva <sidnei.da.silva@canonical.com>
#
# This file is part of the Image Store Proxy.
#
# This program is free software: you can redistribute it and/or modify it 
# under the terms of the GNU General Public License version 3, as published 
# by the Free Software Foundation.
#
# This program is distributed in the hope that it will be useful, but 
# WITHOUT ANY WARRANTY; without even the implied warranties of 
# MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR 
# PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along 
# with this program.  If not, see <http://www.gnu.org/licenses/>.
#
import os
import re
import time
import glob
import zipfile
import subprocess
import tempfile
import shutil
import base64
import commands
import logging
import random

from imagestore.lib.service import (
    ServiceError, ServiceTask, ThreadedService, taskHandlerInThread)


class EucaServiceError(ServiceError):
    pass

class EucaToolsError(EucaServiceError):
    pass


class GetEucaInfo(ServiceTask):

    def __init__(self, username):
        self.username = username


class EucaInfo(object):
    userId = None
    accessKey = None
    secretKey = None
    privateKeyPath = None
    certificatePath = None
    cloudCertificatePath = None
    urlForS3 = None
    urlForEC2 = None


class FakeEucaInfo(EucaInfo):
    userId = "01234567890"
    accessKey = "AcCeSsKeY123"
    privateKeyPath = "/fake/path"
    certificatePath = "/fake/path"
    cloudCertificatePath = "/fake/path"
    urlForS3 = "http://fake/url"
    urlForEC2 = "http://fake/url"


class BundleAndUploadImageTask(ServiceTask):

    def __init__(self, imageRegistration):
        self.imageRegistration = imageRegistration


class EucaService(ThreadedService):

    def __init__(self, reactor, basePath, eucalyptusPrefix="",
                 fakeMode=False, fakeSecretKey=None, createTime=time.time):
        ThreadedService.__init__(self, reactor)
        self._basePath = basePath
        self._eucaInfos = {}
        self._eucalyptusPrefix = eucalyptusPrefix
        self._fakeMode = fakeMode
        self._fakeSecretKey = fakeSecretKey
        self._createTime = createTime

    def clearEucaInfoCache(self):
        self._eucaInfos.clear()

    @taskHandlerInThread(BundleAndUploadImageTask)
    def _bundleAndUploadImage(self, task):
        # XXX This method needs tests.

        imageRegistration = task.imageRegistration

        if self._fakeMode:
            logging.warning("Faking the bundling and registration of an "
                            "image in Eucalyptus.")

            suffix = random.randint(100000, 999999)
            # These attribute names could be better.
            imageRegistration.eki.eid = "eki-%d" % suffix
            imageRegistration.eri.eid = "eri-%d" % suffix
            imageRegistration.emi.eid = "emi-%d" % suffix
            return imageRegistration

        eucaTools = EucaTools(self._getEucaInfo("admin"))

        # XXX This must be run in an external thread to avoid locking
        #     the GetEucaInfo() handler, but it shouldn't bundle two
        #     images together.

        # XXX We really need a new bucket naming scheme here.  There are
        #     a few different issues at play:
        #
        #     1) Buckets by default support only 5GB, so we can't use a
        #        single bucket.
        #     2) The admin interface is awful with long image names, so
        #        we give a little help by renaming the manifest.
        #     3) In the future, it shouldn't blow up if the bucket already
        #        exists.
        bucketName = "image-store-%d" % int(self._createTime())

        bundlePath = tempfile.mkdtemp()
        try:
            manifest = eucaTools.bundleKernel(imageRegistration.eki.path,
                                              bundlePath, "kernel")
            eucaTools.uploadBundle(manifest, bucketName)
            eki = eucaTools.registerBundle(manifest, bucketName)
        finally:
            shutil.rmtree(bundlePath)

        bundlePath = tempfile.mkdtemp()
        try:
            manifest = eucaTools.bundleRamdisk(imageRegistration.eri.path,
                                               bundlePath, "ramdisk")
            eucaTools.uploadBundle(manifest, bucketName)
            eri = eucaTools.registerBundle(manifest, bucketName)
        finally:
            shutil.rmtree(bundlePath)

        bundlePath = tempfile.mkdtemp()
        try:
            manifest = eucaTools.bundleImage(imageRegistration.emi.path,
                                             bundlePath, eki, eri, "image")
            eucaTools.uploadBundle(manifest, bucketName)
            emi = eucaTools.registerBundle(manifest, bucketName)
        finally:
            shutil.rmtree(bundlePath)

        # These attribute names could be better.
        imageRegistration.eki.eid = eki
        imageRegistration.eri.eid = eri
        imageRegistration.emi.eid = emi

        return imageRegistration

    @taskHandlerInThread(GetEucaInfo)
    def _getEucaInfoHandler(self, task):
        return self._getEucaInfo(task.username)

    def _getEucaInfo(self, username):
        if self._fakeMode:
            eucaInfo = FakeEucaInfo()
            eucaInfo.secretKey = self._fakeSecretKey
            logging.warning("Faking the retrieval of eucalyptus credentials.")
            return eucaInfo

        eucaInfo = self._eucaInfos.get(username)
        if eucaInfo is not None:
            return eucaInfo

        credentialsPath = os.path.join(self._basePath, "%s-credentials.zip" % 
                                                       username)
        writeCredentialsZip(username, credentialsPath,
                            self._eucalyptusPrefix)

        try:
            zip = zipfile.ZipFile(credentialsPath)
            zip.testzip()
        except (IOError, zipfile.BadZipfile):
            raise EucaServiceError("Credentials zip file is corrupted")

        eucaInfo = EucaInfo()

        # Hide eucarc inside the closure to ensure that any missing
        # entries are properly propagated as an error.
        eucarcContent = self._readFromZip(zip, "eucarc")
        def get(setting, _eucarc=self._parseEucaRC(eucarcContent)):
            if setting not in _eucarc:
                raise EucaServiceError("%s not found in eucarc" % setting)
            return _eucarc[setting]

        # User ID and query keys.
        eucaInfo.userId = get("EC2_USER_ID")
        eucaInfo.accessKey = get("EC2_ACCESS_KEY")
        eucaInfo.secretKey = get("EC2_SECRET_KEY")

        # S3 and EC2 URLs.
        eucaInfo.urlForS3 = get("S3_URL")
        eucaInfo.urlForEC2 = get("EC2_URL")

        # User private key.
        eucaInfo.privateKeyPath = "%s/%s-user-private-key.pem" % \
                                  (self._basePath, username)
        content = self._readFromZip(zip, get("EC2_PRIVATE_KEY"))
        self._writeSecureFile(eucaInfo.privateKeyPath, content)

        # User certificate.
        eucaInfo.certificatePath = "%s/%s-user-certificate.pem" % \
                                   (self._basePath, username)
        content = self._readFromZip(zip, get("EC2_CERT"))
        self._writeSecureFile(eucaInfo.certificatePath, content)

        # Cloud certificate.
        eucaInfo.cloudCertificatePath = "%s/cloud-certificate.pem" % \
                                        (self._basePath,)
        content = self._readFromZip(zip, get("EUCALYPTUS_CERT"))
        self._writeSecureFile(eucaInfo.cloudCertificatePath, content)

        self._eucaInfos[username] = eucaInfo
        return eucaInfo

    def _writeSecureFile(self, path, content):
        fd = os.open(path, os.O_WRONLY | os.O_TRUNC | os.O_CREAT, 0600)
        try:
            os.write(fd, content)
        finally:
            os.close(fd)

    def _readFromZip(self, zip, fileName):
        fileName = os.path.split(fileName)[1]
        for info in zip.infolist():
            if os.path.split(info.filename)[1] == fileName:
                return zip.read(info)
        raise EucaServiceError("%r not found in credentials zip file" %
                               fileName)

    def _parseEucaRC(self, eucarcContent):
        eucarc = {}
        for line in eucarcContent.splitlines():
            match = EXPORT_LINE_RE.match(line)
            if match:
                eucarc[match.group(1)] = match.group(2)
            elif "EC2_USER_ID" not in eucarc:
                # During development of 1.6, this setting went away.
                # Here we try to capture it from the ec2-bundle-image
                # command that still contained it.
                match = USER_ID_RE.match(line)
                if match:
                    eucarc["EC2_USER_ID"] = match.group(1)
        return eucarc


EXPORT_LINE_RE = re.compile("^export\s+(\w+)='?(.*?)'?\s*$")
USER_ID_RE = re.compile(".*--user (\d+)")


class EucaTools(object):

    def __init__(self, eucaInfo):
        self._eucaInfo = eucaInfo
        self._paths = {}

    def setToolPath(self, toolName, path):
        """Allow replacing of the tool path for debugging purposes."""
        self._paths[toolName] = path

    def _getToolPath(self, toolName):
        return self._paths.get(toolName, toolName)

    def bundleKernel(self, file, bundlePath, prefix=None):
        return self._bundleImage(file, bundlePath, kernel=True, prefix=prefix)

    def bundleRamdisk(self, file, bundlePath, prefix=None):
        return self._bundleImage(file, bundlePath, ramdisk=True, prefix=prefix)

    def bundleImage(self, file, bundlePath, kernel, ramdisk, prefix=None):
        return self._bundleImage(file, bundlePath, kernel, ramdisk,
                                 prefix=prefix)

    def _bundleImage(self, file, bundlePath, kernel=None, ramdisk=None,
                     prefix=None):
        args = ["--user", self._eucaInfo.userId,
                "--image", file,
                "--destination", bundlePath]
        if kernel is True:
            args.extend(["--kernel", "true"])
        elif kernel:
            args.extend(["--kernel", kernel])
        if ramdisk is True:
            args.extend(["--ramdisk", "true"])
        elif ramdisk:
            args.extend(["--ramdisk", ramdisk])
        if prefix is not None:
            args.extend(["--prefix", prefix])

        output = self._runTool("euca-bundle-image", args)

        manifest = self._findManifest(bundlePath)
        if manifest is None:
            message = "Manifest wasn't generated by bundle command:\n%s" \
                      % (output,)
            logging.error(message)
            raise EucaToolsError(message)

        return manifest

    def uploadBundle(self, manifest, bucket):
        if not os.path.isfile(manifest):
            message = "Bundle manifest %r not found for upload." % (manifest,)
            logging.error(message)
            raise EucaToolsError(message)
        args = ["--manifest", manifest, "--bucket", bucket]
        output = self._runTool("euca-upload-bundle", args)

    def registerBundle(self, manifest, bucket):
        manifestName = os.path.split(manifest)[1]
        args = ["%s/%s" % (bucket, manifestName)]
        output = self._runTool("euca-register", args)
        tokens = output.split()
        if len(tokens) != 2 or tokens[0] != "IMAGE":
            message = "Unexpected output registering image:\n%s" % (output,)
            logging.error(message)
            raise EucaToolsError(message)
        return tokens[1]

    def _runTool(self, toolName, args):
        command = self._getToolPath(toolName)
        args = [command] + args
        logArgs = []
        for arg in args:
            if " " in arg:
                logArgs.append("'%s'" % (arg,))
            else:
                logArgs.append(arg)
        logging.info("Running command: " + " ".join(logArgs))
        process = subprocess.Popen(args, env=self._getEnv(),
                                   stdout=subprocess.PIPE,
                                   stderr=subprocess.STDOUT)
        output = process.stdout.read()
        status = process.wait()
        if status != 0:
            message = "Command %r returned status code %d" % \
                      (toolName, status)
            if output.strip():
                message += ":\n" + output
            logging.error(message)
            raise EucaToolsError(message)
        else:
            logging.debug("Command finished with status code 0.")
        return output

    def _getEnv(self):
        return dict(
            PATH="/bin:/usr/bin/:/sbin:/usr/sbin",
            EC2_USER_ID=self._eucaInfo.userId,
            EC2_ACCESS_KEY=self._eucaInfo.accessKey,
            EC2_SECRET_KEY=self._eucaInfo.secretKey,
            EC2_PRIVATE_KEY=self._eucaInfo.privateKeyPath,
            EC2_CERT=self._eucaInfo.certificatePath,
            EUCALYPTUS_CERT=self._eucaInfo.cloudCertificatePath,
            EC2_URL=self._eucaInfo.urlForEC2,
            S3_URL=self._eucaInfo.urlForS3,
            )

    def _findManifest(self, bundlePath):
        matches = glob.glob("%s/*.manifest.xml" % (bundlePath,))
        if not matches:
            return None
        return matches[0]


def writeCredentialsZip(username, outputFilename, eucalyptusPrefix=""):
    assert username == "admin", "Only admin supported for now."
    if os.path.isfile(outputFilename):
        logging.debug("Erasing old credentials file before replacing it.")
        os.unlink(outputFilename)
    command = "%s/usr/sbin/euca_conf --get-credentials %s" \
              % (eucalyptusPrefix, outputFilename)
    logging.info("Running command: " + command)
    status, output = commands.getstatusoutput(command)
    if status != 0:
        message = "Error from euca_conf --get-credentials:\n%s" % output
        logging.error(message)
        raise EucaToolsError(message)
    else:
        logging.debug("Command finished with status code 0.")
