/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ivolumeviewsubject.h"


#include "ibuffer.h"
#include "iclipplane.h"
#include "icolorbars.h"
#include "icommoneventobservers.h"
#include "icontrolmodule.h"
#include "idatahelper.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idatasubject.h"
#include "ierror.h"
#include "ifunctionmapping.h"
#include "ihistogrammaker.h"
#include "ipalette.h"
#include "ipiecewisefunction.h"
#include "ireplicatedgriddata.h"
#include "ireplicatedvolume.h"
#include "iviewmodule.h"
#include "iviewsubjectobserver.h"
#include "ivolumedataconverter.h"

#include <vtkToolkits.h>

#include <vtkStructuredPoints.h>
#include <vtkVolumeProperty.h>
#include <vtkVolumeRayCastCompositeFunction.h>
#include <vtkVolumeRayCastMIPFunction.h>
#include <vtkVolumeRayCastMapper.h>
#include <vtkVolumeTextureMapper2D.h>

#ifndef IVTK_44
#include <vtkFixedPointVolumeRayCastMapper.h> 
//#include <vtkVolumeShearWarpMapper.h> 
#include <vtkVolumeTextureMapper3D.h>
#endif

//
//  Allow for the IFrIT own support for VolumePro1000
//
#ifdef I_CUSTOM_VP1000
	#ifndef VTK_USE_VOLUMEPRO
		#define VTK_USE_VOLUMEPRO
	#endif
	#include "ivolumepromapper.h"
#else
	#ifdef VTK_USE_VOLUMEPRO
		#include <vtkVolumeProHelper.h>
	#else
		#define vtkVolumeProHelper int
	#endif
	#define iVolumeProHelper vtkVolumeProHelper
#endif


//
//  Templates
//
#include "iarraytemplate.h"
#include "ibuffertemplate.h"


using namespace iParameter;


IVIEWSUBJECT_DEFINE_TYPE(iVolumeViewSubject,Volume,w);

IOBJECT_DEFINE_KEY(iVolumeViewSubject,BlendMode,bm,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,ImageDownsampleFactor,di,Float,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,DepthDownsampleFactor,dd,Float,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,InterpolationType,it,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,Method,m,OffsetInt,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,MethodNames,mn,String,0);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,OpacityFunction,of,Any,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,Palette,p,Int,1);
IOBJECT_DEFINE_KEY(iVolumeViewSubject,Var,v,OffsetInt,1);

//
//  Inherited keys
//
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_COMMON(iVolumeViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_SHADING(iVolumeViewSubject);
IVIEWSUBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iVolumeViewSubject);


namespace iVolumeViewSubject_Private
{
	class UniformGridVolumeRenderingHelper : public iVolumeRenderingHelper
	{

	public:

		UniformGridVolumeRenderingHelper(iVolumeViewSubject *parent, const iString &name) : iVolumeRenderingHelper(parent,name,false)
		{
		}

		virtual bool IsUsingData(vtkDataSet *ds)
		{
			return (ds!=0 && ds->IsA("vtkImageData")!=0);
		}
	};


	class RayCastHelper : public UniformGridVolumeRenderingHelper
	{

	public:

		RayCastHelper(iVolumeViewSubject *parent, vtkImageData *input) : UniformGridVolumeRenderingHelper(parent,"Ray casting")
		{
			mMapper = vtkVolumeRayCastMapper::New(); IERROR_ASSERT(mMapper);
			mMapper->AddObserver(vtkCommand::ProgressEvent,parent->GetViewModule()->GetAbortRenderEventObserver());
			mMapper->IntermixIntersectingGeometryOn();
			mMapper->AutoAdjustSampleDistancesOn();
#ifdef IVTK_44
			mMapper->UseImageClipperOff();
#endif
			mMapper->SetInput(input);

			mCF = vtkVolumeRayCastCompositeFunction::New(); IERROR_ASSERT(mCF);
			mCF->SetCompositeMethodToInterpolateFirst();
			
			mMF = vtkVolumeRayCastMIPFunction::New(); IERROR_ASSERT(mMF);
			
			mMapper->SetVolumeRayCastFunction(mCF);
		}

		virtual ~RayCastHelper()
		{
			mCF->Delete();
			mMF->Delete();
			mMapper->Delete();
		}

		virtual vtkAbstractVolumeMapper* GetMapper()
		{
			return mMapper;
		}

		virtual void SetBlendMode(int m)
		{
			switch(m)
			{
			case 0:
				{
					mMapper->SetVolumeRayCastFunction(mCF);
					break;
				}
			case 1:
				{
					mMapper->SetVolumeRayCastFunction(mMF);
					break;
				}
			}
		}

	protected:

		virtual void AdjustSpacing()
		{
			if(mMapper->GetAutoAdjustSampleDistances() == 0)
			{
				mMapper->SetImageSampleDistance(mImageDownsampleFactor);
			}
			mMapper->SetSampleDistance(mDepthDownsampleFactor*mDataSpacing);
		}

	private:

		vtkVolumeRayCastMapper *mMapper;
		vtkVolumeRayCastCompositeFunction *mCF;
		vtkVolumeRayCastMIPFunction *mMF;
	};


	class Texture2DHelper : public UniformGridVolumeRenderingHelper
	{

	public:

		Texture2DHelper(iVolumeViewSubject *parent, vtkImageData *input) : UniformGridVolumeRenderingHelper(parent,"2D textures")
		{
			mMapper = vtkVolumeTextureMapper2D::New(); IERROR_ASSERT(mMapper);
			mMapper->AddObserver(vtkCommand::ProgressEvent,parent->GetViewModule()->GetAbortRenderEventObserver());
#ifdef IVTK_44
			mMapper->UseImageClipperOff();
#endif
			mMapper->SetInput(input);
		}

		~Texture2DHelper()
		{
			mMapper->Delete();
		}

		virtual vtkAbstractVolumeMapper* GetMapper()
		{
			return mMapper;
		}

		virtual void SetBlendMode(int m)
		{
#ifndef IVTK_44
			mMapper->SetBlendMode(m);
#endif
		}

	private:

		vtkVolumeTextureMapper2D *mMapper;
	};


#ifndef IVTK_44
	class Texture3DHelper : public UniformGridVolumeRenderingHelper
	{

	public:

		Texture3DHelper(iVolumeViewSubject *parent, vtkImageData *input) : UniformGridVolumeRenderingHelper(parent,"3D textures")
		{
			mMapper = vtkVolumeTextureMapper3D::New(); IERROR_ASSERT(mMapper);
			mMapper->AddObserver(vtkCommand::ProgressEvent,parent->GetViewModule()->GetAbortRenderEventObserver());
			mMapper->SetInput(input);
		}

		~Texture3DHelper()
		{
			mMapper->Delete();
		}

		virtual bool IsUsingData(vtkDataSet *ds)
		{
			if(!this->UniformGridVolumeRenderingHelper::IsUsingData(ds)) return false;
			return (mMapper->IsRenderSupported(mParent->GetVolume()->GetProperty()) != 0);
		}

		virtual vtkAbstractVolumeMapper* GetMapper()
		{
			return mMapper;
		}

		virtual void SetBlendMode(int m)
		{
			mMapper->SetBlendMode(m);
		}

	protected:

		virtual void AdjustSpacing()
		{
			mMapper->SetSampleDistance(mDepthDownsampleFactor*mDataSpacing);
		}

	private:

		vtkVolumeTextureMapper3D *mMapper;
	};


	class FixedPointRayCastHelper  : public UniformGridVolumeRenderingHelper
	{

	public:

		FixedPointRayCastHelper(iVolumeViewSubject *parent, vtkImageData *input) : UniformGridVolumeRenderingHelper(parent,"Fixed point ray casting")
		{
			mMapper = vtkFixedPointVolumeRayCastMapper::New(); IERROR_ASSERT(mMapper);
			mMapper->AddObserver(vtkCommand::ProgressEvent,parent->GetViewModule()->GetAbortRenderEventObserver());
			mMapper->IntermixIntersectingGeometryOn();
			mMapper->AutoAdjustSampleDistancesOn();
			mMapper->SetInput(input);
			mMapper->SetBlendModeToComposite();
		}

		virtual ~FixedPointRayCastHelper()
		{
			mMapper->Delete();
		}

		virtual vtkAbstractVolumeMapper* GetMapper()
		{
			return mMapper;
		}

		virtual void SetBlendMode(int m)
		{
			mMapper->SetBlendMode(m);
		}

	protected:

		virtual void AdjustSpacing()
		{
			if(mMapper->GetAutoAdjustSampleDistances() == 0)
			{
				mMapper->SetImageSampleDistance(mImageDownsampleFactor);
			}
			mMapper->SetSampleDistance(mDepthDownsampleFactor*mDataSpacing);
		}

	private:

		vtkFixedPointVolumeRayCastMapper *mMapper;
	};

/*
	class ShearWarpRayCastHelper : public UniformGridVolumeRenderingHelper
	{

	public:

		ShearWarpRayCastHelper(iVolumeViewSubject *parent, vtkImageData *input) : UniformGridVolumeRenderingHelper(parent,"Shear warp ray casting")
		{
			mMapper = vtkVolumeShearWarpMapper::New(); IERROR_ASSERT(mMapper);
			mMapper->AddObserver(vtkCommand::ProgressEvent,parent->GetViewModule()->GetAbortRenderEventObserver());
			mMapper->IntermixIntersectingGeometryOn();
			mMapper->AutoAdjustSampleDistancesOn();
			mMapper->SetInput(input);
			mMapper->SetFunctionTypeToComposite();
		}

		virtual ~ShearWarpRayCastHelper()
		{
			mMapper->Delete();
		}

		virtual vtkAbstractVolumeMapper* GetMapper()
		{
			return mMapper;
		}

		virtual void SetBlendMode(int m)
		{
			switch(m)
			{
			case 0:
				{
					mMapper->SetFunctionTypeToComposite();
					break;
				}
			case 1:
				{
					mMapper->SetFunctionTypeToMIP();
					break;
				}
			}
		}

	protected:

		virtual void AdjustSpacing()
		{
			if(mMapper->GetAutoAdjustSampleDistances() == 0)
			{
				mMapper->SetImageSampleDistance(mImageDownsampleFactor);
			}
		}

	private:

		vtkVolumeShearWarpMapper *mMapper;
	};
*/
#endif


#ifdef VTK_USE_VOLUMEPRO
	class VolumeProHelper : public UniformGridVolumeRenderingHelper
	{

	public:

		VolumeProHelper(iVolumeViewSubject *parent, vtkImageData *input) : UniformGridVolumeRenderingHelper(parent,"VolumePro board")
		{
			mMapper = iVolumeProHelper::New(); IERROR_ASSERT(mThisMapper);
			mMapper->AddObserver(vtkCommand::ProgressEvent,parent->GetViewModule()->GetAbortRenderEventObserver());
			mMapper->IntermixIntersectingGeometryOn();
			mMapper->SuperSamplingOn();
#ifdef IVTK_44
			mMapper->UseImageClipperOff();
#endif
			mMapper->SetInput(input);
		}

		~VolumeProHelper()
		{
			mMapper->Delete();
		}

		virtual bool IsUsingData(vtkDataSet *ds)
		{
			if(!this->UniformGridVolumeRenderingHelper::IsUsingData(ds)) return false;
			return (mMapper->GetNoHardware()==0 && mMapper->GetWrongVLIVersion()==0 && mMapper->GetNumberOfBoards()>0);
		}

		virtual vtkAbstractVolumeMapper* GetMapper()
		{
			return mMapper;
		}

		virtual void SetBlendMode(int m)
		{
			mMapper->SetBlendMode(m);
		}

	protected:

		virtual void AdjustSpacing()
		{
			double d[3];
			mMapper->GetSuperSamplingFactor(d);
			mMapper->SetSuperSamplingFactor(1.0/mImageDownsampleFactor,1.0/mImageDownsampleFactor,1.0/mDepthDownsampleFactor);
		}

	private:

		iVolumeProHelper *mMapper;
	};
#endif
};


using namespace iVolumeViewSubject_Private;


//
// iVolumeViewSubject class
// 
iVolumeViewSubject::iVolumeViewSubject(iViewModule *vm, const iDataType &type, const iString &name) : iViewSubject(vm,type,name,0U)
{
	mObjectType = _ObjectTypeVolume;

	mVar = 0;
	mPalette = 1;
	mMethod = 1;
	
	mOpacityFunction = iPiecewiseFunction::New(0.0,1.0); IERROR_ASSERT(mOpacityFunction);
	
	mBlendMode = 0;
	mInterpolationType = 1; 

	mImageDownsampleFactor = mDepthDownsampleFactor = 1.0;
	mBlendMode = 0;
	
	mVolume = iReplicatedVolume::New(this); IERROR_ASSERT(mVolume);
	mVolume->SetPosition(0.0,0.0,0.0);
	
	mVolume->GetProperty()->SetShade(mShading?1:0);
	mVolume->GetProperty()->SetScalarOpacity(mOpacityFunction->GetVTKFunction());
	mVolume->GetProperty()->SetColor(this->GetViewModule()->GetControlModule()->GetPalette(iMath::Abs(mPalette)-1)->GetColorTransferFunction(mPalette < 0));
	mVolume->GetProperty()->SetInterpolationTypeToLinear();
	this->UpdateMaterialProperties();
	
	mDataConverter = iVolumeDataConverter::New(this); IERROR_ASSERT(mDataConverter);
	mDataReplicated = iReplicatedGridData::New(this); IERROR_ASSERT(mDataReplicated);
	mHistogramMaker = this->GetViewModule()->GetReader()->GetSubject(this->GetDataType())->CreateHistogramMaker(); IERROR_ASSERT(mHistogramMaker);
	mFunctionMappings = new iFunctionMapping(mOpacityFunction,mHistogramMaker,0); IERROR_ASSERT(mFunctionMappings);
	
	mDataConverter->AddObserver(vtkCommand::ProgressEvent,this->GetViewModule()->GetAbortRenderEventObserver());
	mDataReplicated->AddObserver(vtkCommand::ProgressEvent,this->GetViewModule()->GetAbortRenderEventObserver());

	mDataReplicated->SetInput(mDataConverter->GetOutput());

	//
	//  Add observer to keep information about this object
	//
	mVolume->AddObserver(vtkCommand::UserEvent,mObjectObserver);

	mVolume->SetVisibility(0);
	this->GetViewModule()->AddObject(mVolume);
}


iVolumeViewSubject::~iVolumeViewSubject()
{
	this->GetViewModule()->RemoveObject(mVolume);

	while(mHelpers.Size() > 0) delete mHelpers.RemoveLast();

	mHistogramMaker->Delete();
	delete mFunctionMappings;

	mOpacityFunction->Delete();
	mDataConverter->Delete();
	mDataReplicated->Delete();
	mVolume->Delete();
}


void iVolumeViewSubject::ConfigureBody()
{
	iVolumeRenderingHelper *m;

	//
	//  Create helpers; do it here so that children can replace all mappers
	//
	m = new RayCastHelper(this,mDataReplicated->GetOutput()); IERROR_ASSERT(m); mHelpers.Add(m);

#ifndef IVTK_44
	m = new FixedPointRayCastHelper(this,mDataReplicated->GetOutput()); IERROR_ASSERT(m); mHelpers.Add(m);
//	m = new ShearWarpRayCastHelper(this,mDataReplicated->GetOutput()); IERROR_ASSERT(m); mHelpers.Add(m);
	mMethod = 2;
#endif

	m = new Texture2DHelper(this,mDataReplicated->GetOutput()); IERROR_ASSERT(m); mHelpers.Add(m);
#ifndef IVTK_44
	m = new Texture3DHelper(this,mDataReplicated->GetOutput()); IERROR_ASSERT(m); mHelpers.Add(m);
#endif
#ifdef VTK_USE_VOLUMEPRO
	m = new VolumeProHelper(this,mDataReplicated->GetOutput()); IERROR_ASSERT(m); mHelpers.Add(m);
#endif
}


void iVolumeViewSubject::FinishInitialization()
{
	this->SetMethod(mMethod);
	this->SetBlendMode(mBlendMode);
}


void iVolumeViewSubject::Reset()
{
	this->SyncWithData(this->RequestAll());

	//
	//  Un-initialize, if needed
	//
	if(mIsInitialized)
	{
		this->ShowColorBars(false);
		mVolume->SetVisibility(0);
		mIsInitialized = false;
	}
}


void iVolumeViewSubject::ShowClipPlane(bool s)
{
	int i;

	if(s && !mClipPlaneOn)
	{
		for(i=0; i<mHelpers.Size(); i++) mHelpers[i]->GetMapper()->AddClippingPlane(this->GetViewModule()->GetClipPlane());
		mClipPlaneOn = true;
	} 
	if(!s && mClipPlaneOn)
	{
		for(i=0; i<mHelpers.Size(); i++) mHelpers[i]->GetMapper()->RemoveClippingPlane(this->GetViewModule()->GetClipPlane());
		mClipPlaneOn = false;
	}
	this->ClearCache();
}


void iVolumeViewSubject::ShowColorBars(bool show)
{
	if(!this->IsVisible()) return;
	this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPriorityVolume,mVar,this->GetDataType(),mPalette,show);
	this->ClearCache();
}


void iVolumeViewSubject::SetVar(int v)
{
	if(v>=0 && v<this->GetLimits()->GetNumVars())
	{
		this->ShowColorBars(false);
		mVar = v;
		this->ShowColorBars(true);
		this->UpdateVar();
		this->SyncWithData(this->Request(mVar));
		this->ClearCache();
	}
}


void iVolumeViewSubject::UpdateVar()
{
	int i;

	for(i=0; i<mHelpers.Size(); i++)
	{
		mHelpers[i]->SetVar(mVar);
	}

	mDataConverter->SetCurrentVar(mVar);
	mFunctionMappings->AttachToLimits(this->GetLimits(),mVar);
	mHistogramMaker->SetInputComponent(mVar);
}


void iVolumeViewSubject::SetPalette(int p)
{ 
	if(p!=0 && p>=-this->GetViewModule()->GetControlModule()->GetNumberOfPalettes() && p<=this->GetViewModule()->GetControlModule()->GetNumberOfPalettes())
	{
		this->ShowColorBars(false);
		mPalette = p;
		this->ShowColorBars(true);
		mVolume->GetProperty()->SetColor(this->GetViewModule()->GetControlModule()->GetPalette(iMath::Abs(mPalette)-1)->GetColorTransferFunction(mPalette < 0));
		this->ClearCache();
	}
}


void iVolumeViewSubject::UpdateMaterialProperties()
{
	if(mShading) mVolume->GetProperty()->SetShade(1); else mVolume->GetProperty()->SetShade(0);

	mVolume->GetProperty()->SetAmbient(mAmbient);
	mVolume->GetProperty()->SetDiffuse(mDiffuse);
	mVolume->GetProperty()->SetSpecular(mSpecular);
	mVolume->GetProperty()->SetSpecularPower(mSpecularPower);
	this->ClearCache();
}


void iVolumeViewSubject::ShowBody(bool show)
{
	if(show)
	{
		mVolume->SetVisibility(1);
		this->ShowColorBars(true);
	} 
	else 
	{
		this->ShowColorBars(false);
		mVolume->SetVisibility(0);
	}
}


void iVolumeViewSubject::SetMethod(int m)
{
	if(m>=0 && m<mHelpers.Size())
	{
		if(this->IsThereData() && !mHelpers[m]->IsUsingData(this->GetData()))
		{
			int i;
			for(i=mHelpers.MaxIndex(); i>=0; i--)
			{
				if(mHelpers[i]->IsUsingData(this->GetData())) break;
			}
			m = i;
		}
		mMethod = m;
		if(mMethod != -1)
		{
            mVolume->SetMapper(mHelpers[mMethod]->GetMapper());
		}
		else
		{
			this->Show(false);
		}
		this->ClearCache();
	}
}


void iVolumeViewSubject::SetInterpolationType(int m) 
{
	if(m>=0 && m<2) 
	{
		mInterpolationType = m;
		switch(m)
		{
		case 0:
			{
				mVolume->GetProperty()->SetInterpolationTypeToNearest();
				break;
			}
		case 1:
			{
				mVolume->GetProperty()->SetInterpolationTypeToLinear();
				break;
			}
		}
		this->ClearCache();
	}
}


void iVolumeViewSubject::SetDepthDownsampleFactor(float m) 
{
	if(m < 1.0e-3) return;

	int i;

	for(i=0; i<mHelpers.Size(); i++)
	{
		mHelpers[i]->SetDepthDownsampleFactor(m);
	}

	mDepthDownsampleFactor = m;
	this->ClearCache();
}


void iVolumeViewSubject::SetImageDownsampleFactor(float m) 
{
	if(m < 1.0e-3) return;

	int i;

	for(i=0; i<mHelpers.Size(); i++)
	{
		mHelpers[i]->SetImageDownsampleFactor(m);
	}

	mImageDownsampleFactor = m;
	this->ClearCache();
}


void iVolumeViewSubject::SetBlendMode(int m) 
{
	if(m<0 || m>1) return;
	
	int i;

	for(i=0; i<mHelpers.Size(); i++)
	{
		mHelpers[i]->SetBlendMode(m);
	}

	mBlendMode = m;
	this->ClearCache();
}


bool iVolumeViewSubject::IsVisible() const
{
	return (mVolume->GetVisibility() == 1);
}


float iVolumeViewSubject::GetMemorySize() const
{
	int i;
	float s = 0.0;

	for(i=0; i<mHelpers.Size(); i++)
	{
		s += mHelpers[i]->GetMemorySize();
	}
	s += mDataConverter->GetMemorySize();
	s += mDataReplicated->GetMemorySize();
	return s;
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVolumeViewSubject::ViewSubjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyVar(),mVar);
	this->PackValue(s,KeyMethod(),mMethod);
	this->PackValue(s,KeyPalette(),mPalette);
	this->PackValue(s,KeyBlendMode(),mBlendMode);
	this->PackValue(s,KeyInterpolationType(),mInterpolationType);

	this->PackValue(s,KeyImageDownsampleFactor(),mImageDownsampleFactor);
	this->PackValue(s,KeyDepthDownsampleFactor(),mDepthDownsampleFactor);

	this->PackValuePiecewiseFunction(s,KeyOpacityFunction(),mOpacityFunction);

	//
	//  Query keys
	//
	int i;
	iBuffer<iString> sbuf;

	for(i=0; i<mHelpers.Size(); i++) sbuf[i] = mHelpers[i]->GetName();
	this->PackValue(s,KeyMethodNames(),sbuf,mHelpers.Size());
}


void iVolumeViewSubject::ViewSubjectUnPackStateBody(const iString &s)
{
	int i; float f;
		
	if(this->UnPackValue(s,KeyVar(),i)) this->SetVar(i);
	if(this->UnPackValue(s,KeyMethod(),i)) this->SetMethod(i);
	if(this->UnPackValue(s,KeyPalette(),i)) this->SetPalette(i);
	if(this->UnPackValue(s,KeyBlendMode(),i)) this->SetBlendMode(i);
	if(this->UnPackValue(s,KeyInterpolationType(),i)) this->SetInterpolationType(i);
	
	if(this->UnPackValue(s,KeyImageDownsampleFactor(),f)) this->SetImageDownsampleFactor(f);
	if(this->UnPackValue(s,KeyDepthDownsampleFactor(),f)) this->SetDepthDownsampleFactor(f);

	this->UnPackValuePiecewiseFunction(s,KeyOpacityFunction(),mOpacityFunction); // it resets the function only if successfull
}


const iString& iVolumeViewSubject::GetMethodName(int i) const
{
	static const iString none;

	if(i>=0 && i<mHelpers.Size()) return mHelpers[i]->GetName(); else return none;
}


bool iVolumeViewSubject::IsMethodAvailable(int i) const
{
	if(i>=0 && i<mHelpers.Size())
	{
		if(!this->IsThereData()) return true; else return mHelpers[i]->IsUsingData(this->GetData());
	}
	else return false;
}


iFunctionMapping* iVolumeViewSubject::GetOpacityFunction() const
{
	mCacheInvalid = true;
	return mFunctionMappings;
}


bool iVolumeViewSubject::CanBeShown() const
{
	if(mMethod == -1) return false;

	iDataHelper *h = this->CreateDataHelper(this->GetData());
	bool ret = (h->IsThereScalarData() && mVar<this->GetLimits()->GetNumVars());
	delete h;
	return ret;
}


void iVolumeViewSubject::ViewSubjectSyncWithData(const iDataSyncRequest &)
{
	int i;

	if(this->IsThereData())
	{
		//
		//  Input is ImageData
		//
		vtkImageData *id = vtkImageData::SafeDownCast(this->GetData());
	 	if(id != 0)
		{
			mDataConverter->SetInput(id);

			double s = id->GetSpacing()[2];
			for(i=0; i<mHelpers.Size(); i++)
			{
				mHelpers[i]->SetDataSpacing(s);
			}
		}

		//
		//  Reset input
		//
		for(i=0; i<mHelpers.Size(); i++) if(mHelpers[i]->IsAttachedDirectly())
		{
			mHelpers[i]->GetMapper()->SetInput(this->GetData());
		}

		this->SetMethod(mMethod); // may need to reset the mapper.

		mHistogramMaker->SetInput(this->GetData(),mVar);
	}

    mFunctionMappings->AttachToLimits(this->GetLimits(),mVar);
}


//
//  Helper class
//
iVolumeRenderingHelper::iVolumeRenderingHelper(iVolumeViewSubject *parent, const iString &name, bool attachedDirectly) : mParent(parent), mName(name), mAttachedDirectly(attachedDirectly)
{
	IERROR_ASSERT(mParent);

	mDataSpacing = 1.0;
	mImageDownsampleFactor = mDepthDownsampleFactor = 1.0f;
}


iVolumeRenderingHelper::~iVolumeRenderingHelper()
{
}


void iVolumeRenderingHelper::SetDataSpacing(double v)
{
	if(v > 0.0)
	{
		mDataSpacing  = v;
		this->AdjustSpacing();
	}
}


void iVolumeRenderingHelper::SetImageDownsampleFactor(float v)
{
	if(v > 0.0f)
	{
		mImageDownsampleFactor  = v;
		this->AdjustSpacing();
	}
}


void iVolumeRenderingHelper::SetDepthDownsampleFactor(float v)
{
	if(v > 0.0f)
	{
		mDepthDownsampleFactor  = v;
		this->AdjustSpacing();
	}
}

