/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iactor.h"


#include "idatalimits.h"
#include "ierror.h"
#include "ilookuptable.h"
#include "ipalette.h"

#include <vtkCamera.h>
#include <vtkLookupTable.h>
#include <vtkMapperCollection.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkProperty.h>
#include <vtkRenderer.h>

//
//  Templates
//
#include "iarraytemplate.h"


//
//  Actor class
//
iActor* iActor::New(bool scaled)
{
	return new iActor(scaled);
}


iActor::iActor(bool scaled)
{
	mBasicScale = mAxisScale[0] = mAxisScale[1] = mAxisScale[2] = 1.0;
	mScaled = scaled;
	mCurrentSubject = 0;

	mDummySubject = new iActorSubject(this,0); IERROR_ASSERT(mDummySubject);
	mDummySubjectPointer = mDummySubject;

	mDefaultMapper = vtkPolyDataMapper::New(); IERROR_ASSERT(mDefaultMapper);
	this->SetMapper(mDefaultMapper);
	mDefaultMapper->Delete();

	mLookupTable = iLookupTable::New(); IERROR_ASSERT(mLookupTable);
	mDefaultMapper->SetLookupTable(mLookupTable);
	mDefaultMapper->UseLookupTableScalarRangeOn();

	this->CreateOwnLODs();
	mDefaultLODMappers = this->LODMappers; // saving for proper deletion
	vtkMapper *m;
	mDefaultLODMappers->InitTraversal();
	while((m = mDefaultLODMappers->GetNextItem()) != 0)
	{
		m->SetLookupTable(mLookupTable);
		m->UseLookupTableScalarRangeOn();
	}
}


iActor::~iActor()
{
	while(mSubjects.Size() > 0) this->ReleaseGraphicsResources(mSubjects.Last()->mWindow);
	delete mDummySubject;

	mLookupTable->Delete();

	this->Mapper = mDefaultMapper;
	this->LODMappers = mDefaultLODMappers;
}


bool iActor::SetCurrentWindow(vtkWindow *win)
{
	int i;

#ifdef I_CHECK1
	IERROR_ASSERT(win);
#endif

	if(mCurrentSubject!=0 && mCurrentSubject->mWindow==win) return true; // already current

	//
	//  Find this window
	//
	mDummySubject->mWindow = win;
	i = mSubjects.Find(mDummySubjectPointer);
	if(i<0 || i>=mSubjects.Size()) return false;

	mCurrentSubject = mSubjects[i].Pointer();
	this->Mapper = mCurrentSubject->mMapper;
	this->LODMappers = mCurrentSubject->mLODMappers;
	return true;
}


void iActor::ReleaseGraphicsResources(vtkWindow *win)
{
	if(win==0 || !this->SetCurrentWindow(win)) return; // already released

	this->vtkLODActor::ReleaseGraphicsResources(win);

	if(mSubjects.Remove(mCurrentSubject)) delete mCurrentSubject;
	mCurrentSubject = 0;
	this->Mapper = mDefaultMapper;
	this->LODMappers = mDefaultLODMappers;
}



void iActor::Render(vtkRenderer *ren, vtkMapper * /*m*/)
{
	static vtkActor *q = vtkActor::New();

	vtkWindow *win = ren->GetVTKWindow();

	if(win == 0) return; // nothing to do

	if(!this->SetCurrentWindow(win))
	{
		//
		//  This window has not been attached yet
		//
		mCurrentSubject = new iActorSubject(this,win); IERROR_ASSERT(mCurrentSubject);
		mSubjects.Add(mCurrentSubject);

		this->Mapper = mCurrentSubject->mMapper;
		this->LODMappers = mCurrentSubject->mLODMappers;
	}

	if(mScaled)
	{
		vtkCamera *c = ren->GetActiveCamera();
		
		double ps;
		if(c->GetParallelProjection() == 0)
		{
			vtkCamera *cam = ren->GetActiveCamera();
			ps = cam->GetDistance()*tan(cam->GetViewAngle()/2.0*0.017453292)/1.6;
		}
		else
		{
			ps = c->GetParallelScale()/1.6;
		}
		ps *= mBasicScale;
		this->SetScale(ps*mAxisScale[0],ps*mAxisScale[1],ps*mAxisScale[2]);
	}
	else
	{
		this->SetScale(mAxisScale[0],mAxisScale[1],mAxisScale[2]);
	}
	
	//
	//  Fix VTK bug
	//
	if(this->GetProperty()->GetMTime() > this->Mapper->GetMTime())
	{
		this->Mapper->Modified();
	}
	vtkLODActor::Render(ren,this->Mapper);
}


void iActor::SetScaled(bool s)
{
	if(s != mScaled)
	{
		mScaled = s;
		this->Modified();
	}
}


void iActor::SetBasicScale(double s)
{
	if(s > 0.0)
	{
		mBasicScale = s;
		this->Modified();
	}
}


void iActor::SetAxisScale(double sx, double sy, double sz)
{
	if(sx>0.0 && sy>0.0 && sz>0.0)
	{
		mAxisScale[0] = sx;
		mAxisScale[1] = sy;
		mAxisScale[2] = sz;
		this->Modified();
	}
}


//
//  Mapper functionality
//
void iActor::SetInput(vtkPolyData *data)
{
	int i;
	vtkMapper *save = this->Mapper;
	mDefaultMapper->SetInput(data);
	this->Mapper = mDefaultMapper;
	this->UpdateOwnLODs();
	this->Mapper = save;
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->mMapper->SetInput(data);
}
	

void iActor::SetScalarVisibility(bool s)
{
	int i;
	mDefaultMapper->SetScalarVisibility(s?1:0);
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->mMapper->SetScalarVisibility(s?1:0);
}


void iActor::SetScalarRange(float min, float max)
{
	int i;

	if(mDefaultMapper->GetUseLookupTableScalarRange() == 0)
	{
		mDefaultMapper->SetScalarRange(min,max);
		for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->mMapper->SetScalarRange(min,max);
	}
	mLookupTable->SetTableRange(min,max);
}

	
void iActor::SyncWithLimits(iDataLimits *l, int v, float scale)
{
	if(l!=0 && scale>0.0f && v>=0 && v<l->GetNumVars())
	{
		mLookupTable->SetStretch(iParameter::_StretchLin);
		this->SetScalarRange(l->GetLowerLimit(v)/scale,l->GetUpperLimit(v)/scale);
		mLookupTable->SetStretch(l->GetStretch(v));
	}
}


void iActor::AddMapperObserver(unsigned long event, vtkCommand *obs)
{
	int i;
	mDefaultMapper->AddObserver(event,obs);
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->mMapper->AddObserver(event,obs);
}


void iActor::AddClippingPlane(vtkPlane *plane)
{
	int i;
	mDefaultMapper->AddClippingPlane(plane);
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->mMapper->AddClippingPlane(plane);
}


void iActor::RemoveClippingPlane(vtkPlane *plane)
{
	int i;
	mDefaultMapper->RemoveClippingPlane(plane);
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->mMapper->RemoveClippingPlane(plane);
}


void iActor::ColorByArrayComponent(int arrayNum, int component)
{
	int i;
	mDefaultMapper->ColorByArrayComponent(arrayNum,component);
	for(i=0; i<mSubjects.Size(); i++) mSubjects[i]->mMapper->ColorByArrayComponent(arrayNum,component);
}


//
//  Helper classes
//
iActorSubject::iActorSubject(iActor *parent, vtkWindow *win)
{
	IERROR_ASSERT(parent);
 
	mWindow = win;

	if(win == 0) // dummy
	{
		mMapper = 0;
		mLODMappers = 0;
	}
	else
	{
		mMapper = vtkPolyDataMapper::New(); IERROR_ASSERT(mMapper);
		mMapper->ShallowCopy(parent->GetMapper());

		mLODMappers = vtkMapperCollection::New(); IERROR_ASSERT(mLODMappers);

		vtkMapper *m1;
		vtkPolyDataMapper *m2;
		vtkMapperCollection *pm = parent->GetLODMappers();
		pm->InitTraversal();
		while((m1 = pm->GetNextItem()) != 0)
		{
			m2 = vtkPolyDataMapper::New(); IERROR_ASSERT(m2);
			m2->ShallowCopy(m1);
			mLODMappers->AddItem(m2);
			m2->Delete();
		}
	}
}


iActorSubject::~iActorSubject()
{
	if(mMapper != 0) mMapper->Delete();
	if(mLODMappers != 0) mLODMappers->Delete();
}


iActorSubjectPointer::iActorSubjectPointer(iActorSubject *subject)
{
	mPointer = subject;
}


void iActorSubjectPointer::operator=(iActorSubject *p)
{
	mPointer = p;
}


void iActorSubjectPointer::operator=(const iActorSubjectPointer &p)
{
	mPointer = p.mPointer;
}


bool iActorSubjectPointer::operator==(const iActorSubjectPointer &p) const
{
	if(mPointer!=0 && p.mPointer!=0)
	{
		return (mPointer->mWindow == p.mPointer->mWindow); 
	}
	else return (mPointer == p.mPointer);
}


bool iActorSubjectPointer::operator<(const iActorSubjectPointer &p) const
{
	if(mPointer!=0 && p.mPointer!=0)
	{
		return ((unsigned long)mPointer->mWindow < (unsigned long)p.mPointer->mWindow); 
	}
	else return ((unsigned long)mPointer < (unsigned long)p.mPointer);
}

