/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iggframedatatypeselector.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "iimagefactory.h"
#include "ishell.h"
#include "iviewmodule.h"

#include "iggdatatypeprovider.h"
#include "iggmainwindow.h"
#include "iggwidgetkeyselectionbox.h"
#include "iggwidgetotherbutton.h"

#include "ibgwidgetbuttonsubject.h"
#include "ibgwidgethelper.h"
#include "ibgwidgetselectionboxsubject.h"

#include "iggsubjectfactory.h"
using namespace iParameter;
using namespace iggParameter;


namespace iggFrameDataTypeSelector_Private
{
	class DataTypeComboBox : public iggWidget
	{

	public:

		DataTypeComboBox(const iString &title, iggFrameDataTypeSelector *selector, iggFrame *parent) : iggWidget(parent)
		{
			mSelector = selector;
			mSubject = iggSubjectFactory::CreateWidgetComboBoxSubject(this,title,true);
			this->SetBaloonHelp("Select the type of data to work with","This control selects the type of data that is affected by the widgets on this page.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			mSubject->Clear();

			int i;
			const iDataInfo *info(mSelector->GetDataInfo());
			for(i=0; i<info->Count(); i++)
			{
				mSubject->InsertItem(info->Type(i).GetTextName());
			}
			mSubject->SetValue(mSelector->GetCurrentDataTypeIndex());
			this->Enable(info->Count() > 1);
		}

		void OnInt1Body(int v)
		{
			mSelector->SetCurrentDataTypeIndex(v);
		}

		ibgWidgetComboBoxSubject *mSubject;
		iggFrameDataTypeSelector *mSelector;
	};


	class ApplyButton : public iggWidgetSimpleButton
	{

	public:

		ApplyButton(iggFrameDataTypeSelector *selector, iggFrame *parent) : iggWidgetSimpleButton("Apply",parent)
		{
			mSelector = selector;
			this->SetBaloonHelp("Apply the selection","Press this button to apply the selection made in the combo box.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			mWidgetHelper->Enable(mSelector->IsModified());
		}

		virtual void Execute()
		{
			mSelector->Apply();
		}

		iggFrameDataTypeSelector *mSelector;
	};


	class RestrictButton : public iggWidgetSimpleButton
	{

	public:

		RestrictButton(iggFrameDataTypeSelector *selector, iggFrame *parent) : iggWidgetSimpleButton("",parent,true)
		{
			mAllIcon = iImageFactory::FindIcon("singleline.png");
			mSetIcon = iImageFactory::FindIcon("multiline.png");
			if(mAllIcon==0 || mSetIcon==0)
			{
				IERROR_LOW("Icon images are not found.");
			}

			mSubject->SetFlat(true);

			mSelector = selector;
			this->SetBaloonHelp("Restrict the list to types with loaded data","This button toggles showing all types or only types with data actually loaded into memory.");
        }

	protected:

		virtual void UpdateWidgetBody()
		{
			switch(mSelector->GetMode())
			{
			case _ListDataTypesWithData:
				{
					if(mSetIcon != 0) mSubject->SetIcon(*mSetIcon);
					break;
				}
			default:
				{
					if(mAllIcon != 0) mSubject->SetIcon(*mAllIcon);
					break;
				}
			}
		}

		virtual void Execute()
		{
			switch(mSelector->GetMode())
			{
			case _ListDataTypesWithData:
				{
					mSelector->SetMode(_ListDataTypesAll);
					break;
				}
			default:
				{
					mSelector->SetMode(_ListDataTypesWithData);
					break;
				}
			}
			this->UpdateWidget();
		}

		const iImage *mAllIcon, *mSetIcon;
		iggFrameDataTypeSelector *mSelector;
	};
};


using namespace iggFrameDataTypeSelector_Private;


iggFrameDataTypeSelector::iggFrameDataTypeSelector(iggDataTypeProvider *provider, bool withapply, const iString &title, iggFrameBase *parent, int mode) : iggFrame(parent)
{
	const bool hor = true;

	mMode = mode;
	mProvider = provider; IERROR_ASSERT(provider);
	mCurrentDataTypeIndex = 0;
	mInfoPointer = new iDataInfo;

	if(hor)
	{
		mBase = new iggFrame(this,3);
		mBase->AddSpace(1); // improves the layout
	}
	else
	{
		mBase = new iggFrame(title,this,3);
		mBase->AddSpace(1); // improves the layout
	}

	iggWidget *cb = new DataTypeComboBox(hor?title:"",this,mBase);
	RestrictButton *rb = 0;
	if(mode == iggParameter::_ListDataTypesAll) rb = new RestrictButton(this,mBase);

	if(withapply)
	{
		iggWidget *ab = new ApplyButton(this,mBase);
		mBase->AddLine(cb,ab,rb);
		cb->AddDependent(ab);
	}
	else
	{
		mBase->AddLine(cb,2,rb,1);
	}
	mBase->AddSpace(1); // improves the layout
	mBase->SetColStretch(0,10);
	this->AddLine(mBase);

	mAutoApplying = !withapply;

	cb->UpdateWidget();
}

	
iggFrameDataTypeSelector::~iggFrameDataTypeSelector()
{
	delete mInfoPointer;
}

	
/*void iggFrameDataTypeSelector::ChangeProvider(iggDataTypeProvider *provider)
{
	mProvider = provider; IERROR_ASSERT(provider);
	mCurrentDataTypeIndex = mProvider->GetActiveDataTypeIndex();

	this->UpdateWidget();
}*/


const iDataInfo* iggFrameDataTypeSelector::GetDataInfo()
{
	const iDataInfo &pi(mProvider->GetDataInfo());
	int i, n = pi.Count();
	iDataReader *dr = this->GetShell()->GetControlModule()->GetViewModule()->GetReader();

	switch(mMode)
	{
	case _ListDataTypesWithData:
		{
			mInfoPointer->Erase();
			for(i=0; i<n; i++) if(dr->IsThereData(pi.Type(i)))
			{
				*mInfoPointer += pi.Type(i);
			}
			break;
		}
	case _ListDataTypesWithVariables:
		{
			mInfoPointer->Erase();
			for(i=0; i<n; i++) if(dr->IsThereData(pi.Type(i)) && dr->GetLimits(pi.Type(i))!=0 && dr->GetLimits(pi.Type(i))->GetNumVars()>0)
			{
				*mInfoPointer += pi.Type(i);
			}
			break;
		}
	case _ListDataTypesAll:
	default:
		{
			*mInfoPointer = pi;
			break;
		}
	}

	bool done = false;
	for(i=0; !done && i<mInfoPointer->Count(); i++) if(mInfoPointer->Type(i) == mProvider->GetActiveDataType())
	{
        mCurrentDataTypeIndex = i;
		done = true;
	}
	if(!done) this->UpdateProvider();

	return mInfoPointer;
}


void iggFrameDataTypeSelector::SetMode(int m)
{
	if(m>=_ListDataTypesAll && m<=_ListDataTypesWithVariables)
	{
		mMode = m;
		mBase->UpdateWidget();
	}
}


void iggFrameDataTypeSelector::SetCurrentDataTypeIndex(int v)
{
	mCurrentDataTypeIndex = v;
	if(mAutoApplying) this->Apply();
}


void iggFrameDataTypeSelector::UpdateProvider()
{
	if(mCurrentDataTypeIndex>=0 && mCurrentDataTypeIndex<mInfoPointer->Count())
	{
		const iDataInfo &pi(mProvider->GetDataInfo());
		int i, n = pi.Count();
		for(i=0; i<n; i++) if(pi.Type(i) == mInfoPointer->Type(mCurrentDataTypeIndex))
		{
			mProvider->SetActiveDataTypeIndex(i);
			break;
		}
	}
}


void iggFrameDataTypeSelector::Apply()
{
	this->UpdateProvider();

	if(mCurrentDataTypeIndex>=0 && mCurrentDataTypeIndex<mInfoPointer->Count())
	{
		if(mInfoPointer->Type(mCurrentDataTypeIndex) != mProvider->GetActiveDataType())
		{
			//
			//  Setting a data type does not necessarily succeeds, since the data may not
			//  be available. So, we need to update the widget after each execution,
			//  to make sure that it is up-to-date. 
			//
			this->UpdateWidget();
		}

		this->GetShell()->GetControlModule()->Render(_RenderOptionClones);
	}
}


bool iggFrameDataTypeSelector::IsModified() const
{
    return mCurrentDataTypeIndex != mProvider->GetActiveDataTypeIndex();
}


void iggFrameDataTypeSelector::UpdateWidgetBody()
{
	const iDataInfo *info(this->GetDataInfo()); // that will make us up-to-date

	if(mAutoApplying)
	{
		int i, n = info->Count();
		mCurrentDataTypeIndex = -1;
		for(i=0; mCurrentDataTypeIndex==-1 && i<n; i++)
		{
			if(info->Type(i) == mProvider->GetActiveDataType()) mCurrentDataTypeIndex = i;
		}
	}

	mBase->Show(mCurrentDataTypeIndex>=0 && mCurrentDataTypeIndex<info->Count());

	this->iggFrame::UpdateWidgetBody();
}
