/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  This class is based on vtkFunctionParser class, but it was throughly rewritten and optimized
//  (for array calculations the speedup is a factor of 20 or so!).
//


#include "iexpressionparser.h"


#include "ierror.h"

#include <ctype.h>
#include <stdlib.h>
#include <imath.h>

#include "iarraytemplate.h"


namespace iExpressionParser_Private
{
	static const int operatorsMax = 14;
	static const char operators[] = { '|', '&', '<', '>', (char)128, (char)129, (char)130, (char)131, '+', '-', '.', '*', '/', '^' };
};


using namespace iExpressionParser_Private;

iExpressionParser* iExpressionParser::New()
{
	return new iExpressionParser;
}


iExpressionParser::iExpressionParser()
{
	mStackSize = mStackPointer = 0;
	mErrorByteCode = mErrorPosition = -1;
	mModified = mValuesChanged = true;
	mIgnoreDivZero = false;
}


iExpressionParser::~iExpressionParser() 
{
}


void iExpressionParser::Delete()
{
	delete this;
}


void iExpressionParser::SetFunction(const iString &function)
{
	mFunction = function;
	mModified = true;
}


int iExpressionParser::Parse()
{
	int result;
	int i;
	
	mValuesChanged = true;
	mErrorMessage.Clear();
	mErrorByteCode = mErrorPosition = -1;

	if(mFunction == 0)
    {
		this->Error("No function has been set.");
		return 0;
    }
	
	this->RemoveSpaces();
	
	result = this->CheckSyntax();
	if(!result)
    {
		return 0;
    }
	
	result = this->BuildInternalFunctionStructure();
	if(!result)
    {
		return 0;
    }
	
	// need to make sure that the ambiguous operators are correct
	// - scalar/vector +
	// - scalar/vector -
	// - scalar/vector unary minus
	// - * (2 scalars) or scalar multiple (scalar, vector)
	result = this->DisambiguateOperators();
	if(!result)
    {
		return 0;
    }
	
	// need to recalculate stack size based on number of vector variables
	// in byte code
	for(i=0; i<mByteCode.Size(); i++)
    {
		if(mByteCode[i] >= VTK_PARSER_BEGIN_VARIABLES+mScalarVariables.Size())
		{
			mStackSize += 2;
		}
    }
	
	while(mStack.Size() < mStackSize) mStack.Add(0.0);
	
	return 1;
}


int iExpressionParser::DisambiguateOperators()
{
	unsigned char* tempStack = new unsigned char[mByteCode.Size()]; IERROR_ASSERT(tempStack);
	int i;
	int tempStackPtr = -1;
	
	// using 0 for scalars and 1 for vectors
	for (i = 0; i < mByteCode.Size(); i++)
    {
		switch (mByteCode[i])
		{
		case VTK_PARSER_IMMEDIATE:
			tempStackPtr++;
			tempStack[tempStackPtr] = 0;
			break;
		case VTK_PARSER_UNARY_MINUS:
			if (tempStack[tempStackPtr] != 0)
			{
				mByteCode[i] = VTK_PARSER_VECTOR_UNARY_MINUS;
			}
			break;
		case VTK_PARSER_OR:
		case VTK_PARSER_AND:
		case VTK_PARSER_NOT:
		case VTK_PARSER_LT:
		case VTK_PARSER_GT:
		case VTK_PARSER_LE:
		case VTK_PARSER_GE:
		case VTK_PARSER_EQ:
		case VTK_PARSER_NE:
			if (tempStack[tempStackPtr] == 1 || tempStack[tempStackPtr-1] == 1)
			{
				this->Error("Error: logical operations are undefined for vectors.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_ADD:
			if (tempStack[tempStackPtr] != 0 && tempStack[tempStackPtr-1] != 0)
			{
				mByteCode[i] = VTK_PARSER_VECTOR_ADD;
			}
			else if ((tempStack[tempStackPtr] == 0 &&
				tempStack[tempStackPtr-1] != 0) ||
				(tempStack[tempStackPtr] != 0 &&
				tempStack[tempStackPtr-1] == 0))
			{
				this->Error("Error: addition expects either 2 vectors or 2 scalars.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_SUBTRACT:
			if (tempStack[tempStackPtr] != 0 && tempStack[tempStackPtr-1] != 0)
			{
				mByteCode[i] = VTK_PARSER_VECTOR_SUBTRACT;
			}
			else if ((tempStack[tempStackPtr] == 0 &&
				tempStack[tempStackPtr-1] != 0) ||
				(tempStack[tempStackPtr] != 0 &&
				tempStack[tempStackPtr-1] == 0))
			{
				this->Error("Error: addition expects either 2 vectors or 2 scalars.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_MULTIPLY:
			if (tempStack[tempStackPtr-1] == 0 && tempStack[tempStackPtr] == 1)
			{
				mByteCode[i] = VTK_PARSER_SCALAR_MULTIPLE;
				tempStack[tempStackPtr-1] = 1;
			}
			else if (tempStack[tempStackPtr] == 1)
			{
				this->Error("Error: expecting either 2 scalars or a scalar followed by a vector.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_DIVIDE:
			if (tempStack[tempStackPtr] == 1 || tempStack[tempStackPtr-1] == 1)
			{
				this->Error("Error: can't divide vectors.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_POWER:
			if (tempStack[tempStackPtr] == 1)
			{
				this->Error("Error: can't raise a vector to a power.",i);
				return 0;
			}
		case VTK_PARSER_ABSOLUTE_VALUE:
		case VTK_PARSER_EXPONENT:
		case VTK_PARSER_CEILING:
		case VTK_PARSER_FLOOR:
		case VTK_PARSER_LOGARITHM:
		case VTK_PARSER_SQUARE_ROOT:
		case VTK_PARSER_SINE:
		case VTK_PARSER_COSINE:
		case VTK_PARSER_TANGENT:
		case VTK_PARSER_ARCSINE:
		case VTK_PARSER_ARCCOSINE:
		case VTK_PARSER_ARCTANGENT:
		case VTK_PARSER_HYPERBOLIC_SINE:
		case VTK_PARSER_HYPERBOLIC_COSINE:
		case VTK_PARSER_HYPERBOLIC_TANGENT:
			if (tempStack[tempStackPtr] == 1)
			{
				this->Error("Error: expecting a scalar, but got a vector.",i);
				return 0;
			}
			break;
		case VTK_PARSER_VECTOR_UNARY_MINUS:
			if (tempStack[tempStackPtr] == 0)
			{
				mByteCode[i] = VTK_PARSER_UNARY_MINUS;
			}
			break;
		case VTK_PARSER_DOT_PRODUCT:
			if (tempStack[tempStackPtr] == 0 || tempStack[tempStackPtr] == 0)
			{
				this->Error("Error: dot product does not operate on scalars.",i);
				return 0;
			}
			tempStack[tempStackPtr-1] = 0;
			tempStackPtr--;
			break;
		case VTK_PARSER_VECTOR_ADD:
			if (tempStack[tempStackPtr] != 1 && tempStack[tempStackPtr-1] != 1)
			{
				mByteCode[i] = VTK_PARSER_ADD;
			}
			else if ((tempStack[tempStackPtr] == 0 &&
				tempStack[tempStackPtr-1] != 0) ||
				(tempStack[tempStackPtr] != 0 &&
				tempStack[tempStackPtr-1] == 0))
			{
				this->Error("Error: addition expects either 2 vectors or 2 scalars.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_VECTOR_SUBTRACT:
			if (tempStack[tempStackPtr] != 1 && tempStack[tempStackPtr-1] != 1)
			{
				mByteCode[i] = VTK_PARSER_SUBTRACT;
			}
			else if ((tempStack[tempStackPtr] == 0 &&
				tempStack[tempStackPtr-1] != 0) ||
				(tempStack[tempStackPtr] != 0 &&
				tempStack[tempStackPtr-1] == 0))
			{
				this->Error("Error: subtraction expects either 2 vectors or 2 scalars.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_SCALAR_MULTIPLE:
			if (tempStack[tempStackPtr] == 0 && tempStack[tempStackPtr-1] == 0)
			{
				mByteCode[i] = VTK_PARSER_MULTIPLY;
			}
			else if (tempStack[tempStackPtr-1] != 0 ||
				tempStack[tempStackPtr] != 1)
			{
				this->Error("Error: scalar multiple expects a scalar followed by a vector.",i);
				return 0;
			}
			tempStackPtr--;
			break;
		case VTK_PARSER_MAGNITUDE:
			if (tempStack[tempStackPtr] == 0)
			{
				this->Error("Error: magnitude expects a vector, but got a scalar.",i);
				return 0;
			}
			tempStack[tempStackPtr] = 0;
			break;
		case VTK_PARSER_NORMALIZE:
			if (tempStack[tempStackPtr] == 0)
			{
				this->Error("Error: normalize expects a vector, but got a scalar.",i);
				return 0;
			}
			break;
		default:
			if (mByteCode[i] < VTK_PARSER_BEGIN_VARIABLES+mScalarVariables.Size())
			{
				tempStackPtr++;
				tempStack[tempStackPtr] = 0;
			}
			else
			{
				tempStackPtr++;
				tempStack[tempStackPtr] = 1;
			}
		}
    }
	
	delete [] tempStack;
	return 1;
}


void iExpressionParser::Evaluate()
{
	int numBytesProcessed;
	int numImmediatesProcessed = 0;
	int stackPosition = -1;
	double magnitude;
	
	if(mModified && this->Parse()==0)
	{
		return;
	}

	if(!mValuesChanged) return;

	mStackPointer = -1;
	for(numBytesProcessed=0; numBytesProcessed<mByteCode.Size(); numBytesProcessed++)
    {
		switch (mByteCode[numBytesProcessed])
		{
		case VTK_PARSER_IMMEDIATE:
			mStack[++stackPosition] = mImmediates[numImmediatesProcessed++];
			break;
		case VTK_PARSER_UNARY_MINUS:
			mStack[stackPosition] =- mStack[stackPosition];
			break;
			
		case VTK_PARSER_OR:
			mStack[stackPosition-1] = (round(mStack[stackPosition-1]) || round(mStack[stackPosition]));
			stackPosition--;
			break;
		case VTK_PARSER_AND:
			mStack[stackPosition-1] = (round(mStack[stackPosition-1]) && round(mStack[stackPosition]));
			stackPosition--;
			break;
		case VTK_PARSER_NOT:
			mStack[stackPosition] = !round(mStack[stackPosition]);
			break;
			
		case VTK_PARSER_LT:
			mStack[stackPosition-1] = (mStack[stackPosition-1] < mStack[stackPosition]);
			stackPosition--;
			break;
		case VTK_PARSER_GT:
			mStack[stackPosition-1] = (mStack[stackPosition-1] > mStack[stackPosition]);
			stackPosition--;
			break;
		case VTK_PARSER_LE:
			mStack[stackPosition-1] = !(mStack[stackPosition-1] > mStack[stackPosition]);
			stackPosition--;
			break;
		case VTK_PARSER_GE:
			mStack[stackPosition-1] = !(mStack[stackPosition-1] < mStack[stackPosition]);
			stackPosition--;
			break;
		case VTK_PARSER_EQ:
			mStack[stackPosition-1] = !(fabs(mStack[stackPosition-1]-mStack[stackPosition])>0.0);
			stackPosition--;
			break;
		case VTK_PARSER_NE:
			mStack[stackPosition-1] = (fabs(mStack[stackPosition-1]-mStack[stackPosition])>0.0);
			stackPosition--;
			break;
			
			
		case VTK_PARSER_ADD:
			mStack[stackPosition-1] += mStack[stackPosition];
			stackPosition--;
			break;
		case VTK_PARSER_SUBTRACT:
			mStack[stackPosition-1] -= mStack[stackPosition];
			stackPosition--;
			break;
		case VTK_PARSER_MULTIPLY:
			mStack[stackPosition-1] *= mStack[stackPosition];
			stackPosition--;
			break;
		case VTK_PARSER_DIVIDE:
			if (fabs(mStack[stackPosition]) < 1.0e-300)
			{
				if(!mIgnoreDivZero)
				{
					this->Error("Error: dividing by zero.",numBytesProcessed);
					return;
				}
				else
				{
					mStack[stackPosition] = 1.0;
				}
			}
			mStack[stackPosition-1] /= mStack[stackPosition];
			stackPosition--;
			break;
		case VTK_PARSER_POWER:
			mStack[stackPosition-1] = pow(mStack[stackPosition-1],
				mStack[stackPosition]);
			stackPosition--;
			break;
		case VTK_PARSER_ABSOLUTE_VALUE:
			mStack[stackPosition] = fabs(mStack[stackPosition]);
			break;
		case VTK_PARSER_EXPONENT:
			mStack[stackPosition] = exp(mStack[stackPosition]);
			break;
		case VTK_PARSER_CEILING:
			mStack[stackPosition] = ceil(mStack[stackPosition]);
			break;
		case VTK_PARSER_FLOOR:
			mStack[stackPosition] = floor(mStack[stackPosition]);
			break;
		case VTK_PARSER_LOGARITHM:
			if (mStack[stackPosition]<=0)
			{
				this->Error("Error: taking a logarithm of a negative value.",numBytesProcessed);
				return;
			}
			mStack[stackPosition] = log(mStack[stackPosition]);
			break;
		case VTK_PARSER_SQUARE_ROOT:
			if (mStack[stackPosition] < 0)
			{
				this->Error("Error: taking a square root of a negative value.",numBytesProcessed);
				return;
			}
			mStack[stackPosition] = sqrt(mStack[stackPosition]);
			break;
		case VTK_PARSER_SINE:
			mStack[stackPosition] = sin(mStack[stackPosition]);
			break;
		case VTK_PARSER_COSINE:
			mStack[stackPosition] = cos(mStack[stackPosition]);
			break;
		case VTK_PARSER_TANGENT:
			mStack[stackPosition] = tan(mStack[stackPosition]);
			break;
		case VTK_PARSER_ARCSINE:
			if (mStack[stackPosition] < -1 || mStack[stackPosition] > 1)
			{
				this->Error("Error: taking asin of a value < -1 or > 1.",numBytesProcessed);
				return;
			}
			mStack[stackPosition] = asin(mStack[stackPosition]);
			break;
		case VTK_PARSER_ARCCOSINE:
			if(mStack[stackPosition]<-1 || mStack[stackPosition]>1)
			{
				this->Error("Error: taking acos of a value < -1 or > 1.",numBytesProcessed);
				return;
			}
			mStack[stackPosition] = acos(mStack[stackPosition]);
			break;
		case VTK_PARSER_ARCTANGENT:
			mStack[stackPosition] = atan(mStack[stackPosition]);
			break;
		case VTK_PARSER_HYPERBOLIC_SINE:
			mStack[stackPosition] = sinh(mStack[stackPosition]);
			break;
		case VTK_PARSER_HYPERBOLIC_COSINE:
			mStack[stackPosition] = cosh(mStack[stackPosition]);
			break;
		case VTK_PARSER_HYPERBOLIC_TANGENT:
			mStack[stackPosition] = tanh(mStack[stackPosition]);
			break;
		case VTK_PARSER_VECTOR_UNARY_MINUS:
			mStack[stackPosition] = -mStack[stackPosition];
			mStack[stackPosition-1] = -mStack[stackPosition-1];
			mStack[stackPosition-2] = -mStack[stackPosition-2];
			break;
		case VTK_PARSER_DOT_PRODUCT:
			mStack[stackPosition-3] *= mStack[stackPosition];
			mStack[stackPosition-4] *= mStack[stackPosition-1];
			mStack[stackPosition-5] *= mStack[stackPosition-2];
			mStack[stackPosition-5] = mStack[stackPosition-5] +
				mStack[stackPosition-4] + mStack[stackPosition-3];
			stackPosition -= 5;
			break;
		case VTK_PARSER_VECTOR_ADD:
			mStack[stackPosition-3] += mStack[stackPosition];
			mStack[stackPosition-4] += mStack[stackPosition-1];
			mStack[stackPosition-5] += mStack[stackPosition-2];
			stackPosition -= 3;
			break;
		case VTK_PARSER_VECTOR_SUBTRACT:
			mStack[stackPosition-3] -= mStack[stackPosition];
			mStack[stackPosition-4] -= mStack[stackPosition-1];
			mStack[stackPosition-5] -= mStack[stackPosition-2];
			stackPosition -= 3;
			break;
		case VTK_PARSER_SCALAR_MULTIPLE:
			mStack[stackPosition] *= mStack[stackPosition-3];
			mStack[stackPosition-1] *= mStack[stackPosition-3];
			mStack[stackPosition-2] *= mStack[stackPosition-3];
			mStack[stackPosition-3] = mStack[stackPosition-2];
			mStack[stackPosition-2] = mStack[stackPosition-1];
			mStack[stackPosition-1] = mStack[stackPosition];
			stackPosition--;
			break;
		case VTK_PARSER_MAGNITUDE:
			mStack[stackPosition-2] =
				sqrt(pow(mStack[stackPosition], 2) +
				pow(mStack[stackPosition-1], 2) +
				pow(mStack[stackPosition-2], 2));
			stackPosition -= 2;
			break;
		case VTK_PARSER_NORMALIZE:
			magnitude = sqrt(pow(mStack[stackPosition], 2) +
				pow(mStack[stackPosition-1], 2) +
				pow(mStack[stackPosition-2], 2));
			if (magnitude == 0)
			{
				this->Error("Error: the magnitude of this vector is zero; can't normalize.",numBytesProcessed);
				return;
			}
			mStack[stackPosition] /= magnitude;
			mStack[stackPosition-1] /= magnitude;
			mStack[stackPosition-2] /= magnitude;
			break;
		default:
			if (mByteCode[numBytesProcessed] < VTK_PARSER_BEGIN_VARIABLES+mScalarVariables.Size())
			{
				mStack[++stackPosition] = mScalarVariables[mByteCode[numBytesProcessed]-VTK_PARSER_BEGIN_VARIABLES].Value;
			}
			else
			{
				double *v = mVectorVariables[mByteCode[numBytesProcessed]-VTK_PARSER_BEGIN_VARIABLES-mScalarVariables.Size()].Value;
				mStack[++stackPosition] = v[0];
				mStack[++stackPosition] = v[1];
				mStack[++stackPosition] = v[2];
			}
		}
    }
	mStackPointer = stackPosition;
	
	mModified = false;
}


bool iExpressionParser::GetScalarResult(double &result)
{
	this->Evaluate();
	if(mStackPointer==0 && mErrorMessage.IsEmpty())
	{
		result = mStack[0];
		return true;
	}
	else return false;
}


bool iExpressionParser::GetVectorResult(double result[3])
{
	this->Evaluate();
	if(mStackPointer==2 && mErrorMessage.IsEmpty())
	{
		result[0] = mStack[0];
		result[1] = mStack[1];
		result[2] = mStack[2];
		return true;
	}
	else return false;
}


int iExpressionParser::IsVariableName(int currentIndex) const
{
	int i;
	
	for(i=0; i<mScalarVariables.Size(); i++)
    {
		if(mFunction.Part(currentIndex,mScalarVariables[i].Name.Length()) == mScalarVariables[i].Name)
		{
			return 1;
		}
    }
	for(i=0; i<mVectorVariables.Size(); i++)
    {
		if(mFunction.Part(currentIndex,mVectorVariables[i].Name.Length()) == mVectorVariables[i].Name)
		{
			return 1;
		}
    }
	return 0;
}


int iExpressionParser::IsElementaryOperator(int op) const
{
	return strchr(operators,op) != 0;
}


void iExpressionParser::SetScalarVariableValue(const iString &variableName, double value)
{
	Scalar tmp;
	tmp.Name = variableName;
	int i = mScalarVariables.Find(tmp);
	if(i >= 0)
	{
		mScalarVariables[i].Value = value;
		mValuesChanged = true;
		return;
    }
	
	tmp.Value = value;
	mScalarVariables.Add(tmp);
	mModified = true;
}


void iExpressionParser::SetVectorVariableValue(const iString &variableName, double xValue, double yValue, double zValue)
{
	Vector tmp;
	tmp.Name = variableName;
	int i = mVectorVariables.Find(tmp);
	if(i >= 0)
	{
		mVectorVariables[i].Value[0] = xValue;
		mVectorVariables[i].Value[1] = yValue;
		mVectorVariables[i].Value[2] = zValue;
		mValuesChanged = true;
		return;
    }
	
	tmp.Value[0] = xValue;
	tmp.Value[1] = yValue;
	tmp.Value[2] = zValue;
	mVectorVariables.Add(tmp);
	mModified = true;
}


void iExpressionParser::RemoveSpaces()
{
	int i, n = 0;

	mFunction.ReduceWhiteSpace();
	char *p = new char[mFunction.Length()+1]; IERROR_ASSERT(p);

	for(i=0; i<mFunction.Length(); i++) if(mFunction[i] != ' ')
	{
		p[n++] = mFunction[i];
	}
	p[n] = 0;
	mFunction = iString(p);
}


int iExpressionParser::CheckSyntax()
{
	int index = 0, parenthesisCount = 0, currentChar;
	char* ptr;
	int functionNumber;
	
	while(1)
    {
		currentChar = mFunction[index];
		
		// Check for valid operand (must appear)
		
		// Check for leading -
		if (currentChar == '-')
		{
			currentChar = mFunction[++index];  
			if(index == mFunction.Length())
			{
				this->Error("Syntax error: unary minus with no operand.",-1,index);
				return 0;
			}
		}
		
		// Check for leading !
		if(currentChar == '!')
		{
			currentChar = mFunction[++index];  
			if(index == mFunction.Length())
			{
				this->Error("Syntax error: not operation with no operand.",-1,index);
				return 0;
			}
		}
		
		// Check for math function
		if((functionNumber = this->GetMathFunctionNumber(index)))
		{
			index += this->GetMathFunctionStringLength(functionNumber);
			currentChar = mFunction[index];
			if(currentChar != '(')
			{
				this->Error("Syntax error: input to math function not in parentheses.",-1,index);
				return 0;
			}
		}
		
		// Check for opening parenthesis
		if(currentChar == '(')
		{
			parenthesisCount++;
			index++;
			continue;
		}
		
		// Check for number
		if(isdigit(currentChar) || (currentChar=='.' && isdigit(mFunction[index+1])))
		{
			strtod(mFunction.ToCharPointer(index),&ptr);
			index += int(ptr-mFunction.ToCharPointer(index));
			currentChar = mFunction[index];
		}
		else
		{ 
			// Check for variable
			if(!this->IsVariableName(index))
			{
				this->Error("Syntax error: expecting a variable name.",-1,index);
				return 0;
			}
			index += this->GetVariableNameLength(this->GetOperandNumber(index)-VTK_PARSER_BEGIN_VARIABLES);
			currentChar = mFunction[index];
		}
		
		// Check for closing parenthesis
		while(currentChar == ')')
		{
			parenthesisCount--;
			if(parenthesisCount < 0)
			{
				this->Error("Syntax Error: mismatched parenthesis.",-1,index);
				return 0;
			}
			if(mFunction[index-1] == '(')
			{
				this->Error("Syntax Error: empty parentheses.",-1,index);
				return 0;
			}
			currentChar = mFunction[++index];
		}
		
		// If we Get here, we have a legal operand and now a legal operator or
		// end of string must follow.
		
		// Check for EOS
		// The only way to end the checking loop without error.
		if (index == mFunction.Length())
		{
			break;
		}
		// Check for operator
		if(!this->IsElementaryOperator(currentChar))
		{
			this->Error("Syntax error: operator expected.",-1,index);
			return 0;
		}
		
		// If we Get here, we have an operand and an operator; the next loop will
		// check for another operand (must appear)
		index++;
    } // while
	
	// Check that all opened parentheses are also closed
	if(parenthesisCount > 0)
    {
		this->Error("Syntax Error: missing closing parenthesis.",-1,index);
		return 0;
    }
	
	// The string is ok
	return 1;
}


int iExpressionParser::BuildInternalFunctionStructure()
{
	mStack.Clear();
	mImmediates.Clear();
	mByteCode.Clear(); 
	mBytePosition.Clear();

	mStackSize = 0;
	mStackPointer = 0;
	this->BuildInternalSubstringStructure(0,mFunction.Length()-1);  
	
	return 1;
}


void iExpressionParser::BuildInternalSubstringStructure(int beginIndex, int endIndex)
{
	int mathFunctionNum, beginIndex2;
	int opNum, parenthesisCount, i;
	
	if (this->IsSubstringCompletelyEnclosed(beginIndex, endIndex))
    {
		this->BuildInternalSubstringStructure(beginIndex+1, endIndex-1);
		return;
    }
	
	if (mFunction[beginIndex] == '-')
    {
		if (this->IsSubstringCompletelyEnclosed(beginIndex+1, endIndex))
		{
			this->BuildInternalSubstringStructure(beginIndex+2, endIndex-1);
			this->AddInternalByte(VTK_PARSER_UNARY_MINUS,beginIndex);
			return;
		}
		if (this->GetMathFunctionNumber(beginIndex+1) > 0 &&
			this->FindEndOfMathFunction(beginIndex+1) == endIndex)
		{
			this->BuildInternalSubstringStructure(beginIndex+1, endIndex);
			this->AddInternalByte(VTK_PARSER_UNARY_MINUS,beginIndex);
			return;
		}
    }
	
	if (mFunction[beginIndex] == '!')
    {
		if (this->IsSubstringCompletelyEnclosed(beginIndex+1, endIndex))
		{
			this->BuildInternalSubstringStructure(beginIndex+2, endIndex-1);
			this->AddInternalByte(VTK_PARSER_NOT,beginIndex);
			return;
		}
		if (this->GetMathFunctionNumber(beginIndex+1) > 0 &&
			this->FindEndOfMathFunction(beginIndex+1) == endIndex)
		{
			this->BuildInternalSubstringStructure(beginIndex+1, endIndex);
			this->AddInternalByte(VTK_PARSER_NOT,beginIndex);
			return;
		}
    }
	
	if (isalpha(mFunction[beginIndex]))
    {
		mathFunctionNum = this->GetMathFunctionNumber(beginIndex);
		if (mathFunctionNum > 0)
		{
			beginIndex2 = beginIndex;
			while (mFunction[beginIndex2] != '(')
			{
				beginIndex2++;
			}
			if (this->IsSubstringCompletelyEnclosed(beginIndex2, endIndex))
			{
				this->BuildInternalSubstringStructure(beginIndex2+1, endIndex-1);
				this->AddInternalByte(mathFunctionNum,beginIndex);
				return;
			}
		}
    }
	
	for (opNum = 0; opNum < operatorsMax; opNum++)
    {
		parenthesisCount = 0;
		for (i = endIndex; i > beginIndex; i--)
		{
			if (mFunction[i] == ')')
			{
				parenthesisCount++;
			}
			else if (mFunction[i] == '(')
			{
				parenthesisCount--;
			}
			if (parenthesisCount == 0 &&
				mFunction[i] == operators[opNum] &&
				!(mFunction[i] == '-' &&
				(this->IsElementaryOperator(i-1) || mFunction[i-1] == '(' ||
				(mFunction[i-1] == 'e' && i > 1 &&
				isdigit(mFunction[i-2])))) &&
				!(mFunction[i] == '.' &&
				(i+1 < mFunction.Length()) &&
				(isdigit(mFunction[i+1]))))
			{
				this->BuildInternalSubstringStructure(beginIndex, i-1);
				this->BuildInternalSubstringStructure(i+1, endIndex);
				this->AddInternalByte(this->GetElementaryOperatorNumber(operators[opNum]),i);
				mStackPointer--;
				return;
			}
		}
    }
	
	beginIndex2 = beginIndex;
	if(mFunction[beginIndex]=='-' || mFunction[beginIndex]=='!')
    {
		beginIndex2++;
    }
	
	this->AddInternalByte(this->GetOperandNumber(beginIndex2),beginIndex2);
	mStackPointer++;
    
	if (mStackPointer > mStackSize)
    {
		mStackSize++;
    }
	if (beginIndex2 > beginIndex)
    {
		if (mFunction[beginIndex] == '-')
	    {
			this->AddInternalByte(VTK_PARSER_UNARY_MINUS,beginIndex);
		} 
		else if (mFunction[beginIndex] == '!')
		{
			this->AddInternalByte(VTK_PARSER_NOT,beginIndex);
	    }
    }
}


void iExpressionParser::AddInternalByte(unsigned char newByte, int pos)
{
	mByteCode.Add(newByte);
	mBytePosition.Add(pos);
}


int iExpressionParser::IsSubstringCompletelyEnclosed(int beginIndex, int endIndex) const
{
	int i, parenthesisCount;
	
	if ( mFunction[beginIndex] == '(' && mFunction[endIndex]== ')' )
    {
		parenthesisCount = 1;
		for (i = beginIndex + 1; i < endIndex; i++)
		{
			if (mFunction[i] == '(' )
			{
				parenthesisCount++;
			}
			else if(mFunction[i] == ')' )
			{
				parenthesisCount--;
			}
			if (parenthesisCount == 0)
			{
				break;
			}
		}
		if (i == endIndex)
		{
			return 1;
		}
    }
	return 0;
}


int iExpressionParser::GetMathFunctionNumber(int currentIndex) const
{
	if(mFunction.Part(currentIndex,3) == "abs")
    {
		return VTK_PARSER_ABSOLUTE_VALUE;
    }
	if(mFunction.Part(currentIndex,3) == "exp")
    {
		return VTK_PARSER_EXPONENT;
    }
	if(mFunction.Part(currentIndex,4) == "ceil")
    {
		return VTK_PARSER_CEILING;
    }
	if(mFunction.Part(currentIndex,5) == "floor")
    {
		return VTK_PARSER_FLOOR;
    }
	if(mFunction.Part(currentIndex,3) == "log")
    {
		return VTK_PARSER_LOGARITHM;
    }
	if(mFunction.Part(currentIndex,4) == "sqrt")
    {
		return VTK_PARSER_SQUARE_ROOT;
    }
	if(mFunction.Part(currentIndex,3) == "sin")
    {
		if(mFunction.Part(currentIndex,4) == "sinh")
		{
			return VTK_PARSER_HYPERBOLIC_SINE;
		}
		return VTK_PARSER_SINE;
    }
	if(mFunction.Part(currentIndex,3) == "cos")
    {
		if(mFunction.Part(currentIndex,4) == "cosh")
		{
			return VTK_PARSER_HYPERBOLIC_COSINE;
		}
		return VTK_PARSER_COSINE;
    }
	if(mFunction.Part(currentIndex,3) == "tan")
    {
		if(mFunction.Part(currentIndex,4) == "tanh")
		{
			return VTK_PARSER_HYPERBOLIC_TANGENT;
		}
		return VTK_PARSER_TANGENT;
    }
	if(mFunction.Part(currentIndex,4) == "asin")
    {
		return VTK_PARSER_ARCSINE;
    }
	if(mFunction.Part(currentIndex,4) == "acos")
    {
		return VTK_PARSER_ARCCOSINE;
    }
	if(mFunction.Part(currentIndex,4) == "atan")
    {
		return VTK_PARSER_ARCTANGENT;
    }
	if(mFunction.Part(currentIndex,3) == "mag")
    {
		return VTK_PARSER_MAGNITUDE;
    }
	if(mFunction.Part(currentIndex,4) == "norm")
    {
		return VTK_PARSER_NORMALIZE;
    }
	
	return 0;
}


int iExpressionParser::GetMathFunctionStringLength(int mathFunctionNumber) const
{
	switch (mathFunctionNumber)
    {
    case VTK_PARSER_ABSOLUTE_VALUE:
    case VTK_PARSER_EXPONENT:
    case VTK_PARSER_LOGARITHM:
    case VTK_PARSER_SINE:
    case VTK_PARSER_COSINE:
    case VTK_PARSER_TANGENT:
    case VTK_PARSER_MAGNITUDE:
		return 3;
    case VTK_PARSER_CEILING:
    case VTK_PARSER_SQUARE_ROOT:
    case VTK_PARSER_ARCSINE:
    case VTK_PARSER_ARCCOSINE:
    case VTK_PARSER_ARCTANGENT:
    case VTK_PARSER_HYPERBOLIC_SINE:
    case VTK_PARSER_HYPERBOLIC_COSINE:
    case VTK_PARSER_HYPERBOLIC_TANGENT:
    case VTK_PARSER_NORMALIZE:
		return 4;
    case VTK_PARSER_FLOOR:
		return 5;
    default:
		return 0;
    }
}


int iExpressionParser::GetVariableNameLength(int variableNumber) const
{
	if(variableNumber < mScalarVariables.Size())
    {
		return mScalarVariables[variableNumber].Name.Length();
    }
	else
    {
		return mVectorVariables[variableNumber-mScalarVariables.Size()].Name.Length();
    }
}


int iExpressionParser::FindEndOfMathFunction(int beginIndex) const
{
	int i = beginIndex, parenthesisCount;
	
	while (mFunction[i] != '(' )
    {
		i++;
    }
	
	for (parenthesisCount = 1; parenthesisCount > 0; ++i)
    {
		parenthesisCount += (mFunction[i] == '(' ? 1 :	(mFunction[i] == ')' ? -1 : 0));
    }
	return i - 1;
}


bool iExpressionParser::CheckSyntax(const iString &s)
{
	this->SetFunction(s.ToCharPointer());
	return (this->Parse() != 0);
}


int iExpressionParser::GetElementaryOperatorNumber(char op) const
{
	int i;
	
	for(i = 0; i < operatorsMax; i++)
    {
		if (operators[i] == op)
		{
			return VTK_PARSER_OR + i;
		}
    }
	
	return 0;
}


int iExpressionParser::GetOperandNumber(int currentIndex)
{
	int i, variableIndex = -1;
	
	if(isdigit(mFunction[currentIndex]) || mFunction[currentIndex] == '.') // Number
    {
		mImmediates.Add(atof(mFunction.ToCharPointer(currentIndex)));
		return VTK_PARSER_IMMEDIATE;
    }
	
	for(i=0; i<mScalarVariables.Size(); i++)
    { 
		// Variable
		if(mFunction.Part(currentIndex).Find(mScalarVariables[i].Name) == 0)
		{
			if(variableIndex==-1 || mScalarVariables[i].Name.Length()>mScalarVariables[variableIndex].Name.Length())
			{
				variableIndex = i;
			}
		}
    }
	if(variableIndex >= 0)
    {
		return VTK_PARSER_BEGIN_VARIABLES + variableIndex;
    }
	
	for(i=0; i<mVectorVariables.Size(); i++)
    { 
		// Variable
		if(mFunction.Part(currentIndex).Find(mVectorVariables[i].Name) == 0)
		{
			if(variableIndex==-1 || mVectorVariables[i].Name.Length()>mVectorVariables[variableIndex].Name.Length())
			{
				variableIndex = i;
			}
		}
    }
	if (variableIndex >= 0)
    {
		return VTK_PARSER_BEGIN_VARIABLES + variableIndex + mScalarVariables.Size();
    }
	
	return 0;
}


void iExpressionParser::RemoveAllVariables()
{
	mScalarVariables.Clear();	
	mVectorVariables.Clear();	
}


const char iExpressionParser::GetElementaryOperatorByteEncoding(ParserTypes op) const
{
	if(op>=VTK_PARSER_OR && op<=VTK_PARSER_POWER) return operators[op-VTK_PARSER_OR]; else return ' ';
}


void iExpressionParser::Error(const iString &text, int bytecode, int position)
{
	mErrorMessage += text + "\n";
	if(mErrorByteCode < 0) mErrorByteCode = bytecode;
	if(mErrorPosition < 0) mErrorPosition = position;
}


int iExpressionParser::GetErrorPosition() const
{
	int pos = -1;
	if(mErrorPosition >= 0) pos = mErrorPosition;
	if(mErrorByteCode>=0 && mErrorByteCode<mByteCode.Size() && mBytePosition[mErrorByteCode]<pos) pos = mBytePosition[mErrorByteCode];
	return pos;
}

