/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icommandinterpreter.h"


#include "icontrolmodule.h"
#include "icontrolscript.h"
#include "idirectory.h"
#include "ifile.h"
#include "ierrorstatus.h"
#include "ishell.h"


using namespace iParameter;


iCommandInterpreter::iCommandInterpreter(iShell *s) : mShell(s)
{
	mBreakFlag = mExecFlag = mWorkFlag = false;
	mReturnState = 0;
}


iCommandInterpreter::~iCommandInterpreter()
{
}


void iCommandInterpreter::Start()
{
	this->DislayLineOfText("IFrIT Command Interpreter.",_CommandInterpreterTextTypeResult);
	this->DislayLineOfText("Type commands one per line.",_CommandInterpreterTextTypeResult);
	this->DislayLineOfText("Type 'help' for a list of available commands.",_CommandInterpreterTextTypeResult);

	this->Initialize();
}


int iCommandInterpreter::Exec()
{
	iString line;

	mScriptText.Clear();
	mBreakFlag = false;
	mReturnState = 0;

	while(!mBreakFlag)
	{
		this->DisplayPrompt();

		//
		//  Read in single line from "stdin"
		//
		line.Clear();
		this->ReadLineOfText(line);
		line.ReduceWhiteSpace();

		this->ExecLineBody(line);
	}

	return mReturnState;
}


void iCommandInterpreter::ExecLine(iString &line)
{
	//
	//  Reset the flags
	//
	mBreakFlag = false;
	mReturnState = 0;

	this->ExecLineBody(line);
}


void iCommandInterpreter::ExecLineBody(iString &line)
{
	//
	//  Clear the error status
	//
	this->Script()->GetErrorStatus()->Clear();

	//
	//  Is this an exec line?
	//
	mExecFlag = false;
	mWorkFlag = false;
	if(line[line.Length()-1] == ';')
	{
		mExecFlag = true;
		//
		//  Remove the semi-colon
		//
		line = line.Part(0,line.Length()-1);
		line.ReduceWhiteSpace();
	}

	//
	//  Shell-specific commands
	//
	this->AnalyseSpecialCommands(line);
	if(mBreakFlag)
	{
		mExecFlag = false;
		return;
	}

	//
	//  Redirection sign reads from an external file
	//
	if(line[0] == '<')
	{
		iString s = line.Part(1);
		s.ReduceWhiteSpace();
		iDirectory::ExpandFileName(s,this->GetShell()->GetEnvironment(_EnvironmentBase));

		iFile F(s);
		if(F.Open(iFile::_Read,iFile::_Text))
		{
			iString s;
			while(F.ReadLine(s)) if(!s.IsEmpty())
			{
				this->DislayLineOfText("==> "+s,_CommandInterpreterTextTypeCode);
				if(s[s.Length()-1] == ';')
				{
					mExecFlag = true;
					//
					//  Remove the semi-colon
					//
					s = s.Part(0,s.Length()-1);
				}
				mScriptText += s + "\n";
			}
			F.Close();
		}
		else
		{
			this->DislayLineOfText("File does not exist.",_CommandInterpreterTextTypeError);
		}
		line.Clear();
	}

	//
	//  Help, print, and list are executed at once
	//
	if(this->IsHelpRequest(line))
	{
		mExecFlag = false;
		mWorkFlag = true;
		this->Script()->Run(line,false,false); // do not reset the script!!!
		if(!this->Script()->GetErrorStatus()->Message().IsEmpty())
		{
			this->DislayLineOfText("ERROR: "+this->Script()->GetErrorStatus()->Message(),_CommandInterpreterTextTypeError);
		}
		return;
	}

	//
	// Form a script
	//
	if(!line.IsEmpty()) mScriptText += line + "\n";
	if(mScriptText.IsEmpty()) mExecFlag = false;
	if(mExecFlag)
	{
		//
		//  Execute the script - must mutex it since it is called from a thread
		//
		mWorkFlag = true;
		this->Script()->Run(mScriptText,false,false); // do not reset the script!!!
		mLastExecutedScriptText = mScriptText;
		mScriptText.Clear();
		if(this->Script()->GetErrorStatus()->IsError())
		{
			this->DislayLineOfText("ERROR: "+this->Script()->GetErrorStatus()->Message(),_CommandInterpreterTextTypeError);
		}
	}

	this->CheckBreak();
}


void iCommandInterpreter::Break(int ret)
{
	mReturnState = ret;
	mBreakFlag = true;
}


void iCommandInterpreter::Stop()
{
	this->Finalize();
}


iControlScript* iCommandInterpreter::Script() const
{
	return this->GetShell()->GetControlModule()->GetControlScript();
}


bool iCommandInterpreter::IsHelpRequest(const iString &line) const
{
	return (line.Part(0,4)=="help" && (line.Length()==4 || line[4]==' ')) ||
		line.Part(0,5)=="list " ||
		line.Part(0,6)=="print " ||
		(line.Part(0,2)==".h" && (line.Length()==2 || line[2]==' ')) ||
		line.Part(0,4)==".hp " || 
		line.Part(0,4)==".ho " || 
		line.Part(0,4)==".lp " || 
		(line.Part(0,3)==".lo" && (line.Length()==3 || line[3]==' ')) ||
		line.Part(0,3)==".p ";
}

