/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ianimatorscript.h"


#include "ianimator.h"
#include "ibasicdatasubjects.h"
#include "icamera.h"
#include "icontrolmodule.h"
#include "icontrolscript.h"
#include "icrosssectionviewsubject.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "idirectory.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "iextensionfactory.h"
#include "ifile.h"
#include "imarker.h"
#include "imarkerfamily.h"
#include "iparticlegroup.h"
#include "iparticlesviewsubject.h"
#include "irendertool.h"
#include "ishell.h"
#include "ishellfactory.h"
#include "isurfaceviewsubject.h"
#include "itensorfieldviewsubject.h"
#include "ivectorfieldviewsubject.h"
#include "iviewmodule.h"
#include "iviewobject.h"
#include "iviewobjectfamily.h"
#include "ivolumeviewsubject.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkRenderer.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "iviewfamilytemplate.h"

//
//  macros
//
#include "ianimatorscriptmacro.h"
#include "ibasescriptmacro.h"


using namespace iParameter;


#define BACKWARD_COMPATIBLE


iValue* iAnimatorScript::valIS = 0;
iValue* iAnimatorScript::valRS = 0;
iValue* iAnimatorScript::valRP = 0;
iValue* iAnimatorScript::valVA3 = 0;

#define curScript	iRequiredCast<iAnimatorScript>(INFO,me)


iAnimatorScript* iAnimatorScript::New(iAnimator *an)
{
	IERROR_ASSERT(an);
	return iExtensionFactory::CreateAnimatorScript(an);
}


iAnimatorScript::iAnimatorScript(iAnimator *an) : iBaseScript(an->GetViewModule()->GetControlModule()->GetControlScript()) // always embedded
{
	mEmbeddedControlScript = iShellFactory::CreateControlScript(an->GetViewModule()->GetControlModule(),this);
	mAnimator = an;
	mInRender = false;

	//
	// Init TMP values
	//
	if(mCounter == 1)
	{
		valIS = iValue::New(0,"tmpIS",iValue::_ByValue,0);
		valRS = iValue::New(0,"tmpRS",iValue::_ByValue,0.0);
		valRP = iValue::New(0,"tmpRP",iValue::_ByValue,static_cast<double *>(0));
		valVA3 = this->TransformSingleTokenToValue("(0.0,0.0,0.0)");
	}

	//
	//  Alias words (mostly for backward compatibility)
	//
#ifdef BACKWARD_COMPATIBLE
	this->CreateAliasWord("x-section-speed","cross-section-speed"); 
	this->CreateAliasWord("xsection-speed","cross-section-speed"); 
	this->CreateAliasWord("fly-by-speed","flyby-speed"); 

	this->CreateAliasWord("mesh-file","uniform-scalars-file");
	this->CreateAliasWord("particle-file","basic-particles-file");
	this->CreateAliasWord("vector-file","uniform-vectors-file");
	this->CreateAliasWord("tensor-file","uniform-tensors-file");
	
	this->CreateAliasWord("xsection","cross-section"); 
	this->CreateAliasWord("vector","vector-field"); 
	this->CreateAliasWord("tensor","tensor-field"); 

	this->CreateAliasWord("mesh-variable-lower-limit","uniform-scalars-lower-limit");
	this->CreateAliasWord("mesh-variable-upper-limit","uniform-scalars-upper-limit");
	this->CreateAliasWord("particle-attribute-lower-limit","basic-particles-lower-limit");
	this->CreateAliasWord("particle-attribute-upper-limit","basic-particles-upper-limit");
	this->CreateAliasWord("vector-upper-limit","uniform-vectors-upper-limit");
	this->CreateAliasWord("tensor-upper-limit","uniform-tensors-upper-limit");
	this->CreateAliasWord("scalar-field-lower-limit","uniform-scalars-lower-limit");
	this->CreateAliasWord("scalar-field-upper-limit","uniform-scalars-upper-limit");
	this->CreateAliasWord("particle-set-lower-limit","basic-particles-lower-limit");
	this->CreateAliasWord("particle-set-upper-limit","basic-particles-upper-limit");
	this->CreateAliasWord("vector-field-upper-limit","uniform-vectors-upper-limit");
	this->CreateAliasWord("tensor-field-upper-limit","uniform-tensors-upper-limit");

	this->CreateAliasWord("xsection-var","cross-section-var");
	this->CreateAliasWord("xsection-dir","cross-section-dir");
	this->CreateAliasWord("xsection-position","cross-section-position");

	this->CreateAliasWord("rotate-scale","rotate");
	this->CreateAliasWord("scale","zoom");
	this->CreateAliasWord("fly-by","flyby");
	this->CreateAliasWord("camera-path","camerapath");
	this->CreateAliasWord("upper-left-corner","ulcorner");
	this->CreateAliasWord("upper-right-corner","urcorner");
	this->CreateAliasWord("lower-left-corner","llcorner");
	this->CreateAliasWord("lower-right-corner","lrcorner");
#endif

	//
	//  Parameter words
	//
	this->CreateParameterWord(iValue::New(this,"static",iValue::_ByValue,0));
	this->CreateParameterWord(iValue::New(this,"rotate",iValue::_ByValue,1));
	this->CreateParameterWord(iValue::New(this,"tumble",iValue::_ByValue,2));
	this->CreateParameterWord(iValue::New(this,"flyby",iValue::_ByValue,3));
	this->CreateParameterWord(iValue::New(this,"camerapath",iValue::_ByValue,4));
	this->CreateParameterWord(iValue::New(this,"next",iValue::_ByValue,-1));
	this->CreateParameterWord(iValue::New(this,"parallel",iValue::_ByValue,false));
	this->CreateParameterWord(iValue::New(this,"perspective",iValue::_ByValue,true));
	this->CreateParameterWord(iValue::New(this,"hidden",iValue::_ByValue,false));
	this->CreateParameterWord(iValue::New(this,"visible",iValue::_ByValue,true));
	this->CreateParameterWord(iValue::New(this,"current",iValue::_ByValue,-1));
	this->CreateParameterWord(iValue::New(this,"marker",iValue::_ByFunction,0,iValue::New(this,"",iValue::_ByValue,0),0,iAnimatorScript::GetMarkerPosition));
	this->CreateParameterWord(iValue::New(this,"ulcorner",iValue::_ByValue,0));
	this->CreateParameterWord(iValue::New(this,"urcorner",iValue::_ByValue,1));
	this->CreateParameterWord(iValue::New(this,"llcorner",iValue::_ByValue,2));
	this->CreateParameterWord(iValue::New(this,"lrcorner",iValue::_ByValue,3));

	//
	//  Command words
	//

	//
	//  Assignment operations hidden as statements
	//
	this->CreateCommandWord(iOperation::New(this,"","render",1,iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::Render,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","render-all",1,iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::RenderAll,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","render-set",1,iValue::_Absolute,iValue::NewArray(this),iAnimatorScript::MultiRender,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","load",1,iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::Load,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"","reset",1,iAnimatorScript::Reset,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"load","uniform-scalars-file",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::LoadUniformScalarsFile,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"load","basic-particles-file",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::LoadBasicParticlesFile,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"load","uniform-vectors-file",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::LoadUniformVectorsFile,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"load","uniform-tensors-file",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::LoadUniformTensorsFile,(unsigned char)1));

	this->CreateCommandWord(iOperation::New(this,"","execute-control-script",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::ExecuteControlScript,(unsigned char)1));
	this->CreateCommandWord(iOperation::New(this,"embed","control-script",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::EmbedControlScript,(unsigned char)1));

	//
	//  Genuine assignment operations to pre-defined variables
	//
	this->CreateCommandWord(iOperation::New(this,"","style",1,iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::Style,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","frames-per-file",1,iValue::_RelativeAddMult,iValue::NewInt(this),iAnimatorScript::FramesPerFile,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","blended-frames",1,iValue::_RelativeAddMult,iValue::NewInt(this),iAnimatorScript::NumBlendedFrames,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","transition-frames",1,iValue::_RelativeAddMult,iValue::NewInt(this),iAnimatorScript::NumTransitionFrames,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","flyby-speed",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::FlybySpeed,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","rotation-phi",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::RotationPhi,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","rotation-theta",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::RotationTheta,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","rotation-roll",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::RotationRoll,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","scale",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::Scale,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","slide-speed",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::CrossSectionSpeed,(unsigned char)3));

	this->CreateCommandWord(iOperation::New(this,"","camera-view-up",1,iValue::_RelativeAdd,iValue::NewArray(this,3),iAnimatorScript::CameraViewUp,(unsigned char)5));
	this->CreateCommandWord(iOperation::New(this,"","camera-scale",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::CameraScale,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","camera-focal-point",1,iValue::_RelativeAdd,iValue::NewArray(this,3),iAnimatorScript::CameraFocus,(unsigned char)5));
	this->CreateCommandWord(iOperation::New(this,"","camera-position",1,iValue::_RelativeAdd,iValue::NewArray(this,3),iAnimatorScript::CameraPosition,(unsigned char)5));
	this->CreateCommandWord(iOperation::New(this,"","projection",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::Projection,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","stop-on-path",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::StopOnPath,(unsigned char)3));

	this->CreateCommandWord(iOperation::New(this,"","record-label",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::TimeLabel,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","record-label-value",1,iValue::_Absolute,iValue::NewReal(this),iAnimatorScript::TimeLabelValue,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","color-bars",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::ColorBars,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","bounding-box",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::BoundingBox,(unsigned char)3));

	this->CreateCommandWord(iOperation::New(this,"","title-page-file",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::TitlePageFile,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","title-page-number-of-frames",1,iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::TitlePageNumFrames,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","title-page-number-of-blended-frames",1,iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::TitlePageNumBlendedFrames,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","logo-file",1,iValue::_Absolute,iValue::NewString(this,false),iAnimatorScript::LogoFile,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","logo-opacity",1,iValue::_Absolute,iValue::NewReal(this),iAnimatorScript::LogoOpacity,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","logo-position",1,iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::LogoPosition,(unsigned char)3));

	this->CreateCommandWord(iOperation::New(this,"","surface",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::ShowSurface,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","cross-section",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::ShowCrossSection,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","volume",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::ShowVolume,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","particles",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::ShowParticles,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","vector-field",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::ShowVectorField,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","tensor-field",1,iValue::_Absolute,iValue::NewBool(this),iAnimatorScript::ShowTensorField,(unsigned char)3));

	//
	// This is depreciated: use embed control-script instead
	//
	this->CreateCommandWord(iOperation::New(this,"","uniform-scalars-lower-limit",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::UniformScalarsLowerLimit,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","uniform-scalars-upper-limit",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::UniformScalarsUpperLimit,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","basic-particles-lower-limit",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::BasicParticlesLowerLimit,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","basic-particles-upper-limit",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::BasicParticlesUpperLimit,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","uniform-vectors-upper-limit",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::UniformVectorsUpperLimit,(unsigned char)3));
	this->CreateCommandWord(iOperation::New(this,"","uniform-tensors-upper-limit",1,iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::UniformTensorsUpperLimit,(unsigned char)3));

	this->CreateCommandWord(iOperation::New(this,"","surface-var",1,iValue::NewIndex(this),iValue::_Absolute,iValue::NewIndex(this),iAnimatorScript::SurfaceVar,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","surface-level",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::SurfaceLevel,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","surface-opacity",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::SurfaceOpacity,(unsigned char)4));

	this->CreateCommandWord(iOperation::New(this,"","cross-section-var",1,iValue::NewIndex(this),iValue::_Absolute,iValue::NewIndex(this),iAnimatorScript::CrossSectionVar,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","cross-section-dir",1,iValue::NewIndex(this),iValue::_Absolute,iValue::NewInt(this),iAnimatorScript::CrossSectionDir,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","cross-section-position",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::CrossSectionPos,(unsigned char)4));

	this->CreateCommandWord(iOperation::New(this,"","volume-var",1,iValue::NewIndex(this),iValue::_Absolute,iValue::NewIndex(this),iAnimatorScript::VolumeVar,(unsigned char)4));

	this->CreateCommandWord(iOperation::New(this,"","particles-opacity",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::ParticlesOpacity,(unsigned char)4));

	this->CreateCommandWord(iOperation::New(this,"","marker-position",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewArray(this,3),iAnimatorScript::MarkerPosition,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","marker-size",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::MarkerSize,(unsigned char)4));
	this->CreateCommandWord(iOperation::New(this,"","marker-opacity",1,iValue::NewIndex(this),iValue::_RelativeAddMult,iValue::NewReal(this),iAnimatorScript::MarkerOpacity,(unsigned char)4));
}


iAnimatorScript::~iAnimatorScript()
{
	//
	// Delete TMP values
	//
	if(mCounter == 1)
	{
		valIS->Delete();
		valRS->Delete();
		valRP->Delete();
		valVA3->Delete();
	}
	mEmbeddedControlScript->Delete();
}


//
//  Access to line-by-line driver
//
void iAnimatorScript::Start()
{
	this->RunStart(true);
}


bool iAnimatorScript::Continue()
{
	return this->RunOneLine(false) && this->GetErrorStatus()->NoError();
}


void iAnimatorScript::Finish()
{
	this->RunStop(false);
}


//
//  Turn an unknown token into a string
//
//iValue* iAnimatorScript::TransformUnknownTokenToValue(const iString &s, bool acceptArrays, bool forceFloat)
//{
//	mErrorMessage.Clear();  // clean errors accumulated earlier
//	return iValue::New(this,"",iValue::_ByValue,s);
//}

//
//  Render scene
//
void iAnimatorScript::Render(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int i, v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	curScript->SetInRender(true);

	if(v == -1) v = iMath::_IntMax;
	me->GetErrorStatus()->Monitor(curAnimator->GetErrorStatus(),true);
	for(i=0; i<v && !me->CheckAbort(i,v,1) && me->GetErrorStatus()->NoError(); i++) 
	{
		if(!curAnimator->RenderImages()) me->GetErrorStatus()->Set("Animation finished.",-1);
	}

	curScript->SetInRender(false);
}

void iAnimatorScript::RenderAll(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int i, k, v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	curScript->SetInRender(true);

	for(i=0; i<v && !me->CheckAbort(i,v,2) && me->GetErrorStatus()->NoError(); i++) 
	{
		for(k=0; !me->CheckAbort(k,curControl->GetNumberOfViewModules(),1) && me->GetErrorStatus()->NoError() && k<curControl->GetNumberOfViewModules(); k++) if(k != curControl->GetCurrentViewModuleIndex())
		{
			me->GetErrorStatus()->Monitor(curControl->GetViewModule(k)->GetAnimator()->GetErrorStatus(),true);
			//
			//  Force other animators to continue, even if they are done
			//
			curControl->GetViewModule(k)->GetAnimator()->RenderImages(false);
		}
		if(me->GetErrorStatus()->NoError())
		{
			me->GetErrorStatus()->Monitor(curAnimator->GetErrorStatus(),true);
			if(!curAnimator->RenderImages()) me->GetErrorStatus()->Set("Animation finished.",-1);
		}
	}

	curScript->SetInRender(false);
}

void iAnimatorScript::MultiRender(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	//
	//  Did we Get an array?
	//
	if(v0->GetType() != iValue::_Value_VA)
	{
		me->GetErrorStatus()->Set("Value must be an array."); 
		return;
	}

	int n = static_cast<iValue_VA *>(v0)->GetSize();
	int *v = new int[n]; IERROR_ASSERT(v);
	if(v == 0)
	{
		me->GetErrorStatus()->Set("Unable to allocate memory."); 
		return;
	}

	if(!v0->GetValue(n,v)) 
	{
		me->GetErrorStatus()->Set("Invalid array value."); 
		delete [] v;
		return; 
	} 

	curScript->SetInRender(true);

	int i;
	for(i=0; i<n; i++) v[i]--;
	for(i=0; me->GetErrorStatus()->NoError() && !me->CheckAbort(i,n,3) && i<n; i++) if(v[i]>=0 && v[i]<curControl->GetNumberOfViewModules() && v[i]!=curControl->GetCurrentViewModuleIndex())
	{
		me->GetErrorStatus()->Monitor(curControl->GetViewModule(v[i])->GetAnimator()->GetErrorStatus(),true);
		curControl->GetViewModule(v[i])->GetAnimator()->RenderImages(false);
	}
	for(i=0; me->GetErrorStatus()->NoError() && !me->CheckAbort(i,n,3) && i<n; i++) if(v[i] == curControl->GetCurrentViewModuleIndex())
	{
		me->GetErrorStatus()->Monitor(curAnimator->GetErrorStatus(),true);
		if(!curAnimator->RenderImages()) me->GetErrorStatus()->Set("Animation finished.",-1);
		break;
	}
	delete [] v;

	curScript->SetInRender(false);
}

//
// Embed control script
//
void iAnimatorScript::ExecuteControlScript(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	if(curScript->mParent != 0) return;

	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	if(v.IsEmpty())
	{
		me->GetErrorStatus()->Set("A name of a file with the Control Script must be specified.");
	}
	else
	{
		iDirectory::ExpandFileName(v,curControl->GetShell()->GetEnvironment(_EnvironmentBase));

		iFile f(v);
		if(!f.Open(iFile::_Read,iFile::_Text))
		{
			me->GetErrorStatus()->Set("Control script file is not found.");
			return;
		}

		iString st, tmp;
		while(f.ReadLine(tmp)) st += tmp + "\n";
		if(st.IsEmpty())
		{
			me->GetErrorStatus()->Set("Control script file is empty or unreadable.");
			return;
		}

		f.Close();

		curScript->mAllowChildAccess = true;
		me->GetErrorStatus()->Monitor(curScript->mEmbeddedControlScript->GetErrorStatus(),true,"Control script: ");
		curScript->mEmbeddedControlScript->Run(v,false,false); // do not reset the script!!!
		curScript->mAllowChildAccess = false;
	}
}


void iAnimatorScript::EmbedControlScript(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	//
	//  Replace %% with \n
	//
	v.Replace("%%","\n");

	if(curScript->mEmbeddedControlScript != 0)
	{
		curScript->mAllowChildAccess = true;
		me->GetErrorStatus()->Monitor(curScript->mEmbeddedControlScript->GetErrorStatus(),true,"Control script: ");
		curScript->mEmbeddedControlScript->Run(v,false,false); // do not reset the script!!!
		curScript->mAllowChildAccess = false;
	}
}

//
//  Load Sets
//
void iAnimatorScript::Load(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	curReader->LoadRecord(v,0,curAnimator->GetDebugFlag()>1);
	if(curReader->GetErrorStatus()->NoError()) curAnimator->ResetCurrentFile();
}

//
//  reset Animator Settings
//
void iAnimatorScript::Reset(iScript *me)
{
	curAnimator->ResetState();
	curAnimator->Reset();
	curVisual->GetRenderTool()->GetCamera()->Reset();
}

//
//  Animator Settings
//
void iAnimatorScript::Style(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curAnimator->SetMode(v);
	curAnimator->Reset();
}

void iAnimatorScript::FramesPerFile(iScript *me, iValue *v0, iValue::Assignment rel)
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetNumberOfFrames());
	curAnimator->SetNumberOfFrames(v);
}

void iAnimatorScript::NumBlendedFrames(iScript *me, iValue *v0, iValue::Assignment rel)
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetNumberOfBlendedFrames());
	curAnimator->SetNumberOfBlendedFrames(v);
}

void iAnimatorScript::NumTransitionFrames(iScript *me, iValue *v0, iValue::Assignment rel)
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetNumberOfTransitionFrames());
	curAnimator->SetNumberOfTransitionFrames(v);
}

void iAnimatorScript::FlybySpeed(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetFlybySpeed());
	curAnimator->SetFlybySpeed(v);
}

void iAnimatorScript::RotationPhi(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetPhi());
	curAnimator->SetPhi(v);
}

void iAnimatorScript::RotationTheta(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetTheta());
	curAnimator->SetTheta(v);
}

void iAnimatorScript::RotationRoll(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetRoll());
	curAnimator->SetRoll(v);
}

void iAnimatorScript::Scale(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetZoom());
	curAnimator->SetZoom(v);
}

void iAnimatorScript::CrossSectionSpeed(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;
	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curAnimator->GetSlideSpeed());
	curAnimator->SetSlideSpeed(v);
}

//
//  variable limits
//
void iAnimatorScript::UniformScalarsLowerLimit(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	n--;

	if(n>=0 && n<curLimits(iUniformScalarsDataSubject::DataType())->GetNumVars())
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curLimits(iUniformScalarsDataSubject::DataType())->GetLowerLimit(n));
		curLimits(iUniformScalarsDataSubject::DataType())->SetLowerLimit(n,v);
	}
	else me->GetErrorStatus()->Set("Invalid variable index.");
}

void iAnimatorScript::UniformScalarsUpperLimit(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	n--;

	if(n>=0 && n<curLimits(iUniformScalarsDataSubject::DataType())->GetNumVars())
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curLimits(iUniformScalarsDataSubject::DataType())->GetUpperLimit(n));
		curLimits(iUniformScalarsDataSubject::DataType())->SetUpperLimit(n,v);
	}
	else me->GetErrorStatus()->Set("Invalid variable index.");
}

void iAnimatorScript::BasicParticlesLowerLimit(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	n--;

	if(n>=0 && n<curLimits(iBasicParticlesDataSubject::DataType())->GetNumVars())
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curLimits(iBasicParticlesDataSubject::DataType())->GetLowerLimit(n));
		curLimits(iBasicParticlesDataSubject::DataType())->SetLowerLimit(n,v);
	}
	else me->GetErrorStatus()->Set("Invalid attribute index.");
}

void iAnimatorScript::BasicParticlesUpperLimit(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	n--;

	if(n>=0 && n<curLimits(iBasicParticlesDataSubject::DataType())->GetNumVars())
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curLimits(iBasicParticlesDataSubject::DataType())->GetUpperLimit(n));
		curLimits(iBasicParticlesDataSubject::DataType())->SetUpperLimit(n,v);
	}
	else me->GetErrorStatus()->Set("Invalid attribute index.");
}

void iAnimatorScript::UniformVectorsUpperLimit(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;

	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curLimits(iUniformVectorsDataSubject::DataType())->GetUpperLimit(0));
	curLimits(iUniformVectorsDataSubject::DataType())->SetUpperLimit(0,v);
}

void iAnimatorScript::UniformTensorsUpperLimit(iScript *me, iValue *v0, iValue::Assignment rel)
{
	float v;

	IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curLimits(iUniformTensorsDataSubject::DataType())->GetUpperLimit(0));
	curLimits(iUniformTensorsDataSubject::DataType())->SetUpperLimit(0,v);
}

//
//  focal point & camera location
//
void iAnimatorScript::CameraFocus(iScript *me, iValue *v0, iValue::Assignment rel)
{
	double v[3];
 	double *p = curCamera->GetFocalPoint();

	IBASESCRIPT_UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		
	curCamera->SetFocalPoint(v);
	curCamera->OrthogonalizeViewUp();
}

void iAnimatorScript::CameraPosition(iScript *me, iValue *v0, iValue::Assignment rel)
{
	double v[3];
	double *p = curCamera->GetPosition();

	IBASESCRIPT_UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		
	curCamera->SetPosition(v);
	curCamera->OrthogonalizeViewUp();
}

void iAnimatorScript::CameraViewUp(iScript *me, iValue *v0, iValue::Assignment rel)
{
	double v[3];
	double *p = curCamera->GetViewUp();

	IBASESCRIPT_UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		
	double *dop = curCamera->GetDirectionOfProjection();
	double c[3];
	double vup[3];
	vup[0] = v[0]; vup[1] = v[1]; vup[2] = v[2];
	vtkMath::Cross(dop,vup,c);

	if((v[0]*v[0]+v[1]*v[1]+v[2]*v[2])>1.0e-8 && (c[0]*c[0]+c[1]*c[1]+c[2]*c[2])>1.0e-8)
	{
		curCamera->SetViewUp(v);
		curCamera->OrthogonalizeViewUp();
	}
}

void iAnimatorScript::CameraScale(iScript *me, iValue *v0, iValue::Assignment rel)
{
	double v;
    IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,curCamera->GetParallelScale());
	curCamera->SetParallelScale(v);
}

void iAnimatorScript::Projection(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curCamera->SetParallelProjection(!v?1:0);
}

//
//  Features
//
void iAnimatorScript::StopOnPath(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curAnimator->SetStopOnPath(v);
}

void iAnimatorScript::TimeLabel(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curVisual->ShowLabel(v); 
}

void iAnimatorScript::TimeLabelValue(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	float v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curVisual->UpdateLabel(&v); 
}

void iAnimatorScript::ColorBars(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curVisual->ShowColorBars(v); 
}

void iAnimatorScript::BoundingBox(iScript *me, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curVisual->ShowBoundingBox(v); 
}

void iAnimatorScript::TitlePageFile(iScript *me, iValue* v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	if(!curAnimator->SetTitlePageFile(v)) me->GetErrorStatus()->Set("Unable to load title page image.");
}


void iAnimatorScript::TitlePageNumFrames(iScript *me, iValue* v0, iValue::Assignment)
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curAnimator->SetNumberOfTitlePageFrames(v);
}


void iAnimatorScript::TitlePageNumBlendedFrames(iScript *me, iValue* v0, iValue::Assignment)
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curAnimator->SetNumberOfTitlePageBlendedFrames(v);
}

void iAnimatorScript::LogoFile(iScript *me, iValue* v0, iValue::Assignment)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	if(!curAnimator->SetLogoFile(v)) me->GetErrorStatus()->Set("Unable to load logo image.");
}

void iAnimatorScript::LogoOpacity(iScript *me, iValue* v0, iValue::Assignment)
{
	float v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curAnimator->SetLogoOpacity(v);
}

void iAnimatorScript::LogoPosition(iScript *me, iValue* v0, iValue::Assignment)
{
	int v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curAnimator->SetLogoPosition(v);
}

//
// Manupulation of iViewSubjects
//
void iAnimatorScript::SurfaceVar(iScript *me, iValue *n0, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int n;
	int v;
	iSurfaceViewSubject *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curObject(Surface); else s = indObject(Surface,n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
		s->SetIsoSurfaceVar(v-1);
	}
}

void iAnimatorScript::SurfaceLevel(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	iSurfaceViewSubject *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curObject(Surface); else s = indObject(Surface,n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,s->GetIsoSurfaceLevel());
		s->SetIsoSurfaceLevel(v);
	}
}

void iAnimatorScript::SurfaceOpacity(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	iSurfaceViewSubject *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curObject(Surface); else s = indObject(Surface,n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,s->GetOpacity(0));
		s->SetOpacity(0,v);
		s->SetOpacity(1,v);
	}
}

void iAnimatorScript::CrossSectionVar(iScript *me, iValue *n0, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int n;
	int v;
	iCrossSectionViewSubject *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curObject(CrossSection); else s = indObject(CrossSection,n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
		s->SetVar(v-1);
	}
}

void iAnimatorScript::CrossSectionDir(iScript *me, iValue *n0, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int n;
	int v;
	iCrossSectionViewSubject *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curObject(CrossSection); else s = indObject(CrossSection,n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
		s->SetDir(v);
	}
}

void iAnimatorScript::CrossSectionPos(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	double v;
	iCrossSectionViewSubject *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curObject(CrossSection); else s = indObject(CrossSection,n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,(double)s->GetLocation());
		s->SetLocation(v);
	}
}

void iAnimatorScript::VolumeVar(iScript *me, iValue *n0, iValue *v0, iValue::Assignment vtkNotUsed(rel))
{
	int n;
	int v;
	iVolumeViewSubject *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curObject(Volume); else s = indObject(Volume,n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
		s->SetVar(v-1);
	}
}

void iAnimatorScript::ParticlesOpacity(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	iParticleGroup *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVisual->GetParticlesViewSubject()->GetCurrentGroup(); else s = curVisual->GetParticlesViewSubject()->GetGroup(n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,s->GetOpacity(0));
		s->SetOpacity(0,v);
	}
}

void iAnimatorScript::MarkerPosition(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	iMarker *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVisual->GetMarker(); else s = curVisual->GetMarker(n-1);
	if(s != 0)
	{
		double v[3];
		const double *p = s->GetPosition();
		IBASESCRIPT_UPDATE_POSITIONVALUE(v0,rel,3,v,p);
		iPosition tmp(s->GetViewModule());
		tmp = v;
		s->SetPosition(tmp);
	}
}

void iAnimatorScript::MarkerSize(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	double v;
	iMarker *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVisual->GetMarker(); else s = curVisual->GetMarker(n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,(double)s->GetSize());
		s->SetSize(v);
	}
}

void iAnimatorScript::MarkerOpacity(iScript *me, iValue *n0, iValue *v0, iValue::Assignment rel)
{
	int n;
	float v;
	iMarker *s;
	IBASESCRIPT_UPDATE_VALUE_ABS(n0,n);
	if(n <= 0) s = curVisual->GetMarker(); else s = curVisual->GetMarker(n-1);
	if(s != 0)
	{
		IBASESCRIPT_UPDATE_VALUE_REL(v0,rel,v,s->GetOpacity(0));
		s->SetOpacity(0,v);
	}
}


//
//  Global variable function
//
iValue* iAnimatorScript::GetMarkerPosition(iScript *me, int i)
{
	int j;

	if(i>=0 && i<=curVisual->GetMarkerFamily()->GetMaxMemberIndex())
	{
		const double *p = curVisual->GetMarkerFamily()->GetMember(i)->GetPosition();
		for(j=0; j<3; j++) static_cast<iValue_RS*>(static_cast<iValue_VA*>(valVA3)->GetComponent(j+1))->SetNumericValue(p[j]);
		return valVA3;
	}
	else
	{
		return 0;
	}
}


//
//  File loading functions - for batch operations
//
void iAnimatorScript::LoadFileByType(iScript *me, iValue* v0, const iDataType &type)
{
	iString v;
	IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);

	//
	//  If the string quoted, remove quotes
	//
	if(v[0]=='\'' && v[v.Length()-1]=='\'') v = v.Part(1,v.Length()-2);

	iDirectory::ExpandFileName(v,type.GetEnvironment(curControl->GetShell()));
	me->GetErrorStatus()->Monitor(curReader->GetErrorStatus(),true);
	curReader->LoadFile(type,v);
}


void iAnimatorScript::LoadUniformScalarsFile(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	LoadFileByType(me,v0,iUniformScalarsDataSubject::DataType());
}


void iAnimatorScript::LoadBasicParticlesFile(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	LoadFileByType(me,v0,iBasicParticlesDataSubject::DataType());
}


void iAnimatorScript::LoadUniformVectorsFile(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	LoadFileByType(me,v0,iUniformVectorsDataSubject::DataType());
}


void iAnimatorScript::LoadUniformTensorsFile(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	LoadFileByType(me,v0,iUniformTensorsDataSubject::DataType());
}


//
//  Object showing/hiding functions - for batch operations
//
void iAnimatorScript::ShowSurface(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curObjectFamily(Surface)->Show(v); 
}


void iAnimatorScript::ShowCrossSection(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curObjectFamily(CrossSection)->Show(v); 
}


void iAnimatorScript::ShowVolume(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curObjectFamily(Volume)->Show(v); 
}


void iAnimatorScript::ShowParticles(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curObjectFamily(Particles)->Show(v); 
}


void iAnimatorScript::ShowVectorField(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curObjectFamily(VectorField)->Show(v); 
}


void iAnimatorScript::ShowTensorField(iScript *me, iValue* v0, iValue::Assignment vtkNotUsed(rel))
{
	bool v;
    IBASESCRIPT_UPDATE_VALUE_ABS(v0,v);
	curObjectFamily(TensorField)->Show(v); 
}

