/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "igenericparticlesdatasubject.h"


#include "ibuffer.h"
#include "idata.h"
#include "idatalimits.h"
#include "ierror.h"
#include "ierrorstatus.h"
#include "ieventobserver.h"
#include "ihistogrammaker.h"
#include "iparallel.h"
#include "iparallelmanager.h"
#include "iparallelworker.h"
#include "iprobefilter.h"
#include "iviewmodule.h"

#include <vtkCellArray.h>
#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkPolyData.h>

//
//  Templates
//
#include "ibuffertemplate.h"


//
//  Helper classes for parallel execution & templates
//
class iGenericParticlesDataHelper : protected iParallelWorker
{

public:

	iGenericParticlesDataHelper(iGenericParticlesDataSubject *subject);

	void ShiftData(bool paf, int dim, long n, bool per[3], float dr, float *pf, double *pd);
	void FindRange(int dim, long n, float *pf, float *amin, float *amax);

protected:

	virtual int ExecuteStep(int step, iParallel::ProcessorInfo &p);

	iGenericParticlesDataSubject *mSubject;

	iBuffer<float> mMin, mMax;

	bool *mPeriodic;
	float *mFarr;
	double *mDarr;

	int mDim, mNumProcs;
	bool mPaf;
	long mTot;
	float mDr;
};


//
//  Main class
//
iGenericParticlesDataSubject::iGenericParticlesDataSubject(iDataReader *r, const iDataType &type) : iPrimaryDataSubject(r,type)
{
	mHaveNormals = false;

	mDownsampleMode = 0;
	mDownsampleFactor = 1;

	mHelper = new iGenericParticlesDataHelper(this); IERROR_ASSERT(mHelper);
}


iGenericParticlesDataSubject::~iGenericParticlesDataSubject()
{
	delete mHelper;
}


void iGenericParticlesDataSubject::SetDownsampleMode(int v)
{
	if(v>=0 && v<=3 && v!=mDownsampleMode)
	{
		mDownsampleMode = v;
	}
}


void iGenericParticlesDataSubject::SetDownsampleFactor(int v)
{
	if(v > 0)
	{
		mDownsampleFactor = v;
	}
}


void iGenericParticlesDataSubject::FinalizePolyData(vtkPolyData *data)
{
	if(data==0 || data->GetPoints()==0) return;

	int i;
	vtkIdType l, loff, ntot = data->GetNumberOfPoints();

	//
	//  Boundary condition
	//
	for(i=0; i<3; i++) this->SetDirectionPeriodic(i,this->IsBoxPeriodic());

	if(this->IsBoxPeriodic())
	{
		if(data->GetPoints()->GetDataType() == VTK_FLOAT)
		{
			float *xptrF = (float *)data->GetPoints()->GetVoidPointer(0);
			for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i))
			{
				for(l=0; l<ntot; l++)
				{
					loff = i + 3*l;
					xptrF[loff] -= 2.0*floor(0.5*(1.0+xptrF[loff]));
				}
			}
		}
		else if(data->GetPoints()->GetDataType() == VTK_DOUBLE)
		{
			double *xptrD = (double *)data->GetPoints()->GetVoidPointer(0);
			for(i=0; i<3; i++) if(this->IsDirectionPeriodic(i))
			{
				for(l=0; l<ntot; l++)
				{
					loff = i + 3*l;
					xptrD[loff] -= 2.0*floor(0.5*(1.0+xptrD[loff]));
				}
			}
		}
		else
		{
			this->GetErrorStatus()->Set("Internal bug: ivalid points data type.");
		}
	}

	if(mHaveNormals)
	{
		vtkFloatArray *newNormals;
		newNormals = vtkFloatArray::New(); IERROR_ASSERT(newNormals);
		newNormals->SetNumberOfComponents(3);
		// Allocates and Sets MaxId
		newNormals->SetNumberOfTuples(ntot);
		float *p = (float *)newNormals->GetVoidPointer(0);
		if(p != 0)
		{
			for(l=0; l<ntot; l++)
			{
				p[3*l+0] = p[3*l+1] = 0.0f;
				p[3*l+2] = 1.0f;
			}
			data->GetPointData()->SetNormals(newNormals);
		}
		newNormals->Delete();
	}

	//
	//  Check for overflow
	//
	if(data->GetPointData()->GetScalars() != 0)
	{
		float *p = (float *)data->GetPointData()->GetScalars()->GetVoidPointer(0);
		int natt = data->GetPointData()->GetScalars()->GetNumberOfComponents();
		if(p != 0)
		{
			for(l=0; l<ntot; l++)
			{
				for(i=0; i<natt; i++)
				{
					if(p[natt*l+i] < -iMath::_LargeFloat)
					{
						p[natt*l+i] = -iMath::_LargeFloat;
						mOverflow = true;
					}
					if(p[natt*l+i] >  iMath::_LargeFloat)
					{
						p[natt*l+i] =  iMath::_LargeFloat;
						mOverflow = true;
					}
				}
			}
		}
	}
}


void iGenericParticlesDataSubject::ShiftPolyData(vtkPolyData *data, int d, double dx)
{
	if(data==0 || data->GetPoints()==0 || d<0 || d>2) return;

	long n = data->GetPoints()->GetNumberOfPoints();
	float *pf = (float *)data->GetPoints()->GetVoidPointer(0);
	double *pd = (double *)data->GetPoints()->GetVoidPointer(0);
		
	if(pf!=0 && pd!=0)
	{
		mHelper->ShiftData(data->GetPoints()->GetDataType()==VTK_FLOAT,d,n,mPeriodic,dx,pf,pd);
		data->Modified();
	}
}


vtkIdType iGenericParticlesDataSubject::PrepareMask(long ntot0, int df, vtkIdType offm, long offt)
{
	if(df < 1) df = 1;

	wMask.NumTotal = ntot0;
	wMask.NumMasked = (ntot0+df-1)/df;
	wMask.NumUnmasked = wMask.NumTotal - wMask.NumMasked;
	wMask.OffsetTotal = offt;
	wMask.OffsetMasked = offm;
	wMask.DownsampleFactor = df;
	wMask.RandomThreshold = 1.0/df;

	return wMask.NumMasked;
}


bool iGenericParticlesDataSubject::ReadFortranRecordWithMask(iFile &F, long ntot0, int comp, vtkFloatArray *array, float updateStart, float updateDuration, float scale, float offset)
{
	return this->ReadFortranRecordWithMaskTemplate(F,ntot0,comp,array,updateStart,updateDuration,scale,offset);
}


bool iGenericParticlesDataSubject::ReadFortranRecordWithMask(iFile &F, long ntot0, int comp, vtkDoubleArray *array, float updateStart, float updateDuration, double scale, double offset)
{
	return this->ReadFortranRecordWithMaskTemplate(F,ntot0,comp,array,updateStart,updateDuration,scale,offset);
}


template<class T>
bool iGenericParticlesDataSubject::ReadFortranRecordWithMaskTemplate(iFile &F, long ntot0, int comp, vtkDataArray *array, float updateStart, float updateDuration, T scale, T offset)
{
	int j;
	long lrec1, lrec2;
	vtkIdType l, idm, idt;

	//
	//  Is component valid?
	//
	int nComp;
	if(array==0 || comp<0 || comp>=(nComp = array->GetNumberOfComponents()))
	{
		this->GetErrorStatus()->Set("Invalid call to iGenericParticlesDataSubject::ReadFortranRecordWithMask.");
		return false;
	}

	//
	//  Read the header
	//
	if(!this->ReadFortranHeaderFooter(F,lrec1) || lrec1!=sizeof(T)*ntot0)
	{
		this->GetErrorStatus()->Set("Corrupted data.");
		return false;
	}

	//
	//  Direct access to data
	//
	T *ptr = (T *)array->GetVoidPointer(0) + comp;
	if(ptr == 0)
	{
		this->GetErrorStatus()->Set("Not enough memory to create the data.");
		return false;
	}

	//
	// Decide how to read
	//
	vtkIdType ntot = array->GetNumberOfTuples();
	vtkIdType lpiece1, lpiece = ntot0/1000;
	if(lpiece < 1000) lpiece = 1000;
	int npieces = (ntot0+lpiece-1)/lpiece;

	//
	//  Create tmp array
	//
	T *d = new T[lpiece];
	if(d == 0) 
	{ 
		this->GetErrorStatus()->Set("Not enough memory to create the data.");
		return false;
	}

	//
	//  parameters for the Progress Bar
	//
	updateDuration /= npieces;

	//
	//  Read piece by piece
	//
	idm = idt = 0;
	for(j=0; j<npieces; j++)
	{
		if(j < npieces-1)
		{
			lpiece1 = lpiece;
		}
		else
		{
			//
			//  Correct for the last record
			//
			lpiece1 = ntot0 - j*lpiece;
		}
		if(!this->ReadBlock(F,d,lpiece1,updateStart+j*updateDuration,updateDuration))
		{
			this->GetErrorStatus()->Set("Corrupted data.");
			return false;
		}
		if(this->GetObserver()->IsAborted())
		{
			this->GetErrorStatus()->SetAbort();
			return false;
		}
		for(l=0; l<lpiece1; l++, idt++) if(this->IsMasked(idm,idt))
		{
#ifdef I_CHECK1
			if(idm>=ntot || idt>=ntot0)
			{
				IERROR_REPORT_BUG;
			}
#endif
			ptr[nComp*idm] = d[l];
			idm++;
		}
	}

	//
	//  Read the footer
	//
	if(!this->ReadFortranHeaderFooter(F,lrec2) || lrec1!=lrec2)
	{
		this->GetErrorStatus()->Set("Corrupted data.");
		return false;
	}

	delete [] d;

	//
	//  Do we need to scale?
	//
	if(fabs(scale) > iMath::_DoubleMin)
	{
		if(fabs(offset) > iMath::_DoubleMin)
		{
			//
			//  Scale as positions
			//
			for(l=0; l<ntot; l++)
			{
				ptr[nComp*l] = -1.0 + scale*(ptr[nComp*l]-offset);
			}
		}
		else
		{
			//
			//  Just scale
			//
			for(l=0; l<ntot; l++)
			{
				ptr[nComp*l] *= scale;
			}
		}
	}

	if(this->GetObserver()->IsAborted())
	{
		this->GetErrorStatus()->SetAbort();
		return false;
	}

	return true;
}


iProbeFilter* iGenericParticlesDataSubject::CreateProbeFilter(iViewSubject *vo) const
{
	return iProbeFilter::New(vo);
}


iHistogramMaker* iGenericParticlesDataSubject::CreateHistogramMaker() const
{
	return iHistogramMaker::New(this->GetViewModule());
}


void iGenericParticlesDataSubject::CreateArrays(vtkIdType ntot, int natt, bool paf, vtkPolyData *data, vtkPoints* &newPoints, vtkCellArray* &newVerts, vtkFloatArray* &newScalars)
{
	vtkPoints *oldPoints = data->GetPoints();
	vtkCellArray *oldVerts = data->GetVerts();
	vtkFloatArray *oldScalars = iRequiredCast<vtkFloatArray>(data->GetPointData()->GetScalars());

	int type = paf ? VTK_FLOAT : VTK_DOUBLE;

	if(oldPoints==0 || oldPoints->GetNumberOfPoints()!=ntot || oldPoints->GetDataType()!=type)
	{
		data->SetPoints(0);
		newPoints = vtkPoints::New(type); IERROR_ASSERT(newPoints);
		// Allocates and Sets MaxId
		newPoints->SetNumberOfPoints(ntot);
	}
	else
	{
		newPoints = oldPoints;
		newPoints->Register(0);
	}

	if(oldVerts==0 || oldVerts->GetNumberOfCells()!=ntot)
	{
		data->SetVerts(0);
		newVerts = vtkCellArray::New(); IERROR_ASSERT(newVerts);
		// This allocates but does not Set Max Id
		newVerts->Allocate(newVerts->EstimateSize(ntot,1));
		vtkIdType l;
		for(l=0; l<ntot; l++)
		{
			newVerts->InsertNextCell(1);
			newVerts->InsertCellPoint(l);
		}
	}
	else
	{
		newVerts = oldVerts;
		newVerts->Register(0);
	}

	if(oldScalars==0 || oldScalars->GetNumberOfTuples()!=ntot || oldScalars->GetNumberOfComponents()!=natt)
	{
		data->GetPointData()->SetScalars(0);
		newScalars = vtkFloatArray::New(); IERROR_ASSERT(newScalars);
		newScalars->SetNumberOfComponents(natt);
		newScalars->SetNumberOfTuples(ntot);
	}
	else
	{
		newScalars = oldScalars;
		newScalars->Register(0);
	}
}

	
void iGenericParticlesDataSubject::AttachArrays(vtkPolyData *data, vtkPoints *newPoints, vtkCellArray *newVerts, vtkFloatArray *newScalars)
{
	data->SetPoints(newPoints);
	newPoints->Delete();

	data->SetVerts(newVerts);
	newVerts->Delete();

	data->GetPointData()->SetScalars(newScalars);
	newScalars->Delete();

	//
	//  Attribute limits.
	//
	if(mAdjustableLimits)
	{
		vtkFloatArray *att = iRequiredCast<vtkFloatArray>(data->GetPointData()->GetScalars());

		if(att->GetNumberOfComponents() == this->GetLimits()->GetNumVars()) // just in case
		{
			iBuffer<float> aMin, aMax;
			aMin.Extend(att->GetNumberOfComponents());
			aMax.Extend(att->GetNumberOfComponents());

			mHelper->FindRange(att->GetNumberOfComponents(),att->GetNumberOfTuples(),att->GetPointer(0),aMin,aMax);

			int i;
			for(i=0; i<att->GetNumberOfComponents(); i++)
			{
				this->GetLimits()->SetMin(i,aMin[i]);
				this->GetLimits()->SetMax(i,aMax[i]);
			}
		}
	}
}


//
//  Helper class
//
iGenericParticlesDataHelper::iGenericParticlesDataHelper(iGenericParticlesDataSubject *subject) : iParallelWorker(subject->GetViewModule()->GetParallelManager())
{
	mSubject = subject;
}


void iGenericParticlesDataHelper::ShiftData(bool paf, int dim, long n, bool per[3], float dr, float *pf, double *pd)
{
	mPaf = paf;
	mDim = dim;
	mTot = n;
	mDr = dr;
	mFarr = pf;
	mDarr = pd;
	mPeriodic = per;

	this->ParallelExecute(1);
}


void iGenericParticlesDataHelper::FindRange(int dim, long tot, float *pf, float *fMin, float *fMax)
{
	mDim = dim;
	mTot = tot;
	mFarr = pf;

	mNumProcs = this->GetManager()->GetNumberOfProcessors();
	mMin.Extend(mDim*mNumProcs);
	mMax.Extend(mDim*mNumProcs);
	
	this->ParallelExecute(2);

	int i, n;
	for(n=0; n<dim; n++)
	{
		fMin[n] = mMin[n];
		fMax[n] = mMax[n];
		for(i=1; i<mNumProcs; i++)
		{
			if(fMin[n] > mMin[n+dim*i]) fMin[n] = mMin[n+dim*i];
			if(fMax[n] < mMax[n+dim*i]) fMax[n] = mMax[n+dim*i];
		}
	}
}


int iGenericParticlesDataHelper::ExecuteStep(int step, iParallel::ProcessorInfo &p)
{
	long l, kstp, kbeg, kend;
	int d = mDim;

	iParallel::SplitRange(p,mTot,kbeg,kend,kstp);

	switch(step)
	{
	case 1:
		{
			float dr = mDr;

			if(mPaf)
			{
				float *x = mFarr + 3*kbeg;
				for(l=kbeg; l<kend; l++)
				{
					if(l%1000 == 0)
					{
						if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress((d+(float)(l-kbeg)/(kend-kbeg))/3.0);
						if(mSubject->GetObserver()->IsAborted()) return 2;
					}
					x[d] += 2.0*dr;
					if(mPeriodic[d])
					{
						if(x[d] >  1.0) x[d] -= 2.0;
						if(x[d] < -1.0) x[d] += 2.0;
					}
					x += 3;
				}
			}
			else
			{
				double *x = mDarr + 3*kbeg;
				for(l=kbeg; l<kend; l++)
				{
					if(l%1000 == 0)
					{
						if(this->IsMaster(p)) mSubject->GetObserver()->SetProgress((float)(l-kbeg)/(kend-kbeg));
						if(mSubject->GetObserver()->IsAborted()) return 2;
					}
					x[d] += 2.0*dr;
					if(mPeriodic[d])
					{
						if(x[d] >  1.0) x[d] -= 2.0;
						if(x[d] < -1.0) x[d] += 2.0;
					}
					x += 3;
				}
			}
			return 0;
		}
	case 2:
		{
			int j;
			float *f = mFarr + d*kbeg, *fmin = mMin + d*p.ThisProc, *fmax = mMax + d*p.ThisProc;
			for(j=0; j<d; j++) fmin[j] = fmax[j] = f[j];
			for(l=kbeg+1; l<kend; l++)
			{
				f += d;
				for(j=0; j<d; j++)
				{
					if(fmin[j] > f[j]) fmin[j] = f[j];
					if(fmax[j] < f[j]) fmax[j] = f[j];
				}
			}
			return 0;
		}
	default:
		{
			return 1;
		}
	}
}

