/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


//
//  The different kinds of data are distinguished by an associated class DataType. One instance
//  of DataType class describes a specific type of data loaded from a single data file (like 
//  UniformScalars, BasicParticles, etc). DataType(s) can be queried by string keywords. A
//  keyword must be in the form <qualifier>:<specifier>[,specifier...], where a specifier
//  is one of followig four strings: "ScalarField", "VectorFied", "TensorField", and 
//  "ParticleSet". A qualifier can be any string. Quailifiers are registered as new DataType(s)
//  are created, and are associated with an unsigned int index. Keywords and names of DataType(s)
//  have to be unique.
//  
//  A group of DataType(s) forms a DataInfo.
//

#ifndef IDATA_H
#define IDATA_H


#include "iarray.h"
#include "istring.h"


class iDataClass;
class iDataInfo;
class iDataType;


namespace iData
{
	const iDataType& None();
	const iDataInfo& Any();

	const iDataClass& ScalarField();
	const iDataClass& VectorField();
	const iDataClass& TensorField();
	const iDataClass& ParticleSet();

	const iDataType& UniformScalars();
	const iDataType& UniformVectors();
	const iDataType& UniformTensors();
	const iDataType& BasicParticles();
};


class iDataClass
{

	friend const iDataClass& iData::ScalarField();
	friend const iDataClass& iData::VectorField();
	friend const iDataClass& iData::TensorField();
	friend const iDataClass& iData::ParticleSet();

public:

	inline const iString& GetName() const { return mName; }
	inline const iString& GetTextName() const { return mTextName; }

	inline bool IsOfType(const iString &name) const { return name == mName; }

private:

	iDataClass(const iString& tname);
	~iDataClass();

	iString mName, mTextName;

	iDataClass(); // not implemented
	iDataClass(const iDataClass &kind); // not implemented
	void operator=(const iDataClass&); // not implemented
};


class iDataType
{

public:

	iDataType(int id, const iString& tname, const iString& sname, int priority, const iString &keywords, bool primary = true, unsigned int rank = 0U);
	~iDataType();

	bool operator==(const iDataType &type) const;
	bool operator!=(const iDataType &type) const;

	inline int GetId() const { return mId; }
	inline bool IsNull() const { return mId == 0; }
	inline unsigned int GetRank() const { return mRank; }

	inline int GetPriority() const { return mPriority; }
	inline const iString& GetName() const { return mName; }
	inline const iString& GetTextName() const { return mTextName; }
	inline const iString& GetShortName() const { return mShortName; }

	inline bool IsOfType(const iString &name) const { return name==mName || name==mShortName; }

	inline bool IsPrimary() const { return mIsPrimary; }

	bool IsOfClass(const iDataClass &c) const;

	static const iDataType& FindTypeById(int id);
	static const iDataType& FindTypeByName(const iString &name);

	static const iDataType& ReSpecifyType(const iDataType &type, const iString &specifier);
	static const iDataType& FindTypeByKeyword(const iString &keyword);
	static void FindAllTypesByKeywords(iDataInfo &info, const iString &keywords);

private:

	const int mId;
	const int mPriority;
	const bool mIsPrimary;
	const unsigned int mRank;
	const iString mTextName, mShortName, mKeywords;
	iString mName;

	iDataType(); // not implemented
	iDataType(const iDataType &type); // not implemented
	void operator=(const iDataType&); // not implemented
};


class iDataInfo
{

public:

	iDataInfo();
	iDataInfo(const iDataInfo &info);
	iDataInfo(const iDataType &type);
	~iDataInfo();

	void Erase();
	inline int Count() const { return mArr.Size(); }
	const iDataType& Type(int i) const;
	int Index(const iDataType &type) const;

	iDataInfo& operator=(const iDataInfo &info);
	iDataInfo& operator+=(const iDataType &type);
	iDataInfo& operator+=(const iDataInfo &info);
	iDataInfo& operator-=(const iDataType &type);

	bool Includes(const iDataType &type) const;

private:

	iSearchableArray<const iDataType*> mArr;
};


inline bool iDataType::operator==(const iDataType &type) const
{
	return mId == type.mId;
}


inline bool iDataType::operator!=(const iDataType &type) const
{
	return mId != type.mId;
}


inline const iDataInfo operator+(const iDataInfo &info, const iDataType &type)
{
	iDataInfo tmp(info);
	tmp += type;
	return tmp;
}


inline const iDataInfo operator+(const iDataInfo &info1, const iDataInfo &info2)
{
	iDataInfo tmp(info1);
	tmp += info2;
	return tmp;
}


inline const iDataInfo operator-(const iDataInfo &info, const iDataType &type)
{
	iDataInfo tmp(info);
	tmp -= type;
	return tmp;
}

#endif // IDATA_H
