/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  A ViewObject that represents a fixed set of solid surfaces (iActor)
//
#ifndef ISOLIDVIEWOBJECT_H
#define ISOLIDVIEWOBJECT_H


#include "iviewobject.h"


#include "iarray.h"
#include "icolor.h"

class iActor;


class iSolidViewObject : public iViewObject
{
	
public:
	
	static const iObjectType& Type();
	virtual const iObjectType& RealType();
	
	virtual void SetOpacity(int l, float o);
	float GetOpacity(int l) const;
	
	virtual void SetPalette(int l, int p);
	int GetPalette(int l) const;
	
	virtual void SetColor(int l, const iColor& c);
	const iColor& GetColor(int l) const;
	
	void UpdateLookupTables();

	//
	//  Inherited members
	//
	virtual void Reset();
	virtual void ShowClipPlane(bool s);
	virtual void ShowColorBars(bool s);
	virtual void UpdateMaterialProperties();
	virtual bool IsVisible() const;

protected:
	
	iSolidViewObject(iViewModule *vm, const iDataInfo &info, int numActors, unsigned int flags);
	virtual ~iSolidViewObject();

	virtual void ResetInput() = 0;
	virtual void ResetPipeline() = 0;

	virtual void SolidViewObjectPackStateBody(iString &s) const = 0;
	virtual void SolidViewObjectUnPackStateBody(const iString &s) = 0;

	virtual void ShowColorBarsBody(bool s) = 0;

	//
	//  It is more convenient to keep separate components in separate arrays
	//
	iArray<iActor*> mActors;
	iArray<float> mOpacities;
	iArray<int> mPalettes;
	iArray<iColor> mColors;

private:

	const bool mNoColor, mNoOpacity, mNoPalette;

	virtual void ViewObjectPackStateBody(iString &s) const;
	virtual void ViewObjectUnPackStateBody(const iString &s);

	static const iObjectKey& KeyOpacity();
	static const iObjectKey& KeyPalette();
	static const iObjectKey& KeyColor();
};


inline float iSolidViewObject::GetOpacity(int l) const
{ 
	if(l>=0 && l<mOpacities.Size()) return mOpacities[l]; else return 0.0; 
}
	

inline int iSolidViewObject::GetPalette(int l) const
{ 
	if(l>=0 && l<mPalettes.Size()) return mPalettes[l]; else return -1;
}
	

inline const iColor& iSolidViewObject::GetColor(int l) const
{
	static const iColor black;
	if(l>=0 && l<mColors.Size()) return mColors[l]; else return black;
}


//
//  Re-define ViewObject macros
//
#define ISOLIDVIEWOBJECT_DECLARE_INHERITED_KEYS \
	IVIEWOBJECT_DECLARE_INHERITED_KEYS; \
	static const iObjectKey& KeyOpacity(); \
	static const iObjectKey& KeyPalette(); \
	static const iObjectKey& KeyColor()


#define ISOLIDVIEWOBJECT_DECLARE_INHERITED_MEMBERS \
	virtual float GetMemorySize() const; \
	virtual bool CanBeShown() const; \
	protected: \
	virtual void ShowColorBarsBody(bool); \
	virtual void ResetInput(); \
	virtual void ResetPipeline(); \
	virtual void SolidViewObjectPackStateBody(iString &s) const; \
	virtual void SolidViewObjectUnPackStateBody(const iString &s); \
	virtual void ViewObjectSyncWithData(int info, void *data); \
	virtual void ShowBody(bool s); \
	virtual void FinishInitialization(); \
	public: \
	virtual const iObjectType& RealType()


#define IVIEWOBJECT_DEFINE_INHERITED_KEYS_COLOR(_type_,_num_) \
	IOBJECT_DEFINE_INHERITED_KEY(iSolidViewObject,_type_,Color,c,Color,_num_)

#define IVIEWOBJECT_DEFINE_INHERITED_KEYS_OPACITY(_type_,_num_) \
	IOBJECT_DEFINE_INHERITED_KEY(iSolidViewObject,_type_,Opacity,o,Float,_num_)

#define IVIEWOBJECT_DEFINE_INHERITED_KEYS_PALETTE(_type_,_num_) \
	IOBJECT_DEFINE_INHERITED_KEY(iSolidViewObject,_type_,Palette,p,OffsetInt,_num_)

#define IVIEWOBJECT_DEFINE_INHERITED_KEYS_ALL(_type_,_num_) \
	IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(_type_); \
	IVIEWOBJECT_DEFINE_INHERITED_KEYS_SHADING(_type_); \
	IVIEWOBJECT_DEFINE_INHERITED_KEYS_EXTENDING(_type_); \
	IVIEWOBJECT_DEFINE_INHERITED_KEYS_COLOR(_type_,_num_); \
	IVIEWOBJECT_DEFINE_INHERITED_KEYS_OPACITY(_type_,_num_); \
	IVIEWOBJECT_DEFINE_INHERITED_KEYS_PALETTE(_type_,_num_)

#endif // ISOLIDVIEWOBJECT_H

