/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iparticlegroup.h"


#include "icontrolmodule.h"
#include "idata.h"
#include "idatalimits.h"
#include "idatareader.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "iextensionfactory.h"
#include "ifunctionmapping.h"
#include "ihistogrammaker.h"
#include "ilookuptable.h"
#include "ipiecewisefunction.h"
#include "iparticleconnector.h"
#include "iparticledataconverter.h"
#include "iparticlegroupparallelpipeline.h"
#include "iparticlegroupserialpipeline.h"
#include "iparticlesviewobject.h"
#include "ipointglyph.h"
#include "ireplicatedactor.h"
#include "iviewmodule.h"

#include <vtkLookupTable.h>
#include <vtkPolyData.h>
#include <vtkProperty.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"

using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iParticleGroup,ParticleGroup,pg);

IOBJECT_DEFINE_KEY(iParticleGroup,AttributeToColor,ac,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticleGroup,AttributeSizeDirect,ad,Bool,1);
IOBJECT_DEFINE_KEY(iParticleGroup,AttributeSizeExtraFactor,ae,Float,1);
IOBJECT_DEFINE_KEY(iParticleGroup,AttributeToSize,as,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticleGroup,AttributeToConnect,atc,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticleGroup,AttributeToSeparate,ats,OffsetInt,1);
IOBJECT_DEFINE_KEY(iParticleGroup,StretchToColor,stc,Int,1);
IOBJECT_DEFINE_KEY(iParticleGroup,LowerLimitToColor,ltc,Float,1);
IOBJECT_DEFINE_KEY(iParticleGroup,UpperLimitToColor,utc,Float,1);
IOBJECT_DEFINE_KEY(iParticleGroup,StretchToSize,sts,Int,1);
IOBJECT_DEFINE_KEY(iParticleGroup,LowerLimitToSize,lts,Float,1);
IOBJECT_DEFINE_KEY(iParticleGroup,UpperLimitToSize,uts,Float,1);
IOBJECT_DEFINE_KEY(iParticleGroup,LineWidth,lw,Int,1);
IOBJECT_DEFINE_KEY(iParticleGroup,FixedSize,s,Float,1);
IOBJECT_DEFINE_KEY(iParticleGroup,SizeFunction,sf,Any,1);
IOBJECT_DEFINE_KEY(iParticleGroup,Type,t,Int,1);

//
//  Inherited keys
//
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COMMON(iParticleGroup);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_SHADING(iParticleGroup);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_REPLICATING(iParticleGroup);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_COLOR(iParticleGroup,1);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_OPACITY(iParticleGroup,1);
IVIEWOBJECT_DEFINE_INHERITED_KEYS_PALETTE(iParticleGroup,1);


namespace iParticleGroup_Private
{
	//
	//  Helper class
	//
	class SizeFunctionEventObserver : public vtkCommand
	{

	public:

		static SizeFunctionEventObserver* New(iPiecewiseFunction *fun)
		{
			return new SizeFunctionEventObserver(fun);
		}

		void AttachPipeline(iParticleGroupParallelPipeline *pipe)
		{
			mPipeline = pipe;
		}

		virtual void Execute(vtkObject *caller, unsigned long event, void *)
		{
			if(caller==mFunction && event==vtkCommand::ModifiedEvent && mPipeline!=0) mPipeline->UpdateSizeFunction();
		}

	protected:

		SizeFunctionEventObserver(iPiecewiseFunction *fun)
		{
			mFunction = fun; IERROR_ASSERT_NULL_POINTER(fun);
			mPipeline = 0;

			mFunction->AddObserver(vtkCommand::ModifiedEvent,this);
		}

		iPiecewiseFunction *mFunction;
		iParticleGroupParallelPipeline *mPipeline;
	};
}


using namespace iParticleGroup_Private;


iParticleGroup* iParticleGroup::New(iViewModule *vm)
{
	iParticleGroup *tmp = iPointerCast<iParticleGroup,iObject>(iExtensionFactory::CreateObject(Type(),vm));
	IERROR_ASSERT_NULL_POINTER(tmp);
	//
	//  Create pipeline (must be created after the ibject is fully created)
	//	
	tmp->mPipeline = tmp->CreateParallelPipeline(); IERROR_ASSERT_NULL_POINTER(tmp->mPipeline);
	tmp->mActors[0]->SetInput(tmp->mPipeline->GetOutput());
	tmp->mPipeline->SetInput(tmp->mConnector->GetOutput());

	tmp->SetStretchToColor(_StretchLog);
	tmp->SetStretchToSize(_StretchLog);

	return tmp;
}


iParticleGroup::iParticleGroup(iViewModule *vm, const iDataInfo &info) : iSolidViewObject(vm,info,1,0U)
{
	mName = "iParticleGroup";
	mObjectType = _ObjectTypeParticles;

	vtkPolyData *input = dynamic_cast<vtkPolyData*>(this->GetData()); // accepts any PolyData
	if(input == 0)
	{
		IERROR_REPORT_FATAL_ERROR("iParticleGroup is configured incorrectly.");
	}

	mParent = 0;
	mParentIndex = -1;

	mAttSize = mAttColor = -1;

	mAttSizeDirect = false;
	mAttSizeExtraFactor = 1.0;

	mType = _PointTypePoint;
	mSize = 1.0;
	mStretchToColor = _StretchLog;
	mLimitsToColor[0] = 0.1f;
	mLimitsToColor[1] = 10.0f;
	mKeepPoint = false;

	mLineWidth = 1;
	
	mSF = iPiecewiseFunction::New(); IERROR_ASSERT_NULL_POINTER(mSF);
	mSF->SetMinMax(0.0,1.0);
	mSF->MovePoint(0,0.0,1.0);
	mSF->MovePoint(1,1.0,1.0);
	
	mConnector = iParticleConnector::New(); IERROR_ASSERT_NULL_POINTER(mConnector);
	mHistogramMaker = iHistogramMaker::New(this->GetViewModule()); IERROR_ASSERT_NULL_POINTER(mHistogramMaker);
	mFunction = new iFunctionMapping(mSF,mHistogramMaker,0); IERROR_ASSERT_NULL_POINTER(mFunction);
	mSizeFunctionEventObserver = SizeFunctionEventObserver::New(mSF);

	mHistogramMaker->SetInput(mConnector->GetOutput(),-1);

	iAbortRenderEventObserver *obsAbortRender = this->GetViewModule()->GetAbortRenderEventObserver();
	mConnector->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

	mActors[0]->GetProperty()->SetPointSize(mSize);
	mActors[0]->GetProperty()->SetLineWidth(mLineWidth);

	this->SetColor(0,iColor(0,0,0));
}


iParticleGroup::~iParticleGroup()
{
	mSizeFunctionEventObserver->Delete();
	mPipeline->Delete(); // must be deleted before the mSF
	mConnector->Delete();
	mHistogramMaker->Delete();
	mSF->Delete();
	delete mFunction;
}


void iParticleGroup::FinishInitialization()
{
	iPointerCast<SizeFunctionEventObserver,vtkCommand>(mSizeFunctionEventObserver)->AttachPipeline(mPipeline);
}


void iParticleGroup::SetInput(vtkPolyData *input)
{
	mConnector->SetInput(input);
	mHistogramMaker->SetInput(input,mAttSize);
}


void iParticleGroup::ResetInput()
{
}


void iParticleGroup::ResetPipeline()
{
}


void iParticleGroup::SetLineWidth(int w)
{
	if(w>0 && w<100)
	{
		mLineWidth = w;
		mActors[0]->GetProperty()->SetLineWidth(mLineWidth);
		this->ClearCache();
	}
}


void iParticleGroup::SetShading(bool v)
{
	mShading = v;
	this->SetType(mType); 
	this->ClearCache();
}


void iParticleGroup::SetType(int t)
{ 
	mType = t;
	mPipeline->UpdateType();

	switch (mType) 
	{
	case _PointTypePoint:
	case _PointTypeTetrahydron: 
	case _PointTypeCluster:
	case _PointTypeGalaxy: 
		{ 
			mShading = false; 
			break; 
		}
	default:
		{
			mShading = true; 
		}
	}
	this->UpdateMaterialProperties();

	switch (mType)
	{
	case _PointTypeCluster:
	case _PointTypeGalaxy: 
		{ 
			mKeepPoint = true; 
			break; 
		}
	default:
		{
			mKeepPoint = false;
		}
	}
	if(mKeepPoint) mActors[0]->GetProperty()->SetPointSize(1.0); else mActors[0]->GetProperty()->SetPointSize(mSize); 
	this->ClearCache();
}


void iParticleGroup::SetAttributeToColor(int a)
{ 
	if(a>=0 && a<this->GetLimits()->GetNumVars()) mAttColor = a; else mAttColor = -1;
	if(mAttColor >= 0) 
	{
		mActors[0]->SetScalarVisibility(true); 
		mActors[0]->ColorByArrayComponent(0,mAttColor);
		this->SetLowerLimitToColor(mLimitsToColor[0]);
		this->SetUpperLimitToColor(mLimitsToColor[1]);
	} 
	else 
	{
		mActors[0]->SetScalarVisibility(false); 
	}
	this->UpdateColorLookupTable();
	this->ClearCache();
}


void iParticleGroup::SetAttributeToSize(int a)
{ 
	if(a>=0 && a<this->GetLimits()->GetNumVars())
	{
		mAttSize = a;
	}
	else mAttSize = -1;
	mPipeline->UpdateAttToSize();
	mFunction->AttachToLimits(this->GetLimits(),mAttSize);
	mHistogramMaker->SetInputComponent(mAttSize);
	this->ClearCache();
}


void iParticleGroup::SetAttributeSizeDirect(bool s)
{
	mAttSizeDirect = s;
	mPipeline->UpdateAttToSize();
	this->ClearCache();
}


void iParticleGroup::SetAttributeSizeExtraFactor(float f)
{
	if(f > 0.0)
	{
		mAttSizeExtraFactor = f;
		mPipeline->UpdateAttToSize();
		this->ClearCache();
	}
}


void iParticleGroup::SetFixedSize(float s)
{ 
	if(s>0.0 && s<1000.1)
	{
		mSize = s;
		if(mKeepPoint) mActors[0]->GetProperty()->SetPointSize(1.0); else mActors[0]->GetProperty()->SetPointSize(mSize); 
		mPipeline->UpdateFixedSize();
		this->ClearCache();
	}
}


void iParticleGroup::SetStretchToColor(int s)
{
	mStretchToColor = s;
	this->UpdateColorLookupTable();
	this->ClearCache();
}


void iParticleGroup::SetLowerLimitToColor(float s)
{
	if(mAttColor >= 0)
	{
		this->GetLimits()->UpdateLowerLimit(s,mAttColor,mLimitsToColor[0],mLimitsToColor[1]);
		this->UpdateColorLookupTable();
		this->ClearCache();
	}
}


void iParticleGroup::SetUpperLimitToColor(float s)
{
	if(mAttColor >= 0)
	{
		this->GetLimits()->UpdateUpperLimit(s,mAttColor,mLimitsToColor[0],mLimitsToColor[1]);
		this->UpdateColorLookupTable();
		this->ClearCache();
	}
}


void iParticleGroup::UpdateColorLookupTable()
{
	if(mAttColor >= 0) 
	{
		mActors[0]->SetScalarRange(mLimitsToColor[0],mLimitsToColor[1]);
		mActors[0]->GetLookupTable()->SetStretch(mStretchToColor);
		this->ClearCache();
	}
}


void iParticleGroup::SetStretchToSize(int s)
{ 
	if(mAttSize>=0 && mAttSize<this->GetLimits()->GetNumVars())
	{
		this->GetLimits()->SetStretch(mAttSize,s);
		mPipeline->UpdateAttToSize();
		this->ClearCache();
	}
}


int iParticleGroup::GetStretchToSize() const
{ 
	if(mAttSize>=0 && mAttSize<this->GetLimits()->GetNumVars())
	{
		return this->GetLimits()->GetStretch(mAttSize);
	}
	else
	{
		return _StretchLin;
	}
}


void iParticleGroup::SetLowerLimitToSize(float s)
{
	if(mAttSize>=0 && mAttSize<this->GetLimits()->GetNumVars())
	{
		this->GetLimits()->SetLowerLimit(mAttSize,s);
		mPipeline->UpdateAttToSize();
		this->ClearCache();
	}
}


float iParticleGroup::GetLowerLimitToSize() const
{
	if(mAttSize>=0 && mAttSize<this->GetLimits()->GetNumVars())
	{
		return this->GetLimits()->GetLowerLimit(mAttSize);
	}
	else
	{
		return iMath::_FloatMin;
	}
}


void iParticleGroup::SetUpperLimitToSize(float s)
{
	if(mAttSize>=0 && mAttSize<this->GetLimits()->GetNumVars())
	{
		this->GetLimits()->SetUpperLimit(mAttSize,s);
		mPipeline->UpdateAttToSize();
		this->ClearCache();
	}
}


float iParticleGroup::GetUpperLimitToSize() const
{
	if(mAttSize>=0 && mAttSize<this->GetLimits()->GetNumVars())
	{
		return this->GetLimits()->GetUpperLimit(mAttSize);
	}
	else
	{
		return iMath::_FloatMax;
	}
}


void iParticleGroup::ShowBody(bool show)
{
	if(show)
	{
		this->SetStretchToColor(mStretchToColor);
		mActors[0]->VisibilityOn();
	} 
	else 
	{
		mActors[0]->VisibilityOff();
	}
}


float iParticleGroup::GetMemorySize() const
{
	float s = 0.0;
	s += mPipeline->GetMemorySize();
	s += mConnector->GetOutput()->GetActualMemorySize();
	return s;
}


//
//  Two functions used in saving/restoring the state and in creating new instances with - ask the parent to help
//
void iParticleGroup::SolidViewObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyType(),mType);
	this->PackValue(s,KeyAttributeToColor(),mAttColor);
	this->PackValue(s,KeyAttributeToSize(),mAttSize);
	this->PackValue(s,KeyFixedSize(),mSize);
	this->PackValue(s,KeyStretchToColor(),mStretchToColor);
	this->PackValue(s,KeyLowerLimitToColor(),mLimitsToColor[0]);
	this->PackValue(s,KeyUpperLimitToColor(),mLimitsToColor[1]);
	this->PackValue(s,KeyStretchToSize(),this->GetStretchToSize());
	this->PackValue(s,KeyLowerLimitToSize(),this->GetLowerLimitToSize());
	this->PackValue(s,KeyUpperLimitToSize(),this->GetUpperLimitToSize());
	this->PackValue(s,KeyAttributeSizeDirect(),mAttSizeDirect);
	this->PackValue(s,KeyAttributeSizeExtraFactor(),mAttSizeExtraFactor);
	this->PackValue(s,KeyLineWidth(),mLineWidth);

	this->PackValue(s,KeyAttributeToConnect(),this->GetAttributeToConnect());
	this->PackValue(s,KeyAttributeToSeparate(),this->GetAttributeToSeparate());

	this->PackValuePiecewiseFunction(s,KeySizeFunction(),mSF);
}


void iParticleGroup::SolidViewObjectUnPackStateBody(const iString &s)
{
	int i; bool b; float f;

	if(this->UnPackValue(s,KeyType(),i)) this->SetType(i);
	if(this->UnPackValue(s,KeyAttributeToColor(),i)) this->SetAttributeToColor(i);
	if(this->UnPackValue(s,KeyAttributeToSize(),i)) this->SetAttributeToSize(i);
	if(this->UnPackValue(s,KeyFixedSize(),f)) this->SetFixedSize(f);
	if(this->UnPackValue(s,KeyStretchToColor(),i)) this->SetStretchToColor(i);
	if(this->UnPackValue(s,KeyLowerLimitToColor(),f)) this->SetLowerLimitToColor(f);
	if(this->UnPackValue(s,KeyUpperLimitToColor(),f)) this->SetUpperLimitToColor(f);
	if(this->UnPackValue(s,KeyStretchToSize(),i)) this->SetStretchToSize(i);
	if(this->UnPackValue(s,KeyLowerLimitToSize(),f)) this->SetLowerLimitToSize(f);
	if(this->UnPackValue(s,KeyUpperLimitToSize(),f)) this->SetUpperLimitToSize(f);
	if(this->UnPackValue(s,KeyAttributeSizeDirect(),b)) this->SetAttributeSizeDirect(b);
	if(this->UnPackValue(s,KeyAttributeSizeExtraFactor(),f)) this->SetAttributeSizeExtraFactor(f);
	if(this->UnPackValue(s,KeyLineWidth(),i)) this->SetLineWidth(i);
	if(this->UnPackValue(s,KeyAttributeToConnect(),i)) this->SetAttributeToConnect(i);
	if(this->UnPackValue(s,KeyAttributeToSeparate(),i)) this->SetAttributeToSeparate(i);
	if(this->UnPackValuePiecewiseFunction(s,KeySizeFunction(),mSF)) this->ClearCache(); // this unpacks into the appropriate function automatically.
}


void iParticleGroup::SetAttributeToConnect(int a)
{
	mConnector->SetAttributeToConnect(a);
	this->ClearCache();
}


int iParticleGroup::GetAttributeToConnect() const
{
	return mConnector->GetAttributeToConnect();
}


void iParticleGroup::SetAttributeToSeparate(int a)
{
	mConnector->SetAttributeToSeparate(a);
	this->ClearCache();
}


int iParticleGroup::GetAttributeToSeparate() const
{
	return mConnector->GetAttributeToSeparate();
}


void iParticleGroup::ShowColorBarsBody(bool)
{
}


iParticleGroupParallelPipeline* iParticleGroup::CreateParallelPipeline()
{
	iParticleGroupParallelPipeline *tmp = new iParticleGroupParallelPipeline(this); IERROR_ASSERT_NULL_POINTER(tmp);
	tmp->FinishInitialization();
	return tmp;
}


iParticleGroupSerialPipeline* iParticleGroup::CreateSerialPipeline()
{
	return new iParticleGroupSerialPipeline(this);
}


bool iParticleGroup::CanBeShown() const
{
	return true;
}


void iParticleGroup::ViewObjectSyncWithData(int n, void *)
{
	if(mAttColor > this->GetLimits()->GetNumVars()) this->SetAttributeToColor(-1);
	if(mAttSize > this->GetLimits()->GetNumVars()) this->SetAttributeToSize(-1);
	if(n>0 && n==mAttColor)
	{
		this->SetLowerLimitToColor(mLimitsToColor[0]);
		this->SetUpperLimitToColor(mLimitsToColor[1]);
	}
	if(n>0 && n==mAttSize)
	{
		mPipeline->UpdateAttToSize();
	}
	this->UpdateColorLookupTable();

	mFunction->AttachToLimits(this->GetLimits(),mAttSize);
}

