/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "ifieldviewobject.h"

#include "icolorbars.h"
#include "idata.h"
#include "idatareader.h"
#include "idatalimits.h"
#include "igriddataconverter.h"
#include "ireplicatedactor.h"
#include "iviewmodule.h"

#include <vtkStructuredPoints.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"


using namespace iParameter;


IVIEWOBJECT_DEFINE_TYPE(iFieldViewObject,FieldViewObject,-fvo);

IOBJECT_DEFINE_KEY(iFieldViewObject,ColorBy,cb,OffsetInt,1);
IOBJECT_DEFINE_KEY(iFieldViewObject,GlyphSampleRate,gr,Int,1);
IOBJECT_DEFINE_KEY(iFieldViewObject,ConnectedToScalars,cs,Bool,1);
IOBJECT_DEFINE_KEY(iFieldViewObject,DataOffset,-off,OffsetInt,1);

IOBJECT_DEFINE_DISTANCE_KEY(iFieldViewObject,GlyphSize,gs);


//
// iFieldViewObject class
//
iFieldViewObject::iFieldViewObject(iViewModule *vm, const iDataInfo &info, int numActors, unsigned int rank, unsigned int flags) : iSolidViewObject(vm,info,numActors,flags), mGlyphSize(vm,true)
{
	mName = "iFieldViewObject";

	if(rank==0U || rank>2U)
	{
		IERROR_REPORT_FATAL_ERROR("Incorrect rank for iFieldViewObject.");
	}

	mColorBy = -1;
	mGlyphSize = 0.1;
	mGlyphSampleRate = 16;
	mColorByOffset = 0;

	//
	//  Do VTK stuff
	//	
	mDataConverter = iGridDataConverter::New(this); IERROR_ASSERT_NULL_POINTER(mDataConverter);
	mDataConverter->SetDataRank(rank);
}


iFieldViewObject::~iFieldViewObject()
{
	mDataConverter->Delete();
}


void iFieldViewObject::ResetInput()
{
	mDataConverter->SetScalarInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformScalars())));
	switch(mDataConverter->GetDataRank())
	{
	case 1U:
		{
			mDataConverter->SetVectorInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformVectors())));
			break;
		}
	case 2U:
		{
			mDataConverter->SetTensorInput(iPointerCast<vtkImageData,vtkDataSet>(this->GetViewModule()->GetReader()->GetOutput(iDataType::UniformTensors())));
			break;
		}
	}
	this->ResetPipelineInput(mDataConverter->GetOutput());
}


void iFieldViewObject::SetGlyphSize(const iDistance &v)
{ 
	if(v < 1.0e5)
	{
		mGlyphSize = v;
		this->UpdateGlyphSize();
		this->ClearCache();
	}
}


void iFieldViewObject::SetColorBy(int v)
{ 
	if(v != mColorBy) 
	{
		if(mColorBy >= 0) this->ShowColorBars(false);
		mColorBy = v;
		if(v >= 0) this->ShowColorBars(true);
		if(this->IsColoredByScalars())
		{
			mDataConverter->SetCurrentVar(mColorBy);
			mActors[0]->SyncWithLimits(this->GetLimits(this->GetScalarDataType()),mColorBy);
			mActors[0]->ColorByArrayComponent(0,mColorByOffset); 
			mActors[0]->SetScalarVisibility(true);
		}
		else
		{
			mActors[0]->SetScalarVisibility(false);
			this->UpdateColorByExtra();
		}
		this->ClearCache();
	}
}


void iFieldViewObject::SetGlyphSampleRate(int v)
{ 
	if(v>0 && v<1600 && v!=mGlyphSampleRate) 
	{
		mGlyphSampleRate = v;
		this->UpdateGlyphSampleRate();
		this->ClearCache();
	}
}


void iFieldViewObject::ShowColorBarsBody(bool show)
{
	if(this->IsColoredByScalars()) this->GetViewModule()->GetColorBars()->ShowBar(_ColorBarsPriorityField,mColorBy,this->GetScalarDataType(),mPalettes[0],show);
}


//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iFieldViewObject::SolidViewObjectPackStateBody(iString &s) const
{
	this->PackValue(s,KeyColorBy(),mColorBy);
	this->PackValue(s,KeyGlyphSampleRate(),mGlyphSampleRate);
	this->PackValue(s,KeyConnectedToScalars(),this->IsConnectedToScalars());
	this->PackValue(s,KeyDataOffset(),this->GetDataOffset());

	this->PackValueDistance(s,KeyGlyphSize(),mGlyphSize);

	this->FieldViewObjectPackStateBody(s);
}


void iFieldViewObject::SolidViewObjectUnPackStateBody(const iString &s)
{
	int i;

	if(this->UnPackValue(s,KeyColorBy(),i)) this->SetColorBy(i);
	if(this->UnPackValue(s,KeyGlyphSampleRate(),i)) this->SetGlyphSampleRate(i);
	
	if(this->UnPackValueDistance(s,KeyGlyphSize(),mGlyphSize)) this->SetGlyphSize(mGlyphSize);

	this->FieldViewObjectUnPackStateBody(s);
}


const iDataType& iFieldViewObject::GetScalarDataType() const
{
	return iDataType::UniformScalars();
}


bool iFieldViewObject::CanBeShown() const
{
	return (this->GetDataOffset()>=0 && this->GetDataOffset()<this->GetLimits()->GetNumVars());
}


bool iFieldViewObject::IsConnectedToScalars() const
{
	return mDataConverter->GetScalarStatus() != iGridDataConverter::_NoScalars;
}


bool iFieldViewObject::IsColoredByScalars() const
{
	return (this->IsConnectedToScalars() && this->GetLimits(this->GetScalarDataType())!=0 && mColorBy>=0 && mColorBy<this->GetLimits(this->GetScalarDataType())->GetNumVars());
}


void iFieldViewObject::ViewObjectSyncWithData(int info, void *data)
{
	if((info==0 || info==1+mColorBy) && this->IsColoredByScalars())
	{
		mActors[0]->SyncWithLimits(this->GetLimits(this->GetScalarDataType()),mColorBy);
	}

	this->FieldViewObjectSyncWithData(info,data);
}

