/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

//
//  This class is a helper for DataReader that deals with just one data type for one data channel.
//  It is effectively an extended struct
//
#ifndef IDATASUBJECT_H
#define IDATASUBJECT_H


#include "iobject.h"
#include "ierrorstatusholder.h"
#include "iviewmodulecomponent.h"
#include "ideletable.h"


#include "imath.h"
#include "iparameter.h"

class iDataLimits;
class iDataReader;
struct iDataSubjectBuffer;
class iDataSet;
class iDataType;
class iFile;
class iProgressEventObserver;

class vtkDataSet;


//
//  Useful macro to declare all keys that have to be present in children too
//
#define IDATASUBJECT_DECLARE_INHERITED_KEYS \
	static const iObjectKey& KeyResizeLimits(); \
	static const iObjectKey& KeyFixedLimits(); \
	static const iObjectKey& KeyDataPresent(); \
	static const iObjectKey& KeyFileName(); \
	static const iObjectKey& KeyNumVars(); \
	static const iObjectKey& KeyStretch(); \
	static const iObjectKey& KeyMin(); \
	static const iObjectKey& KeyMax(); \
	static const iObjectKey& KeyLowerLimit(); \
	static const iObjectKey& KeyUpperLimit(); \
	static const iObjectKey& KeyFixedStretch(); \
	static const iObjectKey& KeyName()

//
//  Helper macros
//
#define	IDATASUBJECT_DECLARE_SWAP_BYTES_FUNCTIONS(_type_) \
	static void SwapBytes(_type_ &p); \
	static void SwapBytesRange(_type_ *data, long count)

#define	IDATASUBJECT_DECLARE_READ_BLOCK_FUNCTIONS(_type_) \
	bool ReadBlock(iFile& F, _type_ *p, long len, float updateStart, float updateDuration); \
	ReadingBuffer Buffer(_type_ &d) const; \
	ReadingBuffer Buffer(_type_ *d, long l) const


class iDataSubject : public iObject, public iErrorStatusHolder, public iViewModuleComponent, public iDeletable
{

public:

	//
	//  Helper struct
	//
	struct ReadingBuffer
	{
		void *Data;
		long Length;  // does not make sense to make in unsigned as vtkIdType is signed anyway
		int Type, Size;
		ReadingBuffer(void *d, long l, int t, int s)
		{
			Data = d;
			Length = l;
			Type = t;
			Size = s;
		}
	};

	static const iObjectType& Type();
	static bool IsTypeMatch(const iString &prefix);
	static iString GetDataSubjectName(const iString &prefix);

	IDATASUBJECT_DECLARE_INHERITED_KEYS;

	//
	//  Overloads for flexibility
	//
	IDATASUBJECT_DECLARE_SWAP_BYTES_FUNCTIONS(int);
	IDATASUBJECT_DECLARE_SWAP_BYTES_FUNCTIONS(long);
	IDATASUBJECT_DECLARE_SWAP_BYTES_FUNCTIONS(float);
	IDATASUBJECT_DECLARE_SWAP_BYTES_FUNCTIONS(double);

	//
	//  Operation on data
	//
	void ReadFile(const iString &fname, const iString &root, const iString &suffix, int record);
	void Finalize();
	void EraseData();
	void DetachData();
	void Reset();
	void ShiftData(double dr[3]);

	virtual bool IsThereData() const;
	virtual bool IsAborted() const;
	inline bool OwnsData() const { return mOwnsData; }
	inline bool IsAnimatable() const { return (mRecord >= 0); }

	inline const iString& GetLastFileName() const { return mLastFileName; }
	inline const iString& GetFileRoot() const { return mFileRoot; }
	inline const iString& GetFileSuffix() const { return mFileSuffix; }
	inline int GetRecord() const { return mRecord; }

	virtual float GetMemorySize() const;

	void SetAdjustableLimits(bool s);
	inline bool GetAdjustableLimits() const { return mAdjustableLimits; }

	//
	//  Boundary conditions
	//
	void SetBoundaryConditions(int s);
	void SetDirectionPeriodic(int d, bool s);
	inline int GetBoundaryConditions() const {	return mBoundaryConditions; }
	inline bool IsBoxPeriodic() const {	return (mBoundaryConditions == iParameter::_BoundaryConditionsPeriodic); }
	inline bool IsDirectionPeriodic(int d) const { if(d>=0 && d<3) return this->IsBoxPeriodic() && mPeriodic[d]; else return false; }

	//
	//  Access to components
	//
	iDataLimits* GetLimits() const;
	virtual vtkDataSet* GetData() const;
	inline iProgressEventObserver* GetObserver() const { return mObserver; }

	inline const iDataType& GetDataType() const { return mDataType; }
	bool IsUsingData(const iDataType &type) const;

	virtual const iObjectType& RealType();

	virtual void PackCompleteState(iString &s) const;
	virtual void UnPackCompleteState(const iString &s);

protected:

	iDataSubject(iDataReader *r, const iDataType &type);
	virtual ~iDataSubject();

	virtual void DataSubjectPackStateBody(iString &s) const = 0;
	virtual void DataSubjectUnPackStateBody(const iString &s) = 0;

	virtual void ReadFileBody(const iString &suffix, const iString &fname) = 0;
	virtual void FinalizeBody() = 0;
	virtual void ShiftDataBody(int d, double dx) = 0;
	virtual void Polish(){}; // optional post-finalize

	virtual iDataLimits* CreateLimits() const = 0;
	virtual void NotifyDependencies();

	void ResetLimits();

	bool DetectFileStructureFromFirstRecord(iFile &F, long len);

	IDATASUBJECT_DECLARE_READ_BLOCK_FUNCTIONS(int);
	IDATASUBJECT_DECLARE_READ_BLOCK_FUNCTIONS(long);
	IDATASUBJECT_DECLARE_READ_BLOCK_FUNCTIONS(float);
	IDATASUBJECT_DECLARE_READ_BLOCK_FUNCTIONS(double);

	bool SkipFortranRecord(iFile& F, long len);
	bool ReadFortranRecord(iFile& F, const ReadingBuffer &b, float updateStart, float updateDuration, bool autoswap = true);
	bool ReadFortranRecord(iFile& F, const ReadingBuffer &b1, const ReadingBuffer &b2, float updateStart, float updateDuration, bool autoswap = true);
	bool ReadFortranRecord(iFile& F, const ReadingBuffer &b1, const ReadingBuffer &b2, const ReadingBuffer &b3, float updateStart, float updateDuration, bool autoswap = true);

	bool ReadFortranHeaderFooter(iFile& F, long &lrec);

	static void SwapBytesError();

	//
	//  Components
	//
	const iDataType &mDataType;

	int mBoundaryConditions;
	bool mPeriodic[3];

	iString mFileRoot, mFileSuffix, mLastFileName;
	bool mTwoCopies, mOwnsData, mIsBigEndian, mAdjustableLimits, mOverflow;
	iProgressEventObserver *mObserver;
	double mShift[3];
	int mRecord, mFortranHeaderFooterLength;

	iDataReader *mReader;
	vtkDataSet *mInternalData, *mExternalData;

private:

	bool mUpdated; // should not be used by children
	virtual void PackStateBody(iString &s) const;
	virtual void UnPackStateBody(const iString &s);
	//
	// mLimits should only be accesses via GetLimits() function
	//
	mutable iDataLimits *mLimits;
};


//
//  Useful macros to declare all members that have to be overwritten in children
//
#define IDATASUBJECT_DECLARE_INHERITED_MEMBERS \
	protected: \
	virtual void DataSubjectPackStateBody(iString &s) const; \
	virtual void DataSubjectUnPackStateBody(const iString &s); \
	virtual void ReadFileBody(const iString &suffix, const iString &fname); \
	virtual void FinalizeBody(); \
	virtual void ShiftDataBody(int d, double dx); \
	virtual iDataLimits* CreateLimits() const; \
	public: \
	virtual const iObjectType& RealType()

#define IDATASUBJECT_DEFINE_TYPE(_type_,_fname_,_sname_) \
	IOBJECT_DEFINE_TYPE(_type_,Data-_fname_,d-_sname_,iObjectType::_Helper); \
	const iObjectType& _type_::RealType(){ return _type_::Type(); }

#define IDATASUBJECT_DEFINE_INHERITED_KEYS(_type_) \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,ResizeLimits,-rl,Int,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,FixedLimits,fl,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,DataPresent,dp,Bool,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,FileName,fn,String,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,NumVars,n,Int,1); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Stretch,s,Int,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Min,min,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Max,max,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,LowerLimit,lo,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,UpperLimit,up,Float,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,FixedStretch,fs,Bool,0); \
	IOBJECT_DEFINE_INHERITED_KEY(iDataSubject,_type_,Name,na,String,0)

#endif

