/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2006 Nick Gnedin 
All rights reserved.

This file may be distributed and/or modified under the terms of the
GNU General Public License version 2 as published by the Free Software
Foundation and appearing in the file LICENSE.GPL included in the
packaging of this file.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "icrosssectionserialpipeline.h"


#include "icrosssectionviewobject.h"
#include "ierror.h"
#include "ieventobserver.h"
#include "igriddataconverter.h"
#include "iorthopolygonplanefilter.h"
#include "iorthoslicer.h"
#include "iorthotextureplanefilter.h"
#include "iviewmodule.h"

#include <vtkPlaneSource.h>
#include <vtkPolyData.h>
#include <vtkProbeFilter.h>
#include <vtkStructuredPoints.h>
#include <vtkTexture.h>

//
//  Templates (needed for some compilers)
//
#include "iarraytemplate.h"
#include "igenericfiltertemplate.h"


//
// Main class
//
iCrossSectionSerialPipeline::iCrossSectionSerialPipeline(iCrossSectionViewObject *s)  : iGridDataToPolyDataPipeline(1,s)
{
	mParent = s;

	//
	//  Do VTK stuff
	//	
	mDataConverter = iGridDataConverter::New(mParent); IERROR_ASSERT_NULL_POINTER(mDataConverter);
	mSlicer = iOrthoSlicer::New(mParent); IERROR_ASSERT_NULL_POINTER(mSlicer);
	mOrthoPlane = iOrthoPolygonPlaneFilter::New(); IERROR_ASSERT_NULL_POINTER(mOrthoPlane);
	mTextureConverter = iOrthoTexturePlaneFilter::New(); IERROR_ASSERT_NULL_POINTER(mTextureConverter);
	
	mDataConverter->SetCurrentVar(mParent->GetVar());

	mSlicer->SetInput(mDataConverter->GetOutput());
	mSlicer->SetDir(mParent->GetDir());
	mOrthoPlane->SetInput(mSlicer->GetOutput());
	mTextureConverter->SetInput(mSlicer->GetOutput());
	
	this->UpdateAll();
	mDataConverter->SetScalarInput(0);
}


iCrossSectionSerialPipeline::~iCrossSectionSerialPipeline()
{
	mSlicer->Delete();
	mDataConverter->Delete();
	mOrthoPlane->Delete();
	mTextureConverter->Delete();
}


void iCrossSectionSerialPipeline::AttachObservers(bool master)
{
	iEventObserver *obsAbortRender;
	
	if(master)
	{
		obsAbortRender = mParent->GetViewModule()->GetAbortRenderEventObserver();
	}
	else
	{
		obsAbortRender = mParent->GetViewModule()->GetSlaveAbortRenderEventObserver();
	}
	mDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mSlicer->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mOrthoPlane->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	mTextureConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
}


void iCrossSectionSerialPipeline::SetTexturePiece(vtkImageData *texture, int offx, int offy)
{
	mTextureConverter->SetTexturePiece(texture,offx,offy);
	this->Modified();
}


void iCrossSectionSerialPipeline::ProduceOutput(vtkImageData *input, vtkPolyData *output)
{
	if(mDataConverter->GetScalarInput() != input)
	{
		mDataConverter->SetScalarInput(input);
	}
	
	switch(mParent->GetActualMethod())
	{
	case 0:
		{
			mOrthoPlane->Update();
			output->ShallowCopy(mOrthoPlane->GetOutput());
			break;
		}
	case 1: 
		{
			mTextureConverter->Update(); 
			output->ShallowCopy(mTextureConverter->GetOutput());
			break;
		}
	default: return;
	}
}


void iCrossSectionSerialPipeline::UpdateMethod()
{
	switch(mParent->GetActualMethod())
	{
	case 0:
		{
			mOrthoPlane->Modified();
			break;
		}
	case 1: 
		{
			mTextureConverter->Modified(); 
			break;
		}
	default: return;
	}
	this->Modified();
}


void iCrossSectionSerialPipeline::UpdateDir()
{ 
	mSlicer->SetDir(mParent->GetDir());
	mSlicer->SetPos(mParent->GetPos());
	this->Modified();
}


void iCrossSectionSerialPipeline::UpdateVar()
{ 
	mDataConverter->SetCurrentVar(mParent->GetVar());
	this->Modified();
}


void iCrossSectionSerialPipeline::UpdatePos()
{ 
	mSlicer->SetPos(mParent->GetPos());
	this->Modified();
}


void iCrossSectionSerialPipeline::UpdateInterpolateData()
{
	mOrthoPlane->SetCorrectForFlatShading(!mParent->GetInterpolateData());
	mSlicer->SetInterpolation(mParent->GetInterpolateData());
	this->Modified();
}


void iCrossSectionSerialPipeline::UpdateResampleFactor()
{
	mSlicer->SetResampleFactor(mParent->GetResampleFactor());
	this->Modified();
}


float iCrossSectionSerialPipeline::GetMemorySize() const
{
	float s;
	s = mSlicer->GetOutput()->GetActualMemorySize();
	s += mOrthoPlane->GetOutput()->GetActualMemorySize();
	s += mTextureConverter->GetOutput()->GetActualMemorySize();
	return s;
}


iGridDataToPolyDataPipeline* iCrossSectionSerialPipeline::Copy() const
{
	iCrossSectionSerialPipeline *tmp = this->mParent->CreateSerialPipeline();	
	if(tmp == 0) return 0;

	//
	//  Update the pipeline
	//
	tmp->UpdateAll();

	return tmp;
}


void iCrossSectionSerialPipeline::UpdateAll()
{
	this->UpdateMethod();
	this->UpdateDir();
	this->UpdateVar();
	this->UpdatePos();
	this->UpdateResampleFactor();
	this->UpdateInterpolateData();
}

