/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
**  Implementation of ivtk.h
*/

#include "iglobals.h"

#include "ivtk.h"
#include "ivtkwindow.h"

#include "ienvironment.h"
#include "ipalette.h"
#include "iglobals.h"

#include "idialogpopup.h"
#include "ivtkcallback.h"
#include "idatareader.h"
#include "idatagateway.h"
#include "ianimator.h"
#include "iruler.h"
#include "imeasuringbox.h"
#include "imeasuringboxinteractorstyle.h"
#include "icolorbars.h"
#include "itextactor.h"
#include "iparticlessplitter.h"
#include "imath.h"
#include "ilimits.h"
#include "ilightkit.h"
#include "icubeaxesactor.h"
#include "iuniformmeshdata.h"
#include "ipolygonaldata.h"
#include "ipicker.h"
#include "ipostscriptwriter.h"
#include "imarker.h"
#include "ilegend.h"
#include "icaptioninteractorstyle.h"
#include "iscaledlodactor.h"
#include "idialogimagecomposer.h"

#include "isurface.h"
#include "ixsection.h"
#include "ivolume.h"
#include "iparticles.h"
#include "ivector.h"
#include "itensor.h"
#include "iobjectfactory.h"

#include <vtkRenderer.h>
#include <vtkRenderLargeImage.h>
#include <vtkDebugLeaks.h>
#include <vtkCylinderSource.h>
#include <vtkCubeSource.h>
#include <vtkPolyData.h>
#include <vtkPolyDataMapper.h>
#include <vtkLODActor.h>
#include <vtkActor.h>
#include <vtkInteractorStyleFlight.h>
#include <vtkInteractorStyleTrackballCamera.h>
#include <vtkArrowSource.h>
#include <vtkPlane.h>
#include <vtkCommand.h>
#include <vtkCamera.h>
#include <vtkTransform.h>
#include <vtkFloatArray.h>
#include <vtkPoints.h>
#include <vtkCellArray.h>
#include <vtkFloatArray.h>
#include <vtkImageWriter.h>
#include <vtkJPEGWriter.h>
#include <vtkPNMWriter.h>
#include <vtkBMPWriter.h>
#include <vtkPNGWriter.h>
#include <vtkTIFFWriter.h>
#include <vtkPostScriptWriter.h>
#include <vtkCallbackCommand.h>
#include <vtkLightCollection.h>
#include <vtkProperty2D.h>
#include <vtkPropCollection.h>
#include <vtkPointData.h>
#include <vtkProperty.h>
#include <vtkStructuredPoints.h>
#include <vtkActor2D.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkCoordinate.h>
#include <vtkDiskSource.h>
#include <vtkRenderLargeImage.h>
#include <vtkSphereSource.h>
#include <vtkUnsignedCharArray.h>
#include <vtkInteractorStyle.h>
#include <vtkPerspectiveTransform.h>
#include <vtkTransform.h>
#include <vtkImageData.h>

#ifndef I_VTK_VERSION_40
#include <vtkTextProperty.h>
#include "iinteractoreventrecorder.h"
#include <vtkPointWidget.h>
#endif

//
//  Qt - related stuff
//
#include <qpixmap.h>
QPixmap image(char *s);
void reportNullPointer(int ec);


//
//  Declare static members
//
int iVTK::im_icur = 1;
int iVTK::im_acur = 1;
iString iVTK::im_inam = iString("ifrit_im");
iString iVTK::im_anam = iString("ifrit_an");
bool iVTK::autoRaise = false;
bool iVTK::focusCurrent = true;
bool iVTK::broadcastMouseEvents = false;
bool iVTK::eventsBlocked = false;
int iVTK::idCounter = 0;
bool iVTK::endiness = false;

QWidget *iVTK::bottomWidget = 0;
QButtonGroup *iVTK::windowSizeButtonGroup = 0;
QButtonGroup *iVTK::windowTileButtonGroup = 0;

//
// These are C-style fuction neede to use SetAbortCheckMethod for vtkRenderWindow
// since it does NOT emit AbortCheckEvent in reality, contrary to what User's Guide claims.
//
void iVTK_AbortRenderPseudoObserver(void *arg)
{
	
	iVTKAbortRenderEventObserver *self = (iVTKAbortRenderEventObserver*)arg;
	self->docheck();
	
}
//
//  helper class
//
#ifndef I_VTK_VERSION_40

#include "iqt.h"
#include "iqtwindow.h"

class iMarkerEventObserver : public vtkCommand, private iVTKSource
{
	
public:
	
	static iMarkerEventObserver *New(iVTK *m) { return new iMarkerEventObserver(m); }

	virtual void Execute(vtkObject *caller, unsigned long, void *callData);

protected:

	iMarkerEventObserver(iVTK *m);
	virtual ~iMarkerEventObserver(){}
	
};


iMarkerEventObserver::iMarkerEventObserver(iVTK *m) : iVTKSource(m)
{
}


void iMarkerEventObserver::Execute(vtkObject *caller, unsigned long eventId, void *vtkNotUsed(callData))
{
	switch (eventId)
	{
	case vtkCommand::InteractionEvent:
		{
			vtkFloat *x = ((vtkPointWidget *)caller)->GetPosition();
			double xi[3];
			for(int j=0; j<3; j++) xi[j] = x[j];
			myVTK->getMarkerFamily()->getCurrentMember()->setPosition(xi);
			iQTWindow::getCurrentWindow()->displayCurrentMarkerPosition();
		}
	}
}

#endif

//
//  ----------------------  end of C-style functions ------------------------
//

iVTK* iVTK::New(iVTK *)
{
	return iObjectFactory::createVTK();
}


iVTK::iVTK(QWidget *parent, const char *name) : iRenderWindow(parent,name)
{
	int i, j;
	
	id = idCounter++;

	imageType = I_IMAGETYPE_PNG;

	fontSize = 16;
	fontType = 0;
	boxType = 0;
	imageMagnification = 1;
	imageMethod = 0;

	closeable = false;
	bottomWidget = 0;

	isFlight = isDebug = false;
	parallelProjection = true;
	antialiasing = false;
	
	blockRender = false;
	rate = 0.0;
	boxSize = -1.0;
	clone = false;

	isBoundingBoxOn = true;
	isLabelOn = true;
	isColorBarsOn = true;
	isMeasuringBoxOn = false;
	isRulerOn = false;
	isClipPlaneOn = false;
	isGlassClipPlaneOn = false;

	longRenderProgress = 0;

	labelJustificationLeft = false;

	renMagnification = 1;
	//
	//  Window stuff
	//
	win = this;
	iren = iRenderWindowInteractor::New();
	if(iren == 0) reportNullPointer(6401);
	ren = vtkRenderer::New();
	if(ren == 0) reportNullPointer(6402);

	win->AddRenderer(ren);
	
	ren->SetBackground(1.0,1.0,1.0);     
	this->setParallelProjection(parallelProjection);

	//
	//  It is important to switch the backing store off for the
	//  stereo mode to work properly
	//
	ren->BackingStoreOff();
	ren->TwoSidedLightingOff();
	ren->LightFollowCameraOn();

	if(this->doubleBuffer()) win->DoubleBufferOn(); else win->DoubleBufferOff();
	win->StereoCapableWindowOn();
	win->SetStereoTypeToRedBlue();
	ren->GetActiveCamera()->SetEyeAngle(2.0);
	
	iren->SetRenderWindow(win);
	//
	// iVisualObject collections
	//
	objectCollection = vtkPropCollection::New();
	if(objectCollection == 0) reportNullPointer(6450);
	objectCollectionVisible = vtkPropCollection::New();
	if(objectCollectionVisible == 0) reportNullPointer(6452);
	//
	//  Create and initiate lights
	//
	lights = iLightKit::New();
	if(lights == 0) reportNullPointer(6403);
	lights->AddLightsToRenderer(ren);
	lights->SetKeyLightIntensity(1.0);
	lights->SetKeyToFillRatio(3.0);
	lights->SetKeyToHeadRatio(6.0);
	lights->SetKeyLightAzimuth(0.0);
	lights->SetKeyLightElevation(45.0);
	lights->SetFillLightAzimuth(0.0);
	lights->SetFillLightElevation(-75.0);
	//
	//  File reader, gateway, volume converter
	//
	dataReader = iDataReader::New(this);
	if(dataReader == 0) reportNullPointer(6404);
	dataGateway = iDataGateway::New();
	if(dataGateway == 0) reportNullPointer(6405);
	animator = iAnimator::New(this);
	if(animator == 0) reportNullPointer(6408);
	particlesSplitter = iParticlesSplitter::New(this);
	if(particlesSplitter == 0) reportNullPointer(6451);
	
	endiness = dataReader->getEndinessOfMachine();
	dataGateway->SetInput(dataReader->getMeshOutput());
	particlesSplitter->SetInput(dataReader->getParticlesOutput());
	//
	//  Image writer
	//
	switch ( imageType ) 
	{
	case I_IMAGETYPE_PNM: { writer = vtkPNMWriter::New(); break; }
	case I_IMAGETYPE_BMP: { writer = vtkBMPWriter::New(); break; }
	case I_IMAGETYPE_PNG: { writer = vtkPNGWriter::New(); break; }
	case I_IMAGETYPE_TIF: { writer = vtkTIFFWriter::New(); break; }
	case I_IMAGETYPE_PS:  { writer = iPostScriptWriter::New(); }
	default:			  { writer = vtkJPEGWriter::New(); break; }
	}
	if(writer == 0) reportNullPointer(6409);

	imageData = vtkImageData::New();
	if(imageData == 0) reportNullPointer(6409);

	imageData->SetScalarType(VTK_UNSIGNED_CHAR);
	imageData->SetNumberOfScalarComponents(3);
	imageData->SetOrigin(0.0,0.0,0.0);
	imageData->SetSpacing(1.0,1.0,1.0);

	//
	//  Bounding box 1
	//
	float diam = 0.02;
	for(i=0; i<12; i++) 
	{
		box1Actor[i] = vtkActor::New();
		if(box1Actor[i] == 0) reportNullPointer(6410);
		box1Mapper[i] = vtkPolyDataMapper::New();
		if(box1Mapper[i] == 0) reportNullPointer(6411);
		box1Source[i] = vtkCylinderSource::New();
		if(box1Source[i] == 0) reportNullPointer(6412);
		box1Source[i]->SetResolution(6);
		box1Source[i]->SetHeight(2.0+2.0*diam);
		box1Source[i]->SetRadius(diam);
		box1Mapper[i]->SetInput(box1Source[i]->GetOutput());
	}
	
	for(i=0; i<4; i++) 
	{
		box1Actor[i]->GetProperty()->SetColor(0.0,0.0,1.0);
		box1Actor[i]->RotateX(90.0);
	}
	for(i=4; i<8; i++) box1Actor[i]->GetProperty()->SetColor(0.0,1.0,0.0);
	for(i=8; i<12; i++) 
	{
		box1Actor[i]->GetProperty()->SetColor(1.0,0.0,0.0);
		box1Actor[i]->RotateZ(90.0);
	}
	
	box1Actor[ 0]->SetPosition(-1.0,-1.0,0.0);
	box1Actor[ 1]->SetPosition(-1.0, 1.0,0.0);
	box1Actor[ 2]->SetPosition( 1.0,-1.0,0.0);
	box1Actor[ 3]->SetPosition( 1.0, 1.0,0.0);
	box1Actor[ 4]->SetPosition(-1.0,0.0,-1.0);
	box1Actor[ 5]->SetPosition(-1.0,0.0, 1.0);
	box1Actor[ 6]->SetPosition( 1.0,0.0,-1.0);
	box1Actor[ 7]->SetPosition( 1.0,0.0, 1.0);
	box1Actor[ 8]->SetPosition(0.0,-1.0,-1.0);
	box1Actor[ 9]->SetPosition(0.0,-1.0, 1.0);
	box1Actor[10]->SetPosition(0.0, 1.0,-1.0);
	box1Actor[11]->SetPosition(0.0, 1.0, 1.0);
	
	for(i=0; i<12; i++) 
	{
		box1Actor[i]->GetProperty()->SetAmbient(0.2);
		box1Actor[i]->GetProperty()->SetDiffuse(0.2);
		box1Actor[i]->GetProperty()->SetSpecular(0.2);
		box1Actor[i]->SetMapper(box1Mapper[i]);
		box1Actor[i]->VisibilityOff();
		box1Actor[i]->PickableOff();
		ren->AddActor(box1Actor[i]);
	}
	//
	//  Bounding box 2
	//
	box2Actor = vtkActor::New();
	if(box2Actor == 0) reportNullPointer(6413);
	box2Mapper = vtkPolyDataMapper::New();
	if(box2Mapper == 0) reportNullPointer(6414);
	
	static float x[8][3]={{-1,-1,-1}, {1,-1,-1}, {1,1,-1}, {-1,1,-1},
	{-1,-1,1}, {1,-1,1}, {1,1,1}, {-1,1,1}};
	//static vtkIdType pts[6][4]={{0,1,2,3}, {4,5,6,7}, {0,1,5,4},
	//{1,2,6,5}, {2,3,7,6}, {3,0,4,7}};
	static vtkIdType lns[12][2]={{0,1}, {1,2}, {2,3}, {3,0}, {0,4}, {1,5},
	{2,6}, {3,7}, {4,5}, {5,6}, {6,7}, {7,4}};
	
	// We'll create the building blocks of polydata including data attributes.
	box2Source = vtkPolyData::New();
	if(box2Source == 0) reportNullPointer(6415);
	vtkPoints *points = vtkPoints::New(VTK_FLOAT);
	if(points == 0) reportNullPointer(6416);
	vtkCellArray *polys = vtkCellArray::New();
	if(polys == 0) reportNullPointer(6417);
	vtkFloatArray *scalars = vtkFloatArray::New();
	if(scalars == 0) reportNullPointer(6418);
	
	// Load the point, cell, and data attributes.
	for (i=0; i<8; i++) points->InsertPoint(i,x[i]);
	for (i=0; i<8; i++) scalars->InsertTuple1(i,(i<4));
	//
	//  Do full cube as a wireframe
	//
	//	for (i=0; i<6; i++) polys->InsertNextCell(4,pts[i]);
	//	box2Source->SetPolys(polys);
	//	box2Actor->GetProperty()->SetRepresentationToWireframe();
	//
	//  Do box as 12 lines
	//
	for (i=0; i<12; i++) polys->InsertNextCell(2,lns[i]);
	box2Source->SetLines(polys);
	polys->Delete();
	
	// We now assign the pieces to the vtkPolyData.
	box2Source->SetPoints(points);
	points->Delete();
	box2Source->GetPointData()->SetScalars(scalars);
	scalars->Delete();
	
	// Now we'll look at it.
	box2Mapper->SetInput(box2Source);
	box2Mapper->SetScalarRange(0,1);
	
	box2Actor->SetMapper(box2Mapper);
	box2Actor->GetProperty()->SetAmbient(1.0);
	box2Actor->GetProperty()->SetDiffuse(0.0);
	box2Actor->GetProperty()->SetSpecular(0.0);
	box2Actor->GetProperty()->SetLineWidth(4.0);
	box2Actor->GetProperty()->SetColor(0.0,0.0,0.0);
	
	box2Actor->VisibilityOff();
	box2Actor->PickableOff();
	ren->AddActor(box2Actor);
	//
	//  Bounding box 3
	//
	diam = 0.02;
	for(i=0; i<3; i++) 
	{
		box3Actor[i] = vtkActor::New();
		if(box3Actor[i] == 0) reportNullPointer(6419);
		box3Mapper[i] = vtkPolyDataMapper::New();
		if(box3Mapper[i] == 0) reportNullPointer(6420);
		box3ArrowSource[i] = vtkArrowSource::New();
		if(box3ArrowSource[i] == 0) reportNullPointer(6421);
		box3ArrowSource[i]->SetShaftResolution(6);
		box3ArrowSource[i]->SetTipResolution(6);
		box3ArrowSource[i]->SetTipLength(0.15);
		box3ArrowSource[i]->SetShaftRadius(0.5*diam);
		box3ArrowSource[i]->SetTipRadius(1.5*diam);
		box3Mapper[i]->SetInput(box3ArrowSource[i]->GetOutput());
		box3Actor[i]->SetScale(2.5);
	}
	
	diam *= 0.75;
	for(i=0; i<9; i++) 
	{
		box3Actor[3+i] = vtkActor::New();
		if(box3Actor[3+i] == 0) reportNullPointer(6422);
		box3Mapper[3+i] = vtkPolyDataMapper::New();
		if(box3Mapper[3+i] == 0) reportNullPointer(6423);
		box3CylinderSource[i] = vtkCylinderSource::New();
		if(box3CylinderSource[i] == 0) reportNullPointer(6424);
		box3CylinderSource[i]->SetResolution(6);
		box3CylinderSource[i]->SetHeight(2.0+2.0*diam);
		box3CylinderSource[i]->SetRadius(diam);
		box3Mapper[3+i]->SetInput(box3CylinderSource[i]->GetOutput());
	}
	
	box3Actor[0]->GetProperty()->SetColor(0.0,0.0,1.0);
	box3Actor[0]->RotateY(-90.0);
	box3Actor[1]->GetProperty()->SetColor(0.0,1.0,0.0);
	box3Actor[1]->RotateZ(90.0);
	box3Actor[2]->GetProperty()->SetColor(1.0,0.0,0.0);
	
	for(i=3; i<6; i++) 
	{
		box3Actor[i]->GetProperty()->SetColor(0.0,0.0,1.0);
		box3Actor[i]->RotateX(90.0);
	}
	for(i=6; i<9; i++) box3Actor[i]->GetProperty()->SetColor(0.0,1.0,0.0);
	for(i=9; i<12; i++) 
	{
		box3Actor[i]->GetProperty()->SetColor(1.0,0.0,0.0);
		box3Actor[i]->RotateZ(90.0);
	}
	
	box3Actor[ 0]->SetPosition(-1.0,-1.0,-1.0);
	box3Actor[ 1]->SetPosition(-1.0,-1.0,-1.0);
	box3Actor[ 2]->SetPosition(-1.0,-1.0,-1.0);
	
	box3Actor[ 3]->SetPosition(-1.0, 1.0,0.0);
	box3Actor[ 4]->SetPosition( 1.0,-1.0,0.0);
	box3Actor[ 5]->SetPosition( 1.0, 1.0,0.0);
	
	box3Actor[ 6]->SetPosition(-1.0,0.0, 1.0);
	box3Actor[ 7]->SetPosition( 1.0,0.0,-1.0);
	box3Actor[ 8]->SetPosition( 1.0,0.0, 1.0);
	
	box3Actor[ 9]->SetPosition(0.0, 1.0,-1.0);
	box3Actor[10]->SetPosition(0.0,-1.0, 1.0);
	box3Actor[11]->SetPosition(0.0, 1.0, 1.0);
	
	for(i=0; i<12; i++) 
	{
		box3Actor[i]->GetProperty()->SetAmbient(0.2);
		box3Actor[i]->GetProperty()->SetDiffuse(0.2);
		box3Actor[i]->GetProperty()->SetSpecular(0.2);
		box3Actor[i]->SetMapper(box3Mapper[i]);
		box3Actor[i]->VisibilityOff();
		box3Actor[i]->PickableOff();
		ren->AddActor(box3Actor[i]);
	}
	
	box3Cube = iCubeAxesActor::New();
	if(box3Cube == 0) reportNullPointer(6425);
	box3Cube->SetCamera(ren->GetActiveCamera());
	this->setBox3Axes("X","Y","Z",-1.0,1.0,-1.0,1.0,-1.0,1.0);
	
	box3Cube->VisibilityOff();
	box3Cube->PickableOff();
	ren->AddProp(box3Cube);
	//
	//  Set bounding box
	//	
	this->setBoxType(boxType);
	//
	//  ruler actor
	//
	rulerActor = iRuler::New();
	if(rulerActor == 0) reportNullPointer(6426);
	rulerActor->VisibilityOff();
	rulerActor->PickableOff();
	
	ren->AddProp(rulerActor);
	//
	//  Text label actors
	//
	labelActor = iTextActor::New();
	if(labelActor == 0) reportNullPointer(6427);
	labelActor->setText("");
	labelActor->setBold(true);
	labelActor->setHJustification(ITEXT_RIGHT);
	labelActor->setVJustification(ITEXT_TOP);
	labelActor->setColor(0.0,0.0,0.0);
	labelActor->setPosition(0.95,0.9);
	labelActor->VisibilityOff();
	labelActor->PickableOff();
	ren->AddActor(labelActor);
	isLabelOn = true;
	//
	//  Measuring box
	//
	measuringBox = iMeasuringBox::New();
	if(measuringBox == 0) reportNullPointer(6428);
	istyleMeasuringBox = iMeasuringBoxInteractorStyle::New();
	if(istyleMeasuringBox == 0) reportNullPointer(6429);
	
	measuringBox->VisibilityOff();
	ren->AddActor(measuringBox);
	istyleMeasuringBox->SetMeasuringBox(measuringBox);
	//
	//  Color bars
	//
	colorBars = iColorBars::New(this);
	if(colorBars == 0) reportNullPointer(6430);
	colorBars->PickableOff();
	ren->AddActor(colorBars);
	//
	//  Clipping plane
	//
	clipPlane = vtkPlane::New();
	if(clipPlane == 0) reportNullPointer(6431);
	clipPlane->SetNormal(0.0,0.0,-1.0);
	vtkFloat n[3];
	clipPlane->GetNormal(n);
	n[0] *= (-1.75*0);
	n[1] *= (-1.75*0);
	n[2] *= (-1.75*0);
	clipPlane->SetOrigin(n);
	
	clipPlaneSource = vtkPolyData::New();
	if(clipPlaneSource == 0) reportNullPointer(6432);
	clipPlaneMapper = vtkPolyDataMapper::New();
	if(clipPlaneMapper == 0) reportNullPointer(6433);
	clipPlaneActor = vtkActor::New();
	if(clipPlaneActor == 0) reportNullPointer(6434);
	
	static float x1[8][3] = { {1.0,0.0,0.0}, {0.7071,0.7071,0.0}, {0.0,1.0,0.0}, {-0.7071,0.7071,0.0}, 
	{-1.0,0.0,0.0}, {-0.7071,-0.7071,0.0}, {0.0,-1.0,0.0}, {0.7071,-0.7071,0.0} };
	static vtkIdType pts1[8] = { 0, 1, 2, 3, 4, 5, 6, 7 };
	
	for(i=0; i<8; i++) for(j=0; j<3; j++) x1[i][j] *= 2.5;
	
	points = vtkPoints::New(VTK_FLOAT);
	if(points == 0) reportNullPointer(6435);
	for (i=0; i<8; i++) points->InsertPoint(i,x1[i]);
	clipPlaneSource->SetPoints(points);
	points->Delete();
	
	polys = vtkCellArray::New();
	if(polys == 0) reportNullPointer(6436);
	polys->InsertNextCell(8,pts1);
	clipPlaneSource->SetPolys(polys);
	polys->Delete();
	
	vtkFloatArray *normals = vtkFloatArray::New();
	if(normals == 0) reportNullPointer(6437);
	normals->SetNumberOfComponents(3);
	normals->SetNumberOfTuples(1);
	clipPlane->GetNormal(n);
	normals->SetTuple(0,n);
	clipPlaneSource->GetPointData()->SetNormals(normals);
	normals->Delete();
	
	clipPlaneMapper->SetInput(clipPlaneSource);	
	
	clipPlaneActor->SetMapper(clipPlaneMapper);
	clipPlaneActor->VisibilityOff();
	clipPlaneActor->PickableOff();
	
	clipPlaneActor->GetProperty()->SetOpacity(0.5);
	clipPlaneActor->GetProperty()->SetAmbient(1.0);
	clipPlaneActor->GetProperty()->SetDiffuse(1.0);
	clipPlaneActor->GetProperty()->SetSpecular(0.7);
	clipPlaneActor->GetProperty()->SetSpecularPower(50.0);
	clipPlaneActor->GetProperty()->SetColor(0.7,0.8,0.8);
	
	ren->AddActor(clipPlaneActor);
	//
	//  Progress callback functions
	//
	obsStart = iVTKStartEventObserver::New();
	if(obsStart == 0) reportNullPointer(6438);
	obsProgress = iVTKProgressEventObserver::New();
	if(obsProgress == 0) reportNullPointer(6439);
	obsEnd = iVTKEndEventObserver::New();
	if(obsEnd == 0) reportNullPointer(6440);
	
	obsAbortRender = iVTKAbortRenderEventObserver::New();
	if(obsAbortRender == 0) reportNullPointer(6441);
	obsAbortRender->setRenderWindow(win);
	//
	//  Interactor styles
	//
	istyleFlight = vtkInteractorStyleFlight::New();
	if(istyleFlight == 0) reportNullPointer(6442);
	istyleFlight->SetAngleStepSize(0.1);
	istyleFlight->SetMotionStepSize(0.001);
#ifndef I_VTK_VERSION_40
	//
	//  Work around for a VTK bug: undefined angles in vtkInteractorStyleFlight.
	//
	{
		double p[3], f[3];
		ren->GetActiveCamera()->GetPosition(p);
		ren->GetActiveCamera()->GetFocalPoint(f);
		istyleFlight->SetCurrentRenderer(ren);
		istyleFlight->SetInteractor(iren);
		istyleFlight->ForwardFly();
		ren->GetActiveCamera()->SetPosition(p);
		ren->GetActiveCamera()->SetFocalPoint(f);
		ren->GetActiveCamera()->SetViewUp(0.0,1.0,0.0);
		istyleFlight->SetInteractor(0);
	}
#endif
	istyleView = vtkInteractorStyleTrackballCamera::New();
	if(istyleView == 0) reportNullPointer(6443);

	iren->SetInteractorStyle(istyleView);
	//
	//  Debug renderer and props
	//
	diam = 0.2;
	for(i=0; i<3; i++) 
	{
		debugActor[i] = vtkActor::New();
		if(debugActor[i] == 0) reportNullPointer(6447);
		debugMapper[i] = vtkPolyDataMapper::New();
		if(debugMapper[i] == 0) reportNullPointer(6448);
		debugArrowSource[i] = vtkArrowSource::New();
		if(debugArrowSource[i] == 0) reportNullPointer(6449);
		debugArrowSource[i]->SetShaftResolution(6);
		debugArrowSource[i]->SetTipResolution(18);
		debugArrowSource[i]->SetTipLength(0.5);
		debugArrowSource[i]->SetShaftRadius(0.7*diam);
		debugArrowSource[i]->SetTipRadius(1.3*diam);
		debugMapper[i]->SetInput(debugArrowSource[i]->GetOutput());
		debugActor[i]->SetScale(0.5);
		debugActor[i]->GetProperty()->SetAmbient(0.25);
		debugActor[i]->GetProperty()->SetDiffuse(0.25);
		debugActor[i]->GetProperty()->SetSpecular(0.5);
		debugActor[i]->SetMapper(debugMapper[i]);
		debugActor[i]->VisibilityOff();
		debugActor[i]->PickableOff();
		ren->AddActor(debugActor[i]);
	}
	
	debugActor[0]->GetProperty()->SetColor(0.0,0.0,1.0);
	debugActor[0]->RotateY(-90.0);
	debugActor[1]->GetProperty()->SetColor(0.0,1.0,0.0);
	debugActor[1]->RotateZ(90.0);
	debugActor[2]->GetProperty()->SetColor(1.0,0.0,0.0);
	
	debugActor[ 0]->SetPosition(0.0,0.0,0.0);
	debugActor[ 1]->SetPosition(0.0,0.0,0.0);
	debugActor[ 2]->SetPosition(0.0,0.0,0.0);
	//
	// icon
	//
	iconActor = vtkActor2D::New();
	iconMapper = vtkPolyDataMapper2D::New();
	iconCoordinate = vtkCoordinate::New();
	iconSource = vtkDiskSource::New();

	iconSource->SetInnerRadius(0.03);
	iconSource->SetOuterRadius(0.07);
	iconSource->SetRadialResolution(1);
	iconSource->SetCircumferentialResolution(8);

	iconSource->Update();
	int npoi = iconSource->GetOutput()->GetPoints()->GetNumberOfPoints();
	vtkFloatArray *mpoi = vtkFloatArray::New();
	mpoi->SetNumberOfComponents(1);
	mpoi->SetNumberOfTuples(npoi);
	for(i=0; i<npoi; i++)
	{
		mpoi->SetTuple1(i,(i%4)/2);
	}
	iconSource->GetOutput()->GetPointData()->SetScalars(mpoi);
	mpoi->Delete();

	iconCoordinate->SetCoordinateSystemToView();

	iconMapper->SetInput(iconSource->GetOutput());
	iconMapper->SetTransformCoordinate(iconCoordinate);
	iconMapper->SetScalarRange(0,1);

	iconActor->SetMapper(iconMapper);
	iconActor->GetProperty()->SetColor(0.0,0.0,0.0);
	iconActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedDisplay();
	iconActor->SetPosition(-0.45,0.45);
	iconActor->GetProperty()->SetOpacity(0.0);
	iconActor->VisibilityOff();
	iconActor->PickableOff();

	ren->AddActor2D(iconActor);
	//
	//  Picker
	//
	picker = iPicker::New(this);
	if(picker == 0) reportNullPointer(6454);
	iren->SetPicker(picker);
	picker->SetTolerance(0.01);

	obsPick = iVTKPickEventObserver::New();
	if(obsPick == 0) reportNullPointer(6453);

	picker->AddObserver(vtkCommand::StartPickEvent,obsPick);
	picker->AddObserver(vtkCommand::EndPickEvent,obsPick);
	picker->AddObserver(vtkCommand::AbortCheckEvent,obsAbortRender);

	pickedPointActor = iScaledLODActor::New();
	if(pickedPointActor == 0) reportNullPointer(6455);
	pickedPointMapper = vtkPolyDataMapper::New();
	if(pickedPointMapper == 0) reportNullPointer(6456);
	pickedPointSource = vtkSphereSource::New();
	if(pickedPointSource == 0) reportNullPointer(6457);
	pickedPointSource2 = vtkCubeSource::New();
	if(pickedPointSource2 == 0) reportNullPointer(6459);

	pickedPointSize = 0.1;
	this->setPickedPointSize(pickedPointSize);
	pickedPointSource->SetRadius(0.5);
	pickedPointMapper->SetInput(pickedPointSource->GetOutput());
	pickedPointActor->SetMapper(pickedPointMapper);
	pickedPointActor->GetProperty()->SetColor(1.0,1.0,1.0);
	pickedPointActor->VisibilityOff();
	pickedPointActor->PickableOff();

	ren->AddActor(pickedPointActor);
	//
	//  image magnifier
	//
	magnifier = vtkRenderLargeImage::New();
	if(magnifier == 0) reportNullPointer(6461);
	magnifier->SetInput(ren);
	//
	//  Event recorder & marker helpers
	//  
	eventRecorder = 0;
	markerPlacer = 0;
	markerLegend = 0;
	markerEventObserver = 0;
	istyleCaption = 0;

	markerLegendOn = false;
	markerLegendPosition = 0;

#ifndef I_VTK_VERSION_40

	eventRecorder = iInteractorEventRecorder::New();
	if(eventRecorder == 0) reportNullPointer(6463);
	eventRecorder->SetInteractor(iren);
	eventRecorder->SetEnabled(1);

	obsRecord = iVTKRecordEventObserver::New();
	if(obsRecord == 0) reportNullPointer(6464);

	istyleView->AddObserver(vtkCommand::InteractionEvent,obsRecord);
	istyleFlight->AddObserver(vtkCommand::InteractionEvent,obsRecord);
	istyleMeasuringBox->AddObserver(vtkCommand::InteractionEvent,obsRecord);

	markerPlacer = vtkPointWidget::New();
	if(markerPlacer == 0) reportNullPointer(6462);
	markerPlacer->XShadowsOn();
	markerPlacer->YShadowsOn();
	markerPlacer->ZShadowsOn();
	markerPlacer->OutlineOff();
	markerPlacer->TranslationModeOff();
	markerPlacer->SetInteractor(iren);
	markerPlacer->GetProperty()->SetColor(0.0,0.0,0.0);
	markerPlacer->GetProperty()->SetLineWidth(4.0);
//	markerPlacer->GetSelectedProperty()->SetColor(1.0,0.0,0.0);
	markerPlacer->GetSelectedProperty()->SetLineWidth(2.0);
	markerPlacer->PlaceWidget(-1.0,1.0,-1.0,1.0,-1.0,1.0);

	markerEventObserver = iMarkerEventObserver::New(this);
	if(markerEventObserver == 0) reportNullPointer(6465);
	markerPlacer->AddObserver(vtkCommand::InteractionEvent,markerEventObserver);
	//
	//  vtkLegendBoxActor is defined under earlier VTK versions, but
	//  markers cannot be positioned without the vtkPointWidgets, so the legend
	//  is useless then too.
	//
	markerLegend = iLegend::New();
	if(markerLegend == 0) reportNullPointer(6466);
	this->setMarkerLegendPosition(markerLegendPosition);
	ren->AddProp(markerLegend);
	markerLegend->VisibilityOff();

	istyleCaption = iCaptionInteractorStyle::New(this);
	if(istyleCaption == 0) reportNullPointer(6467);

#endif

	//
	//  iVisualObjects
	//
	surfaceFamily = new iVisualObjectFamily<iSurface>(this);
	xsectionFamily = new iVisualObjectFamily<iXsection>(this);
	volumeFamily = new iVisualObjectFamily<iVolume>(this);
	vectorFamily = new iVisualObjectFamily<iVector>(this);
	tensorFamily = new iVisualObjectFamily<iTensor>(this);
	particlesFamily = new iVisualObjectFamily<iParticles>(this);

	particlesFamily->setDataChannel(0);
	particlesFamily->getMember(0)->connectSplitter(0,0);

	markerFamily = new iVisualObjectFamily<iMarker>(this);
	//
	//  Broadcasting helpers
	//
	inBroadcast = false;
	//
	//  Render the scene
	//
	this->resetCamera();

	rulerActor->setInitialScale(ren->GetActiveCamera()->GetParallelScale());
	measuringBox->setInitialScale(ren->GetActiveCamera()->GetParallelScale());
	
	iren->SetDesiredUpdateRate(rate);
	iren->SetStillUpdateRate(0.001);
	
	iren->Initialize();
	//
	//  Set the font size
	//
	this->setFontSize(fontSize);
	this->setFontType(fontType);
	//
	//  Window stuff
	//
	this->resize(640,480);
	this->setMinimumWidth(128);
	this->setMinimumHeight(128);
	this->sizePolicy().setHorData(QSizePolicy::Expanding);
	this->sizePolicy().setVerData(QSizePolicy::Expanding);

	this->setWindowNumber(-1);
	//
	//  QT-dependent members. Classes are declared as such (when possible), and not as pointers,
	//  to emphasize the difference with the rest of iVTK.
	//
	popupDialog = new iDialogPopup;
	if(popupDialog == 0) reportNullPointer(6458);
	imageComposer = 0;

	this->setIcon(image("genie1vtk.png"));

	glPolygonOffset(1.0,1.0);
	//
	//  Debug area
	//

}


iVTK::~iVTK()
{
	// Clean up
	int i;
	//
	//  In case something is highlighted
	//
	blockRender = true;

#ifndef I_VTK_VERSION_40
	if(iren != 0) ((vtkInteractorStyle *)iren->GetInteractorStyle())->HighlightProp(NULL);
	eventRecorder->SetInteractor(NULL);
	eventRecorder->Delete();
	obsRecord->Delete();
	markerEventObserver->Delete();
	markerPlacer->Off();
	markerPlacer->SetInteractor(NULL);
	markerPlacer->Delete();
	markerLegend->Delete();
	istyleCaption->Delete();
#endif
	ren->RemoveActor(pickedPointActor);

	delete surfaceFamily;
	delete xsectionFamily;
	delete volumeFamily;
	delete particlesFamily;
	delete vectorFamily;
	delete tensorFamily;

	delete markerFamily;
	
	delete animator;

	delete popupDialog;

	particlesSplitter->Delete();
	lights->Delete();
	istyleView->Delete();
	istyleFlight->Delete();
	istyleMeasuringBox->Delete();
	for(i=0; i<12; i++) 
	{
		ren->RemoveActor(box1Actor[i]);
		box1Source[i]->Delete();
		box1Mapper[i]->Delete();
		box1Actor[i]->Delete();
	}
	ren->RemoveActor(box2Actor);
	box2Source->Delete();
	box2Mapper->Delete();
	box2Actor->Delete();
	for(i=0; i<3; i++) 
	{
		ren->RemoveActor(box3Actor[i]);
		box3ArrowSource[i]->Delete();
		box3Mapper[i]->Delete();
		box3Actor[i]->Delete();
	}
	for(i=0; i<9; i++) 
	{
		ren->RemoveActor(box3Actor[3+i]);
		box3CylinderSource[i]->Delete();
		box3Mapper[3+i]->Delete();
		box3Actor[3+i]->Delete();
	}
	for(i=0; i<3; i++) 
	{
		ren->RemoveActor(debugActor[i]);
		debugArrowSource[i]->Delete();
		debugMapper[i]->Delete();
		debugActor[i]->Delete();
	}
	ren->RemoveProp(box3Cube);
	box3Cube->Delete();
	ren->RemoveProp(rulerActor);
	rulerActor->Delete();
	ren->RemoveActor(labelActor);
	labelActor->Delete();
	ren->RemoveProp(colorBars);
	colorBars->Delete();
	ren->RemoveProp(measuringBox);
	measuringBox->Delete();
	iconActor->Delete();
	iconMapper->Delete();
	iconCoordinate->Delete();
	iconSource->Delete();
	clipPlane->Delete();
	clipPlaneSource->Delete();
	clipPlaneMapper->Delete();
	ren->RemoveActor(clipPlaneActor);
	clipPlaneActor->Delete();
	if(!clone) dataReader->Delete();
	imageData->Delete();
	writer->Delete();
	obsStart->Delete();
	obsProgress->Delete();
	obsEnd->Delete();
	obsAbortRender->Delete();
	obsPick->Delete();
	picker->Delete();
	pickedPointActor->Delete();
	pickedPointMapper->Delete();
	pickedPointSource->Delete();
	pickedPointSource2->Delete();
	magnifier->Delete();
	ren->Delete();
	iren->SetRenderWindow(NULL) ;
	iren->Delete();
	objectCollection->Delete();
	objectCollectionVisible->Delete();
	
}


iVTK* iVTK::convertFromViewport(vtkViewport *viewport)
{
	return (iVTK *)((vtkRenderer *)viewport)->GetRenderWindow();
}


void iVTK::setWindowNumber(int wNum)
{
	iString windowName = "IFRIT - Visualization Window";
	windowNumber = wNum;
	if(windowNumber >= 0)
	{
		iString s;
		windowName += " #" + s.setNum(windowNumber+1);
	}
	this->setCaption(windowName);
}


bool iVTK::becomeClone(iVTK *v)
{
	bool ret = true;

	if(v == 0)
	{
		//
		//	Unmake the clone
		//
		v = dataReader->getVTK(); 
		if(v != this)
		{
			clone = false;
			dataReader = iDataReader::New(this);
			if(dataReader == 0) reportNullPointer(6404);

			//
			//  Copy dateReaders
			//
			dataReader->copyState(v->getReader());

			surfaceFamily->reset();
			xsectionFamily->reset();
			volumeFamily->reset();
			particlesFamily->reset();
			vectorFamily->reset();
			tensorFamily->reset();
			//
			//  Un-show label
			//
			this->showLabel(isLabelOn); 
		}
	}
	else
	{
		//
		//	Make the clone
		//
		if(dataReader->getVTK() == this)
		{
			clone = true;
			//
			//  Link the Data Reader
			//
			dataReader->Delete();
			dataReader = v->dataReader;
			//
			//  Copy iVisualObject families
			//
			ret = ret & surfaceFamily->copy(v->surfaceFamily);
			ret = ret & xsectionFamily->copy(v->xsectionFamily);
			ret = ret & volumeFamily->copy(v->volumeFamily);
			ret = ret & vectorFamily->copy(v->vectorFamily);
			ret = ret & tensorFamily->copy(v->tensorFamily);
			ret = ret & particlesFamily->copy(v->particlesFamily);
			//
			//  Copy Animator state
			//
			v->animator->saveState();
			animator->copyState(v->animator);
			//
			//  Copy camera location
			//
			ren->GetActiveCamera()->SetPosition(v->ren->GetActiveCamera()->GetPosition());
			ren->GetActiveCamera()->SetFocalPoint(v->ren->GetActiveCamera()->GetFocalPoint());
			ren->GetActiveCamera()->SetViewUp(v->ren->GetActiveCamera()->GetViewUp());
			ren->GetActiveCamera()->SetParallelScale(v->ren->GetActiveCamera()->GetParallelScale());
			//
			//  Show label
			//
			this->showLabel(v->isLabelVisible()); 
		}
	}

	dataGateway->SetInput(dataReader->getMeshOutput());
	particlesSplitter->SetInput(dataReader->getParticlesOutput());
	
	return ret;

}


void iVTK::setDebugMode(bool s)
{
	int i;
	vtkProp *p;

	if(s != isDebug)
	{
		isDebug = s;
		if(s)
		{
			
			objectCollectionVisible->RemoveAllItems();
			objectCollectionVisible->InitTraversal();
			
			objectCollection->InitTraversal();
			while((p = objectCollection->GetNextProp()) != NULL) if(p->GetVisibility())
			{
				objectCollectionVisible->AddItem(p);
				p->VisibilityOff();
			}
			
			for(i=0; i<3; i++) debugActor[i]->VisibilityOn();
			animator->setDebugMode(true);
			
		}
		else
		{
			
			animator->setDebugMode(false);
			for(i=0; i<3; i++) debugActor[i]->VisibilityOff();
			
			objectCollectionVisible->InitTraversal();
			while((p = objectCollectionVisible->GetNextProp()) != NULL)
			{
				p->VisibilityOn();
			}
			
		}
		this->render(true);
	}
}


void iVTK::addProp(vtkProp* p){
	ren->AddProp(p);
	objectCollection->AddItem(p);
}


void iVTK::removeProp(vtkProp* p){
	ren->RemoveProp(p);
	objectCollection->RemoveItem(p);
}
//
// small interface over the real Render call
//
void iVTK::render(bool s)
{
	
	if(s)
	{
#ifndef I_VTK_VERSION_40
		win->AddObserver(vtkCommand::AbortCheckEvent,obsAbortRender);
#else
		win->SetAbortCheckMethod(iVTK_AbortRenderPseudoObserver,(void *)obsAbortRender);
#endif
		obsAbortRender->start();
	}
      	ren->ResetCameraClippingRange();
	this->Render();
	if(s)
	{
		obsAbortRender->finish();
#ifndef I_VTK_VERSION_40
		win->RemoveObserver(vtkCommand::AbortCheckEvent);
#else
		win->SetAbortCheckMethod(0,0);
#endif
	}
	
}
//
//  vtk-style Render call - has to go through it so that QT updates
//  the window during vtk-type Render() call.
//
void iVTK::Render()  
{
	static int ops;

	if(blockRender) return;

	this->startRender();
	//
	//  This way the magnified image creation do not show in actual window
	//
	if(longRenderProgress > 0)
	{	
		iRenderWindow::Render();
		if(longRenderProgress == 1) 
		{
			obsStart->Execute(this,0,0);
			float p = -0.12;
			obsProgress->Execute(this,0,&p);
			ops = imageMagnification*imageMagnification;
		}
		float p = (float)longRenderProgress/ops;
		obsProgress->Execute(this,0,&p);
		longRenderProgress++;
	}
	else this->updateGL();

	this->endRender();

}


void iVTK::setBox3Axes(char *labelX, char *labelY, char *labelZ, float xMin, float xMax, float yMin, float yMax, float zMin, float zMax)
{
	box3Cube->SetXLabel(labelX);
	box3Cube->SetYLabel(labelY);
	box3Cube->SetZLabel(labelZ);
	box3Cube->SetRanges(xMin,xMax,yMin,yMax,zMin,zMax);
}


void iVTK::setFontSize(int s)
{
	labelActor->setFontSize(s); 
	if(s == labelActor->getFontSize())
	{
		fontSize = s;
		rulerActor->setFontSize(s);
		measuringBox->getTextActor()->setFontSize(s);
		colorBars->setFontSize(s);
		box3Cube->setFontSize(s);
		int i;
		for(i=0; i<=markerFamily->getMaxMemberIndex(); i++) markerFamily->getMember(i)->setFontSize(s);
#ifndef I_VTK_VERSION_40
		markerLegend->setFontSize(s);
#endif
	}
}


void iVTK::setFontType(int s)
{
	labelActor->setFontType(s); 
	if(s == labelActor->getFontType())
	{
		fontType = s; 
		//		rulerActor->setFontType(s);
		measuringBox->getTextActor()->setFontType(s);
		colorBars->setFontType(s);
	}
}


void iVTK::showBox(bool s)
{
	int i;
	for(i=0; i<12; i++)
	{
		box1Actor[i]->VisibilityOff(); 
		box3Actor[i]->VisibilityOff();
	}
	box2Actor->VisibilityOff();
	box3Cube->VisibilityOff();
	if(s) this->setBoxType(boxType);
	isBoundingBoxOn = s;
}


void iVTK::showRuler(bool s)
{
	if(s) rulerActor->VisibilityOn(); else rulerActor->VisibilityOff();
	isRulerOn = s;
}


void iVTK::showMeasuringBox(bool s)
{
	if(s)
	{
		this->setInteractorStyleFlight(false);
		measuringBox->VisibilityOn(); 
		iren->SetInteractorStyle(istyleMeasuringBox);
	} 
	else 
	{
		measuringBox->VisibilityOff();
		iren->SetInteractorStyle(istyleView);
	}
	isMeasuringBoxOn = s;
}


void iVTK::showColorBars(bool s)
{
	if(s) colorBars->VisibilityOn(); else colorBars->VisibilityOff();
	isColorBarsOn = s;
}


void iVTK::showGlassClipPlane(bool s)
{
	
	if(!s)
	{
		clipPlaneActor->VisibilityOff();
	} 
	else 
	{
		clipPlaneActor->VisibilityOn();
	}
	isGlassClipPlaneOn = s;
	
}


void iVTK::showClipPlane(bool s)
{
	int i;
	
	for(i=0; i<=surfaceFamily->getMaxMemberIndex(); i++) surfaceFamily->getMember(i)->showClipPlane(s);
	for(i=0; i<=volumeFamily->getMaxMemberIndex(); i++) volumeFamily->getMember(i)->showClipPlane(s);
	for(i=0; i<=particlesFamily->getMaxMemberIndex(); i++) particlesFamily->getMember(i)->showClipPlane(s);
	for(i=0; i<=vectorFamily->getMaxMemberIndex(); i++) vectorFamily->getMember(i)->showClipPlane(s);
	for(i=0; i<=tensorFamily->getMaxMemberIndex(); i++) tensorFamily->getMember(i)->showClipPlane(s);
	isClipPlaneOn = s;
	
}


void iVTK::showLabel(bool s, float *pv)
{
	
	isLabelOn = s;
	if(s && dataReader->isFileAnimatable())
	{
		
		iString s;
		int r = dataReader->getRecordNumber();
		float v;
		if(pv == NULL)
		{
			if(dataReader->getLimits(dataReader->getCurrentDataChannel())->getLabelScale() > 0.0) v = r*dataReader->getLimits(dataReader->getCurrentDataChannel())->getLabelScale(); else v = 1.0/(1.0e-4*((r>0)?r:1))-1.0;
		}
		else
		{
			v = *pv;
		}
		s = dataReader->getLimits(dataReader->getCurrentDataChannel())->getLabelName() + "=" + s.setNum(v,'g',4) + dataReader->getLimits(dataReader->getCurrentDataChannel())->getLabelUnit();
		labelActor->setText((char *)s.latin1());
		labelActor->VisibilityOn(); 
		
	} else labelActor->VisibilityOff();
}


void iVTK::showPickedPoint(bool s, float, double *)
{
	if(s && picker->GetProp()!=0) 
	{
		pickedPointMapper->SetInput(pickedPointSource->GetOutput());
		pickedPointActor->SetPosition(picker->GetPickPosition());
		pickedPointActor->VisibilityOn(); 
	}
	else pickedPointActor->VisibilityOff();
}


void iVTK::setPickedPointSize(float s)
{
	if(s > 0.0)
	{
		pickedPointSize = s;
		pickedPointActor->setBasicScale(pickedPointSize);
	}
}


void iVTK::justifyLabelLeft(bool s)
{
	if(s)
	{
		if(!labelJustificationLeft)
		{
			labelJustificationLeft = true;
			float s[2], *p;
			labelActor->getSize(s);
			labelActor->setHJustification(ITEXT_LEFT);
			p = labelActor->getPosition();
			labelActor->setPosition(p[0]-s[0],p[1]);
		}
	}
	else
	{
		if(labelJustificationLeft)
		{
			labelJustificationLeft = false;
			float s[2], *p;
			labelActor->getSize(s);
			labelActor->setHJustification(ITEXT_RIGHT);
			p = labelActor->getPosition();
			labelActor->setPosition(p[0]+s[0],p[1]);
		}
	}
}


void iVTK::setUpdateRate(float r)
{
	r = (r<0.0) ? 0.0 : r;
	r = (r>60.0) ? 60.0 : r;
	rate = r;
	iren->SetDesiredUpdateRate(rate);
}


void iVTK::setInteractorStyleFlight(bool s)
{
	isFlight = s;
	if(s) 
	{
		this->showMeasuringBox(false);
		iren->SetInteractorStyle(istyleFlight);
		ren->GetActiveCamera()->ParallelProjectionOff();
	} 
	else 
	{
		iren->SetInteractorStyle(istyleView);
		this->setParallelProjection(parallelProjection);
	}
}


void iVTK::setParallelProjection(bool s)
{
	parallelProjection = s;
	if(parallelProjection) ren->GetActiveCamera()->ParallelProjectionOn(); else ren->GetActiveCamera()->ParallelProjectionOff();
}


void iVTK::setBackgroundColor(iColor c)
{
	float r, g, b;

	r = 1.0-c.red()/255.0;
	g = 1.0-c.green()/255.0;
	b = 1.0-c.blue()/255.0;

	ren->SetBackground(1.0-r,1.0-g,1.0-b);
	rulerActor->setColor(r,g,b);
	measuringBox->setColor(r,g,b);
	colorBars->setColor(r,g,b);
	labelActor->setColor(r,g,b);
	box2Actor->GetProperty()->SetColor(r,g,b);
	iconActor->GetProperty()->SetColor(r,g,b);
//	pickedPointActor->GetProperty()->SetColor(r,g,b);
#ifndef I_VTK_VERSION_40
	animator->setCameraPathColor(r,g,b);
	markerLegend->GetProperty()->SetColor(r,g,b);
	markerLegend->GetEntryTextProperty()->SetColor(r,g,b);
	markerPlacer->GetProperty()->SetColor(r,g,b);
	for(int i=0; i<=markerFamily->getMaxMemberIndex(); i++) if(markerFamily->getMember(i)->isColorAutomatic()) markerFamily->getMember(i)->setTextColor(iColor(255-c.red(),255-c.green(),255-c.blue()));
#endif
}


iColor iVTK::getBackgroundColor()
{
	vtkFloat *c;
	c = ren->GetBackground();
	return iColor(round(255*c[0]),round(255*c[1]),round(255*c[2]));	
}


void iVTK::setBoxType(int s)
{
	boxType = s;
	if(s<0 || s>3) return;
	
	int i;
	for(i=0; i<12; i++)
	{
		box1Actor[i]->VisibilityOff(); 
		box3Actor[i]->VisibilityOff();
	}
	box2Actor->VisibilityOff();
	box3Cube->VisibilityOff();
	switch(boxType)
	{
	case 0: { box2Mapper->ScalarVisibilityOn(); box2Actor->GetProperty()->SetLineWidth(4.0); box2Actor->VisibilityOn(); break; }
	case 1: { for(i=0; i<12; i++) box1Actor[i]->VisibilityOn(); break; }
	case 2: { box2Mapper->ScalarVisibilityOff(); box2Actor->GetProperty()->SetLineWidth(1.0); box2Actor->VisibilityOn(); break; }
	case 3: { for(i=0; i<12; i++) box3Actor[i]->VisibilityOn(); box3Cube->VisibilityOn(); break; }
	}
	
}


void iVTK::setImageType(int i)
{
	i = (i<0) ? 0 : i;
	i = (i>I_IMAGETYPE_MAX) ? I_IMAGETYPE_MAX : i;
	if(i != imageType)
	{
		imageType = i;
		writer->Delete();
		switch ( imageType ) 
		{
		case I_IMAGETYPE_PNM: { writer = vtkPNMWriter::New(); break; }
		case I_IMAGETYPE_BMP: { writer = vtkBMPWriter::New(); break; }
		case I_IMAGETYPE_PNG: { writer = vtkPNGWriter::New(); break; }
		case I_IMAGETYPE_TIF: { writer = vtkTIFFWriter::New(); break; }
		case I_IMAGETYPE_PS:  { writer = iPostScriptWriter::New(); break; }
		default:			  { writer = vtkJPEGWriter::New(); }
		}
		if(writer == 0) reportNullPointer(6444);
		writer->AddObserver(vtkCommand::StartEvent,obsStart);
		writer->AddObserver(vtkCommand::ProgressEvent,obsProgress);
		writer->AddObserver(vtkCommand::EndEvent,obsEnd);
	}
}


void iVTK::setAntialiasingOn(bool s)
{
	
	antialiasing = s;
	if(s)
	{
		this->SetLineSmoothing(1);
		this->SetPointSmoothing(1);
		glEnable(GL_LINE_SMOOTH);
		glEnable(GL_POINT_SMOOTH);
    } 
	else 
	{
		this->SetLineSmoothing(0);
		this->SetPointSmoothing(0);
		glDisable(GL_LINE_SMOOTH);
		glDisable(GL_POINT_SMOOTH);
    }
	this->Modified();
	
}


void iVTK::showIcon(int iconMode)
{
	if(windowNumber < 0) 
	{
		iconActor->VisibilityOff(); 
	}
	else 
	{
		switch(iconMode)
		{
		default:
		case I_ICON_NONE: 
			{ 
				iconActor->VisibilityOff(); 
				break; 
			}
		case I_ICON_CLONE: 
			{ 
				iconActor->VisibilityOn(); 
				iconActor->GetProperty()->SetOpacity(0.5); 
				break; 
			}
		case I_ICON_FULL: 
			{ 
				iconActor->VisibilityOn(); 
				iconActor->GetProperty()->SetOpacity(1.0); 
				break; 
			}
		}
	}
}


void iVTK::setImageMagnification(int m)
{
#ifndef I_VTK_VERSION_40
	if(m>0 && m<101)
	{
		imageMagnification = m;
	}
#endif
}


int iVTK::getFullImageWidth()
{
	if(imageComposer != 0)
	{
		return imageComposer->getImageWidth();
	}
	else return this->getThisImageWidth();
}


int iVTK::getFullImageHeight()
{
	if(imageComposer != 0)
	{
		return imageComposer->getImageHeight();
	}
	else return this->getThisImageHeight();
}


int iVTK::getThisImageWidth()
{
	switch(imageMethod)
	{
	case 0: return imageMagnification*this->width();
	case 1: return this->width();  // no magnification for method 1
	default: return 0;
	}
}


int iVTK::getThisImageHeight()
{
	switch(imageMethod)
	{
	case 0: return imageMagnification*this->height();
	case 1: return this->height();  // no magnification for method 1
	default: return 0;
	}
}


int iVTK::createImageData(vtkImageData* &data)
{
	if(imageComposer != 0)
	{
		//
		//  ask composer to composer the image
		//
		data = imageData;
		return imageComposer->compose(this,data);
	}
	else
	{
		return this->renderImageData(data);
	}

}


int iVTK::renderImageData(vtkImageData* &data)
{
	int iconVis = iconActor->GetVisibility();
	iconActor->VisibilityOff();
	
	//
	//  a few more things for better representation
	//
	float box2LineWidth = 4.0;
	if(boxType == 2) box2LineWidth = 1.0;

	//
	//  Status of preparing window for rendering an image: the VTK window must be
	//  visible and raised.
	//	Return values:
	//	0 -  window is ready.
	//  1 -  failed to prepare a window for unknown reason.
	//  2 -  failed to prepare window for a known reason, notification has been displayed.
	//
	int winStatus = 0;
	//
	//  Prepare for render
	//
	switch(imageMethod)
	{
	case 0: 
		{
			winStatus = this->isReadyForDumpingImage();
#ifndef I_VTK_VERSION_40
			box2Actor->GetProperty()->SetLineWidth(box2LineWidth*imageMagnification);
#endif
			break;
		}
	case 1:
		{
			vtkMapper::GlobalImmediateModeRenderingOn();
			break;
		}
	}
	
	//
	//  do the render
	//
	if(winStatus == 0)
	{
		longRenderProgress = 1;
		
		switch(imageMethod)
		{
		case 0:
			{
				renMagnification = imageMagnification;
				magnifier->SetMagnification(imageMagnification);
				magnifier->Modified();
				magnifier->Update();
				renMagnification = 1;
				imageData->DeepCopy(magnifier->GetOutput());
				data = imageData;
				data->Modified();
				break;
			}
		case 1:
			{
				//
				//  This uses Qt functions
				//
				createImageByMethod1();
				data = imageData;
				data->Modified();
				break;
			}
		}
		
		longRenderProgress = 0;
		//
		//  Reset to the present state
		//
		box2Actor->GetProperty()->SetLineWidth(box2LineWidth);
		iconActor->SetVisibility(iconVis);
		//
		// render again to restore the original view
		//
		this->render(true);
		
		switch(imageMethod)
		{
		case 0: 
			{
				break;
			}
		case 1:
			{
				vtkMapper::GlobalImmediateModeRenderingOff();
				break;
			}
		}
		
	}
	
	return winStatus;
	
}


int iVTK::dumpImage(int m, vtkImageData *data)
{
	
	iString nam;
	int cur;
	
	switch ( m ) 
	{
	case 0: { nam = im_inam; cur = im_icur; break; }
	case 1: { nam = im_anam; cur = im_acur; break; }
	default: return -1;
	}

	nam = iEnvironment::getInstance()->getFileName("IMAGE_DIR",(char *)nam.latin1()) + iString("").sprintf(".%04d",cur);
	switch ( imageType ) 
	{
	case I_IMAGETYPE_JPG: { nam = nam + iString(".jpg"); break; }
	case I_IMAGETYPE_PNM: { nam = nam + iString(".ppm"); break; }
	case I_IMAGETYPE_BMP: { nam = nam + iString(".bmp"); break; }
	case I_IMAGETYPE_PNG: { nam = nam + iString(".png"); break; }
	case I_IMAGETYPE_TIF: { nam = nam + iString(".tif"); break; }
	default:              { nam = nam + iString(".ps"); }
	}
	
	int winStatus = 0;
	if(data == 0) winStatus = this->createImageData(data);

	//
	// Create and write the file
	//
	float p = -0.13;
	obsProgress->Execute(this,0,&p);
	p = 0.0;
	obsProgress->Execute(this,0,&p);

	if(data != 0)
	{
		writer->SetInput(data);
		writer->SetFileName(nam.latin1());
		writer->Write();
		switch ( m ) 
		{
		case 0: { im_icur++; break; }
		case 1: { im_acur++; break; }
		}
	}
	else
	{
		if(winStatus == 1) this->reportDumpImageError();
	}
	
	obsEnd->Execute(this,0,0);
	longRenderProgress = 0;

	return winStatus;
	
}


void iVTK::computePositionShiftsUnderMagnification(int winij[2], vtkViewport *)
{
	int i;
	vtkCamera *cam = ren->GetActiveCamera();
	//
	//  Compute position shifts if under magnification
	//
	if(cam!=0 && renMagnification>1)
	{
		double winCenter[2];
		double err, errmax = 0.0;

		cam->GetWindowCenter(winCenter);

		for(i=0; i<2; i++) 
		{
			winij[i] = round(0.5*(winCenter[i]+renMagnification-1));
			err = fabs(winCenter[i]-(2*winij[i]-renMagnification+1));
			if(err > errmax) errmax = err;
		}

		if(errmax > 1.0e-5)
		{
			winij[0] = winij[1] = 0;
			return; // something is very wrong
		}
		
	}
	else winij[0] = winij[1] = 0;

}


void iVTK::updateGL()
{
	//
	//  Reserved for future enhancements
	//
	iRenderWindow::updateGL();
}

	
void iVTK::resetGL()
{
	
    glMatrixMode( GL_MODELVIEW );
    glDepthFunc( GL_LEQUAL );
    glEnable( GL_DEPTH_TEST );
    glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );
    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    glEnable(GL_BLEND);
    glEnable( GL_NORMALIZE );
//    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_true);
    glAlphaFunc(GL_GREATER,0); 
	glPolygonOffset(1.0,1.0);
	
}	


void iVTK::resetCamera()
{

	double *dp = ren->GetActiveCamera()->GetDirectionOfProjection();
	int idir = 1; if(dp[0] > 0.0) idir = -1;
	double dmax = fabs(dp[0]);
	if(fabs(dp[1]) > dmax)
	{
		dmax = fabs(dp[1]);
		idir = 2; if(dp[1] > 0.0) idir = -2;
	}
	if(fabs(dp[2]) > dmax)
	{
		idir = 3; if(dp[2] > 0.0) idir = -3;
	}

	ren->GetActiveCamera()->SetFocalPoint(0.0,0.0,0.0);
	ren->GetActiveCamera()->SetPosition(0.0,0.0,7.0);
	ren->GetActiveCamera()->SetViewUp(0.0,1.0,0.0);
	ren->GetActiveCamera()->SetParallelScale(1.6);
	ren->ResetCameraClippingRange();

	switch (idir)
	{
	case  1: { ren->GetActiveCamera()->Azimuth(90.0); break; }
	case -1: { ren->GetActiveCamera()->Azimuth(-90.0); break; }
	case  2: { ren->GetActiveCamera()->Elevation(90.0); ren->GetActiveCamera()->SetViewUp(0.0,0.0,-1.0); break; }
	case -2: { ren->GetActiveCamera()->Elevation(-90.0); ren->GetActiveCamera()->SetViewUp(0.0,0.0,1.0); break; }
	case  3: { break; }
	case -3: { ren->GetActiveCamera()->Azimuth(180.0); break; }
	}

}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVTK::packState(iString &s)
{

	s = "";

	this->packValue(s,"iVTK::isFlight",isFlight);
	this->packValue(s,"iVTK::isDebug",isDebug);
	this->packValue(s,"iVTK::parallelProjection",parallelProjection);
	this->packValue(s,"iVTK::isLabelOn",isLabelOn);
	this->packValue(s,"iVTK::isColorBarsOn",isColorBarsOn);
	this->packValue(s,"iVTK::isBoundingBoxOn",isBoundingBoxOn);
	this->packValue(s,"iVTK::isRulerOn",isRulerOn);
	this->packValue(s,"iVTK::isMeasuringBoxOn",isMeasuringBoxOn);
	this->packValue(s,"iVTK::isGlassClipPlaneOn",isGlassClipPlaneOn);
	this->packValue(s,"iVTK::isClipPlaneOn",isClipPlaneOn);
	this->packValue(s,"iVTK::antialiasing",antialiasing);
	this->packValue(s,"iVTK::labelJustificationLeft",labelJustificationLeft);
	this->packValue(s,"iVTK::clone",clone);
	this->packValue(s,"iVTK::autoRaise",autoRaise);
	this->packValue(s,"iVTK::focusCurrent",focusCurrent);
	this->packValue(s,"iVTK::broadcastMouseEvents",broadcastMouseEvents);
	this->packValue(s,"iVTK::markerLegendOn",markerLegendOn);

	this->packValue(s,"iVTK::fontSize",fontSize);
	this->packValue(s,"iVTK::fontType",fontType);
	this->packValue(s,"iVTK::boxType",boxType);
	if(clone) this->packValue(s,"iVTK::cloneParent",dataReader->getVTK()->getWindowNumber()); else this->packValue(s,"iVTK::cloneParent",0);
	this->packValue(s,"iVTK::imageType",imageType);
	this->packValue(s,"iVTK::width()",this->width());
	this->packValue(s,"iVTK::height()",this->height());
	this->packValue(s,"iVTK::markerLegendPosition",markerLegendPosition);
	this->packValue(s,"iVTK::pickedPointSize",pickedPointSize);
	this->packValue(s,"iVTK::imageMagnification",imageMagnification);
	this->packValue(s,"iVTK::imageMethod",imageMethod);

	this->packValue(s,"iVTK::background",this->getBackgroundColor());

	this->packValue(s,"iVTK::rate",rate);
	this->packValue(s,"iVTK::boxSize",boxSize);
	//
	//  Camera orientation
	//
	this->packValue(s,"iVTK::camera->GetPosition()",ren->GetActiveCamera()->GetPosition(),3);
	this->packValue(s,"iVTK::camera->GetFocalPoint()",ren->GetActiveCamera()->GetFocalPoint(),3);
	this->packValue(s,"iVTK::camera->GetViewUp()",ren->GetActiveCamera()->GetViewUp(),3);
	this->packValue(s,"iVTK::camera->GetParallelScale()",ren->GetActiveCamera()->GetParallelScale());

}


void iVTK::unpackState(iString s)
{
	int i, i2; bool b; float f; iColor c; vtkFloat v[3];

	if(this->unpackValue(s,"iVTK::isFlight",b)) this->setInteractorStyleFlight(b);
	if(this->unpackValue(s,"iVTK::isDebug",b)) this->setDebugMode(b);
	if(this->unpackValue(s,"iVTK::parallelProjection",b)) this->setParallelProjection(b);
	if(this->unpackValue(s,"iVTK::isLabelOn",b)) this->showLabel(b);
	if(this->unpackValue(s,"iVTK::isColorBarsOn",b)) this->showColorBars(b);
	if(this->unpackValue(s,"iVTK::isBoundingBoxOn",b)) this->showBox(b);
	if(this->unpackValue(s,"iVTK::isRulerOn",b)) this->showRuler(b);
	if(this->unpackValue(s,"iVTK::isMeasuringBoxOn",b)) this->showMeasuringBox(b);
	if(this->unpackValue(s,"iVTK::isGlassClipPlaneOn",b)) this->showGlassClipPlane(b);
	if(this->unpackValue(s,"iVTK::isClipPlaneOn",b)) this->showClipPlane(b);
	if(this->unpackValue(s,"iVTK::antialiasing",b)) this->setAntialiasingOn(b);
//	if(this->unpackValue(s,"iVTK::labelJustificationLeft",b)) this->justifyLabelLeft(b);
	if(this->unpackValue(s,"iVTK::clone",b)) clone = b;
	if(this->unpackValue(s,"iVTK::autoRaise",b)) this->setAutoRaiseOn(b);
	if(this->unpackValue(s,"iVTK::focusCurrent",b)) this->setCurrentWindowUnderFocusOn(b);
	if(this->unpackValue(s,"iVTK::broadcastMouseEvents",b)) this->setBroadcastMouseEventsOn(b);
	if(this->unpackValue(s,"iVTK::markerLegendOn",b)) this->showMarkerLegend(b);

	if(this->unpackValue(s,"iVTK::fontSize",i)) this->setFontSize(i);
	if(this->unpackValue(s,"iVTK::fontType",i)) this->setFontType(i);
	if(this->unpackValue(s,"iVTK::boxType",i)) this->setBoxType(i);
	if(this->unpackValue(s,"iVTK::imageType",i)) this->setImageType(i);
	if(this->unpackValue(s,"iVTK::width()",i) && this->unpackValue(s,"iVTK::height()",i2)) this->resize(i,i2);
	if(this->unpackValue(s,"iVTK::markerLegendPosition",i)) this->setMarkerLegendPosition(i);
	if(this->unpackValue(s,"iVTK::pickedPointSize",f)) this->setPickedPointSize(f);
	if(this->unpackValue(s,"iVTK::imageMagnification",i)) this->setImageMagnification(i);
	if(this->unpackValue(s,"iVTK::imageMethod",i)) imageMethod = i;
	
	if(this->unpackValue(s,"iVTK::background",c)) this->setBackgroundColor(c);

	if(this->unpackValue(s,"iVTK::rate",f)) this->setUpdateRate(f);
	if(this->unpackValue(s,"iVTK::boxSize",f)) 
	{
		this->useOpenGLCoordinates(f < 0.0);
		this->setBoxSize(f);
	}
	//
	//  Camera orientation
	//
	if(this->unpackValue(s,"iVTK::camera->GetPosition()",v,3)) ren->GetActiveCamera()->SetPosition(v);
	if(this->unpackValue(s,"iVTK::camera->GetFocalPoint()",v,3)) ren->GetActiveCamera()->SetFocalPoint(v);
	if(this->unpackValue(s,"iVTK::camera->GetViewUp()",v,3)) ren->GetActiveCamera()->SetViewUp(v);
	if(this->unpackValue(s,"iVTK::camera->GetParallelScale()",f)) ren->GetActiveCamera()->SetParallelScale(f);


	if(this->unpackValue(s,"iVTK::cloneParent",i)) if(clone) becomeClone(iVTKWindow::getWindow(i));

}


void iVTK::placeMarker(bool s)
{
#ifndef I_VTK_VERSION_40
	if(s)
	{
		double *x = markerFamily->getCurrentMember()->getPosition();
		markerPlacer->SetPosition(x[0],x[1],x[2]);
		markerPlacer->On();
	}
	else
	{
		markerPlacer->Off();
	}
#endif
}


void iVTK::moveMarkerCaption(bool s)
{
#ifndef I_VTK_VERSION_40
	if(s)
	{
		prevIstyle = iren->GetInteractorStyle();
		iren->SetInteractorStyle(istyleCaption);
	}
	else
	{
		iren->SetInteractorStyle(prevIstyle);
	}
#endif
}


void iVTK::showMarkerLegend(bool s)
{
#ifndef I_VTK_VERSION_40
	markerLegendOn = s;
	if(s) markerLegend->VisibilityOn(); else markerLegend->VisibilityOff();
#endif
}


void iVTK::buildMarkerLegend(int n)
{
#ifndef I_VTK_VERSION_40
	int i;
	vtkFloat c[3];
	if(n<0 || n>markerFamily->getMaxMemberIndex())
	{
		markerLegend->SetNumberOfEntries(1+markerFamily->getMaxMemberIndex());
		for(i=0; i<=markerFamily->getMaxMemberIndex(); i++)
		{
			c[0] = markerFamily->getMember(i)->getColor().red()/255.0;
			c[1] = markerFamily->getMember(i)->getColor().green()/255.0;
			c[2] = markerFamily->getMember(i)->getColor().blue()/255.0;
		//
		//  Fix a bug in vtkLegendBoxActor
		//
			if(markerFamily->getMember(i)->getCaption().length() > 0) markerLegend->SetEntry(i,markerFamily->getMember(i)->getMarkerObject(),markerFamily->getMember(i)->getCaption().latin1(),c); else markerLegend->SetEntry(i,markerFamily->getMember(i)->getMarkerObject()," ",c);
		}
	}
	else
	{
		c[0] = markerFamily->getMember(n)->getColor().red()/255.0;
		c[1] = markerFamily->getMember(n)->getColor().green()/255.0;
		c[2] = markerFamily->getMember(n)->getColor().blue()/255.0;
		//
		//  Fix a bug in vtkLegendBoxActor
		//
		if(markerFamily->getMember(n)->getCaption().length() > 0) markerLegend->SetEntry(n,markerFamily->getMember(n)->getMarkerObject(),markerFamily->getMember(n)->getCaption().latin1(),c); else markerLegend->SetEntry(n,markerFamily->getMember(n)->getMarkerObject()," ",c);
	}
#endif
}


void iVTK::setMarkerLegendPosition(int p)
{
#ifndef I_VTK_VERSION_40
	if(p>=0 && p<=1)
	{
		markerLegendPosition = p;
		switch (p)
		{
		case 0:
			{
				markerLegend->SetPosition(0.73,0.02);
				break;
			}
		case 1:
			{
				markerLegend->SetPosition(0.02,0.02);
				break;
			}
		}
	}
#endif
}


void iVTK::setCurrentParticlesFamily(int, int)
{
}


void iVTK::setCurrentParticlesFamily(int dc)
{
	if(dc>=0 && dc<=MAXDATACHANNEL) this->setCurrentParticlesFamily(dc,dataReader->getCurrentDataStream(dc));
}


iVisualObjectFamily<iParticles>* iVTK::getParticlesFamily()
{ 
	return particlesFamily; 
}


iVisualObjectFamily<iParticles>* iVTK::getParticlesFamily(int dc, int ds)
{ 
	if(dc==0 && ds==0) return particlesFamily; else return 0;
}


iParticlesSplitter* iVTK::getParticlesSplitter(int dc, int ds)
{ 
	if(dc==0 && ds==0) return particlesSplitter; else return 0;
}


iParticlesSplitter* iVTK::getCurrentParticlesSplitter()
{ 
	return particlesSplitter;
}


void iVTK::synchronizeObjectsAndData()
{
	int i;

	i = 0;
	while(i<=surfaceFamily->getMaxMemberIndex())
	{
		iSurface *s = surfaceFamily->getMember(i);
		if(!dataReader->isThereMeshData(s->getDataChannel(),s->getDataStream()) && surfaceFamily->deleteMember(i)) i--; 
		i++;
	}

	i = 0;
	while(i<=xsectionFamily->getMaxMemberIndex())
	{
		iXsection *s = xsectionFamily->getMember(i);
		if(!dataReader->isThereMeshData(s->getDataChannel(),s->getDataStream()) && xsectionFamily->deleteMember(i)) i--; 
		i++;
	}

	i = 0;
	while(i<=volumeFamily->getMaxMemberIndex())
	{
		iVolume *s = volumeFamily->getMember(i);
		if(!dataReader->isThereMeshData(s->getDataChannel(),s->getDataStream()) && volumeFamily->deleteMember(i)) i--; 
		i++;
	}

	i = 0;
	while(i<=particlesFamily->getMaxMemberIndex())
	{
		iParticles *s = particlesFamily->getMember(i);
		if(!dataReader->isTherePartData(s->getDataChannel(),s->getDataStream()) && particlesFamily->deleteMember(i)) i--; 
		i++;
	}

	i = 0;
	while(i<=vectorFamily->getMaxMemberIndex())
	{
		iVector *s = vectorFamily->getMember(i);
		if(!dataReader->isThereVectData(s->getDataChannel(),s->getDataStream()) && vectorFamily->deleteMember(i)) i--; 
		i++;
	}

	i = 0;
	while(i<=tensorFamily->getMaxMemberIndex())
	{
		iTensor *s = tensorFamily->getMember(i);
		if(!dataReader->isThereTensData(s->getDataChannel(),s->getDataStream()) && tensorFamily->deleteMember(i)) i--; 
		i++;
	}

}


float iVTK::getMemorySize()
{
	float s = 0.0;
	s += dataReader->getMemorySize();
	s += dataGateway->getMemorySize();
	s += surfaceFamily->getMemorySize();
	s += xsectionFamily->getMemorySize();
	s += volumeFamily->getMemorySize();
	s += particlesFamily->getMemorySize();
	s += vectorFamily->getMemorySize();
	s += tensorFamily->getMemorySize();
	return s;
}


//
//  Coordinate transformation functions
//
void iVTK::useOpenGLCoordinates(bool s)
{
	if(s == (boxSize<0.0)) return;
	if(s) 
	{
		boxSize = -fabs(boxSize); 
		box3Cube->SetRanges(-1.0,1.0,-1.0,1.0,-1.0,1.0);
	}
	else 
	{
		boxSize = fabs(boxSize);
		box3Cube->SetRanges(0.0,boxSize,0.0,boxSize,0.0,boxSize);
	}
	if(boxType == 3) this->render(false);
}


void iVTK::setBoxSize(float s)
{ 
	if(s > 0.0)
	{
		if(boxSize < 0) boxSize = -s; else
		{
			boxSize = s;
			box3Cube->SetRanges(0.0,s,0.0,s,0.0,s); 
			if(boxType == 3) this->render(false);
		}
	}
}


double iVTK::transformToInternalCoordinate(double x)
{
	if(boxSize > 0.0) return -1.0 + 2.0*x/boxSize; else return x;
}


double iVTK::transformToExternalCoordinate(double x)
{
	if(boxSize > 0.0) return 0.5*(1.0+x)*boxSize; else return x;
}


void iVTK::transformToInternalCoordinate(int n, double *x)
{
	if(boxSize > 0.0) this->transformToInternalCoordinate(n,x,x);
}


void iVTK::transformToExternalCoordinate(int n, double *x)
{
	if(boxSize > 0.0) this->transformToExternalCoordinate(n,x,x);
}


void iVTK::transformToInternalCoordinate(int n, double *src, double *dest)
{
	int i;
	for(i=0; i<n; i++) dest[i] = this->transformToInternalCoordinate(src[i]);
}


void iVTK::transformToExternalCoordinate(int n, double *src, double *dest)
{
	int i;
	for(i=0; i<n; i++) dest[i] = this->transformToExternalCoordinate(src[i]);
}


//
//  User callback function for window interactor
//
void iRenderWindowInteractor::UserCallback()
{
	iVTKWindow::getCurrentWindow()->dumpImage();
}
