/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "itextvector.h"
#include "imath.h"
#include "ivtk.h"

#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkRenderer.h>
#include <vtkViewport.h>
#include <vtkProperty2D.h>
#include <vtkVectorText.h>
#include <vtkTransform.h>
#include <vtkTransformPolyDataFilter.h>
#include <vtkPolyDataMapper2D.h>
#include <vtkPolyData.h>
#include <vtkActor2D.h>


void reportNullPointer(int);


iTextVector::iTextVector()
{
	
	textSource = vtkVectorText::New();
	if(textSource == 0) reportNullPointer(6101);
	textTransform = vtkTransform::New();
	if(textTransform == 0) reportNullPointer(6102);
	textFilter = vtkTransformPolyDataFilter::New();
	if(textFilter == 0) reportNullPointer(6103);
	textMapper = vtkPolyDataMapper2D::New();
	if(textMapper == 0) reportNullPointer(6104);
	textActor = vtkActor2D::New();
	if(textActor == 0) reportNullPointer(6105);
	textActorShadow = vtkActor2D::New();
	if(textActorShadow == 0) reportNullPointer(6106);
	
	textFilter->SetInput(textSource->GetOutput());
	textFilter->SetTransform(textTransform);
	textMapper->SetInput(textFilter->GetOutput());
	textActor->SetMapper(textMapper);
	textActorShadow->SetMapper(textMapper);

	textActor->GetProperty()->SetPointSize(4.0);
	textActorShadow->GetProperty()->SetPointSize(4.0);

	textActor->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();
	textActorShadow->GetPositionCoordinate()->SetCoordinateSystemToNormalizedViewport();

}


iTextVector::~iTextVector()
{
	
	textSource->Delete();
	textTransform->Delete();
	textFilter->Delete();
	textMapper->Delete();
	textActor->Delete();
	textActorShadow->Delete();

}


void iTextVector::getSize(vtkViewport *viewport, float s[2])
{

	iVTK *vtk = iVTK::convertFromViewport(viewport);
	int mag = vtk->getRenderingMagnification();

	float cosa = cos(angle*0.017453292);
	float sina = sin(angle*0.017453292);

	textSource->SetText(text);
	textSource->Update();
	vtkFloat *b = textSource->GetOutput()->GetBounds();
	
	float w = 0.618*(b[1]-b[0])*fontSize;
	float h = 0.953*fontSize;
	float wt = w*fabs(cosa) + h*fabs(sina);
	float ht = h*fabs(cosa) + w*fabs(sina);

	float c[4][2], x[2];

	c[0][0] = c[3][0] = 0.0;
	c[1][0] = c[2][0] = w;
	c[0][1] = c[1][1] = 0.0;
	c[2][1] = c[3][1] = h;

	float dx = 0.0, dy = 0.0;
	int j;
	for(j=0; j<4; j++)
	{
		x[0] = c[j][0]*cosa - c[j][1]*sina;
		x[1] = c[j][0]*sina + c[j][1]*cosa;
		if(x[0] < dx) dx = x[0];
		if(x[1] < dy) dy = x[1];
	}

	textTransform->Identity();
	textTransform->Translate(-dx*mag,-dy*mag,0.0);
	textTransform->RotateZ(angle);
	textTransform->Scale(0.6*mag*fontSize,0.65*mag*fontSize,1.0);
	textTransform->Translate(-0.17,0.09,0.0);
	textMapper->GetInput()->Update();

	int *ws = viewport->GetSize();
	size[0] = s[0] = wt/ws[0]; 
	size[1] = s[1] = ht/ws[1];

}


void iTextVector::ReleaseGraphicsResources(vtkWindow *win)
{
	textActor->ReleaseGraphicsResources(win);
	textActorShadow->ReleaseGraphicsResources(win);
	this->iText::ReleaseGraphicsResources(win);
}


int iTextVector::RenderOverlay(vtkViewport* viewport)
{
	textActorShadow->RenderOverlay(viewport);
	textActor->RenderOverlay(viewport);
	return 1;
}


int iTextVector::RenderOpaqueGeometry(vtkViewport* viewport)
{
	int *ws = viewport->GetSize();
	float cosa = cos(angle*0.017453292);
	float sina = sin(angle*0.017453292);

	float pshadow[2];
	pshadow[0] = 1.0/ws[0]*(cosa+sina);
	pshadow[1] = -1.0/ws[1]*(cosa-sina);

	this->getSize(viewport,s);

	iVTK *vtk = iVTK::convertFromViewport(viewport);
	int mag = vtk->getRenderingMagnification();

	if(mag == 1)
	{
		
		p[0] = pos[0] - 0.5*hJust*s[0];
		p[1] = pos[1] - 0.5*vJust*s[1];

		textActor->GetProperty()->SetColor(red,green,blue);

		textActor->SetPosition(p[0],p[1]);
		textActor->SetPosition2(s[0],s[1]);
		textActorShadow->SetPosition(p[0]+pshadow[0],p[1]+pshadow[1]);
		textActorShadow->SetPosition2(s[0],s[1]);

	}
	else
	{
		int winij[2];
		vtk->computePositionShiftsUnderMagnification(winij,viewport);
		
		textActor->SetPosition(mag*p[0]-winij[0],mag*p[1]-winij[1]);
		textActor->SetPosition2(mag*s[0],mag*s[1]);
		textActorShadow->SetPosition(mag*(p[0]+pshadow[0])-winij[0],mag*(p[1]+pshadow[1])-winij[1]);
		textActorShadow->SetPosition2(mag*s[0],mag*s[1]);

	}
	//
	//  shadow color
	//
    float cshadow = (red+green+blue)/3.0 > 0.5 ? 0.0 : 1.0;

	textActorShadow->GetProperty()->SetColor(cshadow,cshadow,cshadow);
	textActorShadow->RenderOpaqueGeometry(viewport);
	textActor->GetProperty()->SetColor(red,green,blue);
	textActor->RenderOpaqueGeometry(viewport);

	return 1;
				
}
