/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "ivtk.h"
#include "iqt.h"
#include "ivtkwindow.h"
#include "iqtwindow.h"

#include "iqtdefs.h"

#include "iqt_view.h"
#include "iqt_surf.h"
#include "iqt_xsec.h"
#include "iqt_volv.h"
#include "iqt_part.h"
#include "iqt_vect.h"
#include "iqt_tens.h"
#include "iqt_data.h"
#include "iqt_wins.h"

#include "idialoghelp.h"
#include "idialogpaletteeditor.h"
#include "idialogscripter.h"
#include "idialogeventrecorder.h"
#include "idialogprogress.h"
#include "ianimatorscript.h"
#include "ianimator.h"
#include "ivtkcallback.h"
#include "idatareader.h"
#include "ivolume.h"
#include "idialogpick.h"
#include "idatagateway.h"
#include "ierror.h"
#include "ilabel.h"
#include "imath.h"
#include "iparticlessplitter.h"
#include "isurface.h"
#include "ixsection.h"
#include "iparticles.h"
#include "ivector.h"
#include "itensor.h"
#include "iprogressbar.h"
#include "ienvironment.h"
#include "ipiecewisefunction.h"
#include "ilimits.h"
#include "ipalette.h"
#include "ipostscriptwriter.h"
#include "imarker.h"
#include "idialogparallelcontroller.h"
#include "idialogvariablesexplorer.h"
#include "idialogimagecomposer.h"
#include "ipostscriptwriter.h"


#include "iqtextension.h"
#include "iobjectfactory.h"

#include <vtkCamera.h>
#include <vtkMatrix4x4.h>
#include <vtkImageWriter.h>

#include <qnamespace.h>
#include <qaction.h>
#include <qapplication.h>
#include <qbitmap.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qcursor.h>
#include <qdir.h>
#include <qfile.h>
#include <qgroupbox.h>
#include <qlayout.h>
#include <qlcdnumber.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qmessagebox.h>
#include <qobject.h>
#include <qobjectlist.h>
#include <qprogressbar.h>
#include <qprogressdialog.h>
#include <qpushbutton.h>
#include <qradiobutton.h>
#include <qregexp.h>
#include <qslider.h>
#include <qspinbox.h>
#include <qstatusbar.h>
#include <qtabwidget.h>
#include <qtextedit.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qtoolbutton.h>
#include <qtooltip.h>
#include <qwidgetstack.h>
#include <qworkspace.h>

#include <vtkRenderer.h>



QPixmap image(char *s);
void reportNullPointer(int ec);
float iMemLimit();


//
//  Restorable widgets
//
const int nTypes = 7;
static const char *objectTypes[nTypes] = 
{
	"QComboBox",
	"QCheckBox", 
	"QSlider",
	"QButtonGroup",
	"QAction",
	"QSpinBox",
	"QLabel"
};

//
// *********************************************************************************
//
//  
//   Constructor, destructor, and some auxilary slots
//
//
// *********************************************************************************
//
iQT* iQT::New(iVTK *)
{
	return new iQT;
}
//
//  This function is called at the end of IFRIT constructor
//
void iQT::init()
{
	int i;

	TabWidgetView = 0;
	TabWidgetSurf = 0;
	TabWidgetXsec = 0;
	TabWidgetVolv = 0;
	TabWidgetPart = 0;
	TabWidgetVect = 0;
	TabWidgetTens = 0;
	TabWidgetData = 0;
	
	TabLayoutView = 0;
	TabLayoutSurf = 0;
	TabLayoutXsec = 0;
	TabLayoutVolv = 0;
	TabLayoutPart = 0;
	TabLayoutVect = 0;
	TabLayoutTens = 0;
	TabLayoutData = 0;
	//
	//  Set widget specific parameters
	//    
	inTask = false;
	blockRenderer = false;
	remoteControlled = false;
	idiosyncratic = true;
	optionsAffectAll = false;

	dataChannel = 0;
	stopAnimation = 0;

	reloadLastSetMode = 0;

	histoColor = QColor(160,160,160);

    nlinesTextLog = 1000;
    
    dockState = false;
	FrameLeft->hide();

	workspace = new QWorkspace(FrameLeft);
	workspace->setScrollBarsEnabled(true); 
	workspace->setSizePolicy(QSizePolicy(QSizePolicy::Expanding,QSizePolicy::Expanding));
	FrameLeft->layout()->add(workspace);
    
    scaleSlider1 = 10.0;
    scaleSlider2 = 100.0;
	
	variableLimits = 2;
	//
	//  Specify, whether sliders call Renderer while tracking
	//     
    sliderRenderTracking = false;
    reportMemory = true;
	//
	//  Set icon
	//
	this->setIcon(image("genie1qt.png"));
	//    
    //  Set up pixmaps
	//
    pixmapPaint = QPixmap(256,32);

	pixmapDC = new QPixmap[MAXDATACHANNEL+1];

	pixmapDC[0] = image("mesh.png");

	pixmapBusyOn = image("er_recon.png");
	pixmapBusyOff = QPixmap(22,22); //image("er_stop.png");
	pixmapBusyOff.fill();
    pixmapBusyOff.setMask(pixmapBusyOff.createHeuristicMask());

	//	Volv_Opaci_FunctionPixmap->setScaledContents(true);
	//	Part_Size_FunctionPixmap->setScaledContents(true);
	//
	//  Create the status bar
	//
	busyLabel = new QLabel(statusBar(),"BusyLabel");
	busyLabel->setPixmap(pixmapBusyOff);
	QToolTip::add(busyLabel,"Busy indicator");
    ProgressBar = new iProgressBar(statusBar(),"ProgressBar");
	if(ProgressBar == 0) reportNullPointer(1007);
	QToolTip::add(ProgressBar,"Progress bar");
	//ProgressBar->setCenterIndicator(true);
    QLabel* FileLabel = new QLabel("  File:",statusBar(),"CurrentFileLabel");
	if(FileLabel == 0) reportNullPointer(1008);
    CurrentFileStack = new QWidgetStack(statusBar(),"CurrentFileStack");
	if(CurrentFileStack == 0) reportNullPointer(1009);
	QToolTip::add(CurrentFileStack,"List of visited files");

	this->createVisitedFilesList(iVTKWindow::getCurrentWindowIndex());
	this->updateVisitedFilesList(true);

    statusBar()->addWidget(busyLabel,0,true);
    statusBar()->addWidget(ProgressBar,2,true);
    statusBar()->addWidget(FileLabel,0,true);
    statusBar()->addWidget(CurrentFileStack,3,true);
	//
	//  Send the pointer to progress bar to iVTKcallback observers
	//    
    iVTKWindow::getCurrentWindow()->getStartEventObserver()->setProgressBar(ProgressBar);
    iVTKWindow::getCurrentWindow()->getProgressEventObserver()->setProgressBar(ProgressBar);
    iVTKWindow::getCurrentWindow()->getEndEventObserver()->setProgressBar(ProgressBar);
	//
	//  Create the progress dialog
	//
    ProgressDialog = new iDialogProgress(this,0,true);
	if(ProgressDialog == 0) reportNullPointer(1010);
	//
	//  Send the pointer to progress dialog to iVTKcallback observers
	//    
    iVTKWindow::getCurrentWindow()->getAbortRenderEventObserver()->setProgressDialog(ProgressDialog);
	//
	//  Create animation scripter
	//
	AnimationScripter = new iDialogScripter(NULL,"AnimationScripter");
	if(AnimationScripter == 0) reportNullPointer(1011);
	//
	//  Create help
	//
	Help = new iDialogHelp;
	if(Help == 0) reportNullPointer(1021);
	//
	//  Create picker
	//
	Pick = new iDialogPick;
	if(Pick == 0) reportNullPointer(1022);
    iVTKWindow::getCurrentWindow()->getPickEventObserver()->setDialogPick(Pick);
	//
	//  Create parallel controller
	//
	ParallelController = new iDialogParallelController;
	if(ParallelController == 0) reportNullPointer(1024);
	//
	//  Create variables explorer
	//
	VariablesExplorer = iObjectFactory::createVariablesExplorer();
	if(VariablesExplorer == 0) reportNullPointer(1024);
	//
	//  Create image composer
	//
	ImageComposer = new iDialogImageComposer();
	if(ImageComposer == 0) reportNullPointer(1024);
	iVTKWindow::getCurrentWindow()->setImageComposer(ImageComposer);

	//
	//  Create event recorder
	//
#ifndef I_VTK_VERSION_40
	EventRecorder = new iDialogEventRecorder;
	if(EventRecorder == 0) reportNullPointer(1023);
    iVTKWindow::getCurrentWindow()->getRecordEventObserver()->setDialogEventRecorder(EventRecorder);
#endif
	//
	//  Fill in PostScriptWrite page size menu
	//
	psPaperFormatList = new QAction*[NFORMATS];
	
	iPostScriptWriter *tmp = iPostScriptWriter::New(); // create one to fill in static variables

	for(i=0; i<NFORMATS; i++)
	{
		psPaperFormatList[i] = new QAction(optionsImageFormatPSPaperFormatActionGroup);
		psPaperFormatList[i]->setMenuText(iPostScriptWriter::getPaperFormatName(i));
		psPaperFormatList[i]->setToggleAction(true);
	}
	CALL_FUNCTION1(psPaperFormatList[tmp->getPaperFormat()],setOn,true);

	tmp->Delete();

	if(iVTKWindow::getCurrentWindow()->getImageType() != 5)
	{
		optionsImageFormatPSPaperFormatActionGroup->setEnabled(false);
		optionsImageFormatPSOrientationActionGroup->setEnabled(false);
	}
	//
	//  Define cursor
	//
	this->resetCursor();
	//
	//  Create extension
	//
	ext = iObjectFactory::createQTExtension(this);

	initialized = false;
	
}
//
//  This function is called in the beginning of IFRIT desctructor
//
void iQT::destroy()
{
	int i;

	for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) 
	{
		iVTKWindow::getWindow(i)->getPickEventObserver()->setDialogPick(0);
	    iVTKWindow::getWindow(i)->getAbortRenderEventObserver()->setProgressDialog(0);
#ifndef I_VTK_VERSION_40
		iVTKWindow::getWindow(i)->getRecordEventObserver()->setDialogEventRecorder(0);
#endif
	}

    delete ProgressBar;
    delete CurrentFile;
    delete ProgressDialog;
	delete ParallelController;
	delete VariablesExplorer;
	delete ImageComposer;

	delete AnimationScripter;
	delete Help;
	delete Pick;
#ifndef I_VTK_VERSION_40
	delete EventRecorder;
#endif

	delete[] psPaperFormatList;

	if(ext!=NULL && !dockState) delete ext;
	
}
//
//  Additional initialization
//
void iQT::finishInitialization()
{
    int i,j;
	bool res;
	
	if(initialized) return;
	
	//
	//  No need to destroy those - QT will do it for us.
	//
	TabLayoutView = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_VIEW), 0, 6, ""); 
	TabWidgetView = new iQT_View(TabWidget->page(DISPLAYPAGE_VIEW),"TabWidgetView");
	TabLayoutView->addWidget(TabWidgetView);
	
	TabLayoutSurf = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_SURF), 0, 6, ""); 
	TabWidgetSurf = new iQT_Surf(TabWidget->page(DISPLAYPAGE_SURF),"TabWidgetSurf");
	TabLayoutSurf->addWidget(TabWidgetSurf);
	
	TabLayoutXsec = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_XSEC), 0, 6, ""); 
	TabWidgetXsec = new iQT_Xsec(TabWidget->page(DISPLAYPAGE_XSEC),"TabWidgetXsec");
	TabLayoutXsec->addWidget(TabWidgetXsec);
	
	TabLayoutVolv = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_VOLV), 0, 6, ""); 
	TabWidgetVolv = new iQT_Volv(TabWidget->page(DISPLAYPAGE_VOLV),"TabWidgetVolv");
	TabLayoutVolv->addWidget(TabWidgetVolv);
	
	TabLayoutPart = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_PART), 0, 6, ""); 
	TabWidgetPart = new iQT_Part(TabWidget->page(DISPLAYPAGE_PART),"TabWidgetPart");
	TabLayoutPart->addWidget(TabWidgetPart);
	
	TabLayoutVect = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_VECT), 0, 6, ""); 
	TabWidgetVect = new iQT_Vect(TabWidget->page(DISPLAYPAGE_VECT),"TabWidgetVect");
	TabLayoutVect->addWidget(TabWidgetVect);
	
	TabLayoutTens = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_TENS), 0, 6, ""); 
	TabWidgetTens = new iQT_Tens(TabWidget->page(DISPLAYPAGE_TENS),"TabWidgetTens");
	TabLayoutTens->addWidget(TabWidgetTens);
	
	TabLayoutData = new QVBoxLayout(TabWidget->page(DISPLAYPAGE_DATA), 0, 6, ""); 
	TabWidgetData = new iQT_Data(TabWidget->page(DISPLAYPAGE_DATA),"TabWidgetData");
	TabLayoutData->addWidget(TabWidgetData);

	TabWidget->updateGeometry();

	//
	// Initialize extension
	//
	if(ext != NULL)
	{
		ext->finishInitialization();
	}	
	//
	//  If we have no extension, remove data channel frames
	//
	if(ext==NULL || !ext->isActive())
	{
		DataChannelFrame->hide();
		TabWidgetView->View_Bars_LeftDataChannelFrame->hide();
		TabWidgetView->View_Bars_RightDataChannelFrame->hide();
	}

	//
	//  Disable the VolumePro button if VolumePro board is not available
	//    
    if(!iVolume::isVolumePro()) TabWidgetVolv->Volv_Field_MethodVolumeProButton->setEnabled(false);
	//
	//  Create a list of all variable radio buttons for run-time enabling/disabling.
	//    
    for(i=0; i<NWIDMAX; i++) for(j=0; j<NVARMAX; j++) button[i][j] = 0;
	
    nButtons = 0;
//    button[nButtons][0] = TabWidgetPart->Part_Paint_PaintAtt1;
//    button[nButtons][1] = TabWidgetPart->Part_Paint_PaintAtt2;
//    button[nButtons][2] = TabWidgetPart->Part_Paint_PaintAtt3;
//    nButtons++;
//    button[nButtons][0] = TabWidgetPart->Part_Size_SizeAtt1;
//    button[nButtons][1] = TabWidgetPart->Part_Size_SizeAtt2;
//    button[nButtons][2] = TabWidgetPart->Part_Size_SizeAtt3;
//    nButtons++;
//    button[nButtons][0] = TabWidgetPart->Part_Piece_SplitWithAtt1;
//    button[nButtons][1] = TabWidgetPart->Part_Piece_SplitWithAtt2;
//    button[nButtons][2] = TabWidgetPart->Part_Piece_SplitWithAtt3;
//    nButtons++;
//    button[nButtons][0] = TabWidgetPart->Part_Connt_ConnectWithAtt1;
//    button[nButtons][1] = TabWidgetPart->Part_Connt_ConnectWithAtt2;
//    button[nButtons][2] = TabWidgetPart->Part_Connt_ConnectWithAtt3;
//    nButtons++; 
//    button[nButtons][0] = TabWidgetPart->Part_Connt_SeparateWithAtt1;
//    button[nButtons][1] = TabWidgetPart->Part_Connt_SeparateWithAtt2;
//    button[nButtons][2] = TabWidgetPart->Part_Connt_SeparateWithAtt3;
//    nButtons++; 
	nAttButtons = nButtons;
    button[nButtons][0] = TabWidgetSurf->Surf_Level_SurfVar1;
    button[nButtons][1] = TabWidgetSurf->Surf_Level_SurfVar2;
    button[nButtons][2] = TabWidgetSurf->Surf_Level_SurfVar3;
    nButtons++;
    button[nButtons][0] = TabWidgetSurf->Surf_Paint_PaintVar1;
    button[nButtons][1] = TabWidgetSurf->Surf_Paint_PaintVar2;
    button[nButtons][2] = TabWidgetSurf->Surf_Paint_PaintVar3;
    nButtons++;
    button[nButtons][0] = TabWidgetXsec->Xsec_Field_FieldVar1;
    button[nButtons][1] = TabWidgetXsec->Xsec_Field_FieldVar2;
    button[nButtons][2] = TabWidgetXsec->Xsec_Field_FieldVar3;
    nButtons++;
    button[nButtons][0] = TabWidgetVolv->Volv_Field_PrimaryFieldVar1;
    button[nButtons][1] = TabWidgetVolv->Volv_Field_PrimaryFieldVar2;
    button[nButtons][2] = TabWidgetVolv->Volv_Field_PrimaryFieldVar3;
    nButtons++;
//    button[nButtons][0] = TabWidgetVect->Vect_Paint_PaintVar1;
//    button[nButtons][1] = TabWidgetVect->Vect_Paint_PaintVar2;
//    button[nButtons][2] = TabWidgetVect->Vect_Paint_PaintVar3;
//    nButtons++;
//    button[nButtons][0] = TabWidgetTens->Tens_Paint_PaintVar1;
//    button[nButtons][1] = TabWidgetTens->Tens_Paint_PaintVar2;
//    button[nButtons][2] = TabWidgetTens->Tens_Paint_PaintVar3;
//    nButtons++;
    button[nButtons][0] = TabWidgetData->Data_Calc_VarBoxVar1;
    button[nButtons][1] = TabWidgetData->Data_Calc_VarBoxVar2;
    button[nButtons][2] = TabWidgetData->Data_Calc_VarBoxVar3;
    nButtons++;
	
	//    button[nButtons][0] = Volv_Field_SecondaryFieldVar1;
	//    button[nButtons][1] = Volv_Field_SecondaryFieldVar2;
	//    button[nButtons][2] = Volv_Field_SecondaryFieldVar3;
	//    nButtons++;
	
    for(i=0; i<nButtons; i++) for(j=0; j<3; j++) if(button[i][j]) button[i][j]->setEnabled(false);
	//
	//  Create a list of all palette combo boxes and pixmap labels for run-time editing.
	//    
    for(i=0; i<NWIDMAX; i++) paletteBox[i] = 0;
	
	nPaletteBoxes = 0;
	paletteBox[nPaletteBoxes] = TabWidgetSurf->Surf_Paint_PaletteList; paletteBoxPixmap[nPaletteBoxes++] = TabWidgetSurf->Surf_Paint_PalettePixmap;
	paletteBox[nPaletteBoxes] = TabWidgetXsec->Xsec_Field_PaletteList; paletteBoxPixmap[nPaletteBoxes++] = TabWidgetXsec->Xsec_Field_PalettePixmap;
	paletteBox[nPaletteBoxes] = TabWidgetVolv->Volv_Paint_PaletteList; paletteBoxPixmap[nPaletteBoxes++] = TabWidgetVolv->Volv_Paint_PalettePixmap;
	paletteBox[nPaletteBoxes] = TabWidgetPart->Part_Paint_PaletteList; paletteBoxPixmap[nPaletteBoxes++] = TabWidgetPart->Part_Paint_PalettePixmap;
	paletteBox[nPaletteBoxes] = TabWidgetVect->Vect_Paint_PaletteList; paletteBoxPixmap[nPaletteBoxes++] = TabWidgetVect->Vect_Paint_PalettePixmap;
	paletteBox[nPaletteBoxes] = TabWidgetTens->Tens_Paint_PaletteList; paletteBoxPixmap[nPaletteBoxes++] = TabWidgetTens->Tens_Paint_PalettePixmap;
	paletteBox[nPaletteBoxes] = TabWidgetView->View_Bars_RightPaletteListComboBox; paletteBoxPixmap[nPaletteBoxes++] = NULL;
	paletteBox[nPaletteBoxes] = TabWidgetView->View_Bars_LeftPaletteListComboBox;  paletteBoxPixmap[nPaletteBoxes++] = NULL;
	//
	//  Fix toolbar icons
	//
	fileOpenMeshAction->setIconSet(QIconSet(image("fileopenmesh.png")));
	fileOpenPartAction->setIconSet(QIconSet(image("fileopenpart.png")));
	fileOpenVectAction->setIconSet(QIconSet(image("fileopenvect.png")));
	fileOpenTensAction->setIconSet(QIconSet(image("fileopentens.png")));
	
	fileOpenSetAction->setIconSet(QIconSet(image("fileopenset.png")));
	
	styleDockAction->setIconSet(QIconSet(image("dock.png")));
	
	actionShowSurf->setIconSet(QIconSet(image("surf.png")));
	actionShowXsec->setIconSet(QIconSet(image("xsec.png")));
	actionShowVolv->setIconSet(QIconSet(image("volv.png")));
	actionShowPart->setIconSet(QIconSet(image("part.png")));
	actionShowVect->setIconSet(QIconSet(image("vect.png")));
	actionShowTens->setIconSet(QIconSet(image("tens.png")));

	actionShowWins->setIconSet(QIconSet(image("wins.png")));

	dialogScripterAction->setIconSet(QIconSet(image("scted.png")));
	dialogPaletteEditorAction->setIconSet(QIconSet(image("paled.png")));
	dialogPickerAction->setIconSet(QIconSet(image("picks.png")));
	dialogEventRecorderAction->setIconSet(QIconSet(image("recorder.png")));
	dialogParallelControllerAction->setIconSet(QIconSet(image("parallel.png")));
	dialogVariablesExplorerAction->setIconSet(QIconSet(image("varexp.png")));
	dialogImageComposerAction->setIconSet(QIconSet(image("imcomp.png")));
	//
	//  Add respective icons to second level tabs
	//
	TabWidgetView->TabWidget->changeTab(TabWidgetView->TabWidget->page(0),image("view.png"),QString("Main"));
	TabWidgetView->TabWidget->changeTab(TabWidgetView->TabWidget->page(1),image("view.png"),QString("Scene"));
	TabWidgetView->TabWidget->changeTab(TabWidgetView->TabWidget->page(2),image("view.png"),QString("Animation"));
	TabWidgetView->TabWidget->changeTab(TabWidgetView->TabWidget->page(3),image("view.png"),QString("Windows"));
	
	TabWidgetSurf->TabWidget->changeTab(TabWidgetSurf->TabWidget->page(0),image("surf.png"),QString("Main"));
	TabWidgetSurf->TabWidget->changeTab(TabWidgetSurf->TabWidget->page(1),image("surf.png"),QString("Paint"));
	TabWidgetSurf->TabWidget->changeTab(TabWidgetSurf->TabWidget->page(2),image("surf.png"),QString("Instances"));
	
	TabWidgetXsec->TabWidget->changeTab(TabWidgetXsec->TabWidget->page(0),image("xsec.png"),QString("Main"));
	TabWidgetXsec->TabWidget->changeTab(TabWidgetXsec->TabWidget->page(1),image("xsec.png"),QString("Paint"));
	TabWidgetXsec->TabWidget->changeTab(TabWidgetXsec->TabWidget->page(2),image("xsec.png"),QString("Instances"));
	
	TabWidgetVolv->TabWidget->changeTab(TabWidgetVolv->TabWidget->page(0),image("volv.png"),QString("Main"));
	TabWidgetVolv->TabWidget->changeTab(TabWidgetVolv->TabWidget->page(1),image("volv.png"),QString("Paint"));
	TabWidgetVolv->TabWidget->changeTab(TabWidgetVolv->TabWidget->page(2),image("volv.png"),QString("Opacity"));
	TabWidgetVolv->TabWidget->changeTab(TabWidgetVolv->TabWidget->page(3),image("volv.png"),QString("Extra"));
	
	TabWidgetPart->TabWidget->changeTab(TabWidgetPart->TabWidget->page(0),image("part.png"),QString("Main"));
	TabWidgetPart->TabWidget->changeTab(TabWidgetPart->TabWidget->page(1),image("part.png"),QString("Paint"));
	TabWidgetPart->TabWidget->changeTab(TabWidgetPart->TabWidget->page(2),image("part.png"),QString("Size"));
	TabWidgetPart->TabWidget->changeTab(TabWidgetPart->TabWidget->page(3),image("part.png"),QString("Advanced"));
	
	TabWidgetVect->TabWidget->changeTab(TabWidgetVect->TabWidget->page(0),image("vect.png"),QString("Main"));
	TabWidgetVect->TabWidget->changeTab(TabWidgetVect->TabWidget->page(1),image("vect.png"),QString("Paint"));
	TabWidgetVect->TabWidget->changeTab(TabWidgetVect->TabWidget->page(2),image("vect.png"),QString("Streamlines"));
	
	TabWidgetTens->TabWidget->changeTab(TabWidgetTens->TabWidget->page(0),image("tens.png"),QString("Main"));
	TabWidgetTens->TabWidget->changeTab(TabWidgetTens->TabWidget->page(1),image("tens.png"),QString("Paint"));
	
	TabWidgetData->TabWidget->changeTab(TabWidgetData->TabWidget->page(0),image("data.png"),QString("Adjustment"));
	TabWidgetData->TabWidget->changeTab(TabWidgetData->TabWidget->page(1),image("data.png"),QString("Placement"));
	TabWidgetData->TabWidget->changeTab(TabWidgetData->TabWidget->page(2),image("data.png"),QString("Settings"));
	TabWidgetData->TabWidget->changeTab(TabWidgetData->TabWidget->page(3),image("data.png"),QString("Transform"));
	TabWidgetData->TabWidget->changeTab(TabWidgetData->TabWidget->page(4),image("data.png"),QString("Erase"));

	//
	//  Connect animation scripter
	//
	res = true;
	
	res = res & connect( AnimationScripter, SIGNAL( scriptRun() ) , this, SLOT( animate() ) );
	res = res & connect( AnimationScripter, SIGNAL( scriptFileSet(const QString &) ) , TabWidgetView->View_Anim_ScriptName, SLOT( setText(const QString &) ) ); 
	res = res & connect( (const QObject *)TabWidgetView->View_Anim_EditorButton, SIGNAL( clicked() ) , AnimationScripter, SLOT ( show() ) );
	//
	//  Connect actions to global View buttons
	//
	res = res & connect( actionShowSurf, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Surf(bool) ) );
	res = res & connect( actionShowXsec, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Xsec(bool) ) );
	res = res & connect( actionShowVolv, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Volv(bool) ) );
	res = res & connect( actionShowPart, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Part(bool) ) );
	res = res & connect( actionShowVect, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Vect(bool) ) );
	res = res & connect( actionShowTens, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Tens(bool) ) );
	//
	//  Connect local View buttons to global View buttons
	//
	res = res & connect( (const QObject *)TabWidgetSurf->View_SurfLocal, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Surf(bool) ) );
	res = res & connect( (const QObject *)TabWidgetXsec->View_XsecLocal, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Xsec(bool) ) );
	res = res & connect( (const QObject *)TabWidgetVolv->View_VolvLocal, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Volv(bool) ) );
	res = res & connect( (const QObject *)TabWidgetPart->View_PartLocal, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Part(bool) ) );
	res = res & connect( (const QObject *)TabWidgetVect->View_VectLocal, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Vect(bool) ) );
	res = res & connect( (const QObject *)TabWidgetTens->View_TensLocal, SIGNAL( toggled(bool) ), (const QObject *)TabWidgetView, SLOT( updateView_Tens(bool) ) );

	if(!res) this->popupError("Not all connections can be established.",IFRITERROR_NO_CONNECTIONS);
	//
	//  Edition-specific settings
	//
	this->edition();
	//
	//  Update menu setting as well
	//    
    CALL_FUNCTION1(optionsVarLimitsFixedAction,setOn,(variableLimits==0));
    CALL_FUNCTION1(optionsVarLimitsAdjustableAction,setOn,(variableLimits==1));
    CALL_FUNCTION1(optionsVarLimitsMixedAction,setOn,(variableLimits==2));
	iVTKWindow::getCurrentWindow()->getReader()->setAdjustableBounds(variableLimits>0);
    
	CALL_FUNCTION1(styleBoxIfritAction,setOn,(iVTKWindow::getCurrentWindow()->getBoxType()==0));
    CALL_FUNCTION1(styleBoxClassicAction,setOn,(iVTKWindow::getCurrentWindow()->getBoxType()==1));
    CALL_FUNCTION1(styleBoxHairAction,setOn,(iVTKWindow::getCurrentWindow()->getBoxType()==2));
    CALL_FUNCTION1(styleBoxAxesAction,setOn,(iVTKWindow::getCurrentWindow()->getBoxType()==3));
    if(iVTKWindow::getCurrentWindow()->getBoxType() == 3)	this->loadAxesBoxLabels();
	
    i = iVTKWindow::getCurrentWindow()->getFontSize();
    switch (i) 
	{
	case 10: { CALL_FUNCTION1(styleFontSize10Action,setOn,true); break; }
	case 12: { CALL_FUNCTION1(styleFontSize12Action,setOn,true); break; }
	case 14: { CALL_FUNCTION1(styleFontSize14Action,setOn,true); break; }
	case 16: { CALL_FUNCTION1(styleFontSize16Action,setOn,true); break; }
	case 20: { CALL_FUNCTION1(styleFontSize20Action,setOn,true); break; }
	case 24: { CALL_FUNCTION1(styleFontSize24Action,setOn,true); break; }
	case 32: { CALL_FUNCTION1(styleFontSize32Action,setOn,true); break; }
    }   
	
    i = iVTKWindow::getCurrentWindow()->getFontType();
    switch (i) 
	{
	case 0: { CALL_FUNCTION1(styleFontTypeBitmapAction,setOn,true); break; }
	case 1: { CALL_FUNCTION1(styleFontTypeVectorAction,setOn,true); break; }
    }   
	
   	iVTKWindow::getCurrentWindow()->setAntialiasingOn(styleAllowAntialiasingAction->isOn()); 
	//
	//  Establish contact with the pop-up dialog
	//    
	if(styleInteractorHelpAction->isOn()) iVTKWindow::getCurrentWindow()->setBottomWidget(FrameRight);
	//
	//  Set the style of top-level tabs
	//
	if(styleTabTextAction->isOn()){ this->styleTab(styleTabTextAction); }
	if(styleTabIconAction->isOn()){ this->styleTab(styleTabIconAction); }
	if(styleTabBothAction->isOn()){ this->styleTab(styleTabBothAction); }
	//
	//  Assign palette names
	//
	this->assignPaletteNames();
	//
	//  Finish with updating the widgets
	//    
	variousQTFixes();
	
	this->updateWidgets(-1);

#ifdef I_VTK_VERSION_40
	//
	//  Disable features not supported by earlier VTK versions
	//
	TabWidgetView->View_Anim_CameraPathGroupBox->setEnabled(false);
	TabWidgetView->View_Anim_CameraPathStyleRadioButton->setEnabled(false);
	dialogEventRecorderAction->setEnabled(false);
#endif
	//
	//  Finish
	//
	inUpdate = false;
	this->layout()->activate();
	this->updateGeometry();

	initialized = true;
	
}
//
//  Try to make the cobtrol window smaller
//
void iQT::setSmallWindow(bool s)
{
	
	int layoutMargin, fontSize;
	
	if(s)
	{
		fontSize = 8;
		layoutMargin = 6;
	} 
	else 
	{
		fontSize = 10;
		layoutMargin = 11;
		
	}
	
	QObjectList *list;
	QObject *obj;
	QFont f;
	
	list = this->queryList("QLayout");
	QObjectListIt it = QObjectListIt(*list);
	while ( (obj = it.current()) != 0 ) 
	{
		if(((QLayout *)obj)->margin() > layoutMargin) ((QLayout *)obj)->setMargin(layoutMargin);
		++it;
	}
	delete list;
	
	f = this->font();
	f.setPointSize(fontSize);
	this->setFont(f);
	
	list = this->queryList("QWidget");
	it = QObjectListIt(*list);
	while ( (obj = it.current()) != 0 ) 
	{
		if(((QWidget *)obj)->ownFont() && ((QWidget *)obj)->font().bold())
		{
			f = ((QWidget *)obj)->font();
			f.setPointSize(fontSize);
			((QWidget *)obj)->setFont(f);
		}
		++it;
	}
	delete list; 
	
	this->updateGeometry();
	
}
//
//  Functions called after each mouse event and after the work is done - used to
//  disable user interaction while working on a task.
//
void iQT::startTask()
{
	
	if(remoteControlled) return;
	if(inTask)
	{
		this->popupError("Incorrectly nested startTask function.",IFRITERROR_INCORRECTLY_NESTED_STARTTASK_FUNCTION);
		this->close();
	}
	inTask = true;

	this->blockSignals(true);
	if(ext != NULL) ext->blockSignals(true);

}


void iQT::endTask()
{
	
	if(remoteControlled) return;
	if(!inTask)
	{
		this->popupError("Incorrectly nested endTask function.",IFRITERROR_INCORRECTLY_NESTED_ENDTASK_FUNCTION);
		iVTKWindow::getCurrentWindow()->setCloseable();
		iVTKWindow::getCurrentWindow()->close();
		this->close();
	}
	inTask = false;
	
	this->blockSignals(false);
	if(ext != NULL) ext->blockSignals(false);
	
}
//
//  Close the VTK window before closing the QT window
//
void iQT::closeEvent( QCloseEvent *e )
{
	
	if(idiosyncratic && this->isVisible())
	{
		switch( QMessageBox::warning(this,"IFRIT","Are you sure you want to exit IFRIT?\n\n","Yes", "Cancel",0,0,1) ) 
		{
		case 0: // The user clicked the Retry again button or pressed Enter
			{ break; }
		case 1: // The user clicked the Quit or pressed Escape
			{ e->ignore(); return; }
		}
	}

	AnimationScripter->close();
	Help->close();
	Pick->close();
	ParallelController->close();
#ifndef I_VTK_VERSION_40
	EventRecorder->close();
#endif
	ProgressDialog->close();
	VariablesExplorer->close();
	ImageComposer->close();

	int i;
	for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) 
	{
		iVTKWindow::getWindow(i)->setCloseable();
		iVTKWindow::getWindow(i)->close();
	}

	if(ext != NULL) 
	{
		ext->setCloseable();
		ext->close();
	}

    e->accept();
    
}
//
//  Context-sensitive help
//
void iQT::keyPressEvent(QKeyEvent *e)
{

	switch (e->key())
	{
	case Qt::Key_F1:
	case Qt::Key_Help:
		{
			if(e->state() != 0) e->ignore(); else
			{
				switch(TabWidget->currentPageIndex())
				{
				case 0: { this->showHelp("vtkwview.ihf"); break; }
				case 1: { this->showHelp("vtkwsurf.ihf"); break; }
				case 2: { this->showHelp("vtkwxsec.ihf"); break; }
				case 3: { this->showHelp("vtkwvolv.ihf"); break; }
				case 4: { this->showHelp("vtkwpart.ihf"); break; }
				case 5: { this->showHelp("vtkwvect.ihf"); break; }
				case 6: { this->showHelp("vtkwtens.ihf"); break; }
				case 7: { this->showHelp("vtkwdata.ihf"); break; }
				default: ;
				}
				e->accept();
			}
			break;
		} 
	case Qt::Key_F11:
		{
			this->resetCursor();
		}
	default: e->ignore();
	}
	
}


void iQT::showHelp(const QString &s)
{
	QString ss = QString(s);
	Help->setSource(ss); Help->show();
}
//
//  two functions to report errors and information
//
void iQT::popupInfo(const QString &s)
{
	
	if(this->isVisible()) QMessageBox::information(this,"IFRIT",s); else cerr << s << "\n";
	
}


int iQT::popupError(const QString &s, int errCode)
{
	
	QString q = "Error #" + QString::number(errCode) + ": " + s;
	if(this->isVisible()) QMessageBox::critical(this,"IFRIT",q); else cerr << q << "\n";
	return 0;
	
}
//
//  This is a dummy function to avoid accidental removal of slots in QT Designer
//
void iQT::aaa()
{
	int dummy;
	dummy = 0;
}
//
//  Assign palette names
//
void iQT::assignPaletteNames()
{
	int i, j, curItem;
	
	for(j=0; j<nPaletteBoxes; j++)
	{
		curItem = paletteBox[j]->currentItem();
		paletteBox[j]->blockSignals(true);
		paletteBox[j]->clear();
		for(i=0; i<iPaletteList::getInstance()->getNumPalettes(); i++) 
		{
			paletteBox[j]->insertItem(*(iPaletteList::getInstance()->getPaletteName(i)),-1);
		}
		paletteBox[j]->setCurrentItem(curItem);
		paletteBox[j]->blockSignals(false);
		if(paletteBoxPixmap[j] != NULL)
		{
			updatePalettePixmap(iPaletteList::getInstance()->getPalette(paletteBox[j]->currentItem()),&pixmapPaint);
			paletteBoxPixmap[j]->setPixmap(pixmapPaint);
		}
	}
	
}
//
//  Print a line in the log window
//
void iQT::logLine(const QString &action, const QString &result)
{
    
	if(this->isVisible())
	{
		CALL_FUNCTION1(TextLog,append,"<b>"+action+"</b>:  "+result);
		//
		//  Fix a bug in version 3.0.4 and earlier
		//
		if(QString::compare(QString(qVersion()),QString("3.0.5"))<0) CALL_FUNCTION1(TextLog,append,"  <br><br>");
		
		while(TextLog->lines() > nlinesTextLog) 
		{
			TextLog->blockSignals(true);
			for(int j=0; j<100; j++) TextLog->removeParagraph(0);
			TextLog->blockSignals(false);
		}
		CALL_FUNCTION0(TextLog,scrollToBottom);
	}
	else
	{
		cout << action << ": " << result << endl;
	}
    
}


void iQT::logLine(const QString &action, const QString &result, QColor c)
{
    
	QString cc, cc1;
	cc1.setNum(c.red(),16); cc += cc1.rightJustify(2,'0');
	cc1.setNum(c.green(),16); cc += cc1.rightJustify(2,'0');
	cc1.setNum(c.blue(),16); cc += cc1.rightJustify(2,'0');

	if(this->isVisible())
	{
		CALL_FUNCTION1(TextLog,append,"<b><font color=\"#"+cc+"\">"+action+"</font></b>:  "+result);
		//
		//  Fix a bug in version 3.0.4 and earlier
		//
		if(QString::compare(QString(qVersion()),QString("3.0.5"))<0) CALL_FUNCTION1(TextLog,append,"  <br><br>");
		
		while(TextLog->lines() > nlinesTextLog) 
		{
			TextLog->blockSignals(true);
			for(int j=0; j<100; j++) TextLog->removeParagraph(0);
			TextLog->blockSignals(false);
		}
		CALL_FUNCTION0(TextLog,scrollToBottom);
	}
	else
	{
		cout << action << ": " << result << endl;
	}
    
}
//
//  call renderer here to avoid rendering while sliders are tracking 
//
void iQT::doRender()
{
    
    if(!sliderRenderTracking)
	{
		this->startTask();
		this->render(true);
		this->endTask();
		this->resetCursor();
	}
	
}
//
//  Do render and set the cursor to the hourglass during the rendering process
//
void iQT::render(bool s, bool allvtks)
{
	int i;
    static bool warn50 = true, warn90 = true;
	static float lastMemoryUse = 0.0;
	
	if(blockRenderer) return;
	
	if(allvtks)
	{
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->render(s);
	}
	else
	{
		iVTKWindow::getCurrentWindow()->render(s);
	}
    
    if(reportMemory && s && iVTKWindow::getCurrentWindow()->getRenderer()->GetLastRenderTimeInSeconds()>0.01)
	{
		float s = 0.0;
		
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) s += iVTKWindow::getWindow(i)->getMemorySize();
		
		s /= (1.024*1024.0);
		float sl = iMemLimit();
		
		QString ss, ss1;
		ss.setNum(s,'g',3);
		ss += " MB, time to update the scene: ";
		ss1.setNum(iVTKWindow::getCurrentWindow()->getRenderer()->GetLastRenderTimeInSeconds(),'g',3);
		ss += ss1;
		ss += " sec.";
		logLine("Memory used",ss);
		lastMemoryUse = s;
		
		if(warn90 && s>0.9*sl && sl>0.0)
		{
			warn90 = false;
			warn50 = false;
			this->popupInfo("Critical warning:\n Your memory usage is above 90%.\n IFRIT can fail anytime now.\n");
		}
		
		if(warn50 && s>0.5*sl && sl>0.0)
		{
			warn50 = false;
			this->popupInfo("Important warning:\n Your memory usage is above 50%.\n Just want to let you know.\n");
		}
		
    }
    
}


void iQT::updateFunctionPixmap(QMouseEvent *e, iPiecewiseFunction *mFunction, iLabel *mLabel, int &cpFunction, iPiecewiseFunction *histogram)
{
    static float mouseXold=0, mouseYold=0;
	static QPixmap pix = QPixmap(256,256);
	const int sym = 10;
	static QPen p(Qt::black,1);
	static QBrush b(Qt::black);
    float *x, *y, xl, yl, d;
    int n, i, ix, iy, ixo, iyo, ixmax, iymax;
    
    ixmax = mLabel->geometry().width()-1; iymax = mLabel->geometry().height()-1;
	
	if((1+ixmax)!=pix.width() || (1+iymax)!=pix.height())
	{
		pix = QPixmap(ixmax+1,iymax+1);
	}
	
    z.begin(&pix);
	
    n = mFunction->getFunctionN();
    x = mFunction->getFunctionX();
    y = mFunction->getFunctionY();
    
    if(e != 0 ) 
	{
		
		xl = (float)e->x()/(1.0+ixmax);
		yl = 1.0 - (float)e->y()/(1.0+iymax);
		
		if(e->button()==Qt::LeftButton || e->button()==Qt::RightButton) 
		{
			i = mFunction->findFunctionPoint(xl,yl,d);
			if(i>=0 && i<n && round(ixmax*d)<3*sym) cpFunction = i;
		}
		
		if((e->x()!=mouseXold||e->y()!=mouseYold) && e->state()!=0 && e->stateAfter()!=0) mFunction->movFunctionPoint(cpFunction,xl,yl);
		
		mouseXold = e->x();
		mouseYold = e->y();
		
    }
    
    z.eraseRect(0,0,pix.width(),pix.height());
    
	if(histogram != 0) this->drawHistogram(z,histogram,histoColor,1.0,true,false,0);

    p.setColor(QColor( 255-z.backgroundColor().red(), 255-z.backgroundColor().green(), 255-z.backgroundColor().blue() ));
    b.setColor(QColor( 255-z.backgroundColor().red(), 255-z.backgroundColor().green(), 255-z.backgroundColor().blue() ));
    z.setPen(p);
    z.setBrush(b);
    
    ixo = iyo = -1;
    for(i=0; i<n; i++)  
	{
		ix = round(ixmax*x[i]);
		iy = round(iymax*(1.0-y[i]));
		z.drawEllipse(ix-sym/2,iy-sym/2,sym,sym);
		if(ixo>=0 && iyo >=0) z.drawLine(ixo,iyo,ix,iy);
		ixo = ix; iyo = iy;
    }
	
    b.setColor(QColor( z.backgroundColor().red(), 255-z.backgroundColor().green(), 255-z.backgroundColor().blue() ));
    z.setBrush(b);
    ix = round(ixmax*x[cpFunction]);
    iy = round(iymax*(1.0-y[cpFunction]));
    z.drawEllipse(ix-sym/2,iy-sym/2,sym,sym);
	
    mLabel->setPixmap(pix);
    
    if(e!=0 && e->state()==Qt::RightButton) this->render(true);
    
    z.end();
    
}


void iQT::updatePalettePixmap(iPalette *p, QPixmap* m)
{
    QColor c;
	
    z.begin(m);
    for(int i=0; i<256; i++) 
	{
		c = p->getColor(i);
		z.setPen(c);
		z.drawLine(i,0,i,31);
    }
    z.end();
    
}


void iQT::drawHistogram(QPainter &z, iPiecewiseFunction *mFunction, QColor c, float, bool showTicks, bool stretchLog, float *limits)
{
	static float topOffset = 0.95;
	static const QColor c2 = QColor(60,60,60);    // tick mark color
	static QPen p(black,1);
	static QPen p2(c2,3);
	static int ixmax, iymax, ix1, ix2, iy1, n, iytick;
	static float *x, *y, x1, x2, dx, yoffset;
	int i;

	if(mFunction == 0) return;
	
	n = mFunction->getFunctionN();
	x = mFunction->getFunctionX();
	y = mFunction->getFunctionY();
		
	float ys, ymax, ymin;
	ymax = ymin = y[1];
	for(i=2; i<n-1; i++)
	{
		if(ymax < y[i]) ymax = y[i];
		if(ymin > y[i]) ymin = y[i];
	}

	if(ymax < 1) return;

	float ynorm = 1.0/log(2*ymax);

    ixmax = ((QPixmap *)z.device())->width() - 1; 
	iymax = ((QPixmap *)z.device())->height() - 1;
	
	yoffset = topOffset*iymax;
	iytick = round(yoffset);
	
	p.setColor(c);

	dx = (1.0+(float)ixmax)/(n-2.0);
	x2 = 0.0;
	if(limits != 0)
	{
		dx *= (limits[1]-limits[0]);
		x2 += limits[0]*(1.0+ixmax);
	}
	ix2 = round(x2);

	z.setPen(p);
	for(i=1; i<n-1; i++)  
	{
		x1 = x2;
		ix1 = ix2;
		x2 += dx;
		ix2 = round(x2);
		if(ix2 > ix1)
		{
			if(stretchLog)
			{
				ys = ynorm*log(1.0e-30+2*y[i]);
				if(ys < 0.0) ys = 0.0;
			}
			else
			{
				ys = y[i]/ymax;
			}
			iy1 = round(yoffset*ys);
			if(ix2 > ix1+1)
			{
				z.fillRect(ix1,iymax-iy1,ix2-ix1,iy1,c);
			}
			else
			{
				z.drawLine(ix1,iymax-iy1,ix1,iymax);
			}
			if(showTicks && n<20) 
			{
				z.setPen(p2);
				z.drawLine((ix1+ix2)/2,iymax,(ix1+ix2)/2,iytick);
				z.setPen(p);
			}
		}
	}

}


bool iQT::isDocked()
{
    return dockState;
}


void iQT::loadAxesBoxLabels()
{
	//    
	//  Read the ifrit.lab file here
	//    
    QFile f(iEnvironment::getInstance()->getFileName("DIR","ifrit.lab"));
    QTextStream ts(&f);
	
    if(f.open(IO_ReadOnly))
	{
		QString s, s1, s2, s3;
		
		if(!(s=ts.readLine()).isEmpty()) s1 = s.simplifyWhiteSpace(); else s1 = QString("X");
		if(!(s=ts.readLine()).isEmpty()) s2 = s.simplifyWhiteSpace(); else s2 = QString("Y");
		if(!(s=ts.readLine()).isEmpty()) s3 = s.simplifyWhiteSpace(); else s3 = QString("Z");
		
		float xMin, xMax, yMin, yMax, zMin, zMax;
		xMin = yMin = zMin = 0.0;
		xMax = yMax = zMax = 1.0;
		if(!(s=ts.readLine()).isEmpty())
		{
			int ret = sscanf(s.latin1(),"%g %g %g %g %g %g",&xMin,&xMax,&yMin,&yMax,&zMin,&zMax);
			if(ret < 2)
			{
				xMin = 0.0; xMax = 1.0;
			}
			if(ret < 4)
			{
				yMin = 0.0; yMax = 1.0;
			}
			if(ret < 6)
			{
				zMin = 0.0; zMax = 1.0;
			}
		}
		
		int i;
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->setBox3Axes((char *)s1.latin1(),(char *)s2.latin1(),(char *)s3.latin1(),xMin,xMax,yMin,yMax,zMin,zMax);
		
    }
	
}
//
//
//  Saving and restoring the state
//
//
#define PACK_STATE(object) \
object->packState(state); \
optionsStream << state << " "; 

#define SAVE_IOBJECT(Class,getObjectFamily,ObjectName) \
for(i=0; i<=iVTKWindow::getWindow(k)->getObjectFamily->getMaxMemberIndex(); i++) \
{ \
	Class *s = iVTKWindow::getWindow(k)->getObjectFamily->getMember(i); \
	optionsStream << ObjectName << " " << k << " "; \
    PACK_STATE(s) \
	optionsStream << "\n"; \
}


#define COMPATIBILITY_MODE	10

void iQT::optionsSave()
{
    int i, j, k;
	iString state;
	int nFun;
	float *xFun, *yFun;
	QString *s;
	
	if(QFile::exists(iEnvironment::getInstance()->getFileName("DIR","ifrit.ini")))
	{
		if(this->isVisible())
		{
			switch( QMessageBox::warning(this,"IFRIT",
				"Options file already exists. Would you like to overwrite it?\nIf you answer Ok, the old file will be renamed ifrit.bak.\n\n",
				"Ok", "Cancel", 0, 0, 1 ) ) 
			{
			case 0: // The user clicked the Retry again button or pressed Enter
				{
					QDir d;
					bool ret = true;
					if(QFile::exists(iEnvironment::getInstance()->getFileName("DIR","ifrit.bak"))) ret = d.remove(iEnvironment::getInstance()->getFileName("DIR","ifrit.bak"));
					ret = ret & d.rename(iEnvironment::getInstance()->getFileName("DIR","ifrit.ini"),iEnvironment::getInstance()->getFileName("DIR","ifrit.bak"));
					if(!ret)
					{
						switch( QMessageBox::warning(this,"IFRIT",
							"Cannot save the current ifrit.ini file as ifrit.bak.\nWould you like to continue?\n\n",
							"Ok", "Cancel", 0, 0, 1 ) ) 
						{
						case 0: // The user clicked the Retry again button or pressed Enter
							{
								break;
							}
						case 1: // The user clicked the Quit or pressed Escape
							{ 
								return; 
							}
						}
					}
					break;
				}
			case 1: // The user clicked the Quit or pressed Escape
				{ 
					return; 
				}
			}
		}
		else
		{
			QDir d;
			if(QFile::exists(iEnvironment::getInstance()->getFileName("DIR","ifrit.bak"))) d.remove(iEnvironment::getInstance()->getFileName("DIR","ifrit.bak"));
			d.rename(iEnvironment::getInstance()->getFileName("DIR","ifrit.ini"),iEnvironment::getInstance()->getFileName("DIR","ifrit.bak"));
		}
	}
	
	QFile optionsFile(iEnvironment::getInstance()->getFileName("DIR","ifrit.ini"));
	QTextStream optionsStream(&optionsFile);
	//
	//  Check that we can write the file
	//
	if(!optionsFile.open(IO_WriteOnly)) 
	{
		this->popupError("Could not open the state file for writing.\n The current state will not be saved.",IFRITERROR_CANNOT_SAVE_STATE);
		return;
	}
	
	this->startTask();
	//
	//  Compatibility mode
	//
	optionsStream << "CompatibilityMode " << COMPATIBILITY_MODE << "\n";
	//
	//  Save the QT windows, their & extensions geometries, and Image Composers 
	//
	for(k=0; k<=iQTWindow::getMaxWindowIndex(); k++) 
	{
		//
		//  Window geometry
		//
		optionsStream << "GUI " << k << " " << iQTWindow::getWindow(k)->x() << " " 
			<< iQTWindow::getWindow(k)->y() << " " << iQTWindow::getWindow(k)->width() << " " 
			<< iQTWindow::getWindow(k)->height();
		if(iQTWindow::getWindow(k)->ext != NULL)
		{
			optionsStream << " " << iQTWindow::getWindow(k)->ext->x() << " " 
				<< iQTWindow::getWindow(k)->ext->y() << " " << iQTWindow::getWindow(k)->ext->width() << " " 
				<< iQTWindow::getWindow(k)->ext->height();
		}
		PACK_STATE(iQTWindow::getWindow(k)->ImageComposer);
		optionsStream << "\n";
	}
	//
	//  Save the VTK windows geometries, view transforms, particle splitters, and parameters
	//
	for(k=0; k<=iVTKWindow::getMaxWindowIndex(); k++) 
	{
		//
		//  Window location
		//
		optionsStream << "Window " << k << " " << iVTKWindow::getWindow(k)->x() << " " 
			<< iVTKWindow::getWindow(k)->y() << " ";
		//
		//  Window parameters
		//
		PACK_STATE(iVTKWindow::getWindow(k)) 
		optionsStream << "\n";
		//
		//  Data reader
		//
		optionsStream << "DataReader " << k << " ";
		PACK_STATE(iVTKWindow::getWindow(k)->getReader())
		optionsStream << "\n";
		//
		//  Particle splitters
		//
		optionsStream << "ParticlesSplitters " << k << " "; 
		for(i=0; i<=MAXDATACHANNEL; i++) for(j=0; j<=iVTKWindow::getWindow(k)->getReader()->getMaxDataStream(i); j++) if(iVTKWindow::getWindow(k)->getParticlesSplitter(i,j) != 0)
		{
			optionsStream << "##";
			PACK_STATE(iVTKWindow::getWindow(k)->getParticlesSplitter(i,j));
		}
		optionsStream << "\n";
		//
		//  Save animators
		//
		optionsStream << "Animator " << k << " "; 
		PACK_STATE(iVTKWindow::getWindow(k)->getAnimator()) 
		optionsStream << "\n"; 
		//
		//  Save multiple instances of iVisualObject-s
		//
		SAVE_IOBJECT(iSurface,getSurfaceFamily(),"Surface")
		SAVE_IOBJECT(iXsection,getXsectionFamily(),"Xsection")
		SAVE_IOBJECT(iVolume,getVolumeFamily(),"Volume")
		SAVE_IOBJECT(iParticles,getParticlesFamily(0,0),"Particles")
		SAVE_IOBJECT(iVector,getVectorFamily(),"Vector")
		SAVE_IOBJECT(iTensor,getTensorFamily(),"Tensor")
		//
		//  Don't save single invisible marker
		//
		if(iVTKWindow::getWindow(k)->getMarkerFamily()->getMaxMemberIndex()>0 || iVTKWindow::getWindow(k)->getMarkerFamily()->getMember(0)->isVisible()) SAVE_IOBJECT(iMarker,getMarkerFamily(),"Marker")

		if(ext != NULL) ext->optionsSaveObjects(optionsStream,k);

	}
	//
	// Save currently loaded palettes
	//
	for(j=0; j<iPaletteList::getInstance()->getNumPalettes(); j++)
	{
		
		iPalette *pal = iPaletteList::getInstance()->getPalette(j);

		s = iPaletteList::getInstance()->getPaletteName(j);
		s->replace(QRegExp(" "),"_"); 
		optionsStream << "Palette " << j << " " << s->latin1() << " ";
				
		nFun = pal->getPiecewiseFunctionRed()->getFunctionN();
		xFun = pal->getPiecewiseFunctionRed()->getFunctionX();
		yFun = pal->getPiecewiseFunctionRed()->getFunctionY();
		optionsStream << nFun << " ";
		for(i=0; i<nFun; i++) optionsStream << *(xFun+i) << " " << *(yFun+i) << " ";
		
		nFun = pal->getPiecewiseFunctionGreen()->getFunctionN();
		xFun = pal->getPiecewiseFunctionGreen()->getFunctionX();
		yFun = pal->getPiecewiseFunctionGreen()->getFunctionY();
		optionsStream << nFun << " ";
		for(i=0; i<nFun; i++) optionsStream << *(xFun+i) << " " << *(yFun+i) << " ";
		
		nFun = pal->getPiecewiseFunctionBlue()->getFunctionN();
		xFun = pal->getPiecewiseFunctionBlue()->getFunctionX();
		yFun = pal->getPiecewiseFunctionBlue()->getFunctionY();
		optionsStream << nFun << " ";
		for(i=0; i<nFun; i++) optionsStream << *(xFun+i) << " " << *(yFun+i) << " ";

		optionsStream << "\n";
		
	}
	//
	//  Save states of selected widgets
	//
	for(i=0; i<nTypes; i++)
	{
		
		QObjectList *list = this->queryList(objectTypes[i]);
		QObjectListIt it( *list ); // iterate over the buttons
		QObject *obj;
		

		while ( (obj = it.current()) != 0 ) 
		{

			if(strcmp(obj->name(),"unnamed")==0 || strcmp(obj->name(),"")==0) 
			{
				++it;
				continue;
			}

			switch(i) 
			{
			case 0:	{ optionsStream << obj->name() << " " << ((QComboBox *)obj)->currentItem() << "\n"; break; }
			case 1:	{ optionsStream << obj->name() << " " << ((QCheckBox *)obj)->isChecked() << "\n"; break; }
			case 2:	{ optionsStream << obj->name() << " " << ((QSlider *)obj)->value() << "\n"; break; }
			case 3:	{ optionsStream << obj->name() << " " << ((QButtonGroup *)obj)->id(((QButtonGroup *)obj)->selected()) << "\n"; break; }
			case 4:	{ if(((QAction *)obj)->isToggleAction()) optionsStream << obj->name() << " " << ((QAction *)obj)->isOn() << "\n"; break; }
			case 5:	{ optionsStream << obj->name() << " " << ((QSpinBox *)obj)->value() << "\n"; break; }
			case 6:	{ if(strstr(obj->name(),"ColorLabel") != 0) optionsStream << obj->name() << " " << ((QLabel *)obj)->eraseColor().red() << " " << ((QLabel *)obj)->eraseColor().green() << " " << ((QLabel *)obj)->eraseColor().blue() << " " << "\n"; break; }
			}
			// for each found object...
			++it;
		}
		delete list; // delete the list, not the objects
		
	}
	
	this->endTask();
	
	optionsFile.close();
		
}


#define ERROR_READING_OPTIONS { blockRenderer = false; this->popupError("The state file is corrupted.\n Default values of parameters will be used.",IFRITERROR_CORRUPTED_STATE_FILE); return false; }

#define UNPACK_STATE(object) object->unpackState(s.section(' ',joff));

#define LOAD_IOBJECT(Obj,getObjectFamily,ObjectName,Show) \
if(strcmp(name,ObjectName) == 0) \
{ \
	if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS; \
	if(k>=0 && k<nVTK) \
	{ \
		int i; \
		if(first##Obj[k]) \
		{ \
			first##Obj[k] = false; i = 0; \
		} \
		else i = iVTKWindow::getWindow(k)->getObjectFamily->createMember(); \
		joff = 2; \
		UNPACK_STATE(iVTKWindow::getWindow(k)->getObjectFamily->getMember(i)) \
		if(Show) iVTKWindow::getWindow(k)->getObjectFamily->getMember(i)->show(Show); \
	} \
} 
		

bool iQT::optionsLoad(const char *fname)
{
	QObjectList *list[nTypes];
	int i, j, k, l, joff, maxPalette = -1;
	QString s, s1, s2;
	char name[1025];
	int varInt[100];
	float varFloat[100];
	bool res;
	int nFun;
	int CompatibilityMode = 0;
	
	QFile optionsFile;
	QTextStream optionsStream(&optionsFile);
	if(fname == 0) 
		optionsFile.setName(iEnvironment::getInstance()->getFileName("DIR","ifrit.ini"));
	else 
		optionsFile.setName(fname);
	
	if(!optionsFile.exists()) return false;
	
	if(!optionsFile.open(IO_ReadOnly)) 
	{
		this->popupError("The state file does not exist.\n Default values of parameters will be used.",IFRITERROR_NO_STATE_FILE);
		return false;
	}
	
	for(i=0; i<nTypes; i++) list[i] = this->queryList(objectTypes[i]);
	//
	//  Block renderer and signals for "show" checkboxes and actions to prevent showing objects
	//  without data
	//
	blockRenderer = true;

	TabWidgetView->View_Surf->blockSignals(true);
	TabWidgetSurf->View_SurfLocal->blockSignals(true);
	actionShowSurf->blockSignals(true);
	
	TabWidgetView->View_Xsec->blockSignals(true);
	TabWidgetXsec->View_XsecLocal->blockSignals(true);
	actionShowXsec->blockSignals(true);
	
	TabWidgetView->View_Volv->blockSignals(true);
	TabWidgetVolv->View_VolvLocal->blockSignals(true);
	actionShowVolv->blockSignals(true);

	TabWidgetView->View_Part->blockSignals(true);
	TabWidgetPart->View_PartLocal->blockSignals(true);
	actionShowPart->blockSignals(true);

	TabWidgetView->View_Vect->blockSignals(true);
	TabWidgetVect->View_VectLocal->blockSignals(true);
	actionShowVect->blockSignals(true);

	TabWidgetView->View_Tens->blockSignals(true);
	TabWidgetTens->View_TensLocal->blockSignals(true);
	actionShowTens->blockSignals(true);

	if(ext != NULL) ext->optionsLoadPreProcess();

	DataChannelComboBox->blockSignals(true);
	styleDockAction->blockSignals(true);
	//
	// ***************** Pre-Pass ****************************
	//
	//  Find the Compatibility parameter.
	//
	optionsFile.at(0);
	while(!(s=optionsStream.readLine()).isEmpty()) 
	{
		
		s = s.simplifyWhiteSpace();
		sscanf(s.latin1(),"%s",name);
		
		if(strcmp(name,"CompatibilityMode") == 0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&i) != 2) ERROR_READING_OPTIONS;
			if(i<10 || i>COMPATIBILITY_MODE) ERROR_READING_OPTIONS;
			CompatibilityMode = i;
			break;
		}

	}
	//
	// ***************** Pass 1 ****************************
	//
	//  Count the number of VTK and QT windows and create them with default settings.
	//
	optionsFile.at(0);
	int nVTK = 0, nQT = 0, kVTKMax = 0, kQTMax = 0;
	while(!(s=optionsStream.readLine()).isEmpty()) 
	{
		
		s = s.simplifyWhiteSpace();
		sscanf(s.latin1(),"%s",name);
		
		if(strcmp(name,"Window")==0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS;
			if(k > kVTKMax) kVTKMax = k;
			if(nVTK > 0) 
			{
				int wid = iVTKWindow::createWindow();
				iVTKWindow::getWindow(wid)->getStartEventObserver()->setProgressBar(this->ProgressBar);
				iVTKWindow::getWindow(wid)->getProgressEventObserver()->setProgressBar(this->ProgressBar);
				iVTKWindow::getWindow(wid)->getEndEventObserver()->setProgressBar(this->ProgressBar);
				iVTKWindow::getWindow(wid)->getPickEventObserver()->setDialogPick(this->Pick);
#ifndef I_VTK_VERSION_40
				iVTKWindow::getWindow(wid)->getRecordEventObserver()->setDialogEventRecorder(this->EventRecorder);
#endif
				if(this->remoteControlled) iVTKWindow::getWindow(wid)->getAbortRenderEventObserver()->setProgressDialog(0); else iVTKWindow::getWindow(wid)->getAbortRenderEventObserver()->setProgressDialog(this->ProgressDialog);
				iVTKWindow::getWindow(wid)->setImageComposer(ImageComposer);
				this->createVisitedFilesList(wid);
				s = s.setNum(nVTK+1);
				TabWidgetView->TabWidgetWins->Wins_SetCurrentInstanceList->insertItem("Window "+s,-1);
				TabWidgetView->TabWidgetWins->Wins_DeleteInstanceButton->setEnabled(true);
			}
			nVTK++;
		}
		
		if(strcmp(name,"GUI")==0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS;
			if(k > kQTMax) kQTMax = k;
			if(nQT > 0) 
			{
//				iQTWindow::createWindow();
			}
			nQT++;
		}

	}
	
	if(kVTKMax+1 != nVTK) ERROR_READING_OPTIONS;
	if(kQTMax+1 != nQT) ERROR_READING_OPTIONS;
	if(1 != nQT) ERROR_READING_OPTIONS;  // can be removed in multiple-QTwindow mode
	//
	// ***************** Pass 2 ****************************
	//
	//  Restore VTK and QT windows
	//
	optionsFile.at(0);
	while(!(s=optionsStream.readLine()).isEmpty()) 
	{
		
		s = s.simplifyWhiteSpace();
		sscanf(s.latin1(),"%s",name);
		
		if(strcmp(name,"GUI") == 0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS;
			if(k>=0 && k<nQT)
			{		
				iQT *ss = iQTWindow::getWindow(k); 
				
				if(sscanf(s.latin1(),"%s %d %d %d %d %d",name,&k,
					&varInt[0],&varInt[1],&varInt[2],&varInt[3]) != 6) ERROR_READING_OPTIONS;
				if(varInt[0]>=0 && varInt[0]<qApp->desktop()->width() && varInt[1]>=0 && varInt[1]<qApp->desktop()->height()) ss->move(varInt[0],varInt[1]);
				if(varInt[2]>0 && varInt[2]<=qApp->desktop()->width() && varInt[3]>0 && varInt[3]<qApp->desktop()->height()) ss->resize(varInt[2],varInt[3]);
				joff = 6;

				if(ss->ext != NULL)
				{
					if(sscanf(s.latin1(),"%s %d %d %d %d %d %d %d %d %d",name,&k,
						&varInt[0],&varInt[1],&varInt[2],&varInt[3],
						&varInt[4],&varInt[5],&varInt[6],&varInt[7]) == 10)
					{
						if(varInt[4]>=0 && varInt[4]<qApp->desktop()->width() && varInt[5]>=0 && varInt[5]<qApp->desktop()->height()) ss->ext->move(varInt[4],varInt[5]);
						if(varInt[6]>0 && varInt[6]<=qApp->desktop()->width() && varInt[7]>0 && varInt[7]<qApp->desktop()->height()) ss->ext->resize(varInt[6],varInt[7]);
					}
					else
					{
						ss->ext->move(ss->pos()+QPoint(30,30));
					}
					joff = 10;
				}
				UNPACK_STATE(ss->ImageComposer);
			}
		}

		if(strcmp(name,"Window") == 0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS;
			if(k>=0 && k<nVTK)
			{		
				iVTK *ss = iVTKWindow::getWindow(k); 
				
				if(sscanf(s.latin1(),"%s %d %d %d",name,&k,&varInt[0],&varInt[1]) != 4) ERROR_READING_OPTIONS;
				
				if(varInt[0]>=0 && varInt[0]<qApp->desktop()->width() && varInt[1]>=0 && varInt[1]<qApp->desktop()->height()) ss->move(varInt[0],varInt[1]);
				//
				//  Window parameters
				//
				joff = 4;
				UNPACK_STATE(iVTKWindow::getWindow(k))
				//
				// Set iVisualObject visibility to false to avoid rendering iVisualObjects without the data
				//
//				ss->getSurfaceFamily()->show(false);
//				ss->getXsectionFamily()->show(false);
//				ss->getVolumeFamily()->show(false);
//				ss->getParticlesFamily()->show(false);
//				ss->getVectorFamily()->show(false);
//				ss->getTensorFamily()->show(false);

//				if(ext != NULL) ext->showObjects(false);

			}
		}
		
	}
	//	
	// ***************** Pass 3 ****************************
	//
	//  Restore iVisualObjects-s and Palettes
	//
	iPaletteList *curPaletteList = new iPaletteList;
	if(curPaletteList == 0) reportNullPointer(1012);
	
	optionsFile.at(0);
	bool firstSurf[MAXINSTANCE], firstXsec[MAXINSTANCE], firstVolv[MAXINSTANCE], firstPart[MAXINSTANCE], firstVect[MAXINSTANCE], firstTens[MAXINSTANCE], firstMark[MAXINSTANCE];
	for(i=0; i<MAXINSTANCE; i++) firstSurf[i] = firstXsec[i] = firstVolv[i] = firstPart[i] = firstVect[i] = firstTens[i] = true;
	while(!(s=optionsStream.readLine()).isEmpty()) 
	{
		
		s = s.simplifyWhiteSpace();
		sscanf(s.latin1(),"%s",name);
	
		LOAD_IOBJECT(Surf,getSurfaceFamily(),"Surface",false)
		LOAD_IOBJECT(Xsec,getXsectionFamily(),"Xsection",false)
		LOAD_IOBJECT(Volv,getVolumeFamily(),"Volume",false)
		LOAD_IOBJECT(Part,getParticlesFamily(0,0),"Particles",false)
		LOAD_IOBJECT(Vect,getVectorFamily(),"Vector",false)
		LOAD_IOBJECT(Tens,getTensorFamily(),"Tensor",false)
		LOAD_IOBJECT(Mark,getMarkerFamily(),"Marker",true)

		if(ext != NULL)
		{
			if(!ext->optionsLoadObjects(s,name,joff,nVTK)) return false;
		}

		if(strcmp(name,"DataReader") == 0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS; 
			if(k>=0 && k<nVTK) 
			{ 
				joff = 2; 
				UNPACK_STATE(iVTKWindow::getWindow(k)->getReader()) 
			}
		}		
		
		if(strcmp(name,"Animator") == 0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS; 
			if(k>=0 && k<nVTK) 
			{ 
				joff = 2; 
				UNPACK_STATE(iVTKWindow::getWindow(k)->getAnimator()) 
			}
		}		
		
		if(strcmp(name,"ParticlesSplitters") == 0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&k) != 2) ERROR_READING_OPTIONS; 
			if(k>=0 && k<nVTK) 
			{ 
				joff = 2; 
				l = 0;
				for(i=0; i<=MAXDATACHANNEL; i++) for(j=0; j<=iVTKWindow::getWindow(k)->getReader()->getMaxDataStream(i); j++) if(iVTKWindow::getWindow(k)->getParticlesSplitter(i,j) != 0)
				{
					l++;
					s1 = s.section("##",l,l);
					iVTKWindow::getWindow(k)->getParticlesSplitter(i,j)->unpackState(s1);
				}
			}
		}		
		
		if(strcmp(name,"Palette") == 0) 
		{
			if(sscanf(s.latin1(),"%s %d",name,&i) != 2) ERROR_READING_OPTIONS;
			bool ok = true;
			if(i<0 || i>=MAXPALETTE) ok = false;
			while(i>=curPaletteList->getNumPalettes() && ok)
			{
				ok = curPaletteList->addEmptyPalette();
			}
			if(ok)
			{
				if(i > maxPalette) maxPalette = i;
				s1 = s.section(' ',2,2);
				s1.replace(QRegExp("_")," "); 
				if(!s1.isEmpty()) curPaletteList->setPaletteName(i,s1); else ERROR_READING_OPTIONS;
				joff = 3;
				//
				//  Loop over 3 color functions
				//
				iPiecewiseFunction *f = 0;
				for(k=0; k<3; k++)
				{
					switch (k) 
					{
					case 0: { f = curPaletteList->getPalette(i)->getPiecewiseFunctionRed(); break; }
					case 1: { f = curPaletteList->getPalette(i)->getPiecewiseFunctionGreen(); break; }
					case 2: { f = curPaletteList->getPalette(i)->getPiecewiseFunctionBlue(); break; }
					}
					s1 = s.section(' ',joff,joff);
					if(sscanf(s1.latin1(),"%d",&nFun) != 1) ERROR_READING_OPTIONS;
					joff++;
					for(j=0; j<nFun; j++)
					{
						s1 = s.section(' ',joff+2*j,joff+2*j+1);
						if(sscanf(s1.latin1(),"%f %f",&varFloat[0],&varFloat[1]) != 2) ERROR_READING_OPTIONS;
						if(j==0 || j==nFun-1)
						{
							f->movFunctionPoint(j,varFloat[0],varFloat[1]);
						}
						else
						{
							f->addFunctionPoint(varFloat[0],varFloat[1]);
						}
					}
					joff += 2*nFun;
				}
			}
		}

	}
	
	maxPalette++;
	if(maxPalette > 0)
	{
		for(i=0; i<curPaletteList->getNumPalettes(); i++) curPaletteList->getPalette(i)->update();
		iPaletteList::getInstance()->copy(curPaletteList);
		this->assignPaletteNames();
	}
	
	delete curPaletteList;
	//
	// ***************** Pass 4 ****************************
	//
	//  Restore selected widgets
	//

	//
	//  There is a problem with Particle Splitter - it is already fully set, but widgets will reset
	//  all the options. So, we block signals to the 3 offending widgets
	//
	TabWidgetPart->Part_Piece_Activate->blockSignals(true);
	TabWidgetPart->Part_Piece_Tile->blockSignals(true);
	TabWidgetPart->Part_Piece_SplitWithBox->blockSignals(true);

	optionsFile.at(0);
	while(!(s=optionsStream.readLine()).isEmpty()) 
	{
		
		s = s.simplifyWhiteSpace();
		sscanf(s.latin1(),"%s",name);
		
		if(strcmp(name,"unnamed") == 0) continue;

		for(i=0; i<nTypes; i++)
		{
			
			QObjectListIt it( *list[i] ); // iterate over the buttons
			QObject *obj;
			while ( (obj = it.current()) != 0 ) 
			{
				switch(i) 
				{
					//
					//  QComboBox - reconnect the signal to force highlighted() to be emitted
					//
				case 0:	
					{ 
						if(strcmp(name,obj->name()) == 0)
						{
							sscanf(s.latin1(),"%s %d",name,&varInt[0]);
							if(varInt[0] < ((QComboBox *)obj)->count())
							{
								res = connect( (QComboBox *)obj, SIGNAL( highlighted(int) ) , (QComboBox *)obj, SIGNAL( activated(int) ) );
								((QComboBox *)obj)->setCurrentItem(varInt[0]);
								res = disconnect( (QComboBox *)obj, SIGNAL( highlighted(int) ) , (QComboBox *)obj, SIGNAL( activated(int) ) );
							}
						}
						break; 
					}
					//
					//  QCheckBox - assuming it emits clicked() signal
					//
				case 1:	
					{ 
						if(strcmp(name,obj->name()) == 0)
						{
							sscanf(s.latin1(),"%s %d",name,&varInt[0]);
							((QCheckBox *)obj)->setChecked((bool)varInt[0]);
						}
						break; 
					}
					//
					//  QSlider - assuming it emits valueChanged(int) signal
					//
				case 2:	
					{ 
						if(strcmp(name,obj->name()) == 0)
						{
							sscanf(s.latin1(),"%s %d",name,&varInt[0]);
							((QSlider *)obj)->setValue(varInt[0]);
						}
						break; 
					}
					//
					//  QButtonGroup - reconnect signal to force clicked(int) signal to be emitted
					//
				case 3:	
					{ 
						if(strcmp(name,obj->name()) == 0)
						{
							sscanf(s.latin1(),"%s %d",name,&varInt[0]);
							res = connect( ((QButtonGroup *)obj)->find(varInt[0]), SIGNAL( stateChanged(int) ) , ((QButtonGroup *)obj)->find(varInt[0]), SIGNAL( clicked() ) );
							((QButtonGroup *)obj)->setButton(varInt[0]);
							res = disconnect( ((QButtonGroup *)obj)->find(varInt[0]), SIGNAL( stateChanged(int) ) , ((QButtonGroup *)obj)->find(varInt[0]), SIGNAL( clicked() ) );
						}
						break; 
					}
					//
					//  QAction - assuming it emits toggled() signal
					//
				case 4:	
					{ 
						if(strcmp(name,obj->name()) == 0)
						{
							sscanf(s.latin1(),"%s %d",name,&varInt[0]);
							if(((QAction *)obj)->isToggleAction() && !obj->signalsBlocked()) ((QAction *)obj)->setOn((bool)varInt[0]);
						}
						break; 
					}
					//
					//  QSpinBox - assuming it emits clicked() signal
					//
				case 5:	
					{ 
						if(strcmp(name,obj->name()) == 0)
						{
							sscanf(s.latin1(),"%s %d",name,&varInt[0]);
							((QSpinBox *)obj)->setValue(varInt[0]);
						}
						break; 
					}
					//
					//  QLabel - do it by hand - cannot set color via a signal
					//
				case 6:	
					{ 
						if(strcmp(name,obj->name()) == 0)
						{
							sscanf(s.latin1(),"%s %d %d %d",name,&varInt[0],&varInt[1],&varInt[2]);
							QColor c = QColor(varInt[0],varInt[1],varInt[2]);
							((QLabel *)obj)->setPaletteBackgroundColor(c);
							if(strcmp(name,"View_BackgroundColorLabel") == 0) iVTKWindow::getCurrentWindow()->setBackgroundColor(c);
							if(strcmp(name,"Surf_Mult_ColorLabel") == 0) CURRENT_SURFACE->setColor(c);
							if(strcmp(name,"Part_Main_ColorLabel") == 0) CURRENT_PARTICLES->setColor(c);
							if(strcmp(name,"Vect_Main_HedgeColorLabel") == 0) CURRENT_VECTOR->setColor(c);
							if(strcmp(name,"Tens_Main_GlyphColorLabel") == 0) CURRENT_TENSOR->setColor(c);
							if(ext != NULL) ext->optionsLoadColorLabel(name,c);
						}
						break; 
					}
				}
				++it;
			}
			
		}
		
	}
	
	//
	//  Restore signals to 3 Particle Splitter widgets
	//
	TabWidgetPart->Part_Piece_Activate->blockSignals(false);
	TabWidgetPart->Part_Piece_Tile->blockSignals(false);
	TabWidgetPart->Part_Piece_SplitWithBox->blockSignals(false);

	for(i=0; i<nTypes; i++) delete list[i]; // delete the list, not the objects
	
	optionsFile.close();
	//
	//  Update ParticlesSplitter pixmap as well - it is not updated in updatePart_Widgets()
	//
	TabWidgetPart->updatePart_Piece_SplitterPixmap((QMouseEvent *)0);
	//
	//  Remove visualization tool check marks - to fix VTK bugs
	//
	TabWidgetView->updateView_Surf(false);
	TabWidgetView->updateView_Xsec(false);
	TabWidgetView->updateView_Volv(false);
	TabWidgetView->updateView_Part(false);
	TabWidgetView->updateView_Vect(false);
	TabWidgetView->updateView_Tens(false);
	if(ext != NULL) ext->updateViews(false);
	
	iVTKWindow::setCurrentWindowIndex(0);
	this->setCurrentWindow(0);
	//
	//  Render everything
	//
	blockRenderer = false;

	TabWidgetView->View_Surf->blockSignals(false);
	TabWidgetSurf->View_SurfLocal->blockSignals(false);
	actionShowSurf->blockSignals(false);
	
	TabWidgetView->View_Xsec->blockSignals(false);
	TabWidgetXsec->View_XsecLocal->blockSignals(false);
	actionShowXsec->blockSignals(false);
	
	TabWidgetView->View_Volv->blockSignals(false);
	TabWidgetVolv->View_VolvLocal->blockSignals(false);
	actionShowVolv->blockSignals(false);

	TabWidgetView->View_Part->blockSignals(false);
	TabWidgetPart->View_PartLocal->blockSignals(false);
	actionShowPart->blockSignals(false);

	TabWidgetView->View_Vect->blockSignals(false);
	TabWidgetVect->View_VectLocal->blockSignals(false);
	actionShowVect->blockSignals(false);

	TabWidgetView->View_Tens->blockSignals(false);
	TabWidgetTens->View_TensLocal->blockSignals(false);
	actionShowTens->blockSignals(false);

	if(ext != NULL) ext->optionsLoadPostProcess();

	DataChannelComboBox->blockSignals(false);
	styleDockAction->blockSignals(false);

	return true;
	
}


void iQT::launchPaletteEditor(int n)
{
	
	iDialogPaletteEditor *d = new iDialogPaletteEditor;
	if(d == 0) reportNullPointer(1015);

	d->setCurrentPalette(n);
	
	d->exec();
	delete d;
	
	int i;
    for(i=0; i<=iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMaxMemberIndex(); i++) iVTKWindow::getCurrentWindow()->getParticlesFamily()->getMember(i)->updateColorLookupTable();
	this->assignPaletteNames();
	
}


void iQT::blockAllSignals(QObject *topObj, bool s)
{

	QObjectList *list = topObj->queryList();
    QObjectListIt it(*list); // iterate over the buttons
    QObject *obj;

    while ( (obj = it.current()) != 0 ) {
        // for each found object...
		obj->blockSignals(s);
        ++it;
    }
    delete list; // delete the list, not the objects
    
}


void iQT::showWindowControlPage()
{

	TabWidget->setCurrentPage(0);
	TabWidgetView->TabWidget->setCurrentPage(3);

}


void iQT::resetCursor()
{
	qApp->restoreOverrideCursor();
	this->setCursor(Qt::ArrowCursor);
}


void iQT::launchFromMenuScripter()
{
	AnimationScripter->show();
	AnimationScripter->raise(); 
}


void iQT::launchFromMenuPaletteEditor()
{
	this->launchPaletteEditor(0);
}


void iQT::launchFromMenuPickerDialog()
{
	Pick->showExtended();
	Pick->raise();
}


void iQT::launchFromMenuEventRecorder()
{
#ifndef I_VTK_VERSION_40
	EventRecorder->show();
	EventRecorder->raise();
#endif
}


void iQT::launchFromMenuParallelController()
{
	ParallelController->show();
	ParallelController->raise(); 
}


void iQT::launchFromMenuVariablesExplorer()
{
	VariablesExplorer->updateVariables();
	VariablesExplorer->show();
	VariablesExplorer->raise(); 
}


void iQT::updateVariablesExplorer()
{
	if(VariablesExplorer->isVisible() || VariablesExplorer->isMinimized()) VariablesExplorer->updateVariables();
}


void iQT::launchFromMenuImageComposer()
{
	ImageComposer->updateAll(true);
	ImageComposer->show();
	ImageComposer->raise(); 
}


void iQT::updateImageComposer(bool imageResized)
{
	ImageComposer->updateAll(imageResized);
}


void iQT::showDataChannel(int dc)
{
	if(dc>=0 && dc<=MAXDATACHANNEL && iVTKWindow::getCurrentWindow()->getReader()->isDataChannelActive(dc))
	{
		dataChannel = dc;
		DataChannelApplyButton->setEnabled(false);
		switch(TabWidget->currentPageIndex())
		{
		case DISPLAYPAGE_VIEW: 
			{ 
				break; 
			}
		case DISPLAYPAGE_SURF: 
			{ 
				if(iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(dc,0) && TabWidgetSurf->TabWidget->currentPageIndex()==0 && CURRENT_SURFACE->getDataChannel()!=dc) DataChannelApplyButton->setEnabled(true);
				break; 
			}
		case DISPLAYPAGE_XSEC: 
			{ 
				if(iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(dc,0) && TabWidgetXsec->TabWidget->currentPageIndex()==0 && CURRENT_XSECTION->getDataChannel()!=dc) DataChannelApplyButton->setEnabled(true);
				break; 
			}
		case DISPLAYPAGE_VOLV: 
			{ 
				if(iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(dc,0) && TabWidgetVolv->TabWidget->currentPageIndex()==0 && CURRENT_VOLUME->getDataChannel()!=dc) DataChannelApplyButton->setEnabled(true);
				break; 
			}
		case DISPLAYPAGE_PART: 
			{
				iVTKWindow::getCurrentWindow()->setCurrentParticlesFamily(dc);
				TabWidgetPart->updatePart_Widgets(); 
				break; 
			}
		case DISPLAYPAGE_VECT: 
			{ 
				if(iVTKWindow::getCurrentWindow()->getReader()->isThereVectData(dc,0) && TabWidgetVect->TabWidget->currentPageIndex()==0 && CURRENT_VECTOR->getDataChannel()!=dc) DataChannelApplyButton->setEnabled(true);
				break; 
			}
		case DISPLAYPAGE_TENS: 
			{ 
				if(iVTKWindow::getCurrentWindow()->getReader()->isThereTensData(dc,0) && TabWidgetTens->TabWidget->currentPageIndex()==0 && CURRENT_TENSOR->getDataChannel()!=dc) DataChannelApplyButton->setEnabled(true);
				break; 
			}
		case DISPLAYPAGE_DATA: 
			{ 
				TabWidgetData->updateData_Widgets(); 
				break; 
			}
		}
		//
		//  Update panel
		//
		this->updateVariablesExplorer();
	}
	this->setDataChannel(dataChannel);
}


void iQT::setDataChannel(int dc)
{
	if(dc>=0 && dc<=MAXDATACHANNEL)
	{
		dataChannel = dc;
		CALL_FUNCTION1(DataChannelComboBox,setCurrentItem,dataChannel);
		CALL_FUNCTION1(DataChannelPixmap,setPixmap,pixmapDC[dataChannel]);
	}
}


void iQT::applyDataChannelChange()
{
	switch(TabWidget->currentPageIndex())
	{
	case DISPLAYPAGE_SURF: 
		{ 
			if(TabWidgetSurf->TabWidget->currentPageIndex()==0 && CURRENT_SURFACE->getDataChannel()!=dataChannel && iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(dataChannel,0)) 
			{
				CURRENT_SURFACE->setDataChannel(dataChannel);
				iVTKWindow::broadcastVariableLimitsChange(dataChannel,-1,IVTKWINDOW_BROADCASTADDRESS_SURF);
				this->render(true);
			} 
			if(TabWidgetSurf->TabWidget->currentPageIndex() != 2) TabWidgetSurf->updateSurf_Widgets(); 
			break; 
		}
	case DISPLAYPAGE_XSEC: 
		{ 
			if(TabWidgetXsec->TabWidget->currentPageIndex()==0 && CURRENT_XSECTION->getDataChannel()!=dataChannel && iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(dataChannel,0)) 
			{
				CURRENT_XSECTION->setDataChannel(dataChannel);
				iVTKWindow::broadcastVariableLimitsChange(dataChannel,-1,IVTKWINDOW_BROADCASTADDRESS_XSEC);
				this->render(true);
			} 
			if(TabWidgetXsec->TabWidget->currentPageIndex() != 2) TabWidgetXsec->updateXsec_Widgets(); 
			break; 
		}
	case DISPLAYPAGE_VOLV: 
		{ 
			if(TabWidgetVolv->TabWidget->currentPageIndex()==0 && CURRENT_VOLUME->getDataChannel()!=dataChannel && iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(dataChannel,0)) 
			{
				CURRENT_VOLUME->setDataChannel(dataChannel);
				iVTKWindow::broadcastVariableLimitsChange(dataChannel,-1,IVTKWINDOW_BROADCASTADDRESS_VOLV);
				this->render(true);
			} 
			TabWidgetVolv->updateVolv_Widgets(); 
			break; 
		}
	case DISPLAYPAGE_VECT: 
		{ 
			if(TabWidgetVect->TabWidget->currentPageIndex()==0 && CURRENT_VECTOR->getDataChannel()!=dataChannel && iVTKWindow::getCurrentWindow()->getReader()->isThereVectData(dataChannel,0)) 
			{
				CURRENT_VECTOR->setDataChannel(dataChannel);
				iVTKWindow::broadcastVariableLimitsChange(dataChannel,-1,IVTKWINDOW_BROADCASTADDRESS_VECT);
				this->render(true);
			} 
			TabWidgetVect->updateVect_Widgets(); 
			break; 
		}
	case DISPLAYPAGE_TENS: 
		{ 
			if(TabWidgetTens->TabWidget->currentPageIndex()==0 && CURRENT_TENSOR->getDataChannel()!=dataChannel && iVTKWindow::getCurrentWindow()->getReader()->isThereTensData(dataChannel,0)) 
			{
				CURRENT_TENSOR->setDataChannel(dataChannel);
				iVTKWindow::broadcastVariableLimitsChange(dataChannel,-1,IVTKWINDOW_BROADCASTADDRESS_TENS);
				this->render(true);
			} 
			TabWidgetTens->updateTens_Widgets(); 
			break; 
		}
	}
	DataChannelApplyButton->setEnabled(false);
}


void iQT::updateDataChannelComboBox(int mode)
{
	int i, ndc;

	if(mode != TabWidget->currentPageIndex()) return;

	switch(mode)
	{
		//
		//  View
		//
	case DISPLAYPAGE_VIEW:
		{
			DataChannelComboBox->setEnabled(false);
			break;
		}
		//
		//  Surf
		//
	case DISPLAYPAGE_SURF:
		{
			this->setDataChannel(CURRENT_SURFACE->getDataChannel());
			ndc = 0;
			iDataReader *r = iVTKWindow::getCurrentWindow()->getReader();
			for(i=0; i<=MAXDATACHANNEL; i++) if(r->isThereMeshData(i,0)) ndc++;
			DataChannelComboBox->setEnabled(ndc>1 && (TabWidgetSurf->TabWidget->currentPageIndex()==2 || TabWidgetSurf->TabWidget->currentPageIndex()==0));
			if(ndc>1 && TabWidgetSurf->TabWidget->currentPageIndex()==0) DataChannelApplyButton->show(); else DataChannelApplyButton->hide(); 
			break;
		}
		//
		//  Xsec
		//
	case DISPLAYPAGE_XSEC:
		{
			this->setDataChannel(CURRENT_XSECTION->getDataChannel());
			ndc = 0;
			iDataReader *r = iVTKWindow::getCurrentWindow()->getReader();
			for(i=0; i<=MAXDATACHANNEL; i++) if(r->isThereMeshData(i,0)) ndc++;
			DataChannelComboBox->setEnabled(ndc>1 && (TabWidgetXsec->TabWidget->currentPageIndex()==2 || TabWidgetXsec->TabWidget->currentPageIndex()==0));
			if(ndc>1 && TabWidgetXsec->TabWidget->currentPageIndex()==0) DataChannelApplyButton->show(); else DataChannelApplyButton->hide(); 
			break;
		}
		//
		//  Volv
		//
	case DISPLAYPAGE_VOLV:
		{
			this->setDataChannel(CURRENT_VOLUME->getDataChannel());
			ndc = 0;
			iDataReader *r = iVTKWindow::getCurrentWindow()->getReader();
			for(i=0; i<=MAXDATACHANNEL; i++) if(r->isThereMeshData(i,0)) ndc++;
			DataChannelComboBox->setEnabled(ndc>1 && TabWidgetVolv->TabWidget->currentPageIndex()==0);
			if(ndc>1 && TabWidgetVolv->TabWidget->currentPageIndex()==0) DataChannelApplyButton->show(); else DataChannelApplyButton->hide(); 
			break;
		}
		//
		//  Part
		//
	case DISPLAYPAGE_PART:
		{
			this->setDataChannel(CURRENT_PARTICLES->getDataChannel());
			ndc = 0;
			iDataReader *r = iVTKWindow::getCurrentWindow()->getReader();
			for(i=0; i<=MAXDATACHANNEL; i++) if(r->isTherePartData(i,0)) ndc++;
			DataChannelComboBox->setEnabled(ndc>1 && TabWidgetPart->TabWidget->currentPageIndex()==0);
			DataChannelApplyButton->hide();
			break;
		}
		//
		//  Vect
		//
	case DISPLAYPAGE_VECT:
		{
			this->setDataChannel(CURRENT_VECTOR->getDataChannel());
			ndc = 0;
			iDataReader *r = iVTKWindow::getCurrentWindow()->getReader();
			for(i=0; i<=MAXDATACHANNEL; i++) if(r->isThereVectData(i,0)) ndc++;
			DataChannelComboBox->setEnabled(ndc>1 && TabWidgetVect->TabWidget->currentPageIndex()==0);
			if(ndc>1 && TabWidgetVect->TabWidget->currentPageIndex()==0) DataChannelApplyButton->show(); else DataChannelApplyButton->hide(); 
			break;
		}
		//
		//  Tens
		//
	case DISPLAYPAGE_TENS:
		{
			this->setDataChannel(CURRENT_TENSOR->getDataChannel());
			ndc = 0;
			iDataReader *r = iVTKWindow::getCurrentWindow()->getReader();
			for(i=0; i<=MAXDATACHANNEL; i++) if(r->isThereTensData(i,0)) ndc++;
			DataChannelComboBox->setEnabled(ndc>1 && TabWidgetTens->TabWidget->currentPageIndex()==0);
			if(ndc>1 && TabWidgetTens->TabWidget->currentPageIndex()==0) DataChannelApplyButton->show(); else DataChannelApplyButton->hide(); 
			break;
		}
		//
		//  Data
		//
	case DISPLAYPAGE_DATA:
		{
			DataChannelComboBox->setEnabled(true);
			DataChannelApplyButton->hide();
			this->setDataChannel(DataChannelComboBox->currentItem());
			if(dataChannel == 0)
			{
				//
				//  Make sure that all pages are enabled: the extension may disable some of the pages
				//
				int i, j;
				const int n = 5;
				QTabWidget *t[] = { 
					TabWidgetData->TabWidget, 
					TabWidgetData->TabWidgetAdjustment, 
					TabWidgetData->TabWidgetPlacement, 
					TabWidgetData->TabWidgetSettings,
					TabWidgetData->TabWidgetTransform };

				for(i=0; i<n; i++)
				{
					for(j=0; j<t[i]->count(); j++)
					{
						if(!t[i]->isTabEnabled(t[i]->page(j))) t[i]->setTabEnabled(t[i]->page(j),true);
					}
				}
			}
			break;
		}
	default:
		{
		}
	
	}

	if(ext != NULL) ext->updateDataChannel(mode,dataChannel);

}


void iQT::updateTabWidgetPage(QWidget *page)
{
	if(page == TabWidget->page(DISPLAYPAGE_VIEW)) 
	{
		TabWidgetView->updateView_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_VIEW);
	}
	if(page == TabWidget->page(DISPLAYPAGE_SURF)) 
	{
		TabWidgetSurf->updateSurf_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_SURF);
	}
	if(page == TabWidget->page(DISPLAYPAGE_XSEC)) 
	{
		TabWidgetXsec->updateXsec_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_XSEC);
	}
	if(page == TabWidget->page(DISPLAYPAGE_VOLV)) 
	{
		TabWidgetVolv->updateVolv_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_VOLV);
	}
	if(page == TabWidget->page(DISPLAYPAGE_PART)) 
	{
		TabWidgetPart->updatePart_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_PART);
	}
	if(page == TabWidget->page(DISPLAYPAGE_VECT)) 
	{
		TabWidgetVect->updateVect_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_VECT);
	}
	if(page == TabWidget->page(DISPLAYPAGE_TENS)) 
	{
		TabWidgetTens->updateTens_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_TENS);
	}
	if(page == TabWidget->page(DISPLAYPAGE_DATA)) 
	{
		TabWidgetData->updateData_Widgets();
		if(ext != NULL) ext->updateTabWidgetPage(DISPLAYPAGE_DATA);
	}
}


void iQT::setRemoteControlled()
{
	int i;
	remoteControlled = true;
	for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++)
	{
	    iVTKWindow::getWindow(i)->getAbortRenderEventObserver()->setProgressDialog(0);
	}
	TabWidgetView->TabWidgetWins->windows_MakeWindowUnderFocusCurrent(false);

}


bool iQT::isRemoteControlled()
{
	return remoteControlled;
}


void iQT::updateVisitedFilesList(bool reload)
{
	this->updateVisitedFilesList(iVTKWindow::getCurrentWindowIndex(),reload);
}
	

void iQT::updateVisitedFilesList(int win, bool reload)
{
	static QString margin = "    ";

	if(reload)
	{
		CurrentFileStack->raiseWidget(iVTKWindow::getWindow(win)->getId());
		CurrentFile = (QComboBox *)CurrentFileStack->visibleWidget();
		return;
	}

	QComboBox *fileBox = (QComboBox *)CurrentFileStack->widget(iVTKWindow::getWindow(win)->getId());
	iStringList files = iVTKWindow::getWindow(win)->getReader()->getVisitedFilesList();
	CALL_FUNCTION2(fileBox,insertItem,files[0].mid(1)+margin,0);
	CALL_FUNCTION1(fileBox,setCurrentItem,0);
	//
	//  Make sure the line does not get cut off at the right
	//
	QListBox *list = fileBox->listBox();
	if(list != 0)
	{
		int icut = 1, lastWidth = list->item(0)->width(list) + 1;
		QString s;
		while(icut<999 && list->item(0)->width(list)>fileBox->width() && list->item(0)->width(list)<lastWidth)
		{
			lastWidth = list->item(0)->width(list);
			icut++;
			CALL_FUNCTION2(list,changeItem,files[0].mid(icut)+margin,0);
		}
	}

}


void iQT::createVisitedFilesList(int win)
{
	QComboBox *tmp = CurrentFile = new QComboBox(false,CurrentFileStack);
	if(tmp == 0) reportNullPointer(1009);
	tmp->setMaxCount(10);
	CurrentFileStack->addWidget(tmp,iVTKWindow::getWindow(win)->getId());
	connect( (const QObject *)tmp, SIGNAL( activated(const QString &) ) , this, SLOT ( fileOpenVisited(const QString &) ) );
}


void iQT::deleteVisitedFilesList(int id)
{
	QComboBox *tmp = (QComboBox *)CurrentFileStack->widget(id);
	CurrentFileStack->removeWidget(tmp);
	delete tmp;
}


void iQT::placeNewMarker(double *x)
{
	blockRenderer = true;
	TabWidgetView->updateView_Mark_AddMarkerPushButton();
	iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMember()->setPosition(x);
	blockRenderer = false;
	this->startTask();
	this->render(true);
	this->endTask();
}


void iQT::displayCurrentMarkerPosition()
{
	double ex[3], *x = iVTKWindow::getCurrentWindow()->getMarkerFamily()->getCurrentMember()->getPosition();
	QString s;
	iVTKWindow::getCurrentWindow()->transformToExternalCoordinate(3,x,ex);
	CALL_FUNCTION1(TabWidgetView->View_Mark_MarkerPositionXLineEdit,setText,s.setNum(ex[0]));
	CALL_FUNCTION1(TabWidgetView->View_Mark_MarkerPositionYLineEdit,setText,s.setNum(ex[1]));
	CALL_FUNCTION1(TabWidgetView->View_Mark_MarkerPositionZLineEdit,setText,s.setNum(ex[2]));
}


void iQT::updateCoordinateDisplays()
{
	this->updateVariablesExplorer();
	this->displayCurrentMarkerPosition();
	TabWidgetXsec->updateCoordinateDisplays();
	TabWidgetVect->updateCoordinateDisplays();
	if(Pick->isVisible()) this->launchFromMenuPickerDialog();
	if(ext != NULL) ext->updateCoordinateDisplays();
}


void iQT::resetProgressBar()
{
	ProgressBar->reset();
}


void iQT::updateParallelDisplay()
{
	ParallelController->updateDisplay();
}


void iQT::setIdiosyncratic(bool s)
{
	idiosyncratic = s;
}


void iQT::updateMenuSettings()
{
	int k;

	blockRenderer = true;
	//
	//  Options
	//
	switch(iVTKWindow::getCurrentWindow()->getImageMagnification())
	{
	case   2: { if(!optionsImageZoomX002Action->isOn()) optionsImageZoomX002Action->setOn(true); break; }
	case   3: { if(!optionsImageZoomX002Action->isOn()) optionsImageZoomX003Action->setOn(true); break; }
	case   4: { if(!optionsImageZoomX004Action->isOn()) optionsImageZoomX004Action->setOn(true); break; }
	case   5: { if(!optionsImageZoomX005Action->isOn()) optionsImageZoomX005Action->setOn(true); break; }
	case   6: { if(!optionsImageZoomX006Action->isOn()) optionsImageZoomX006Action->setOn(true); break; }
	case   8: { if(!optionsImageZoomX008Action->isOn()) optionsImageZoomX008Action->setOn(true); break; }
	case  10: { if(!optionsImageZoomX010Action->isOn()) optionsImageZoomX010Action->setOn(true); break; }
	case  15: { if(!optionsImageZoomX015Action->isOn()) optionsImageZoomX015Action->setOn(true); break; }
	case  20: { if(!optionsImageZoomX020Action->isOn()) optionsImageZoomX020Action->setOn(true); break; }
	case  30: { if(!optionsImageZoomX030Action->isOn()) optionsImageZoomX030Action->setOn(true); break; }
	case  40: { if(!optionsImageZoomX040Action->isOn()) optionsImageZoomX040Action->setOn(true); break; }
	case  50: { if(!optionsImageZoomX050Action->isOn()) optionsImageZoomX050Action->setOn(true); break; }
	case  60: { if(!optionsImageZoomX060Action->isOn()) optionsImageZoomX060Action->setOn(true); break; }
	case  80: { if(!optionsImageZoomX080Action->isOn()) optionsImageZoomX080Action->setOn(true); break; }
	case 100: { if(!optionsImageZoomX100Action->isOn()) optionsImageZoomX100Action->setOn(true); break; }
	default: if(!optionsImageZoomX001Action->isOn()) optionsImageZoomX001Action->setOn(true);
	}

	switch(iVTKWindow::getCurrentWindow()->getImageType())
	{
	case I_IMAGETYPE_JPG: { if(!optionsImageFormatJPGAction->isOn()) optionsImageFormatJPGAction->setOn(true); break; }
	case I_IMAGETYPE_PNM: { if(!optionsImageFormatPNMAction->isOn()) optionsImageFormatPNMAction->setOn(true); break; }
	case I_IMAGETYPE_BMP: { if(!optionsImageFormatBMPAction->isOn()) optionsImageFormatBMPAction->setOn(true); break; }
	case I_IMAGETYPE_PNG: { if(!optionsImageFormatPNGAction->isOn()) optionsImageFormatPNGAction->setOn(true); break; }
	case I_IMAGETYPE_TIF: { if(!optionsImageFormatTIFAction->isOn()) optionsImageFormatTIFAction->setOn(true); break; }
	default: { if(!optionsImageFormatPSAction->isOn()) optionsImageFormatPSAction->setOn(true); break; }
	}

	optionsImageFormatPSPaperFormatActionGroup->setEnabled((iVTKWindow::getCurrentWindow()->getImageType()==I_IMAGETYPE_PS));
	optionsImageFormatPSOrientationActionGroup->setEnabled((iVTKWindow::getCurrentWindow()->getImageType()==I_IMAGETYPE_PS));

	if(iVTKWindow::getCurrentWindow()->getWriter()->IsA("vtkPostScriptWriter")) 
	{
		k = ((iPostScriptWriter *)iVTKWindow::getCurrentWindow()->getWriter())->getPaperFormat();
		if(!psPaperFormatList[k]->isOn()) psPaperFormatList[k]->setOn(true);
		switch(((iPostScriptWriter *)iVTKWindow::getCurrentWindow()->getWriter())->getOrientation())
		{
		case 1: { if(!optionsImageFormatPSOrientationLandscapeAction->isOn()) optionsImageFormatPSOrientationLandscapeAction->setOn(true); break; }
		default: if(!optionsImageFormatPSOrientationPortraitAction->isOn()) optionsImageFormatPSOrientationPortraitAction->setOn(true);
		}
	}

	if(iVTKWindow::getCurrentWindow()->getReader()->getAdjustableBounds())
	{
	    if(!optionsVarLimitsAdjustableAction->isOn()) optionsVarLimitsAdjustableAction->setOn(true);
	}
	else
	{
		if(!optionsVarLimitsFixedAction->isOn()) optionsVarLimitsFixedAction->setOn(true);
	}
  
	if(iVTKWindow::getCurrentWindow()->getReader()->getEndinessOfData())
	{
		if(!optionsDataEndinessBigAction->isOn()) optionsDataEndinessBigAction->setOn(true);
	}
	else
	{
		if(!optionsDataEndinessLittleAction->isOn()) optionsDataEndinessLittleAction->setOn(true);
	}

	//
	//  Style
	//
	switch(iVTKWindow::getCurrentWindow()->getBoxType())
	{
	case 1: { if(!styleBoxClassicAction->isOn()) styleBoxClassicAction->setOn(true); break; }
	case 2: { if(!styleBoxHairAction->isOn()) styleBoxHairAction->setOn(true); break; }
	case 3: { if(!styleBoxAxesAction->isOn()) styleBoxAxesAction->setOn(true); break; }
	default: if(!styleBoxIfritAction->isOn()) styleBoxIfritAction->setOn(true);
	}
	
    switch(iVTKWindow::getCurrentWindow()->getFontSize())
	{
	case 12: { if(!styleFontSize12Action->isOn()) styleFontSize12Action->setOn(true); break; }
	case 14: { if(!styleFontSize14Action->isOn()) styleFontSize14Action->setOn(true); break; }
	case 16: { if(!styleFontSize16Action->isOn()) styleFontSize16Action->setOn(true); break; }
	case 20: { if(!styleFontSize20Action->isOn()) styleFontSize20Action->setOn(true); break; }
	case 24: { if(!styleFontSize24Action->isOn()) styleFontSize24Action->setOn(true); break; }
	case 32: { if(!styleFontSize32Action->isOn()) styleFontSize32Action->setOn(true); break; }
	default: if(!styleFontSize10Action->isOn()) styleFontSize10Action->setOn(true);
	}
 	
    switch(iVTKWindow::getCurrentWindow()->getFontType())
	{
	case 1: { if(!styleFontTypeVectorAction->isOn()) styleFontTypeVectorAction->setOn(true); break; }
	default: if(!styleFontTypeBitmapAction->isOn()) styleFontTypeBitmapAction->setOn(true);
	}

	blockRenderer = false;

}


void iQT::updateWidgets(int m)
{
	switch(m)
	{
	case DISPLAYPAGE_VIEW: { TabWidgetView->updateView_Widgets(); break; }
	case DISPLAYPAGE_SURF: { TabWidgetSurf->updateSurf_Widgets(); break; }
	case DISPLAYPAGE_XSEC: { TabWidgetXsec->updateXsec_Widgets(); break; }
	case DISPLAYPAGE_VOLV: { TabWidgetVolv->updateVolv_Widgets(); break; }
	case DISPLAYPAGE_PART: { TabWidgetPart->updatePart_Widgets(); break; }
	case DISPLAYPAGE_VECT: { TabWidgetVect->updateVect_Widgets(); break; }
	case DISPLAYPAGE_TENS: { TabWidgetTens->updateTens_Widgets(); break; }
	case DISPLAYPAGE_DATA: { TabWidgetData->updateData_Widgets(); break; }
	case -1:
		{
			this->updateMenuSettings();
			TabWidgetView->updateView_Widgets();
			TabWidgetSurf->updateSurf_Widgets();
			TabWidgetXsec->updateXsec_Widgets();
			TabWidgetVolv->updateVolv_Widgets();
			TabWidgetPart->updatePart_Widgets();
			TabWidgetVect->updateVect_Widgets();
			TabWidgetTens->updateTens_Widgets(); 
			TabWidgetData->updateData_Widgets();
		    if(ext != NULL) ext->updateWidgets(-2);
		}
	}

}


void iQT::blockProgressDialog(bool s)
{
	int i;
	if(s)
	{
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->getAbortRenderEventObserver()->setProgressDialog(NULL);
	}
	else
	{
		for(i=0; i<=iVTKWindow::getMaxWindowIndex(); i++) iVTKWindow::getWindow(i)->getAbortRenderEventObserver()->setProgressDialog(ProgressDialog);
	}
}


void iQT::showBusy(bool s)
{
	if(s) busyLabel->setPixmap(pixmapBusyOn); else busyLabel->setPixmap(pixmapBusyOff); 
	busyLabel->repaint();
}


#if !EDITION
bool iQT::edition()
{
	return false;
}
#endif


