/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "iglobals.h"
#include "iorthotextureplane.h"

#include "imath.h"

#include <vtkStructuredPoints.h>
#include <vtkPointData.h>
#include <vtkFloatArray.h>
#include <vtkPolyData.h>
#include <vtkCellArray.h>

void reportNullPointer(int);


iOrthoTexturePlane::iOrthoTexturePlane() 
{

	texture = vtkStructuredPoints::New();
	if(texture == 0) reportNullPointer(4901);
	
}


iOrthoTexturePlane::~iOrthoTexturePlane() 
{
	texture->Delete();
}


void iOrthoTexturePlane::ExecuteData(vtkDataObject *)
{
    vtkStructuredPoints *input = (vtkStructuredPoints *)this->GetInput();
    vtkPolyData *output = (vtkPolyData *)this->GetOutput();

	if(input == 0) return;

	output->Initialize();
	
	int dims[3], dimout[3];
	vtkFloat org[3], spa[3];

    input->GetOrigin(org);
    input->GetSpacing(spa);
    input->GetDimensions(dims);

	int i, Axis = -1;
	for(i=0; i<3; i++)
	{
		if(dims[i] == 1) Axis = i;
	}
	if(Axis == -1) return;

    int Uidx=(Axis+1)%3,  Vidx=(Axis+2)%3;
	//
	//  Prepare texture support
	//
	static float tc[4][2] = {{0,0}, {1,0}, {1,1}, {0,1}};
	static vtkIdType pts[4]={0,1,2,3};
	//
	// We'll create the building blocks of polydata including data attributes.
	//
	vtkPoints *points = vtkPoints::New(VTK_FLOAT);
	if(points == 0) reportNullPointer(4903);
	
	vtkCellArray *polys = vtkCellArray::New();
	if(polys == 0) reportNullPointer(4904);
	
	vtkFloatArray* tcoords = vtkFloatArray::New();
	if(tcoords == 0) reportNullPointer(4905);
	
	points->SetNumberOfPoints(4);
	tcoords->SetNumberOfComponents(2);
	tcoords->SetNumberOfTuples(4);
	//
	// Load the cell, and data attributes.
	//
	for (i=0; i<4; i++) tcoords->SetTuple(i,tc[i]);
	polys->InsertNextCell(4,pts);
	//
	//  Place the support plane
	//
	float x1[3];
	x1[Axis] = org[Axis];
	//
	//  LL point 
	//
	x1[Uidx] = org[Uidx];
	x1[Vidx] = org[Vidx];
	points->SetPoint(0,x1);
	//
	//  LR point 
	//
	x1[Uidx] = org[Uidx] + spa[Uidx]*(dims[Uidx]-1);
	x1[Vidx] = org[Vidx];
	points->SetPoint(1,x1);
	//
	//  UR point 
	//
	x1[Uidx] = org[Uidx] + spa[Uidx]*(dims[Uidx]-1);
	x1[Vidx] = org[Vidx] + spa[Vidx]*(dims[Vidx]-1);
	points->SetPoint(2,x1);
	//
	//  UL point 
	//
	x1[Uidx] = org[Uidx];
	x1[Vidx] = org[Vidx] + spa[Vidx]*(dims[Vidx]-1);
	points->SetPoint(3,x1);
	//
	// We now assign the pieces to the vtkPolyData.
	//
	output->SetPolys(polys);
	output->SetPoints(points);
	output->GetPointData()->SetTCoords(tcoords);
	polys->Delete();
	points->Delete();
	tcoords->Delete();
	//	
	//  Fill in the texture
	//
	dimout[2] = 1;
	dimout[0] = dims[Uidx] - 1;
	dimout[1] = dims[Vidx] - 1;

	texture->SetOrigin(org);
    texture->SetSpacing(spa);
    texture->SetDimensions(dimout);

    float *ptrin = (float *)input->GetScalarPointer();
	vtkFloatArray *data = vtkFloatArray::New();
	if(data == 0) reportNullPointer(4902);
	data->SetNumberOfComponents(1);
	data->SetNumberOfTuples(dimout[0]*dimout[1]);

	int j;
	vtkIdType loff, loff1, loff2;
	for(j=0; j<dimout[1]; j++)
	{

		this->UpdateProgress((float)j/dimout[1]);
		if(this->GetAbortExecute()) break;

		loff1 = dimout[0]*j;
		loff2 = dims[Uidx]*j;
		for(i=0; i<dimout[0]; i++)
		{
			loff = i + loff2;
			data->SetTuple1(i+loff1,0.25*(*(ptrin+loff)+*(ptrin+loff+1)+*(ptrin+loff+dims[Uidx])+*(ptrin+loff+1+dims[Uidx])));
		}
	}

	texture->GetPointData()->SetScalars(data);
	data->Delete();

}

