/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of ivolume.h
*/

#include "iglobals.h"
#include "ivolume.h"

#include "ivtk.h"
#include "ilimits.h"
#include "ipalette.h"
#include "iobjectfactory.h"

#include "icolor.h"
#include "idatagateway.h"
#include "idatareader.h"
#include "ipiecewisefunction.h"
#include "ivolumedataconverter.h"
#include "ihistogrammaker.h"
#include "iuniformmeshdataconverter.h"
#include "iuniformmeshdataextender.h"
#include "iuniformmeshdata.h"
#include "ivisualobjectobserver.h"
#include "ivtkcallback.h"
#include "icolorbars.h"

#include <vtkVolume.h>
#include <vtkVolumeProperty.h>
#include <vtkVolumeRayCastCompositeFunction.h>
#include <vtkVolumeRayCastMIPFunction.h>
#include <vtkVolumeRayCastMapper.h>
#include <vtkVolumeTextureMapper2D.h>
#include <vtkPiecewiseFunction.h>
#include <vtkImageData.h>


void reportNullPointer(int ec);


//
// iVolume class
// 

iVolume* iVolume::New(iVTK *mVTK)
{
	return (iVolume *)iObjectFactory::createObject("Volume",mVTK);
}


iVolume::iVolume(iVTK *mVTK) : iVisualObject(mVTK)
{
	
	var = 1;
	secvar = 0;

	pal = 0;
	seccolor = iColor(255,0,0);
	secopa = 1;	
	
#ifdef VTK_USE_VOLUMEPRO
	mode = IVOLUME_MODE_VOLUMEPRO;
#else
	mode = IVOLUME_MODE_TEXTURE;
#endif
	
	opa = new iPiecewiseFunction(0.0,0.996);
	
	compositeMethod = false;
	interpolationType = false; 
	blendMode = 0;

	resXY = resZ = 1.0;
	blendMode = 0;
	
	volume = vtkVolume::New();
	if(volume == 0) reportNullPointer(6301);
	volume->SetPosition(0.0,0.0,0.0);
	
	prop = vtkVolumeProperty::New();
	if(prop == 0) reportNullPointer(6302);
	prop->SetShade(shade);
	prop->SetAmbient(ambient);
	prop->SetDiffuse(diffuse);
	prop->SetSpecular(specular);
	prop->SetSpecularPower(specularPower);
	prop->SetScalarOpacity(opa->getVTKFunction());
	prop->SetColor(iPaletteList::getInstance()->getPalette(pal)->getFunction());
	prop->SetInterpolationTypeToNearest();
	volume->SetProperty(prop);
	
	volumeDataConverter = iVolumeDataConverter::New(this);
	if(volumeDataConverter == 0) reportNullPointer(6308);
	volumeExtender = iUniformMeshDataExtender::New(this);
	if(volumeExtender == 0) reportNullPointer(6309);
	volumeHistogram[0] = iHistogramMaker::New();
	if(volumeHistogram[0] == 0) reportNullPointer(6310);
	volumeHistogramConverter = iUniformMeshDataConverter::New(this);
	if(volumeHistogramConverter == 0) reportNullPointer(6311);
	
	volumeDataConverter->SetInput(myVTK->getGateway()->GetOutput());
	volumeExtender->SetInput(volumeDataConverter->GetOutput());
	volumeHistogramConverter->SetInput(myVTK->getGateway()->GetOutput());
	volumeHistogramConverter->setCurrentVar(var);
	volumeHistogram[0]->SetInput(volumeHistogramConverter->GetOutput(),0);

	iVTKAbortRenderEventObserver *obsAbortRender = myVTK->getAbortRenderEventObserver();

	raycastMapper = vtkVolumeRayCastMapper::New();
	if(raycastMapper == 0) reportNullPointer(6303);
	compositeFunction = vtkVolumeRayCastCompositeFunction::New();
	if(compositeFunction == 0) reportNullPointer(6304);
	compositeFunction->SetCompositeMethodToInterpolateFirst();
	mipFunction = vtkVolumeRayCastMIPFunction::New();
	if(mipFunction == 0) reportNullPointer(6305);

	switch(blendMode)
	{
	case 0: { raycastMapper->SetVolumeRayCastFunction(compositeFunction); break; }
	case 1: { raycastMapper->SetVolumeRayCastFunction(mipFunction); break; }
	}
	//  raycastMapper->SetMinimumImageSampleDistance(0.125);
	//  raycastMapper->SetImageSampleDistance(1.0);
	
	textureMapper = vtkVolumeTextureMapper2D::New();
	if(textureMapper == 0) reportNullPointer(6306);
	
	raycastMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	textureMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	volumeDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	volumeExtender->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);

#ifdef VTK_USE_VOLUMEPRO
	volumeproMapper = iVolumeProMapper::New();
	if(volumeproMapper == 0) reportNullPointer(6307);

	volumeproMapper->IntermixIntersectingGeometryOn();
	volumeproMapper->UseImageClipperOff();
	volumeproMapper->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
#else
	volumeproMapper = 0;
#endif

#ifndef I_VTK_VERSION_40
	raycastMapper->UseImageClipperOff();
	textureMapper->UseImageClipperOff();
#ifdef VTK_USE_VOLUMEPRO
	volumeproMapper->UseImageClipperOff();
#endif
#endif

	textureMapper->SetInput(volumeExtender->GetOutput());
	raycastMapper->SetInput(volumeExtender->GetOutput());
#ifdef VTK_USE_VOLUMEPRO
	volumeproMapper->SetInput(volumeExtender->GetOutput());
#endif

	//
	//  Add observer to keep information about this object
	//
	volume->AddObserver(vtkCommand::UserEvent,objectObserver);
	objectObserver->setTypes(IOBJECTTYPE_VOLV,IDATATYPE_MESH);

	volume->VisibilityOff();

}


iVolume::~iVolume()
{
	
	if(!init) myVTK->removeProp(volume);
	delete opa;
	volume->Delete();
	prop->Delete();
	volumeDataConverter->Delete();
	volumeExtender->Delete();
	volumeHistogram[0]->Delete();
	volumeHistogramConverter->Delete();
	raycastMapper->Delete();
	textureMapper->Delete();
#ifdef VTK_USE_VOLUMEPRO
	volumeproMapper->Delete();
#endif
	compositeFunction->Delete();
	mipFunction->Delete();
	
}


void iVolume::reset()
{
	vtkFloat *s;
	s = myVTK->getReader()->getMeshOutput()->GetSpacing();
	raycastMapper->SetSampleDistance(s[2]*resZ);
	this->updateDataChannel();
	//
	//  Un-initialized, if needed
	//
	if(!init)
	{
		myVTK->removeProp(volume);
		init = true;
	}
}


void iVolume::showClipPlane(bool s)
{

	if(s && !clipPlaneOn)
	{
		raycastMapper->AddClippingPlane(myVTK->getClipPlane());
		textureMapper->AddClippingPlane(myVTK->getClipPlane());
#ifdef VTK_USE_VOLUMEPRO
		volumeproMapper->AddClippingPlane(myVTK->getClipPlane());
#endif
		clipPlaneOn = true;
	} 
	if(!s && clipPlaneOn)
	{
		raycastMapper->RemoveClippingPlane(myVTK->getClipPlane());
		textureMapper->RemoveClippingPlane(myVTK->getClipPlane());
#ifdef VTK_USE_VOLUMEPRO
		volumeproMapper->RemoveClippingPlane(myVTK->getClipPlane());
#endif
		clipPlaneOn = false;
	}

}


void iVolume::showColorBars(bool show)
{
	if(volume->GetVisibility() == 0) return;
	myVTK->getColorBars()->showBar(COLORBARS_PRIORITY_VOLUME,var,dataChannel,pal,show);
}


void iVolume::setVar(int v)
{
	if(v>=1 && v<=this->getLimits()->getNvar() && v!=var)
	{
		this->showColorBars(false);
		var = v;
		this->showColorBars(true);
		volumeDataConverter->setCurrentVar(v);
		volumeHistogramConverter->setCurrentVar(var);
		volumeHistogramConverter->Update();
	}
}


void iVolume::setSecondaryVar(int d)
{ 
	if(d>=0 && d<=this->getLimits()->getNvar()) secvar = d;
}


void iVolume::setPal(int p)
{ 
	p = ( p < 0 ) ? 0 : p;
	p = ( p > iPaletteList::getInstance()->getNumPalettes() ) ? iPaletteList::getInstance()->getNumPalettes() : p;
	this->showColorBars(false);
	pal = p;
	this->showColorBars(true);
	prop->SetColor(iPaletteList::getInstance()->getPalette(pal)->getFunction());
}


void iVolume::setSecondaryColor(iColor c) 
{
	seccolor = c;
}


void iVolume::setSecondaryOpacity(int d)
{
	if(d>=0 && d<=2) secopa = d;
}


void iVolume::setShade(bool v)
{
	shade = v;
	if(v) prop->SetShade(1); else prop->SetShade(0);
}


void iVolume::setAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	ambient = v;
	prop->SetAmbient(v);
}


void iVolume::setDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	diffuse = v;
	prop->SetDiffuse(v);
}


void iVolume::setSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	specular = v;
	prop->SetSpecular(v);
}


void iVolume::setSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	specularPower = v;
	prop->SetSpecularPower(v);
}


void iVolume::show(bool show)
{
	
	if(show)
	{
		if(init)
		{
			switch(mode) 
			{
			case IVOLUME_MODE_TEXTURE: 
				{
					if(dataChannel == 0) volume->SetMapper(textureMapper);
					init = false;
					break; 
				}
			case IVOLUME_MODE_RAYCAST: 
				{
					vtkFloat *s = myVTK->getReader()->getMeshOutput()->GetSpacing();
					raycastMapper->SetSampleDistance(s[2]*resZ);
					if(dataChannel == 0) volume->SetMapper(raycastMapper);
					init = false;
					break; 
				}
			case IVOLUME_MODE_VOLUMEPRO: 
				{
#ifdef VTK_USE_VOLUMEPRO
					if(dataChannel == 0) volume->SetMapper(volumeproMapper);
					init = false;
#endif
					break; 
				}
			}      			
			if(!init)
			{
//				this->setDataChannel(myVTK->getReader()->getCurrentDataChannel());
				myVTK->addProp(volume);
			}
		} 
		if(!init) 
		{
			volume->VisibilityOn();
			this->showColorBars(true);
		}
		volumeHistogramConverter->Update();
	} 
	else 
	{
		if(!init) 
		{
			this->showColorBars(false);
			volume->VisibilityOff();
		}
	}
	
}


int iVolume::changeMode(int m)
{
	
	int mo = mode;
	mode = m;
	switch(m) 
	{
	case IVOLUME_MODE_TEXTURE: 
		{
			if(!init && dataChannel==0)
			{
				volume->SetMapper( textureMapper );
			}
			break; 
		}
	case IVOLUME_MODE_RAYCAST: 
		{
			if(!init) 
			{
				vtkFloat *s = myVTK->getReader()->getMeshOutput()->GetSpacing();
				raycastMapper->SetSampleDistance(s[2]*resZ);
				volume->SetMapper( raycastMapper );
			}
			break; 
		}
	case IVOLUME_MODE_VOLUMEPRO: 
		{
#ifdef VTK_USE_VOLUMEPRO
			if(!init) 
			{
				volume->SetMapper( volumeproMapper );
			}
#else
			mode = mo;
#endif
			break; 
		}
	default: 
		{
			mode = mo; 
		}
	}      			
	
	return mode;
	
}


bool iVolume::isVolumePro()
{
	
#ifdef VTK_USE_VOLUMEPRO
	return true;
#else
	return false;
#endif
	
}


void iVolume::setCompositeMethod(bool m) 
{
	
	if(m != compositeMethod) 
	{
		compositeMethod = m;
		if(!m)
		{ 
			compositeFunction->SetCompositeMethodToInterpolateFirst();
		} 
		else 
		{
			compositeFunction->SetCompositeMethodToClassifyFirst();
		}
	}
	
}


void iVolume::setInterpolationType(bool m) 
{
	
	if(m != interpolationType) 
	{
		interpolationType = m;
		if(!m)
		{ 
			prop->SetInterpolationTypeToNearest();
		} 
		else 
		{
			prop->SetInterpolationTypeToLinear();
		}
	}
	
}


void iVolume::setZResample(float m) 
{
	
	if(m == resZ) return;
	
	resZ = m;
	bool work = false;
	
	switch(mode) 
	{
	case IVOLUME_MODE_RAYCAST: 
		{
			if(!init) raycastMapper->SetSampleDistance(*(myVTK->getReader()->getMeshOutput()->GetSpacing())*resZ);
			work = true; break; 
		}
	case IVOLUME_MODE_VOLUMEPRO: 
		{
#ifdef VTK_USE_VOLUMEPRO
			volumeproMapper->SetSuperSamplingFactor(resXY,resXY,resZ);
			work = true;
#endif
			break; 
		}
	default: { }
	}      			
	
}


void iVolume::setXYResample(float m) 
{
	
	if(m == resXY) return;
	
	resXY = m;
	bool work = false;
	
	switch(mode) 
	{
	case IVOLUME_MODE_RAYCAST: 
		{
			raycastMapper->SetImageSampleDistance(resXY);
			work = true; break; 
		}
	case IVOLUME_MODE_VOLUMEPRO: 
		{
#ifdef VTK_USE_VOLUMEPRO
			volumeproMapper->SetSuperSamplingFactor(resXY,resXY,resZ);
			work = true;
#endif
			break; 
		}
	default: { }
	}      			
	
}


void iVolume::setBlendMode(int m) 
{
	
	if(m==blendMode || m<0 || m>1) return;
	
	blendMode = m;
	bool work = false;
	
	switch(mode) 
	{
	case IVOLUME_MODE_RAYCAST: 
		{
			switch(blendMode){
			case 0: { raycastMapper->SetVolumeRayCastFunction(compositeFunction); break; }
			case 1: { raycastMapper->SetVolumeRayCastFunction(mipFunction); break; }
			}
			work = true; break; 
		}
	case IVOLUME_MODE_VOLUMEPRO: 
		{
#ifdef VTK_USE_VOLUMEPRO
			volumeproMapper->SetBlendMode(blendMode);
			work = true;
#endif
			break; 
		}
	default: { }
	}      			
	
}


float iVolume::getMemorySize()
{
	float s = 0.0;
	s += volumeDataConverter->getMemorySize();
	s += volumeExtender->getMemorySize();
	return s;
}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iVolume::packState(iString &s)
{
	int nFun = opa->getFunctionN();

	iVisualObject::packState(s);

	this->packValue(s,"iVolume::var",var);
	this->packValue(s,"iVolume::mode",mode);
	this->packValue(s,"iVolume::pal",pal);
	this->packValue(s,"iVolume::blendMode",blendMode);
	this->packValue(s,"iVolume::secvar",secvar);
	this->packValue(s,"iVolume::secopa",secopa);
	this->packValue(s,"iVolume::seccolor",seccolor);

	this->packValue(s,"iVolume::compositeMethod",compositeMethod);
	this->packValue(s,"iVolume::interpolationType",interpolationType);

	this->packValue(s,"iVolume::resXY",resXY);
	this->packValue(s,"iVolume::resZ",resZ);

	this->packValue(s,"iVolume::nFun",nFun);
	this->packValue(s,"iVolume::xFun",opa->getFunctionX(),nFun);
	this->packValue(s,"iVolume::yFun",opa->getFunctionY(),nFun);


}


void iVolume::unpackState(iString s)
{
	int i; bool b; float f; iColor c;
		
	iVisualObject::unpackState(s);

	if(this->unpackValue(s,"iVolume::var",i)) this->setVar(i);
	if(this->unpackValue(s,"iVolume::mode",i)) this->changeMode(i);
	if(this->unpackValue(s,"iVolume::pal",i)) this->setPal(i);
	if(this->unpackValue(s,"iVolume::blendMode",i)) this->setBlendMode(i);
	if(this->unpackValue(s,"iVolume::secvar",i)) this->setSecondaryVar(i);
	if(this->unpackValue(s,"iVolume::secopa",i)) this->setSecondaryOpacity(i);
	if(this->unpackValue(s,"iVolume::seccolor",c)) this->setSecondaryColor(c);
	
	if(this->unpackValue(s,"iVolume::compositeMethod",b)) this->setCompositeMethod(b);
	if(this->unpackValue(s,"iVolume::interpolationType",b)) this->setInterpolationType(b);
	
	if(this->unpackValue(s,"iVolume::resXY",f)) this->setXYResample(f);
	if(this->unpackValue(s,"iVolume::resZ",f)) this->setZResample(f);
	
	int nFun;
	if(this->unpackValue(s,"iVolume::nFun",nFun))
	{
		delete opa;
		opa = new iPiecewiseFunction(0.0,0.996);
		if(opa == 0) reportNullPointer(6312);
		prop->SetScalarOpacity(opa->getVTKFunction());
		
		float *xFun = new float[nFun];
		float *yFun = new float[nFun];
		
		if(this->unpackValue(s,"iVolume::xFun",xFun,nFun) && this->unpackValue(s,"iVolume::yFun",yFun,nFun))
		{
			for(i=0; i<nFun; i++)
			{
				if(i==0 || i==nFun-1) 
				{
					opa->movFunctionPoint(i,xFun[i],yFun[i]);
				}
				else
				{
					opa->addFunctionPoint(xFun[i],yFun[i]);
				}
			}
		}
	}

}


vtkImageData* iVolume::getMapperInput()
{
	switch(mode) 
	{
	case IVOLUME_MODE_TEXTURE: return textureMapper->GetInput();
	case IVOLUME_MODE_RAYCAST: return raycastMapper->GetInput();
	case IVOLUME_MODE_VOLUMEPRO: 
		{
#ifdef VTK_USE_VOLUMEPRO
			return volumeproMapper->GetInput();
#endif
		}
	default: ;
	}      			
	
	return 0;

}


