/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

/*
Implementation of iguisurface.h
*/

#include "isurface.h"

#include "ipalette.h"
#include "ivtk.h"
#include "iglobals.h"
#include "iobjectfactory.h"

#include "isurfaceshifter.h"
#include "ipolygonaldataextender.h"
#include "idatareader.h"
#include "idatagateway.h"
#include "ivtkcallback.h"
#include "iuniformmeshdata.h"
#include "iuniformmeshdataconverter.h"
#include "icontourfilter.h"
#include "idecimator.h"
#include "ivisualobjectobserver.h"
#include "ilimits.h"
#include "iprobefilter.h"
#include "icolorbars.h"

#include <vtkCommand.h>
#include <vtkFloatArray.h>
#include <vtkToolkits.h>
#include <vtkPolyData.h>
#include <vtkPolyDataNormals.h>
#include <vtkPolyDataMapper.h>
#include <vtkLODActor.h>
#include <vtkActor.h>
#include <vtkProperty.h>
#include <vtkContourFilter.h>
#include <vtkDecimatePro.h>
#include <vtkQuadricDecimation.h>
#include <vtkQuadricClustering.h>
#include <vtkWindowedSincPolyDataFilter.h>
#include <vtkCleanPolyData.h>
#include <vtkStructuredPoints.h>
#include <vtkLookupTable.h>
#include <vtkReverseSense.h>


void reportNullPointer(int ec);


//
// iSurface class
//

iSurface* iSurface::New(iVTK *mVTK)
{
	return (iSurface *)iObjectFactory::createObject("Surface",mVTK);
}


iSurface::iSurface(iVTK *mVTK) : iVisualObject(mVTK)
{
	int i;
	int nvar = this->getLimits()->getNvar();
	var = 1;
//
//  varpaint[0,1] = 0 means no painting on the exterior,interior.
//
	varpaint[0] = 0; 
	varpaint[1] = 0; 
	float lmin = this->getLimits()->getVarMin(var);
	float lmax = this->getLimits()->getVarMax(var);

	if(var==1 && this->getLimits()->getVarName(1)==iString("Neutral fraction"))
	{
		lev = (float)-0.3; 
		if(nvar > 1) varpaint[0] = 2;
	} 
	else lev = 0.5*(lmin+lmax); 
	lev = (lev < lmin) ? lmin : lev;
	lev = (lev > lmax) ? lmax : lev;
	
	opa = 1.0;
	
	color = NICE_DEFAULT_COLOR;
	
	palpaint[0] = 0;
	palpaint[1] = 0;

	decimate = true;
	smooth = 0;
	normalsFlipped = false;
	postNormals = true;
//
//  Do VTK stuff
//	
	surfaceSource = iContourFilter::New();
	if(surfaceSource == 0) reportNullPointer(7601);
	surfaceDataConverter = iUniformMeshDataConverter::New(this);
	if(surfaceDataConverter == 0) reportNullPointer(7602);
	surfaceShifter = iSurfaceShifter::New();
	if(surfaceShifter == 0) reportNullPointer(7603);

	for(i=0; i<2; i++)
	{
		surfaceActor[i] = vtkLODActor::New();
		if(surfaceActor[i] == 0) reportNullPointer(7604);
		surfaceMapper[i] = vtkPolyDataMapper::New();
		if(surfaceMapper[i] == 0) reportNullPointer(7605);
		surfaceNormals[i] = vtkPolyDataNormals::New();
		if(surfaceNormals[i] == 0) reportNullPointer(7606);
		surfaceCleaner[i] = vtkCleanPolyData::New();
		if(surfaceCleaner[i] == 0) reportNullPointer(7607);
		surfaceSmoother[i] = vtkWindowedSincPolyDataFilter::New();
		if(surfaceSmoother[i] == 0) reportNullPointer(7608);
		surfaceReducer[i] = iDecimator::New();
		if(surfaceReducer[i] == 0) reportNullPointer(7609);
		surfaceProbe[i] = iProbeFilter::New();
		if(surfaceProbe[i] == 0) reportNullPointer(7610);
		surfaceExtender[i] = iPolygonalDataExtender::New(this);
		if(surfaceExtender[i] == 0) reportNullPointer(7611);
		surfaceDataConverterPaint[i] = iUniformMeshDataConverter::New(this);
		if(surfaceDataConverterPaint[i] == 0) reportNullPointer(7612);
		surfaceNormalsFlipper[i] = vtkReverseSense::New();
		if(surfaceNormalsFlipper[i] == 0) reportNullPointer(7614);
	}

	surfaceDataConverter->setCurrentVar(var);

	surfaceSource->SetInput(surfaceDataConverter->GetOutput());
	surfaceSource->UseScalarTreeOn();
	surfaceSource->SetValue(0,lev);

	for(i=0; i<2; i++)
	{

		surfaceDataConverterPaint[i]->setCurrentVar(varpaint[i]);

		surfaceProbe[i]->SetInput(surfaceSource->GetOutput());
		surfaceProbe[i]->SetSource(surfaceDataConverterPaint[i]->GetOutput());
		surfaceProbe[i]->SpatialMatchOn();

		surfaceExtender[i]->SetInput(surfaceProbe[i]->GetPolyDataOutput());

		surfaceCleaner[i]->SetInput(surfaceExtender[i]->GetOutput());
		surfaceCleaner[i]->PointMergingOn();
		surfaceCleaner[i]->SetTolerance(0.0);

		surfaceNormals[i]->SetFeatureAngle(90.0);
		surfaceNormals[i]->ComputeCellNormalsOff();

		surfaceNormalsFlipper[i]->ReverseCellsOff();
		surfaceNormalsFlipper[i]->SetInput(surfaceNormals[i]->GetOutput());

	}

	for(i=0; i<2; i++)
	{
		
		surfaceSmoother[i]->SetNumberOfIterations(10*smooth);
		
		surfaceReducer[i]->SetTargetReduction(0.75);
		surfaceReducer[i]->PreserveTopologyOn();
		
	}
	
	surfaceMapper[0]->SetInput(surfaceNormalsFlipper[0]->GetOutput());

	surfaceShifter->SetInput(surfaceNormalsFlipper[1]->GetOutput());
	surfaceMapper[1]->SetInput(surfaceShifter->GetOutput());
	
	for(i=0; i<2; i++)
	{
		
		if(varpaint[i] == 0) surfaceMapper[i]->ScalarVisibilityOff(); else surfaceMapper[i]->ScalarVisibilityOn();
		surfaceMapper[i]->SetLookupTable(iPaletteList::getInstance()->getPalette(palpaint[i])->getTable());
		surfaceMapper[i]->SetScalarRange(this->getLimits()->getVarLow(varpaint[i]),this->getLimits()->getVarHigh(varpaint[i]));
		surfaceMapper[i]->SetScalarModeToDefault();
		surfaceMapper[i]->UseLookupTableScalarRangeOff();
		
		surfaceActor[i]->SetMapper(surfaceMapper[i]);
		surfaceActor[i]->SetPosition(0.0,0.0,0.0);
		
	}

	surfaceActor[1]->VisibilityOff();

	surfaceProperty = surfaceActor[0]->GetProperty();
	surfaceProperty->SetInterpolationToGouraud();
	surfaceProperty->SetAmbient(ambient);
	surfaceProperty->SetDiffuse(diffuse);
	surfaceProperty->SetSpecular(specular);
	surfaceProperty->SetSpecularPower(specularPower);

	surfaceActor[1]->SetProperty(surfaceProperty);

	iVTKAbortRenderEventObserver *obsAbortRender = myVTK->getAbortRenderEventObserver();
	for(i=0; i<2; i++)
	{
		surfaceMapper[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceNormals[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceCleaner[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceSmoother[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceReducer[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceProbe[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceExtender[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceDataConverterPaint[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
		surfaceNormalsFlipper[i]->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	}
	surfaceSource->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	surfaceDataConverter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	surfaceShifter->AddObserver(vtkCommand::ProgressEvent,obsAbortRender);
	
	surfaceDataConverter->SetInput(myVTK->getGateway()->GetOutput());
	for(i=0; i<2; i++) surfaceDataConverterPaint[i]->SetInput(myVTK->getReader()->getMeshOutput());
	//
	//  Add observer to keep information about this object
	//
	surfaceActor[0]->AddObserver(vtkCommand::UserEvent,objectObserver);
	objectObserver->setTypes(IOBJECTTYPE_SURF,IDATATYPE_MESH);
	objectObserver->setUserValue(lev);

	surfaceActor[0]->VisibilityOff();
	surfaceActor[1]->VisibilityOff();

}


iSurface::~iSurface()
{
	int i;
	for(i=0; i<2; i++)
	{
		if(!init && myVTK!=0) myVTK->removeProp(surfaceActor[i]);
		surfaceActor[i]->Delete();
		surfaceMapper[i]->Delete();
		surfaceNormals[i]->Delete();
		surfaceExtender[i]->Delete();
		surfaceSmoother[i]->Delete();
		surfaceReducer[i]->Delete();
		surfaceCleaner[i]->Delete();
		surfaceProbe[i]->Delete();
		surfaceDataConverterPaint[i]->Delete();
		surfaceNormalsFlipper[i]->Delete();
	}
	surfaceSource->Delete();
	surfaceDataConverter->Delete();
	surfaceShifter->Delete();
}


void iSurface::reset()
{
	int i;
	for(i=0; i<2; i++) surfaceDataConverterPaint[i]->SetInput(myVTK->getReader()->getMeshOutput());
	this->updateDataChannel();
	//
	//  Un-initialized, if needed
	//
	if(!init)
	{
		myVTK->removeProp(surfaceActor[0]);
		myVTK->removeProp(surfaceActor[1]);
		init = true;
	}
}


void iSurface::updatePipeline()
{ 
	int i;

	for(i=0; i<2; i++)
	{
		if(smooth > 0)
		{
			surfaceSmoother[i]->SetInput(surfaceCleaner[i]->GetOutput());
			surfaceSmoother[i]->SetNumberOfIterations(10*smooth);
			if(decimate) 
			{
				surfaceReducer[i]->SetInput(surfaceSmoother[i]->GetOutput());
				surfaceNormals[i]->SetInput(surfaceReducer[i]->GetOutput());
			} 
			else 
			{
				surfaceNormals[i]->SetInput(surfaceSmoother[i]->GetOutput());
			}
		} 
		else 
		{
			if(decimate) 
			{
				surfaceReducer[i]->SetInput(surfaceCleaner[i]->GetOutput());
				surfaceNormals[i]->SetInput(surfaceReducer[i]->GetOutput());
			} 
			else 
			{
				surfaceNormals[i]->SetInput(surfaceCleaner[i]->GetOutput());
			}
		}
	}
}


void iSurface::showClipPlane(bool s)
{

	if(s && !clipPlaneOn)
	{
		surfaceMapper[0]->AddClippingPlane(myVTK->getClipPlane());
		surfaceMapper[1]->AddClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = true;
	} 
	if(!s && clipPlaneOn) 
	{
		surfaceMapper[0]->RemoveClippingPlane(myVTK->getClipPlane());
		surfaceMapper[1]->RemoveClippingPlane(myVTK->getClipPlane());
		clipPlaneOn = false;
	}

}


void iSurface::showColorBars(bool show)
{
	if(surfaceActor[0]->GetVisibility() == 0) return;
	if(varpaint[0] > 0) myVTK->getColorBars()->showBar(COLORBARS_PRIORITY_SURFACE_EXTERIOR,varpaint[0],dataChannel,palpaint[0],show);
	if(varpaint[1] > 0) myVTK->getColorBars()->showBar(COLORBARS_PRIORITY_SURFACE_INTERIOR,varpaint[1],dataChannel,palpaint[1],show);
}


void iSurface::setVar(int v)
{ 
	int nvar = this->getLimits()->getNvar();
	if(v>=1 &&  v<=nvar && v!=var)
	{
		var = v;
		surfaceDataConverter->setCurrentVar(var);
		float lmin = this->getLimits()->getVarMin(var);
		float lmax = this->getLimits()->getVarMax(var);
		lev = (lev < lmin) ? lmin : lev;
		lev = (lev > lmax) ? lmax : lev;
		this->flipNormals(normalsFlipped);
	}
}


void iSurface::setLevel(float l)
{ 
	float lmin, lmax;
	myVTK->getReader()->getLevelLimits(var,lmin,lmax);
	if(lmin < this->getLimits()->getVarMin(var)) lmin = this->getLimits()->getVarMin(var);
	if(lmax > this->getLimits()->getVarMax(var)) lmax = this->getLimits()->getVarMax(var);
	l = (l < lmin) ? lmin : l;
	l = (l > lmax) ? lmax : l;
	lev = l;
	surfaceSource->SetValue(0,lev);
	objectObserver->setUserValue(lev);
}


void iSurface::setOpacity(float o)
{ 
	o = (o < 0.0) ? 0.0 : o;
	o = (o > 1.0) ? 1.0 : o;
	opa = o;
	surfaceProperty->SetOpacity(o);
}


void iSurface::setVarPaint(bool s, int v)
{ 
	int nvar = this->getLimits()->getNvar();
	if(v<0 || v>nvar) v = 0;
	if(v != varpaint[(int)s]) 
	{
		this->showColorBars(false);
		varpaint[(int)s] = v;
		this->showColorBars(true);
		if(!s)
		{
			if(v > 0) 
			{
				surfaceDataConverterPaint[0]->setCurrentVar(v);
				surfaceMapper[0]->SetScalarRange(this->getLimits()->getVarLow(v),this->getLimits()->getVarHigh(v));
				surfaceMapper[0]->ScalarVisibilityOn();
			} 
			else 
			{
				surfaceMapper[0]->ScalarVisibilityOff();
			}
		} 
		else 
		{
			if(v > 0) 
			{
				surfaceDataConverterPaint[1]->setCurrentVar(v);
				surfaceMapper[1]->SetScalarRange(this->getLimits()->getVarLow(v),this->getLimits()->getVarHigh(v));
				surfaceMapper[1]->ScalarVisibilityOn();
				surfaceActor[1]->VisibilityOn();
			} 
			else 
			{
				surfaceMapper[1]->ScalarVisibilityOff();
				surfaceActor[1]->VisibilityOff();
			}
		}
	}
	this->updateDataChannel();
}


void iSurface::setColor(iColor c)
{ 
	color = c;
	surfaceProperty->SetColor((float)(c.red()/255.0),(float)(c.green()/255.0),(float)(c.blue()/255.0));
}


void iSurface::setPalPaint(bool s, int p)
{ 
	p = ( p < 0 ) ? 0 : p;
	p = ( p > iPaletteList::getInstance()->getNumPalettes() ) ? iPaletteList::getInstance()->getNumPalettes() : p;
	int i = (s) ? 1 : 0;
	this->showColorBars(false);
	palpaint[i] = p;
	this->showColorBars(true);
	surfaceMapper[i]->SetLookupTable(iPaletteList::getInstance()->getPalette(palpaint[i])->getTable());
}


void iSurface::setAmbient(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	ambient = v;
	surfaceProperty->SetAmbient(v);
}


void iSurface::setDiffuse(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	diffuse = v;
	surfaceProperty->SetDiffuse(v);
}


void iSurface::setSpecular(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>1.0) ? 1.0 : v;
	specular = v;
	surfaceProperty->SetSpecular(v);
}


void iSurface::setSpecularPower(float v)
{
	v = (v<0.0) ? 0.0 : v;
	v = (v>100.0) ? 100.0 : v;
	specularPower = v;
	surfaceProperty->SetSpecularPower(v);
}


void iSurface::setGeometricNormals(bool s)
{ 
	int i;
	
	postNormals = s; 

	if(postNormals) surfaceSource->ComputeNormalsOff(); else surfaceSource->ComputeNormalsOn();
	for(i=0; i<2; i++) if(postNormals) surfaceNormals[i]->ComputePointNormalsOn(); else surfaceNormals[i]->ComputePointNormalsOff();

	this->flipNormals(normalsFlipped);

} 


void iSurface::setDecimation(bool s)
{ 
	decimate = s; 
	this->updatePipeline();
} 


void iSurface::setSmoothing(int p)
{ 
	p = ( p < 0 ) ? 0 : p;
	p = ( p > 10 ) ? 10 : p;
	smooth = p;
	this->updatePipeline();
}


void iSurface::flipNormals(bool s)
{ 

	normalsFlipped = s;
//	if(!postNormals) s = !s;
	if((s && var!=1) || (!s && var==1)) 
	{
		surfaceNormalsFlipper[0]->ReverseNormalsOn(); 
		surfaceNormalsFlipper[1]->ReverseNormalsOff();
		surfaceShifter->setShift(fabs(surfaceShifter->getShift()));
	} 
	else 
	{
		surfaceNormalsFlipper[0]->ReverseNormalsOff();
		surfaceNormalsFlipper[1]->ReverseNormalsOn();
		surfaceShifter->setShift(-fabs(surfaceShifter->getShift()));
	}
	
}


void iSurface::show(bool show)
{
	
	if(show)
	{
		if(init)
		{
			//
			//  Assign object to the VTK window
			//
			this->setColor(color);
			this->setGeometricNormals(postNormals);
			this->updatePipeline();

			myVTK->addProp(surfaceActor[0]);
			myVTK->addProp(surfaceActor[1]);
			init = false;

		}
		if(!init)
		{
			if(varpaint[0] > 0) 
			{
				surfaceMapper[0]->SetScalarRange(this->getLimits()->getVarLow(varpaint[0]),this->getLimits()->getVarHigh(varpaint[0]));
			}
			if(varpaint[1] > 0) 
			{
				surfaceMapper[1]->SetScalarRange(this->getLimits()->getVarLow(varpaint[1]),this->getLimits()->getVarHigh(varpaint[1]));
				surfaceActor[1]->VisibilityOn();
			}
			surfaceActor[0]->VisibilityOn();
			this->showColorBars(true);
		}
	} 
	else 
	{
		if(!init)
		{
			this->showColorBars(false);
			surfaceActor[0]->VisibilityOff();
			surfaceActor[1]->VisibilityOff();
		}
	}
	
}


float iSurface::getMemorySize()
{
	int i;
	float s, s1;

	if(init) return 0;

	s = surfaceSource->GetOutput()->GetActualMemorySize();

	for(i=0; i<2; i++)
	{
		s1 = (float)surfaceProbe[i]->GetPolyDataOutput()->GetActualMemorySize() - (float)surfaceProbe[i]->GetInput()->GetActualMemorySize();
		if(s1 < 0.0) s1 = 0.0;
		s += s1;
		s += surfaceExtender[i]->getMemorySize();
		s += surfaceCleaner[i]->GetOutput()->GetActualMemorySize();
	}

	for(i=0; i<2; i++)
	{
		if(smooth > 0) s += surfaceSmoother[i]->GetOutput()->GetActualMemorySize();
		if(decimate) s += surfaceReducer[i]->GetOutput()->GetActualMemorySize();
		s1 = (float)surfaceNormalsFlipper[i]->GetOutput()->GetActualMemorySize() - (float)surfaceNormals[i]->GetInput()->GetActualMemorySize();
		if(s1 < 0.0) s1 = 0.0;
		s += s1;
	}

	return s;

}
//
//  Two functions used in saving/restoring the state and in creating new instances with
//
void iSurface::packState(iString &s)
{

	iVisualObject::packState(s);

	this->packValue(s,"iSurface::normalsFlipped",normalsFlipped);
	this->packValue(s,"iSurface::decimate",decimate);

	this->packValue(s,"iSurface::var",var);
	this->packValue(s,"iSurface::color",color);
	this->packValue(s,"iSurface::varpaint",varpaint,2);
	this->packValue(s,"iSurface::palpaint",palpaint,2);
	this->packValue(s,"iSurface::smooth",smooth);

	this->packValue(s,"iSurface::lev",lev);
	this->packValue(s,"iSurface::opa",opa);

}


void iSurface::unpackState(iString s)
{
	int i; bool b; float f; int v[2]; iColor c;

	iVisualObject::unpackState(s);

	if(this->unpackValue(s,"iSurface::normalsFlipped",b)) this->flipNormals(b);
	if(this->unpackValue(s,"iSurface::decimate",b)) this->setDecimation(b);

	if(this->unpackValue(s,"iSurface::var",i)) this->setVar(i);
	if(this->unpackValue(s,"iSurface::color",c)) this->setColor(c);
	if(this->unpackValue(s,"iSurface::varpaint",v,2))
	{
		this->setVarPaint(false,v[0]);
		this->setVarPaint(true,v[1]);
	}
	if(this->unpackValue(s,"iSurface::palpaint",v,2))
	{
		this->setPalPaint(false,v[0]);
		this->setPalPaint(true,v[1]);
	}
	if(this->unpackValue(s,"iSurface::smooth",i)) this->setSmoothing(i);

	if(this->unpackValue(s,"iSurface::lev",f)) this->setLevel(f);
	if(this->unpackValue(s,"iSurface::opa",f)) this->setOpacity(f);

}


