/*=========================================================================

  Program:   Ionization FRont Interactive Tool (IFRIT)
  Language:  C++


Copyright (c) 2002-2003 Nick Gnedin 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Nick Gnedin nor the names of any contributors may be used 
   to endorse or promote products derived from this software without specific
   prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/


#include "idialogvariablesexplorer.h"

#include "ivtk.h"
#include "ivtkwindow.h"
#include "iqt.h"
#include "iqtwindow.h"
#include "ivariablesview.h"
#include "ihistogrammaker.h"
#include "idatareader.h"
#include "ilimits.h"
#include "iuniformmeshdata.h"
#include "iuniformmeshdataconverter.h"
#include "ilabel.h"
#include "ipiecewisefunction.h"
#include "iqtdefs.h"


#include <qapplication.h>
#include <qcolordialog.h>
#include <qcombobox.h>
#include <qcursor.h>
#include <qheader.h>
#include <qimage.h>
#include <qlayout.h>
#include <qlcdnumber.h>
#include <qlineedit.h>
#include <qlistbox.h>
#include <qpixmap.h>
#include <qpopupmenu.h>
#include <qpushbutton.h>

#include <vtkCamera.h>
#include <vtkRenderer.h>


QPixmap image(char *);


namespace iDialogVariablesExplorer_private
{
	
	static QColor defaultColorList[10];

	class VarItem : public QListViewItem
	{
	public:

		VarItem(QListView *parent, QString s1, QString s2) : QListViewItem(parent,s1,s2){}
		iPiecewiseFunction function; 
		QColor color; 
		float opacity; 
		float vMax, vMin, vMed, vAvg, vDis; 
		vtkIdType lMin, lMax;
		double xMin[3], xMax[3];
		int var, dc, ds;
		iVTK *window;
	};


	void displayLineValue(QLineEdit *w, float val)
	{
		w->setText(QString::number((double)val,'g',3));
	}

	void displayComboValue(QComboBox *w, float val, vtkIdType cell, double pos[3])
	{
		w->clear();
		w->insertItem(QString::number((double)val,'g',3));
		w->insertItem("Cell: "+QString::number((long)cell));
		w->insertItem("at X: "+QString::number(pos[0]));
		w->insertItem("at Y: "+QString::number(pos[1]));
		w->insertItem("at Z: "+QString::number(pos[2]));
	}
	
	void displayLCD(QLCDNumber *lcd, float val)
	{
		int i = 4;
		do
		{
			lcd->setNumDigits(++i);
		}
		while(lcd->checkOverflow(val) && i<12);
		if(lcd->checkOverflow(val)) lcd->display("E"); else lcd->display(val);
	}

};
using namespace iDialogVariablesExplorer_private;


class iDialogVariablesExplorerPopup : public QDialog
{

public:

	iDialogVariablesExplorerPopup(iDialogVariablesExplorer *myVE, QWidget *parent) : QDialog(parent,"",true,!WType_Dialog | WType_Popup)
	{
		oldPos = QPoint(0,0);
		moved = false;
		ve = myVE;
		QVBoxLayout *l1 = new QVBoxLayout(this,1,1,"");
		QFrame *f1 = new QFrame(this);
	    f1->setFrameShape(QFrame::StyledPanel);
		f1->setFrameShadow(QFrame::Raised);
		l1->add(f1);
		QHBoxLayout *l2 = new QHBoxLayout(f1,4,-1,"");
		b = new QLabel(f1);
		l2->add(b);
		lv = new QListView(this);
		lv->setFrameShape(QFrame::NoFrame);
		lv->setSorting(-1);
		lv->addColumn("Variable");
		lv->addColumn("Frac. below");
		lv->addColumn("Frac. above");
		lv->setSelectionMode(QListView::NoSelection);
		lv->header()->setClickEnabled(false,0);
		lv->header()->setClickEnabled(false,1);
		lv->header()->setClickEnabled(false,2);
		l1->add(lv);
	}

	void mouseMoveEvent(QMouseEvent *e)
	{ 
		if(e != 0) 
		{
			moved = (e->globalPos() != oldPos);
			oldPos = e->globalPos();
			if(ve != 0) 
			{
				ve->setViewData(b,lv,e->globalPos()); 
				if(lv->contentsWidth()<0.7*ve->width()) this->resize(2+lv->contentsWidth(),this->height());
			}
			this->move(e->globalPos()-QPoint(this->width(),0)); 
		}
	}

	void mouseReleaseEvent(QMouseEvent *)
	{ 
		if(moved) 
		{
			this->hide(); 
			moved = false;
		}
	} 

	void mousePressEvent(QMouseEvent *e)
	{ 
		oldPos = e->globalPos();
		moved = false;
		this->hide(); 
	} 

	void setData(float v)
	{
		b->setText("Value: "+QString::number(v));
	}

private:

	QLabel *b;
	iDialogVariablesExplorer *ve;
	bool moved;
	QPoint oldPos;
	QListView *lv;

};

//
//  Main class
//
void iDialogVariablesExplorer::init()
{

	defaultColorList[0] = red;
	defaultColorList[1] = blue;
	defaultColorList[2] = green;
	defaultColorList[3] = cyan;
	defaultColorList[4] = magenta;
	defaultColorList[5] = darkRed;
	defaultColorList[6] = darkBlue;
	defaultColorList[7] = darkGreen;
	defaultColorList[8] = darkCyan;
	defaultColorList[9] = darkMagenta;

	ShownVariablesList->clear();
	ShownVariablesList->addColumn("Shown Variables");
	ShownVariablesList->addColumn("Window");
	ShownVariablesList->setSelectionMode(QListView::Single);
	ShownVariablesList->setSorting(-1);
//	ShownVariablesList->setColumnWidth(0,QListView::Maximum);
	ShownVariablesList->header()->setClickEnabled(false,0);
	ShownVariablesList->header()->setClickEnabled(false,1);
	if(MAXDATACHANNEL > 0) 
	{
		ShownVariablesList->addColumn("Data channel");
		ShownVariablesList->header()->setClickEnabled(false,2);
	}

	AllVariablesList->clear();

	AddVariablePushButton->setPixmap(image("backward.png"));

	this->setIcon(image("varexp.png"));

	((QBoxLayout *)this->layout())->setStretchFactor(LayoutTop,3);
	((QBoxLayout *)this->layout())->setStretchFactor(LayoutBottom,1);

	connect(View, SIGNAL(resized()), this, SLOT(updateView()) );
	connect(ShownVariablesList, SIGNAL(dropped(QDropEvent *)), this, SLOT(updateDropped(QDropEvent *)) );

	rcm = new QPopupMenu(ShownVariablesList);
	rcm->setCheckable(false);
	rcm->setItemParameter(rcm->insertItem("Remove",this,SLOT(updatePopupMenu(int))),0);
	rcm->setItemParameter(rcm->insertItem("Set color",this,SLOT(updatePopupMenu(int))),1);
	rcm->setItemParameter(rcm->insertItem("Increase opacity",this,SLOT(updatePopupMenu(int))),2);
	rcm->setItemParameter(rcm->insertItem("Decrease opacity",this,SLOT(updatePopupMenu(int))),3);

	connect(ShownVariablesList,SIGNAL(rightButtonPressed(QListViewItem *,const QPoint &,int)),this,SLOT(updateRightButtonPressed(QListViewItem *,const QPoint &,int))); 

	connect(View,SIGNAL(mouseEvent(QMouseEvent *)),this,SLOT(updateMouseEvent(QMouseEvent *)));

	pop = new iDialogVariablesExplorerPopup(this,0);

	
	pix = QPixmap(128,128,-1,QPixmap::BestOptim);
	im0.create(128,128,4);
	im0.setAlphaBuffer(true);
	im1.create(128,128,4);
	im1.setAlphaBuffer(true);

	rcmItem = 0;
	stretchLog = false;

}


void iDialogVariablesExplorer::destroy()
{
	delete pop;
}


void iDialogVariablesExplorer::addVariable()
{
	int i = AllVariablesList->currentItem();
	if(i == -1) return;

	QString s1 = AllVariablesList->currentText();
	QString s2 = " #"+QString::number(iVTKWindow::getCurrentWindowIndex());
	QString s3;
	if(MAXDATACHANNEL > 0) s3 = QString::number(iVTKWindow::getCurrentWindow()->getReader()->getCurrentDataChannel())+"/"+QString::number(iVTKWindow::getCurrentWindow()->getReader()->getCurrentDataStream(iVTKWindow::getCurrentWindow()->getReader()->getCurrentDataChannel()));
	//
	//  Check for duplicates
	//
	QListViewItem *is = ShownVariablesList->firstChild();
	while(is!=0 && (is->text(0)!=s1 || is->text(1)!=s2 || (!s3.isNull() && is->text(2)!=s3))) is = is->itemBelow();
	if(is != 0) 
	{
		ShownVariablesList->setCurrentItem(is);
		return;  // A duplicate!
	}

	//
	//  Create a new entry
	//
	VarItem *it = new VarItem(ShownVariablesList,s1,s2);
	if(MAXDATACHANNEL > 0) it->setText(2,s3);
	if(ShownVariablesList->childCount() <= 10)
	{
		it->color = defaultColorList[ShownVariablesList->childCount()-1];
	}
	else
	{
		it->color = gray;
	}

	QPixmap p = QPixmap(16,16);
	p.fill(it->color);
	it->setPixmap(0,p);
	it->setDragEnabled(true);
	it->setDropEnabled(true);
	it->moveItem(ShownVariablesList->lastItem());

	it->opacity = 0.75;

	it->window = iVTKWindow::getCurrentWindow();
	it->var = i;
	it->dc = iVTKWindow::getCurrentWindow()->getReader()->getCurrentDataChannel();
	it->ds = iVTKWindow::getCurrentWindow()->getReader()->getCurrentDataStream(it->dc);

	this->updateEntry(it);

	this->updateView();

}


bool iDialogVariablesExplorer::updateEntry(QListViewItem *it0)
{
	int j;

	VarItem *it = (VarItem *)it0;

	if(!this->extractFunction(it->window,it->dc,it->ds,it->var,&it->function,it->vMin,it->lMin,it->xMin,it->vMax,it->lMax,it->xMax)) return false;

	float s = 1.0e-35;
	while(it->vMax<it->vMin+s && s<1.0e30) 
	{
		it->vMax += s; 
		s *= 2;
	}

	double sum0 = 0.0, sum1 = 0.0, sum2 = 0.0;
	int n = it->function.getFunctionN();
	float *x = it->function.getFunctionX();
	float *y = it->function.getFunctionY();

	for(j=0; j<n; j++)
	{
		sum0 += y[j];
		sum1 += y[j]*x[j];
		sum2 += y[j]*x[j]*x[j];
	}

	if(sum0 > 0.0)
	{
		it->vAvg = it->vMin + sum1/sum0*(it->vMax-it->vMin);
		it->vDis = sqrt(sum2/sum0-pow(sum1/sum0,2.0))*(it->vMax-it->vMin);
		it->vMed = 0.0;
		sum1 = 0.0;
		for(j=0; j<n; j++)
		{
			if(sum1<0.5*sum0 && (sum1+=y[j])>=0.5*sum0)
			{
				it->vMed = it->vMin + x[j]*(it->vMax-it->vMin);
			}
		}
	}
	else
	{
		it->vAvg = it->vDis = it->vMed = 0.5*(it->vMax+it->vMin);
	}

	return true;

}


void iDialogVariablesExplorer::updateHistogramStretch(int s)
{
	bool v = (s==1);
	if(v != stretchLog)
	{
		stretchLog = v;
		this->updateView();
	}
}


void iDialogVariablesExplorer::updateView()
{
	static QPainter z;

	if(!this->isVisible()) return;
	
    int vw = View->width(); 
	int vh = View->height();
	
	if(vw!=pix.width() || vh!=pix.height())
	{
		pix = QPixmap(vw,vh,-1,QPixmap::BestOptim);
		im0.create(vw,vh,4);
		im1.create(vw,vh,4);
	}
	
	if(ShownVariablesList->firstChild() == 0)
	{
		z.begin(&pix);
		z.eraseRect(0,0,pix.width(),pix.height());
		z.end();
		View->setPixmap(pix);
		return;
	}

	float vmin = 1.0e30, vmax = -1.0e30;
	VarItem *it = (VarItem *)ShownVariablesList->firstChild();
	displayComboValue(MaxValueComboBox,it->vMax,it->lMax,it->xMax);
	displayComboValue(MinValueComboBox,it->vMin,it->lMin,it->xMin);
	displayLineValue(MedValueLineEdit,it->vMed);
	displayLineValue(AvgValueLineEdit,it->vAvg);
	displayLineValue(DisValueLineEdit,it->vDis);

	while(it != 0)
	{
		if(vmin > it->vMin) vmin = it->vMin;
		if(vmax < it->vMax) vmax = it->vMax;
		it = (VarItem *)it->itemBelow();
	}

	displayLCD(MinRangeLCD,vmin);
	displayLCD(MaxRangeLCD,vmax);

	float limits[2];
	bool last = true;
	it = (VarItem *)ShownVariablesList->lastItem();
	while(it != 0)
	{
		limits[0] = (it->vMin-vmin)/(vmax-vmin);
		limits[1] = (it->vMax-vmin)/(vmax-vmin);
		//
		//  Draw a histogram
		//
		z.begin(&pix);
		z.eraseRect(0,0,pix.width(),pix.height());
		iQTWindow::getCurrentWindow()->drawHistogram(z,&(it->function),it->color,it->opacity,false,stretchLog,limits);
		z.end();
		//
		//   Blend on the previous image
		//
		if(last)
		{
			im0 = pix.convertToImage();
			last = false;
		}
		else
		{
			float w0, w1;
			w0 = 1.0 - it->opacity;
			w1 = it->opacity;
			im1 = pix.convertToImage();
			unsigned char *sPtr0 = im0.bits();
			unsigned char *sPtr1 = im1.bits();
			int j, jmax = 4*vh*vw;
			for(j=0; j<jmax; j+=4)
			{
				if(sPtr1[j]!=255 || sPtr1[j+1]!=255 || sPtr1[j+2]!=255)
				{
					sPtr0[j+0] = sPtr0[j+0]*w0 + sPtr1[j+0]*w1;
					sPtr0[j+1] = sPtr0[j+1]*w0 + sPtr1[j+1]*w1;
					sPtr0[j+2] = sPtr0[j+2]*w0 + sPtr1[j+2]*w1;
				}
			}
		}
		//
		//  Go on
		//
		it = (VarItem *)it->itemAbove();
	}
	
	pix.convertFromImage(im0);
	View->setPixmap(pix);

}


void iDialogVariablesExplorer::updateVariables()
{
	int j, jcur, jmax;

	AllVariablesList->clear();

	this->extractVariableList();

	jcur = iVTKWindow::getCurrentWindowIndex();
	jmax = iVTKWindow::getMaxWindowIndex();

	VarItem *itb, *it = (VarItem *)ShownVariablesList->firstChild();
	while(it != 0)
	{
		itb = (VarItem *)it->itemBelow();
		for(j=0; j<=jmax; j++) if(it->window == iVTKWindow::getWindow(j))
		{
			it->setText(1," #"+QString::number(j));
			if(!this->updateEntry(it)) delete it;  //  data could have changed in all readers, not only the current one
		}
		it = itb;
	}

	this->updateView();

}



void iDialogVariablesExplorer::updateDropped(QDropEvent *)
{
	this->updateView();
}


void iDialogVariablesExplorer::updatePopupMenu(int mode)
{
	if(rcmItem == 0) return;

	VarItem *it = (VarItem *)rcmItem;

	switch(mode)
	{
	case 0:
		{
			delete it;
			break;
		}
	case 1:
		{
			it->color = QColorDialog::getColor(it->color,this);
			QPixmap p = QPixmap(16,16);
			p.fill(it->color);
			it->setPixmap(0,p);
			break;
		}
	case 2:
		{
			it->opacity += 0.25;
			if(it->opacity > 1.0) it->opacity = 1.0;
			break;
		}
	case 3:
		{
			it->opacity -= 0.25;
			if(it->opacity < 0.25) it->opacity = 0.25;
			break;
		}
	}

	this->updateView();
	rcmItem = 0;

}


void iDialogVariablesExplorer::updateRightButtonPressed(QListViewItem *it, const QPoint &p, int )
{
	if(it != NULL)
	{
		rcmItem = it;
		rcm->exec(p);
	}
}


void iDialogVariablesExplorer::updateMouseEvent(QMouseEvent *e)
{
	if(e == 0) return;
	if(e->button() == Qt::LeftButton) 
	{
		pop->show();
		QMouseEvent me(QEvent::MouseMove,QPoint(0,0),Qt::LeftButton,Qt::LeftButton);
		QApplication::sendEvent(pop,&me);
		pop->update();
		QApplication::sendEvent(pop,&me);  // do it twice for click-release to work properly
	}
}


void iDialogVariablesExplorer::setViewData(QLabel *l, QListView *v, const QPoint &p)
{
	double sum1, sum2;
	int j, n;
	float *x, *y, xv;

	float val = MinRangeLCD->value() + (MaxRangeLCD->value()-MinRangeLCD->value())/View->width()*View->mapFromGlobal(p).x();
	l->setText("Value: "+QString::number(val));
	if(v != 0)
	{
		QListViewItem *vt1, *vt = v->firstChild();
		VarItem *it = (VarItem *)ShownVariablesList->firstChild();
		while(it != 0)
		{
			if(vt == 0)
			{
				vt = new QListViewItem(v,"","","");
				vt->moveItem(v->lastItem());
			}
		
			vt->setText(0,it->text(0)+" ("+it->text(1)+")");

			n = it->function.getFunctionN();
			x = it->function.getFunctionX();
			y = it->function.getFunctionY();

			xv = (val-it->vMin)/(it->vMax-it->vMin);
			if(xv < 0.0)
			{
				sum1 = 0.0;
				sum2 = 1.0;
			}
			else if(xv > 1.0)
			{
				sum1 = 1.0;
				sum2 = 0.0;
			}
			else
			{
				sum1 = sum2 = 0.0;
				for(j=0; j<n; j++)
				{
					if(xv < x[j]) sum2 += y[j]; else sum1 += y[j]; 
				}
				if(sum1+sum2 < 0.5) sum1 = sum2 = 0.5;
			}

			vt->setText(1,QString::number(sum1/(sum1+sum2)));
			vt->setText(2,QString::number(sum2/(sum1+sum2)));

			vt = vt->itemBelow();
			it = (VarItem *)it->itemBelow();
		}

		while(vt != 0)
		{
			vt1 = vt->itemBelow();
			delete vt;
			vt = vt1;
		}

	}
}

void iDialogVariablesExplorer::updateComboBoxes(int)
{
	CALL_FUNCTION1(MaxValueComboBox,setCurrentItem,0);
	CALL_FUNCTION1(MinValueComboBox,setCurrentItem,0);
}


void iDialogVariablesExplorer::updateMaxCenterOnScreen()
{
	VarItem *it = (VarItem *)ShownVariablesList->firstChild();
	if(it != 0) this->updateCenterOnScreen(it->xMax);
}


void iDialogVariablesExplorer::updateMinCenterOnScreen()
{
	VarItem *it = (VarItem *)ShownVariablesList->firstChild();
	if(it != 0) this->updateCenterOnScreen(it->xMin);
}


void iDialogVariablesExplorer::updateCenterOnScreen(double *xi)
{
	int j;
	vtkCamera *cam = iVTKWindow::getCurrentWindow()->getRenderer()->GetActiveCamera();
	double cf[3], cp[3];
	cam->GetPosition(cp);
	cam->GetFocalPoint(cf);
	for(j=0; j<3; j++) cp[j] += (xi[j]-cf[j]);
	cam->SetFocalPoint(xi);
	cam->SetPosition(cp);
	iVTKWindow::getCurrentWindow()->render(true);
}


//
//  Those two functions need to be overloaded for other data channels
//
void iDialogVariablesExplorer::extractVariableList()
{
	int i;
	if(iVTKWindow::getCurrentWindow()->getReader()->isThereMeshData(0,0))
	{
		for(i=1; i<=iVTKWindow::getCurrentWindow()->getReader()->getLimits(0)->getNvar(); i++)
		{
			AllVariablesList->insertItem(iVTKWindow::getCurrentWindow()->getReader()->getLimits(0)->getVarName(i));
		}
	}
}


bool iDialogVariablesExplorer::extractFunction(iVTK *win, int, int, int var, iPiecewiseFunction *f, float &vmin, vtkIdType &lmin, double *xmin, float &vmax, vtkIdType &lmax, double *xmax)
{
	if(win == 0) return false;

	iHistogramMaker *m = iHistogramMaker::New();
	iUniformMeshDataConverter *dc = iUniformMeshDataConverter::New(0);
	dc->SetInput(win->getReader()->getMeshOutput());
	dc->setCurrentVar(var+1);
	dc->Update();

	if(var+1 != dc->getCurrentVar()) return false; // wrong # of components in the input data

	m->SetInput(dc->GetOutput(),0);
	iPiecewiseFunction *h = m->getHistogram();
	if(h == 0) return false;  // god knows what happened

	f->copy(h);
	vmin = m->getMinValue();
	vmax = m->getMaxValue();

	lmin = m->getMinCellIndex();
	lmax = m->getMaxCellIndex();
	
	win->transformToExternalCoordinate(3,m->getMinCellPosition(),xmin);
	win->transformToExternalCoordinate(3,m->getMaxCellPosition(),xmax);

	m->Delete();
	dc->Delete();

	return true;

}

