/*
 * SpanDSP - a series of DSP components for telephony
 *
 * t31_tests.c - Tests for the T.31 command interpreter.
 *
 * Written by Steve Underwood <steveu@coppice.org>
 *
 * Copyright (C) 2004 Steve Underwood
 *
 * All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: t31_tests.c,v 1.27 2006/05/05 10:14:21 steveu Exp $
 */

/*! \file */

/*! \page t31_tests_page T.31 tests
\section t31_tests_page_sec_1 What does it do?
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#define _GNU_SOURCE

#include <inttypes.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include <string.h>
#include <math.h>
#include <assert.h>
#include <audiofile.h>
#include <tiffio.h>

#include "spandsp.h"
#include "spandsp/t30_fcf.h"

#define OUTPUT_FILE_NAME_T30    "t31_tests_t30.wav"
#define OUTPUT_FILE_NAME_T31    "t31_tests_t31.wav"

#define DLE 0x10
#define ETX 0x03
#define SUB 0x1A

#define MANUFACTURER            "www.soft-switch.org"

#define SAMPLES_PER_CHUNK 160

struct command_response_s
{
    const char *command;
    const char *response;
};

char *decode_test_file = NULL;
int countdown = 0;
int command_response_test_step = -1;
char response_buf[1000];
int response_buf_ptr = 0;
int answered = FALSE;
int kick = FALSE;
int dled = FALSE;

static const struct command_response_s fax_send_test_seq[] =
{
    {"ATE0\r", "ATE0\r\r\nOK\r\n"},
    {"AT+FCLASS=1\r", "\r\nOK\r\n"},
    {"ATD123456789\r", "\r\nCONNECT\r\n"},
    //<NSF frame>         AT+FRH=3 is implied when dialing in AT+FCLASS=1 state
    {"\xFF\x13\x10\x03", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nCONNECT\r\n"},
    //<CSI frame data>
    {"", "\xFF\x03\x40\x32\x32\x32\x32\x32\x32\x32\x32\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x10\x03"},
    {"", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nCONNECT\r\n"},
    //<DIS frame data>
    {"", "\xFF\x13\x80\x01\xCE\xF4\x80\x80\x81\x80\x80\x80\x18\x10\x03"},
    {"", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nNO CARRIER\r\n"},
    {"AT+FTH=3\r", "\r\nCONNECT\r\n"},
    //<TSI frame data>
    {"\xFF\x03\x43\x31\x31\x31\x31\x31\x31\x31\x31\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\xAA\x1F\x10\x03", "\r\nCONNECT\r\n"},
    //<DCS frame data>
    {"\xFF\x13\x83\x01\xC6\x80\x80\x80\x80\x01\xFD\x13\x10\x03", "\r\nOK\r\n"},
    {"AT+FTS=8;+FTM=96\r", "\r\nCONNECT\r\n"},
    //<TCF data pattern>
    {"\x00\x00\x10\x03", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nCONNECT\r\n"},
    //<CFR frame data>
    {"\xFF\x13\x84\x10\x03", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nNO CARRIER\r\n"},
    {"AT+FTM=96\r", "\r\nCONNECT\r\n"},
    //<page image data>
    {"\x00\x00\x10\x03", "\r\nOK\r\n"},
    {"AT+FTS=8;+FTH=3\r", "\r\nCONNECT\r\n"},
    //<EOP frame data>
    {"\xFF\x13\x2E\x10\x03", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nCONNECT\r\n"},
    //<MCF frame data>
    {"\xFF\x13\x8C\x10\x03", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nNO CARRIER\r\n"},
    {"AT+FTH=3\r", "\r\nCONNECT\r\n"},
    // <DCN frame data>
    {"\xFF\x13\xFB\x10\x03", "\r\nOK\r\n"},
    {"ATH0\r", "\r\nOK\r\n"}
};

static const struct command_response_s fax_receive_test_seq[] =
{
    {"ATE0\r", "ATE0\r\r\nOK\r\n"},
    {"AT+FCLASS=1\r", "\r\nOK\r\n"},
    {"", "\r\nRING\r\n"},
    {"ATA\r", "\r\nCONNECT\r\n"},
    //<CSI frame data>
    {"\xFF\x03\x40\x32\x32\x32\x32\x32\x32\x32\x32\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x10\x03", "\r\nCONNECT\r\n"},
    //<DIS frame data>
    {"\xFF\x13\x80\x01\xCE\xF4\x80\x80\x81\x80\x80\x80\x18\x10\x03", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nCONNECT\r\n"},
    //<TSI frame data>
    {"", "\xFF\x03\x43\x31\x31\x31\x31\x31\x31\x31\x31\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\x20\xAA\x1F\x10\x03"},
    {"", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nCONNECT\r\n"},
    //<DCS frame data>
    {"", "\xFF\x13\x83\x00\xC6\x80\x80\x80\x80\x01\xFD\x13\x10\x03"},
    {"", "\r\nOK\r\n"},
    //{"AT+FRH=3\r", "\r\nNO CARRIER\r\n"},
    {"AT+FRM=96\r", "\r\nCONNECT\r\n"},
    //<TCF data>
    {NULL, NULL},
    {"", "\r\nNO CARRIER\r\n"},
    {"AT+FTH=3\r", "\r\nCONNECT\r\n"},
    //<CFR frame data>
    {"\xFF\x13\x84\x10\x03", "\r\nOK\r\n"},
    {"AT+FRM=96\r", "\r\nCONNECT\r\n"},
    //<page image data>
    {NULL, NULL},
    {"", "\r\nNO CARRIER\r\n"},
    {"AT+FRH=3\r", "\r\nCONNECT\r\n"},
    //<EOP frame data>
    //{"", "\xFF\x13\x2E\x00\x00\x10\x03"},
    //<MPS frame data>
    {"", "\xFF\x13\x4F\x35\x05\x10\x03"},
    {"", "\r\nOK\r\n"},
    //{"AT+FRH=3\r", "\r\nNO CARRIER\r\n"},
    {"AT+FTH=3\r", "\r\nCONNECT\r\n"},
    //<MCF frame data>
    {"\xFF\x13\x8C\x10\x03", "\r\nOK\r\n"},
    {"AT+FRH=3\r", "\r\nNO CARRIER\r\n"},
    {"ATH0\r", "\r\nOK\r\n"}
};

int test_seq_ptr = 0;

t31_state_t t31_state;

static int at_send(t31_state_t *s, const char *t)
{
    printf("%s\n", t);
    t31_at_rx(s, t, strlen(t));
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int t31_expect(t31_state_t *s, const char *t)
{
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int at_send_hdlc(t31_state_t *s, uint8_t *t, int len)
{
    uint8_t buf[100];
    int i;
    int j;

    for (i = 0, j = 0;  i < len;  i++)
    {
        if (*t == DLE)
            buf[j++] = DLE;
        buf[j++] = *t++;
    }
    buf[j++] = DLE;
    buf[j++] = ETX;
    t31_at_rx(s, (char *) buf, j);
    return 0;
}
/*- End of function --------------------------------------------------------*/

static void phase_b_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (intptr_t) user_data;
    printf("Phase B handler on channel %d - 0x%X\n", i, result);
}
/*- End of function --------------------------------------------------------*/

static void phase_d_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    t30_stats_t t;
    char ident[21];

    i = (intptr_t) user_data;
    printf("Phase D handler on channel %d - 0x%X\n", i, result);
    t30_get_transfer_statistics(s, &t);
    printf("Phase D: bit rate %d\n", t.bit_rate);
    printf("Phase D: pages transferred %d\n", t.pages_transferred);
    printf("Phase D: image size %d x %d\n", t.width, t.length);
    printf("Phase D: image resolution %d x %d\n", t.x_resolution, t.y_resolution);
    printf("Phase D: bad rows %d\n", t.bad_rows);
    printf("Phase D: longest bad row run %d\n", t.longest_bad_row_run);
    printf("Phase D: coding method %d\n", t.encoding);
    printf("Phase D: image size %d\n", t.image_size);
    t30_get_local_ident(s, ident);
    printf("Phase D: local ident '%s'\n", ident);
    t30_get_far_ident(s, ident);
    printf("Phase D: remote ident '%s'\n", ident);
}
/*- End of function --------------------------------------------------------*/

static void phase_e_handler(t30_state_t *s, void *user_data, int result)
{
    int i;
    
    i = (intptr_t) user_data;
    printf("Phase E handler on channel %d\n", i);
    //exit(0);
}
/*- End of function --------------------------------------------------------*/

static int modem_call_control(t31_state_t *s, void *user_data, int op, const char *num)
{
    switch (op)
    {
    case AT_MODEM_CONTROL_ANSWER:
        printf("\nModem control - Answering\n");
        answered = TRUE;
        break;
    case AT_MODEM_CONTROL_CALL:
        printf("\nModem control - Dialing '%s'\n", num);
        break;
    case AT_MODEM_CONTROL_HANGUP:
        printf("\nModem control - Hanging up\n");
        break;
    case AT_MODEM_CONTROL_OFFHOOK:
        printf("\nModem control - Going off hook\n");
        break;
    case AT_MODEM_CONTROL_DTR:
        printf("\nModem control - DTR %d\n", (int) (intptr_t) num);
        break;
    case AT_MODEM_CONTROL_RTS:
        printf("\nModem control - RTS %d\n", (int) (intptr_t) num);
        break;
    case AT_MODEM_CONTROL_CTS:
        printf("\nModem control - CTS %d\n", (int) (intptr_t) num);
        break;
    case AT_MODEM_CONTROL_CAR:
        printf("\nModem control - CAR %d\n", (int) (intptr_t) num);
        break;
    case AT_MODEM_CONTROL_RNG:
        printf("\nModem control - RNG %d\n", (int) (intptr_t) num);
        break;
    case AT_MODEM_CONTROL_DSR:
        printf("\nModem control - DSR %d\n", (int) (intptr_t) num);
        break;
    default:
        printf("\nModem control - operation %d\n", op);
        break;
    }
    /*endswitch*/
    return 0;
}
/*- End of function --------------------------------------------------------*/

static int at_tx_handler(at_state_t *s, void *user_data, const uint8_t *buf, size_t len)
{
    size_t i;

    i = 0;
    if (fax_receive_test_seq[test_seq_ptr].command == NULL)
    {
        /* TCF or non-ECM image data expected */
        for (  ;  i < len;  i++)
        {
            if (dled)
            {
                if (buf[i] == 0x03)
                {
                    printf("\nFast data ended\n");
                    test_seq_ptr++;
                    response_buf_ptr = 0;
                    response_buf[response_buf_ptr] = '\0';
                    if (fax_receive_test_seq[test_seq_ptr].command)
                        kick = TRUE;
                    break;
                }
                dled = FALSE;
            }
            else
            {
                if (buf[i] == 0x10)
                    dled = TRUE;
            }
        }
        i++;
        if (i >= len)
            return 0;
    }
    for (  ;  i < len;  i++)
    {
        response_buf[response_buf_ptr++] = buf[i];
        putchar(buf[i]);
    }
    response_buf[response_buf_ptr] = '\0';
    printf("Expected ");
    for (i = 0;  i < response_buf_ptr;  i++)
        printf("%02x ", fax_receive_test_seq[test_seq_ptr].response[i] & 0xFF);
    printf("\n");
    printf("Response ");
    for (i = 0;  i < response_buf_ptr;  i++)
        printf("%02x ", response_buf[i] & 0xFF);
    printf("\n");
    if (strncmp(fax_receive_test_seq[test_seq_ptr].response, response_buf, strlen(fax_receive_test_seq[test_seq_ptr].response)) == 0)
    {
        printf("\nMatched\n");
        test_seq_ptr++;
        response_buf_ptr = 0;
        response_buf[response_buf_ptr] = '\0';
        if (fax_receive_test_seq[test_seq_ptr].command)
            kick = TRUE;
        else
            dled = FALSE;
    }
    return 0;
}
/*- End of function --------------------------------------------------------*/

int main(int argc, char *argv[])
{
    int i;
    int outframes;
    fax_state_t fax_state;
    int16_t t30_amp[SAMPLES_PER_CHUNK];
    int16_t t31_amp[SAMPLES_PER_CHUNK];
    int16_t silence[SAMPLES_PER_CHUNK];
    int t30_len;
    int t31_len;
    AFfilesetup filesetup;
    AFfilehandle t30_handle;
    AFfilehandle t31_handle;
    AFfilehandle in_handle;
    
    decode_test_file = NULL;
    for (i = 1;  i < argc;  i++)
    {
        if (strcmp(argv[i], "-d") == 0)
        {
            i++;
            decode_test_file = argv[i];
            continue;
        }
    }

    filesetup = afNewFileSetup();
    if (filesetup == AF_NULL_FILESETUP)
    {
        fprintf(stderr, "    Failed to create file setup\n");
        exit(2);
    }
    afInitSampleFormat(filesetup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, 16);
    afInitRate(filesetup, AF_DEFAULT_TRACK, (float) SAMPLE_RATE);
    afInitFileFormat(filesetup, AF_FILE_WAVE);
    afInitChannels(filesetup, AF_DEFAULT_TRACK, 1);

    memset(silence, 0, sizeof(silence));
 
    in_handle = NULL;
    if (decode_test_file)
    {
        if ((in_handle = afOpenFile(decode_test_file, "r", NULL)) == AF_NULL_FILEHANDLE)
        {
            fprintf(stderr, "    Cannot create wave file '%s'\n", decode_test_file);
            exit(2);
        }
    }
    if ((t30_handle = afOpenFile(OUTPUT_FILE_NAME_T30, "w", filesetup)) == AF_NULL_FILEHANDLE)
    {
        fprintf(stderr, "    Cannot create wave file '%s'\n", OUTPUT_FILE_NAME_T30);
        exit(2);
    }
    if ((t31_handle = afOpenFile(OUTPUT_FILE_NAME_T31, "w", filesetup)) == AF_NULL_FILEHANDLE)
    {
        fprintf(stderr, "    Cannot create wave file '%s'\n", OUTPUT_FILE_NAME_T31);
        exit(2);
    }

    fax_init(&fax_state, TRUE, NULL);
    t30_set_local_ident(&fax_state.t30_state, "11111111");
    t30_set_tx_file(&fax_state.t30_state, "itutests.tif", -1, -1);
    t30_set_phase_b_handler(&fax_state.t30_state, phase_b_handler, (void *) 0);
    t30_set_phase_d_handler(&fax_state.t30_state, phase_d_handler, (void *) 0);
    t30_set_phase_e_handler(&fax_state.t30_state, phase_e_handler, (void *) 0);
    memset(t30_amp, 0, sizeof(t30_amp));

    fax_state.t30_state.logging.level = SPAN_LOG_SHOW_SEVERITY | SPAN_LOG_SHOW_PROTOCOL | SPAN_LOG_SHOW_TAG | SPAN_LOG_FLOW;
    fax_state.t30_state.logging.tag = "XX";
    fax_state.logging.level = SPAN_LOG_SHOW_SEVERITY | SPAN_LOG_SHOW_PROTOCOL | SPAN_LOG_SHOW_TAG | SPAN_LOG_FLOW;
    fax_state.logging.tag = "XX";

    if (t31_init(&t31_state, at_tx_handler, NULL, modem_call_control, NULL) == NULL)
    {
        fprintf(stderr, "    Cannot start the FAX modem\n");
        exit(2);
    }

    t31_state.logging.level = SPAN_LOG_SHOW_SEVERITY | SPAN_LOG_SHOW_PROTOCOL | SPAN_LOG_SHOW_TAG | SPAN_LOG_FLOW;
    t31_state.logging.tag = "YY";

    countdown = 250;
    at_send(&t31_state, "ATE0\r");
    for (;;)
    {
        if (kick)
        {
            kick = FALSE;
            at_send(&t31_state, fax_receive_test_seq[test_seq_ptr].command);
        }
        t30_len = fax_tx(&fax_state, t30_amp, SAMPLES_PER_CHUNK);
        /* The receive side always expects a full block of samples, but the
           transmit side may not be sending any when it doesn't need to. We
           may need to pad with some silence. */
        if (t30_len < SAMPLES_PER_CHUNK)
        {
            memset(t30_amp + t30_len, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - t30_len));
            t30_len = SAMPLES_PER_CHUNK;
        }
        outframes = afWriteFrames(t30_handle, AF_DEFAULT_TRACK, t30_amp, t30_len);
        if (outframes != t30_len)
            break;
        if (t31_rx(&t31_state, t30_amp, t30_len))
            break;
        if (countdown)
        {
            if (answered)
            {
                countdown = 0;
                t31_call_event(&t31_state, AT_CALL_EVENT_ANSWERED);
            }
            else if (--countdown == 0)
            {
                t31_call_event(&t31_state, AT_CALL_EVENT_ALERTING);
                countdown = 250;
            }
        }

        t31_len = t31_tx(&t31_state, t31_amp, SAMPLES_PER_CHUNK);
        if (t31_len < SAMPLES_PER_CHUNK)
        {
            memset(t31_amp + t31_len, 0, sizeof(int16_t)*(SAMPLES_PER_CHUNK - t31_len));
            t31_len = SAMPLES_PER_CHUNK;
        }
        outframes = afWriteFrames(t31_handle, AF_DEFAULT_TRACK, t31_amp, t31_len);
        if (outframes != t31_len)
            break;
        if (fax_rx(&fax_state, t31_amp, SAMPLES_PER_CHUNK))
            break;
            
        usleep(10000);
    }
    if (afCloseFile(t30_handle) != 0)
    {
        fprintf(stderr, "    Cannot close wave file '%s'\n", OUTPUT_FILE_NAME_T30);
        exit(2);
    }
    if (afCloseFile(t31_handle) != 0)
    {
        fprintf(stderr, "    Cannot close wave file '%s'\n", OUTPUT_FILE_NAME_T31);
        exit(2);
    }
    if (decode_test_file)
    {
        if (afCloseFile(in_handle) != 0)
        {
            fprintf(stderr, "    Cannot close wave file '%s'\n", decode_test_file);
            exit(2);
        }
    }
    afFreeFileSetup(filesetup);
    return  0;
}
/*- End of function --------------------------------------------------------*/
/*- End of file ------------------------------------------------------------*/
